/*
 * Convert GNU a.out file to Minix i386 object format.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>
#if __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

struct my_exec {
    unsigned long   a_magic;        /* magic number */
    unsigned long   a_text;         /* size of text segment */
    unsigned long   a_data;         /* size of initialized data */
    unsigned long   a_bss;          /* size of uninitialized data */
    unsigned long   a_syms;         /* size of symbol table */
    unsigned long   a_entry;        /* entry point */
    unsigned long   a_trsize;       /* size of text relocation */
    unsigned long   a_drsize;       /* size of data relocation */
};

struct mnx_hdr {
    unsigned char   a_magic[2];
    unsigned char   a_flags;
    unsigned char   a_cpu;
    unsigned char   a_hdrlen;
    unsigned char   a_unused;
    unsigned short  a_version;
    long	    a_text;
    long	    a_data;
    long	    a_bss;
    long	    a_entry;
    long	    a_total;
    long	    a_syms;
};

char *Program;

/* Output file definitions and such */
static struct mnx_hdr mnx_h;

#define HCLICK_SIZE	  0x10		/* 8086 basic segment size */
#define PAGE_SIZE	0x1000		/* 80386 page size */
#define TOT_HDRSIZE	(sizeof(struct mnx_hdr))

FILE	*input, *output;

int	unresolved;
long	textsize ; 
long	datasize ;
long	bsssize;

#ifdef __STDC__
#define ARGS(list)	list
#else
#define ARGS(list)	()
#endif

long align	ARGS((long a, long b));
void usage	ARGS((void));
int main	ARGS((int argc, char *argv[]));
void fatal	ARGS((char *s, ...));
long get	ARGS((int intsize));
void put	ARGS((long l, int intsize));
void copy	ARGS((unsigned long n));

long align(a,b)
	long a,b;
{
	a += b - 1;
	return a - a % b;
}

void usage()
{
	fprintf(stderr, "Usage: %s <GNU binary> <Minix binary>\n", Program);
	exit(1);
}

int
main(argc, argv)
	int	argc;
	char	*argv[];
{
	long n, total;
	struct my_exec aout_hdr;
	int sep, pal, uzp, nsym;
	int i, outfd;
	long stack= 128 * 1024L;

	Program= argv[0] ;

	i= 1;
	sep = pal = uzp = 0;
	while (i < argc) {
		if (strcmp(argv[i], "-sep") == 0) sep= 1;
		else
		if (strcmp(argv[i], "-pal") == 0) pal= 1;
		else
		if (strcmp(argv[i], "-uzp") == 0) uzp= 1;
		else
		if (strcmp(argv[i], "-S") == 0) {
			char *p, *end;
			long num;

			if (++i == argc) usage();
			p= argv[i];
			if (*p == 0) usage();
			stack= strtol(p, &end, 0);
			if (end == p || stack < 0) usage();
			p= end;
			while (*p != 0) {
				switch (*p++) {
				case 'm':
				case 'M': num= 1024 * 1024L; break;
				case 'k':
				case 'K': num= 1024; break;
				case 'w':
				case 'W': num= 4; break;
				case 'b':
				case 'B': num= 1; break;
				default: usage();
				}
				if (stack > LONG_MAX / num) usage();
				stack*= num;
			}
		} else
			break;
		i++;
	}

	if ((argc - i) != 2) usage();

	if ((input = fopen(argv[i], "r")) == NULL) {
		fatal("cannot open %s\n", argv[i]);
	}

	/* scan the a.out header */
	aout_hdr.a_magic = get(4);
	aout_hdr.a_text = get(4);
	aout_hdr.a_data = get(4);
	aout_hdr.a_bss = get(4);
	aout_hdr.a_syms = get(4);
	aout_hdr.a_entry = get(4);
	aout_hdr.a_trsize = get(4);
	aout_hdr.a_drsize = get(4);

#if DEBUG
	printf("a_magic = 0x%lx\n", aout_hdr.a_magic);
	printf("a_text  = 0x%lx\n", aout_hdr.a_text);
	printf("a_data  = 0x%lx\n", aout_hdr.a_data);
	printf("a_bss   = 0x%lx\n", aout_hdr.a_bss);
	printf("a_entry = 0x%lx\n", aout_hdr.a_entry);
	printf("a_syms  = 0x%lx\n", aout_hdr.a_syms);
#endif

	/* a few sanity checks */
	if (aout_hdr.a_magic != 0x107) {
		fatal("unexpected magic 0x%lx\n", aout_hdr.a_magic);
	}
	if (aout_hdr.a_entry != (pal ? 0x20 : 0) + (uzp ? 0x1000 : 0)) {
		fatal("unexpected entry point 0x%lx\n", aout_hdr.a_entry);
	}

	nsym = (aout_hdr.a_syms != 0);

	mnx_h.a_magic[0]= 0x1;
	mnx_h.a_magic[1]= 0x3;
	mnx_h.a_flags= (uzp ? 0x01 : 0) | (pal ? 0x02 : 0)
			| (nsym ? 0x04 : 0) | (sep ? 0x20 : 0);
	mnx_h.a_cpu= 0x10;
	mnx_h.a_hdrlen= sizeof(mnx_h);
	mnx_h.a_unused= 0;
	mnx_h.a_version= 0;
	mnx_h.a_data= aout_hdr.a_data;
	mnx_h.a_bss= aout_hdr.a_bss;
	mnx_h.a_entry= aout_hdr.a_entry;

	total = mnx_h.a_data + mnx_h.a_bss + stack;

	if (sep) {	/* separate I&D */
		mnx_h.a_flags |= 0x20;
		mnx_h.a_text= align(aout_hdr.a_text, (long)HCLICK_SIZE);
	} else {
		mnx_h.a_text = aout_hdr.a_text;
		total += mnx_h.a_text;
	}

	mnx_h.a_total= total;
	mnx_h.a_syms = aout_hdr.a_syms;

#if DEBUG
	printf("       -->\n");
	printf("a_text  = 0x%lx\n", mnx_h.a_text);
	printf("a_data  = 0x%lx\n", mnx_h.a_data);
	printf("a_bss   = 0x%lx\n", mnx_h.a_bss);
	printf("a_entry = 0x%lx\n", mnx_h.a_entry);
	printf("a_total = 0x%lx\n", mnx_h.a_total);
	printf("a_syms  = 0x%lx\n", mnx_h.a_syms);
#endif

	/* Action at last */
	if ((outfd= open(argv[i+1], O_WRONLY|O_CREAT|O_TRUNC, 0777)) < 0
			|| (output = fdopen(outfd, "w")) == NULL) {
		fatal("cannot open %s\n", argv[i+1]);
	}

	fwrite((char *) &mnx_h, sizeof(char), 8, output);
	put(mnx_h.a_text, 4);
	put(mnx_h.a_data, 4);
	put(mnx_h.a_bss, 4);
	put(mnx_h.a_entry, 4);
	put(mnx_h.a_total, 4);
	put(mnx_h.a_syms, 4);
	if (ferror(output)) {
		fatal("write error on %s\n", argv[i+1]);
	}

	/* Copy the text segment. */
	copy(aout_hdr.a_text);

	/* Pad the text segment if necessary. */
	for (n = mnx_h.a_text - aout_hdr.a_text; n > 0; n--) putc(0, output);

	/* Copy the rest. */
	copy((unsigned long) -1);

	exit(0);
}

long
get(sz)
	int sz;
{
	long l = 0;
	int s= 0;

	while (sz > 0) {
		l |= getc(input) << s;
		s+= 8;
		sz--;
	}
	if (feof(input) || ferror(input)) fatal("read error\n");
	return l;
}

void
put(l, sz)
	long l;
	int sz;
{
	while (sz > 0) {
		if (putc(l & 0xFF, output) == EOF) fatal("write error\n");
		l >>= 8;
		sz--;
	}
}

void
copy(n)
	unsigned long n;
{
	int c;

	while (n > 0 && (c= getc(input)) != EOF) {
		if (putc(c, output) == EOF) fatal("write error\n");
		n--;
	}
	if (ferror(input)) fatal("read error\n");
}

/* VARARGS1 */
#if __STDC__
void
fatal(char *s, ...)
#else
void
fatal(s, va_alist)
	char	*s;
	va_dcl
#endif
{
	va_list ap;

#if __STDC__
	va_start(ap, s);
#else
	va_start(ap);
#endif
	fprintf(stderr,"%s: ", Program) ;
	vfprintf(stderr, s, ap);
	va_end(ap);
	exit(1);
}
