/*
dcore.c

Dump the contents of a corefile in human readable form 

Created 1991-May-4 by Philip Homburg
*/

#include <sys/types.h>
#include <sys/core.h>
#include <errno.h>
#include <a.out.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#if DEBUG
#define where() fprintf(stderr, "%s, %d: ", __FILE__, __LINE__)
#endif

#define CORE_NAME	"core"
#define EXEC_NAME	"a.out"

char *prog_name;
char *exec_file_name;
int core_fd;
struct core_common core_common;
struct core_386 core386;
struct exec exec_header;
struct nlist *nlist_start, *nlist_end;
int nlist_nr;

long text_offset;
u32_t vir_text_base;
u32_t text_len;
long data_offset;
u32_t vir_data_base;
u32_t data_len;
long stack_offset;
u32_t vir_stack_base;
u32_t stack_len;

void main _ARGS(( int argc, char *argv[] ));
void usage _ARGS(( void ));
void read_core_common _ARGS(( void ));
void core_86 _ARGS(( void ));
void core_386 _ARGS(( void ));
u16_t le16 _ARGS(( u16_t *addr ));
u32_t le32 _ARGS(( u32_t *addr ));
u16_t be16 _ARGS(( u16_t *addr ));
u32_t be32 _ARGS(( u32_t *addr ));
void print_386mem _ARGS(( void ));
void print_386regs _ARGS(( void ));
void print_386args _ARGS(( u32_t ebp ));
void init_file_offsets _ARGS((int hs, u32_t tv, u32_t tl,
	u32_t dv, u32_t dl, u32_t sv, u32_t sl ));
void print_386stack _ARGS(( void ));
u32_t get_stack32 _ARGS(( u32_t addr ));
u8_t get_stack8 _ARGS(( u32_t addr ));
char *print_text_sym _ARGS(( u32_t ip ));

void main(argc, argv)
int argc;
char *argv[];
{
	char *core_file_name;
	int argc_i;
	char **argv_i;

	prog_name= argv[0];
	
	core_file_name= 0;
	exec_file_name= 0;

	for (argc_i= argc-1, argv_i= argv+1; argc_i; argc_i--, argv_i++)
	{
		if (!strcmp(*argv_i, "-c"))
		{
			argc_i--;
			argv_i++;
			if (core_file_name || !argc_i)
				usage();
			core_file_name= *argv_i;
			continue;
		}
		if (!strcmp(*argv_i, "-e"))
		{
			argc_i--;
			argv_i++;
			if (exec_file_name || !argc_i)
				usage();
			exec_file_name= *argv_i;
			continue;
		}
		usage();
	}
	if (!core_file_name)
		core_file_name= CORE_NAME;

	core_fd= open(core_file_name, O_RDONLY);
	if (core_fd<0)
	{
		fprintf(stderr, "%s: unable to open '%s': %s\n", prog_name,
			core_file_name, strerror(errno));
		exit(1);
	}
	read_core_common();
	switch(core_common.core_cpu)
	{
	case CORE_CPU_I8086:
		core_86();
		break;
	case CORE_CPU_I80386:
		core_386();
		break;
	default:
		fprintf(stderr, "%s: not supported core file (cpu= %d)\n",
			prog_name, core_common.core_cpu);
		exit(1);
	}
	exit(0);
}

void usage()
{
	fprintf(stderr, "USAGE: %s [-c <core-file>] [-e <executable>]\n",
		prog_name);
	exit(2);
}

void read_core_common()
{
	int r;

	r= read(core_fd, (char *)&core_common, sizeof(core_common));
	if (r != sizeof(core_common))
	{
		fprintf(stderr, "%s: unable to read corefile: %s\n",
			prog_name, r<0 ? strerror(errno) :
			"not enough bytes read");
		exit(1);
	}
	if (core_common.core_magic[0] != CORE_MAGIC0 || 
		core_common.core_magic[1] != CORE_MAGIC1 || 
		core_common.core_magic[2] != CORE_MAGIC2 || 
		core_common.core_magic[3] != CORE_MAGIC3)
	{
		fprintf(stderr, "%s: corefile has wrong magic number\n",
			prog_name);
		exit(1);
	}
}

void core_386()
{
	int r;
	
	printf("got a corefile for 386 executable %.*s\n\n",
		CORE_NAME_SIZE, core_common.core_name);
	if (le16(&core_common.core_len) != sizeof(struct core_386))
	{
		fprintf(stderr, "%s: invalid core header size\n", prog_name);
		exit(1);
	}
	r= read(core_fd, (char *)&core386+sizeof(core_common),
		sizeof(core386)-sizeof(core_common));
	if  (r != sizeof(core386)-sizeof(core_common))
	{
		fprintf(stderr, "%s: unable to read core386 header\n",
			prog_name);
		exit(1);
	}
	nlist_nr= read_nlist(exec_file_name ? exec_file_name : EXEC_NAME,
		&nlist_start);
	if (nlist_nr == -1)
	{
		if (exec_file_name || errno != ENOENT)
			fprintf(stderr, 
			"%s: unable to read nlist of executable '%s': %s\n",
					prog_name, core_common.core_name, 
							strerror(errno));
		nlist_end= nlist_start;
	}
	else
		nlist_end= nlist_start+nlist_nr;
	
	print_386mem();
	print_386regs();
	init_file_offsets(sizeof(core386),
		le32(&core386.c_text.core_vir),
		le32(&core386.c_text.core_len),
		le32(&core386.c_data.core_vir),
		le32(&core386.c_data.core_len),
		le32(&core386.c_stack.core_vir),
		le32(&core386.c_stack.core_len));
	print_386stack();
}

void core_86()
{
	printf("got a corefile for 86/286 executable %.*s\n",
		CORE_NAME_SIZE, core_common.core_name);
}

u16_t le16(addr)
u16_t *addr;
{
	return ((u8_t *)addr)[0] | (((u8_t *)addr)[1] << 8);
}

u32_t le32(addr)
u32_t *addr;
{
	return ((u8_t *)addr)[0] | (((u8_t *)addr)[1] << 8) |
		(((u8_t *)addr)[2] << 16) | (((u8_t *)addr)[3] << 24);
}

u16_t be16(addr)
u16_t *addr;
{
	return ((u8_t *)addr)[1] | (((u8_t *)addr)[0] << 8);
}

u32_t be32(addr)
u32_t *addr;
{
	return ((u8_t *)addr)[3] | (((u8_t *)addr)[2] << 8) |
		(((u8_t *)addr)[1] << 16) | (((u8_t *)addr)[0] << 24);
}

void print_386mem()
{
	printf("\t\tphys_base\tvir_base\tlength\n");
	printf("text segm:\t0x%08lx\t0x%08lx\t0x%08lx\n",
		le32(&core386.c_text.core_phys)-le32(&core386.c_text.
		core_vir), le32(&core386.c_text.core_vir), le32(&core386.
		c_text.core_len));
	printf("data segm:\t0x%08lx\t0x%08lx\t0x%08lx\n",
		le32(&core386.c_data.core_phys)-le32(&core386.c_data.
		core_vir), le32(&core386.c_data.core_vir), le32(&core386.
		c_data.core_len));
	printf("stack segm:\t0x%08lx\t0x%08lx\t0x%08lx\n",
		le32(&core386.c_stack.core_phys)-le32(&core386.c_stack.
		core_vir), le32(&core386.c_stack.core_vir), le32(&core386.
		c_stack.core_len));
	printf("\n");
}

void print_386regs()
{
	struct core_386regs *regs;

	regs= &core386.cr;
	
	printf("    eax         ebx         ecx         edx         esi");
	printf("         edi\n");
	printf("0x%08lx  0x%08lx  0x%08lx  0x%08lx  0x%08lx  0x%08lx\n",
		le32(&regs->core_eax), le32(&regs->core_ebx),
		le32(&regs->core_ecx), le32(&regs->core_edx),
		le32(&regs->core_esi), le32(&regs->core_edi));
	printf("    ebp         esp         eip       eflags\n");
	printf("0x%08lx  0x%08lx  0x%08lx  0x%08lx\n",
		le32(&regs->core_ebp), le32(&regs->core_esp),
		le32(&regs->core_eip), le32(&regs->core_eflags));
	printf("  cs      ds      ss      es      fs      gs\n");
	printf("0x%04x  0x%04x  0x%04x  0x%04x  0x%04x  0x%04x\n",
		le16(&regs->core_cs), le16(&regs->core_ds), 
		le16(&regs->core_ss), le16(&regs->core_es), 
		le16(&regs->core_fs), le16(&regs->core_gs));
	printf("\n");
}

void print_386stack()
{
	u32_t ebp, old_ebp, eip;

	ebp= le32(&core386.cr.core_ebp);
	eip= le32(&core386.cr.core_eip);
	printf("ret addr: %s\n", print_text_sym(eip));
	while (ebp)
	{
		if (ebp < vir_stack_base || ebp >vir_stack_base+stack_len)
		{
			printf("ebp (0x%lx) not is stack space\n", ebp);
			break;
		}
		printf("ebp= 0x%08lx\n", ebp);
		if (ebp+4 < vir_stack_base+stack_len)
		{
			eip= get_stack32(ebp+4);
			eip= le32(&eip);
			printf("ret addr: %s\n", print_text_sym(eip));
		}
		old_ebp= ebp;
		ebp= get_stack32(ebp);
		ebp= le32(&ebp);
		if (ebp && ebp<old_ebp)
		{
			printf("new ebp (0x%lx) < old ebp\n", ebp);
			break;
		}
	}
	printf("\n");
	if (!ebp && old_ebp)
		print_386args(old_ebp);
}

u8_t get_stack8(addr)
u32_t addr;
{
	u8_t res;
	long file_addr;
	int r;
	
	if (addr < vir_stack_base || addr > vir_stack_base+stack_len)
	{
		fprintf(stderr, "%s: illegal request for stack data\n",
			prog_name);
		exit(1);
	}
	file_addr= addr-vir_stack_base+stack_offset;
	r= lseek(core_fd, file_addr, SEEK_SET);
	if (r != file_addr)
	{
		fprintf(stderr, "%s: unable to seek: %s\n",
			prog_name, strerror(errno));
		exit(1);
	}
	r= read(core_fd, (char *)&res, sizeof(res));
	if (r != sizeof(res))
	{
		fprintf(stderr, "%s: unable to read: %s\n",
			prog_name, strerror(errno));
		exit(1);
	}
	return res;
}

u32_t get_stack32(addr)
u32_t addr;
{
	u32_t res;
	long file_addr;
	int r;
	
	if (addr < vir_stack_base || addr > vir_stack_base+stack_len)
	{
		fprintf(stderr, "%s: illegal request for stack data\n",
			prog_name);
		exit(1);
	}
	file_addr= addr-vir_stack_base+stack_offset;
	r= lseek(core_fd, file_addr, SEEK_SET);
	if (r != file_addr)
	{
		fprintf(stderr, "%s: unable to seek: %s\n",
			prog_name, strerror(errno));
		exit(1);
	}
	r= read(core_fd, (char *)&res, sizeof(res));
	if (r != sizeof(res))
	{
		fprintf(stderr, "%s: unable to read: %s\n",
			prog_name, strerror(errno));
		exit(1);
	}
	return res;
}

void init_file_offsets(hs, tv, tl, dv, dl, sv, sl)
int hs;
u32_t tv, tl;
u32_t dv, dl;
u32_t sv, sl;
{
	text_offset= hs;
	vir_text_base= tv;
	text_len= tl;

	data_offset= text_offset+text_len;
	vir_data_base= dv;
	data_len= dl;

	stack_offset= data_offset+data_len;
	vir_stack_base= sv;
	stack_len= sl;
}

char *print_text_sym(ip)
u32_t ip;
{
	static char string[64];
	struct nlist *nlist_p, *nlist_best;
	u32_t best;

	if (!nlist_start)
	{
		sprintf(string, "0x%x", ip);
		return string;
	}
	
	best= 0;
	nlist_best= 0;
	for (nlist_p= nlist_start; nlist_p < nlist_end; nlist_p++)
	{
#if OLD_NLIST
		if ((nlist_p->n_sclass & N_SECT) != N_TEXT ||
			!nlist_p->n_name[0])
			continue;
#else
#if DEBUG
 { where(); fprintf(stderr, "name= %s, value= %d, type= %d\n", 
	nlist_p->n_un.n_name, nlist_p->n_value, nlist_p->n_type); }
#endif
		if ((nlist_p->n_type & N_TYPE) != N_TEXT ||
			nlist_p->n_un.n_name[0] == '\0')
			continue;
#endif
		if (nlist_p->n_value < ip && nlist_p->n_value >= best)
		{
			best= nlist_p->n_value;
			nlist_best= nlist_p;
		}
	}
	if (!nlist_best)				/* ? */
		sprintf(string, "0x%x", ip);
	else
#if OLD_NLIST
		sprintf(string, "%.8s+0x%x (0x%x)", nlist_best->n_name,
			ip-best, ip);
#else
		sprintf(string, "%s+0x%x (0x%x)", nlist_best->n_un.n_name,
			ip-best, ip);
#endif
	return string;
}

void print_386args(ebp)
u32_t ebp;
{
	u32_t argc, argv, avp;
	char c;

	argc= get_stack32(ebp+8);
	argc= le32(&argc);
	printf("argc= %d\n", argc);
	argv= get_stack32(ebp+12);
	argv= le32(&argv);
	while (argc)
	{
		avp= get_stack32(argv);
		avp= le32(&avp);
		while ((c= get_stack8(avp)))
		{
			putchar(c);
			avp++;
		}
		putchar(' ');
		argv += 4;
		argc--;
	}
	puts("");
}
