#!/bin/sh
#
# MAKEDEV 2.18 - Make special devices.			Author: Kees J. Bot

case $1 in
-n)	e=echo; shift ;;	# Just echo when -n is given.
*)	e=
esac

case $#:$1 in
1:std)		# Standard devices.
	set -$- mem fd0 fd1 fd0a fd1a \
		hd0 hd1a hd5 hd6a cd0 cd1a \
		sd0 sd1a sd5 sd6a sd10 sd11a sd15 sd16a sd30 sd31a \
		rsg0 rsg1 rsg2 rsg3 rsg4 rsg5 rsg6 rsg7 st4 st5 \
		dosd0 dosd5 \
		tty ttyc1 vga tty00 tty01 tty02 tty03 ttyp eth daemonlog
	;;
0:|1:-\?)
	cat >&2 <<EOF
Usage:	$0 [-n] key ...
Where key is one of the following:
	ram mem kmem null zero	# One of these makes all these memory devices
	fd0 fd1 ...		# Floppy devices for drive 0, 1, ...
	fd0a fd1a ...		# Make fd0[a-d], fd1[a-d], ...
	hd0 hd5 ...		# Make hd[0-4], hd[5-9], ...
	hd1a hd2a ... hd6a ...	# Make hd1[a-d], hd2[a-d], hd6[a-d], ...
	sd0 sd5 sd1a ...	# Make SCSI disks
	st0 st1 ...		# Make SCSI tapes rst0, nrst0, rst1 ...
	rsg0 rsg1 ...		# Make raw SCSI generic devices
	cd0 cd1a		# Make CD-ROM devices (non SCSI)
	dosd0 dosd5		# Make virtual disks on a dos file
	tty console lp		# One of these makes all three
	ttyc1 ... ttyc3		# Virtual consoles
	profile log vga kbd(aux)# Profiling, console log, VGA, keyboard (aux)
	tty00 ... tty99		# Make serial lines
	ttyp ttyq ...		# Make a series of tty, pty pairs
	eth psip ip tcp udp	# One of these makes lots of TCP/IP devices
	audio mixer		# Make audio devices
	vd0 ...			# Make virtual disk devices
	daemonlog		# A named pipe for syslogd(8)
	std			# All standard devices
EOF
	exit 1
esac

umask 077
ex=0

for dev
do
	case $dev in
	ram|mem|kmem|null|zero)
		# Memory devices.
		#
		$e mknod ram b 1 0;	$e chmod 600 ram
		$e mknod mem c 1 1;	$e chmod 640 mem
		$e mknod kmem c 1 2;	$e chmod 640 kmem
		$e mknod null c 1 3;	$e chmod 666 null
		$e mknod zero c 1 5;	$e chmod 666 zero
		$e chgrp kmem ram mem kmem null zero
		;;
	fd[0-3]|pc[0-3]|at[0-3]|qd[0-3]|ps[0-3]|pat[0-3]|qh[0-3]|PS[0-3])
		# Floppy disk drive n.
		#
		n=`expr $dev : '.*\\(.\\)'`	# Drive number.
		m=$n				# Minor device number.

		$e mknod fd$n  b 2 $m;	m=`expr $m + 4`
		$e mknod pc$n  b 2 $m;	m=`expr $m + 4`
		$e mknod at$n  b 2 $m;	m=`expr $m + 4`
		$e mknod qd$n  b 2 $m;	m=`expr $m + 4`
		$e mknod ps$n  b 2 $m;	m=`expr $m + 4`
		$e mknod pat$n b 2 $m;	m=`expr $m + 4`
		$e mknod qh$n  b 2 $m;	m=`expr $m + 4`
		$e mknod PS$n  b 2 $m;	m=`expr $m + 4`

		$e chmod 666 fd$n pc$n at$n qd$n ps$n pat$n qh$n PS$n
		;;
	fd[0-3][a-d])
		# Floppy disk partitions.
		#
		dev=`expr $dev : '\\(.*\\).'`	# Chop off the letter.
		drive=`expr $dev : '..\\(.\\)'`	# Drive number.
		n=`expr 112 + $drive`		# Partition 'a'.
		alldev=

		for par in a b c d
		do
			$e mknod $dev$par b 2 $n	# Make e.g. fd0a - fd0d
			alldev="$alldev $dev$par"
			n=`expr $n + 4`
		done
		$e chmod 666 $alldev
		;;
	[hs]d[0-9]|[hs]d[123][0-9]|dosd[0-9]|cd[0-4])
		# Hard disk drive & partitions.
		#
		case $dev in
		h*)	name=hd maj=3		# Winchester.
			;;
		c*)	name=cd maj=8		# CD-ROM.
			;;
		s*)	name=sd maj=10		# SCSI.
			;;
		d*)	name=dosd maj=12	# Virtual dos disk.
		esac
		n=`expr $dev : '[^0-9]*\\(.*\\)'`  # Minor device number.
		n=`expr $n / 5 '*' 5`		# Down to a multiple of 5.
		alldev=

		for par in 0 1 2 3 4
		do
			$e mknod $name$n b $maj $n	# Make e.g. hd5 - hd9
			alldev="$alldev $name$n"
			n=`expr $n + 1`
		done
		$e chmod 600 $alldev
		;;
	[hs]d[1-9][a-d]|[hs]d[123][1-9][a-d]|cd[1-4][a-d])
		# Hard disk subpartitions.
		#
		case $dev in
		h*)	name=hd maj=3		# Winchester.
			;;
		c*)	name=cd maj=8		# CD-ROM.
			;;
		s*)	name=sd maj=10		# SCSI.
		esac
		par=`expr $dev : '..\\(.*\\).'`	# Partition number.
		drive=`expr $par / 5`		# Drive number.
		n=`expr $drive '*' 16 + 128`	# Subpartition '1a', '6a', ...
		alldev=

		for par in 1 2 3 4
		do
			dev=$name`expr $drive '*' 5 + $par`
			for sub in a b c d
			do
				# Make e.g. hd6a, hd6b, ... hd9d
				$e mknod $dev$sub b $maj $n
				alldev="$alldev $dev$sub"
				n=`expr $n + 1`
			done
		done
		$e chmod 600 $alldev
		;;
	st[0-7]|rst[0-7]|nrst[0-7])
		# SCSI tape.
		#
		n=`expr $dev : '.*\\(.\\)'`
		m=`expr 64 + $n '*' 2`		# Minor of rstX.

		$e mknod nrst$n c 10 $m
		$e mknod rst$n c 10 `expr $m + 1`
		$e chmod 660 rst$n nrst$n
		;;
	sg[0-7]|rsg[0-7])
		# raw SCSI generic.
		#
		n=`expr $dev : '.*\\(.\\)'`
		m=`expr 120 + $n`		# Minor of rsgX.

		$e mknod rsg$n c 10 $m
		$e chmod 600 rsg$n
		;;
	tty|console|lp)
		# Anonymous tty, console, line printer.
		#
		$e mknod tty c 5 0
		$e chmod 666 tty
		$e mknod console c 4 0
		$e chmod 600 console
		$e chgrp tty console
		$e mknod lp c 6 0
		$e chown daemon lp
		$e chgrp daemon lp
		$e chmod 200 lp
		;;
	ttyc[1-3])
		# Virtual consoles.
		#
		n=`expr $dev : '....\\(.*\\)'`	# Minor device number.
		$e mknod $dev c 4 $n
		$e chgrp tty $dev
		$e chmod 600 $dev
		;;
	profile|log|vga|kbd|kbdaux)
		# Profiling, console log device, VGA screen, raw keyboard,
		# auxiliary keyboard.
		$e mknod profile c 4 16
		$e chmod 660 profile
		$e mknod log c 4 17
		$e chmod 666 log
		$e mknod vga c 4 128
		$e mknod kbd c 4 129
		$e chmod 600 vga kbd
		$e mknod kbdaux c 4 130
		$e chmod 666 kbdaux
		;;
	tty[0-9][0-9])
		# Serial lines.
		#
		n=`expr $dev : '...\\(.*\\)'`	# Minor device number.
		$e mknod $dev c 11 $n
		$e chgrp tty $dev
		$e chmod 600 $dev
		;;
	tty[p-w]|tty[p-w][0-9a-f]|pty[p-w]|pty[p-w][0-9a-f])
		# Pseudo ttys.
		#
		n=`expr $dev : '...\\(.\\)'`	# Which group.
		g=`echo $n | tr 'pqrstuvw' '01234567'`
		i=0
		alldev=

		for m in 0 1 2 3 4 5 6 7 8 9 a b c d e f
		do
			$e mknod tty$n$m c 9 `expr $g '*' 16 + $i`
			$e mknod pty$n$m c 9 `expr $g '*' 16 + $i + 128`
			i=`expr $i + 1`
			alldev="$alldev tty$n$m pty$n$m"
		done
		$e chgrp tty $alldev
		$e chmod 666 $alldev
		;;
	eth|psip|ip|tcp|udp|eth[01]|psip[01]|ip[0-3]|tcp[0-3]|udp[0-3])
		# TCP/IP devices.
		#
		$e mknod eth0 c 7 1		# Ethernet 0
		$e mknod ip0 c 7 2
		$e mknod tcp0 c 7 3
		$e mknod udp0 c 7 4
		$e mknod eth1 c 7 17		# Ethernet 1
		$e mknod ip1 c 7 18
		$e mknod tcp1 c 7 19
		$e mknod udp1 c 7 20
		$e mknod psip0 c 7 33		# Pseudo IP 0
		$e mknod ip2 c 7 34
		$e mknod tcp2 c 7 35
		$e mknod udp2 c 7 36
		$e mknod psip1 c 7 49		# Pseudo IP 1
		$e mknod ip3 c 7 50
		$e mknod tcp3 c 7 51
		$e mknod udp3 c 7 52
		$e chmod 600 eth[01] psip[01] ip[0-3]
		$e chmod 666 tcp[0-3] udp[0-3]
		$e ln -f eth0 eth		# Default interface
		$e ln -f ip0 ip
		$e ln -f tcp0 tcp
		$e ln -f udp0 udp
		;;
	audio|mixer)
		# Audio devices.
		#
		$e mknod audio c 13 0
		$e mknod mixer c 14 0
		$e chmod 666 audio mixer
		;;
	vd[0-9]|vd[1-9][0-9]|vd1[0-9][0-9]|vd2[0-4][0-9]|vd25[0-5])
		# Virtual disk devices.
		#
		n=`expr $dev : 'vd\\(.*\\)'`	# Minor of vdX.

		$e mknod $dev b 15 $n
		$e chmod 600 $dev
		;;
	daemonlog)
		# A named pipe for sending messages to syslogd.
		#
		$e mknod $dev p
		$e chmod 666 $dev
		;;
	*)
		echo "$0: don't know about $dev" >&2
		ex=1
	esac
done

exit $ex
