/*
otab.c

Created:	Aug 18, 1993 by Philip Homburg  <philip@cs.vu.nl>
*/

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/syslog.h>
#include <sys/types.h>
#include <net/hton.h>
#include <net/gen/in.h>
#include <net/gen/inet.h>
#include <net/gen/rip.h>
#include <net/gen/udp.h>
#include <net/gen/udp_hdr.h>

#include "ansi.h"
#include "interface.h"
#include "ip_misc.h"
#include "itab.h"
#include "otab.h"
#include "rip.h"

static otab_t *o_table;
static int otab_trigger;

DEFUN_VOID( void otab_init )
{
	o_table= NULL;
	otab_trigger= 0;
}

DEFUN
(int otab_add_normal, (itab_p),
	itab_t *itab_p
)
{
	otab_t *prev, *curr;
	ipaddr_t dest, subnetmask;
	int i, r;

	dest= itab_p->it_dest;
	subnetmask= itab_p->it_subnetmask;

	for (prev= NULL, curr= o_table; curr; prev= curr, curr= curr->ot_next)
	{
		if (dest > curr->ot_dest)
			continue;
		else if (dest < curr->ot_dest)
		{
			curr= NULL;
			break;
		}

		if (subnetmask > curr->ot_subnetmask)
			continue;
		else if (subnetmask < curr->ot_subnetmask)
		{
			curr= NULL;
			break;
		}
		break;
	}
	if (curr == NULL)
	{
		if ((curr= malloc(sizeof(*curr))) == NULL)
		{
			syslog(LOG_ERR, "otab_add_normal: out of memory");
			return -1;
		}
		if (prev)
		{
			curr->ot_next= prev->ot_next;
			prev->ot_next= curr;
		}
		else
		{
			curr->ot_next= NULL;
			o_table= curr;
		}
	}
	else
	{
		if (curr->ot_type != OTT_DELETED)
			return otab_update_normal(curr, itab_p);
	}

	r= ip_add_route(itab_p->it_dest, itab_p->it_subnetmask,
		itab_p->it_ifno, itab_p->it_gateway, itab_p->it_dist);
	if (r == -1)
	{
		curr->ot_type= OTT_DELETED;
		return -1;
	}
	curr->ot_dest= dest;
	curr->ot_subnetmask= subnetmask;
	curr->ot_dist= itab_p->it_dist;
	curr->ot_type= OTT_NORMAL;
	for (i= 0; i<interface_no; i++)
	{
		curr->ot_if[i].oti_flags= OTIF_TRIGGER;
		curr->ot_if[i].oti_poison_cnt= OTI_MAX_POISON;
		curr->ot_if[i].oti_ifno= itab_p->it_ifno;
		curr->ot_if[i].oti_gateway= itab_p->it_gateway;
	}
	otab_trigger= 1;
	return 0;
}

DEFUN
(int otab_add_subnet, (itab_p),
	itab_t *itab_p
)
{
	otab_t *curr;
	interface_t *ifp;
	ipaddr_t dest, subnetmask;
	int i, r;

	r= otab_add_normal(itab_p);
	if (r == -1)
		return r;

	/* Locate entry to correct the invalid flags */
	dest= itab_p->it_dest;
	subnetmask= itab_p->it_subnetmask;

	for (curr= o_table; curr; curr= curr->ot_next)
	{
		if (dest != curr->ot_dest)
			continue;

		if (subnetmask != curr->ot_subnetmask)
			continue;
		break;
	}

	assert(curr != NULL);

	/* Correct the invalid flags */
	for (i= 0, ifp= interface_table; i<interface_no; i++, ifp++)
	{
		if (((dest ^ ifp->if_addr) & ifp->if_netmask) != 0)
			curr->ot_if[i].oti_flags |= OTIF_INVALID;
	}
	return 0;
}

DEFUN
(int otab_add_agregate, (itab_p),
	itab_t *itab_p
)
{
	otab_t *curr;
	interface_t *ifp;
	ipaddr_t dest, subnetmask;
	int i, r;

	r= otab_add_normal(itab_p);
	if (r == -1)
		return r;

	/* Locate entry to correct the invalid flags, and to mark to route
	 * in the kernel as unreachable. */

	dest= itab_p->it_dest;
	subnetmask= itab_p->it_subnetmask;

	for (curr= o_table; curr; curr= curr->ot_next)
	{
		if (dest != curr->ot_dest)
			continue;

		if (subnetmask != curr->ot_subnetmask)
			continue;
		break;
	}

	assert(curr != NULL);

	/* Correct the invalid flags, and update the routing table */
	for (i= 0, ifp= interface_table; i<interface_no; i++, ifp++)
	{
		if (((dest ^ ifp->if_addr) & ifp->if_netmask) == 0)
			curr->ot_if[i].oti_flags |= OTIF_INVALID;
		ip_add_route(dest, subnetmask, curr->ot_if[i].oti_ifno,
			curr->ot_if[i].oti_gateway, RIP_INFINITY);
	}
	return 0;
}

DEFUN
(int otab_update_normal, (otab_p, itab_p),
	otab_t *otab_p AND
	itab_t *itab_p
)
{
	itab_t *i_tmp;
	ipaddr_t dest, subnetmask, gateway;
	int i, j, r, gotone;
	int ifno, dist;

	syslog(LOG_DEBUG, "otab_update_normal(...)");

	dest= itab_p->it_dest;
	subnetmask= itab_p->it_subnetmask;
	dist= itab_p->it_dist;
	ifno= itab_p->it_ifno;
	gateway= itab_p->it_gateway;

	if (otab_p == NULL)
	{
		for (otab_p= o_table; otab_p; otab_p= otab_p->ot_next)
		{
			if (dest != otab_p->ot_dest)
				continue;

			if (subnetmask != otab_p->ot_subnetmask)
				continue;
			break;
		}
		assert(otab_p != NULL);
	}
	if (dist < otab_p->ot_dist)
	{
		r= ip_add_route(dest, subnetmask, ifno, gateway, dist);
		if (r == -1)
			return r;
		for (i= 0; i<interface_no; i++)
		{
			for (j= i+1; j<interface_no; j++)
			{
				if (otab_p->ot_if[j].oti_ifno == 
					otab_p->ot_if[i].oti_ifno &&
					otab_p->ot_if[j].oti_gateway == 
					otab_p->ot_if[i].oti_gateway)
				{
					break;
				}
			}
			if (j == interface_no)
			{
				ip_del_route(otab_p->ot_dest,
					otab_p->ot_subnetmask,
					otab_p->ot_if[i].oti_ifno,
					otab_p->ot_if[i].oti_gateway);
			}
			otab_p->ot_if[i].oti_flags |= OTIF_TRIGGER;
			otab_p->ot_if[i].oti_poison_cnt= OTI_MAX_POISON;
			otab_p->ot_if[i].oti_ifno= itab_p->it_ifno;
			otab_p->ot_if[i].oti_gateway= itab_p->it_gateway;
		}
		otab_p->ot_dist= dist;
		return 0;
	}
	if (dist == otab_p->ot_dist)
	{
		if (otab_p->ot_if[ifno].oti_ifno == ifno)
			return 0;
		
		r= ip_add_route(dest, subnetmask, ifno, gateway, dist);
		if (r == -1)
			return r;

		for (j= 0; j<interface_no; j++)
		{
			if (j == ifno)
				continue;
			if (otab_p->ot_if[j].oti_ifno == 
				otab_p->ot_if[ifno].oti_ifno &&
				otab_p->ot_if[j].oti_gateway == 
				otab_p->ot_if[ifno].oti_gateway)
			{
				break;
			}
		}
		if (j == interface_no)
		{
			ip_del_route(otab_p->ot_dest,
				otab_p->ot_subnetmask,
				otab_p->ot_if[ifno].oti_ifno,
				otab_p->ot_if[ifno].oti_gateway);
		}
		otab_p->ot_if[ifno].oti_flags |= OTIF_TRIGGER;
		otab_p->ot_if[ifno].oti_poison_cnt= OTI_MAX_POISON;
		otab_p->ot_if[ifno].oti_ifno= itab_p->it_ifno;
		otab_p->ot_if[ifno].oti_gateway= itab_p->it_gateway;
		return 0;
	}

	/* Now we know that the itab entry is worse than the current route.
	 * we have three possibilities:
	 * - the itab entry is not part of the otab entry and we can ignore
	 *   the update.
	 * - the itab entry is part of the otab entry together with other 
	 *   entries which can take it's place, but we have to check the itab
	 *   to see if there is an entry at the same interface that needs to
	 *   be used now.
	 * - the itab entry is the only entry. We set the otab distance to
	 *   the new distance of the itab entry, simulate an update of all 
	 *   itab entries for this route, and than update the kernel route
	 *   if the itab entry is still in the otab.
	 */
	for (i= 0; i<interface_no; i++)
	{
		if (otab_p->ot_if[i].oti_ifno != ifno ||
			otab_p->ot_if[i].oti_gateway != gateway)
		{
			break;
		}
	}
	if (i == interface_no)
	{
		/* The third possibility. */
		otab_p->ot_dist= dist;

		for (i_tmp= i_table; i_tmp; i_tmp= i_tmp->it_next)
		{
			if (i_tmp->it_type != ITT_NORMAL && 
				i_tmp->it_type != ITT_SUBNET)
			{
				continue;
			}
			if (i_tmp->it_dest != dest || 
				i_tmp->it_subnetmask != subnetmask)
			{
				continue;
			}
			if (i_tmp->it_dist > dist)
				continue;

			r= otab_update_normal(otab_p, i_tmp);
			if (r == -1)
				i_tmp->it_type= ITT_DELETED;
		}

		/* If the current entry is still present, we update the
		 * kernel routing table.
		 */
		for (i= 0; i<interface_no; i++)
		{
			if (otab_p->ot_if[i].oti_ifno == ifno &&
				otab_p->ot_if[i].oti_gateway == gateway)
			{
				break;
			}
		}
		if (i != interface_no)
		{
			(void) ip_add_route(dest, subnetmask, ifno,
				gateway, dist);
		}
		return 0;
	}

	/* The first or the second case. */
	gotone= 0;
	for (j= 0; j<interface_no; j++)
	{
		if (otab_p->ot_if[j].oti_ifno == ifno &&
			otab_p->ot_if[j].oti_gateway == gateway)
		{
			gotone= 1;
			otab_p->ot_if[j].oti_flags |= OTIF_TRIGGER;
			otab_p->ot_if[j].oti_poison_cnt= OTI_MAX_POISON;
			otab_p->ot_if[j].oti_ifno=
				otab_p->ot_if[i].oti_ifno;
			otab_p->ot_if[j].oti_gateway=
				otab_p->ot_if[i].oti_gateway;
		}
	}
	if (!gotone)
	{
		/* First case */
		return 0;
	}
	(void) ip_del_route(dest, subnetmask, ifno, gateway);
	for (i_tmp= i_table; i_tmp; i_tmp= i_tmp->it_next)
	{
		if (i_tmp->it_type != ITT_NORMAL && 
			i_tmp->it_type != ITT_SUBNET)
		{
			continue;
		}
		if (i_tmp->it_dest != dest || 
			i_tmp->it_subnetmask != subnetmask)
		{
			continue;
		}
		if (i_tmp->it_dist > dist)
			continue;

		r= otab_update_normal(otab_p, i_tmp);
		if (r == -1)
			i_tmp->it_type= ITT_DELETED;
	}
	return 0;
}

DEFUN
(int otab_update_subnet, (otab_p, itab_p),
	otab_t *otab_p AND
	itab_t *itab_p
)
{
	syslog(LOG_ERR, "otab_update_subnet: not implemented");
	itab_print(LOG_ERR, itab_p);
	/* abort(); XXX */
	return 0;
}

DEFUN
(int otab_update_agregate, (otab_p, itab_p),
	otab_t *otab_p AND
	itab_t *itab_p
)
{
	syslog(LOG_ERR, "otab_update_agregate: not implemented");
	itab_print(LOG_ERR, itab_p);
	/* abort(); XXX */
	return 0;
}

DEFUN
(int otab_remove_normal, (itab_p),
	itab_t *itab_p
)
{
	otab_t *otab_p;
	ipaddr_t dest, subnetmask;

	dest= otab_p->ot_dest;
	subnetmask= otab_p->ot_subnetmask;

	for (otab_p= o_table; otab_p; otab_p= otab_p->ot_next)
	{
		if (dest != otab_p->ot_dest)
			continue;

		if (subnetmask != otab_p->ot_subnetmask)
			continue;
		break;
	}
	assert(otab_p != NULL);
	
	itab_p->it_dist= 2 * RIP_INFINITY;
	(void) otab_update_normal(otab_p, itab_p);
	if (otab_p->ot_dist == itab_p->it_dist)
	{
		(void) ip_del_route(dest, subnetmask, itab_p->it_ifno,
			itab_p->it_gateway);
		otab_p->ot_type= OTT_DELETED;
	}
	return 0;
}


DEFUN
(int otab_remove_subnet, (itab_p),
	itab_t *itab_p
)
{
	syslog(LOG_ERR, "otab_remove_subnet: not implemented");
	itab_print(LOG_ERR, itab_p);
	/* abort(); XXX */
	return 0;
}


DEFUN
(int otab_remove_agregate, (itab_p),
	itab_t *itab_p
)
{
	syslog(LOG_ERR, "otab_remove_agregate: not implemented");
	itab_print(LOG_ERR, itab_p);
	/* abort(); XXX */
	return 0;
}


DEFUN
(void otab_send, (trigger),
	int trigger
)
{
	otab_sendto(HTONL((ipaddr_t)-1), HTONS(RIP_UDP_PORT), trigger,
		/* no request */ 0);
}


DEFUN
(void otab_sendto, (dest_addr, dest_port, trigger, request),
	ipaddr_t dest_addr AND
	udpport_t dest_port AND
	int trigger AND
	int request
)
{
	otab_t *ent;
	pkt_hdr_t *pkt_hdr;
	udp_io_hdr_t *udp_io_hdr;
	rip_hdr_t *rip_hdr;
	rip_entry_t *rip_entry;
	char *packet;
	int entry_cnt;
	int i;

	packet= NULL;
	for (i= 0; i<interface_no; i++)
	{
		for (ent= o_table; ent; ent= ent->ot_next)
		{
			if (ent->ot_type == OTT_DELETED)
				continue;
			if (ent->ot_if[i].oti_flags & OTIF_INVALID)
				continue;
			if (trigger &&
				!(ent->ot_if[i].oti_flags & OTIF_TRIGGER))
			{
				continue;
			}
			if (ent->ot_if[i].oti_ifno == i &&
					ent->ot_if[i].oti_poison_cnt == 0)
			{
				continue;
			}
			if (packet == NULL)
			{
				packet= malloc(sizeof(pkt_hdr_t) +
					sizeof(udp_io_hdr_t) +
					sizeof(rip_hdr_t) +
					RIP_ENTRY_MAX*sizeof(rip_entry_t));
				if (packet == NULL)
				{
					syslog(LOG_ERR,
						"otab_send: out of memory");
					return;
				}
				pkt_hdr= (pkt_hdr_t *)packet;
				udp_io_hdr= (udp_io_hdr_t *)(&pkt_hdr[1]);
				rip_hdr= (rip_hdr_t *)(&udp_io_hdr[1]);
				rip_entry= (rip_entry_t *)(&rip_hdr[1]);

				pkt_hdr->ph_datasize= 0;
				pkt_hdr->ph_next= NULL;
				udp_io_hdr->uih_dst_addr= dest_addr;
				udp_io_hdr->uih_src_port= HTONS(RIP_UDP_PORT);
				udp_io_hdr->uih_dst_port= dest_port;
				udp_io_hdr->uih_ip_opt_len= 0;
				udp_io_hdr->uih_data_len= 0;
				rip_hdr->rh_command= RHC_RESPONSE;
				rip_hdr->rh_version= RH_VERSION;
				rip_hdr->rh_zero= 0;
				entry_cnt= 0;

				if (dest_addr == -1)
				{
					udp_io_hdr->uih_dst_addr=
					htonl(interface_table[i].if_addr |
					~interface_table[i].if_subnetmask);
				}
			}
			if (!request)
				ent->ot_if[i].oti_flags &= ~OTIF_TRIGGER;
			rip_entry->re_family= HTONS(RIP_FAMILY_IP);
			rip_entry->re_zero0= 0;
			rip_entry->re_address= htonl(ent->ot_dest);
			rip_entry->re_zero1= 0;
			rip_entry->re_zero2= 0;
			rip_entry->re_metric= htonl(ent->ot_dist);
			if (ent->ot_if[i].oti_ifno == i)
			{
				rip_entry->re_metric= HTONL(RIP_INFINITY);
				if (request || 
					ent->ot_if[i].oti_poison_cnt == 0)
				{
					continue;
				}
				ent->ot_if[i].oti_poison_cnt--;
			}
			rip_entry++;
			entry_cnt++;
			if (entry_cnt == RIP_ENTRY_MAX)
			{
				udp_io_hdr->uih_data_len= (char *)rip_entry-
					(char *)rip_hdr;
				pkt_hdr->ph_datasize= (char *)rip_entry-
					(char *)udp_io_hdr;
				interface_send(i, packet);
				packet= NULL;
			}
		}
		if (packet != NULL)
		{
			udp_io_hdr->uih_data_len= (char *)rip_entry-
				(char *)rip_hdr;
			pkt_hdr->ph_datasize= (char *)rip_entry-
				(char *)udp_io_hdr;
			interface_send(i, packet);
			packet= NULL;
		}
	}
}

DEFUN
(int otab_lookup, (ifno, addr),
int ifno AND
ipaddr_t addr
)
{
	otab_t *ent;

	for (ent= o_table; ent; ent= ent->ot_next)
	{
		if (ent->ot_type == OTT_DELETED)
			continue;
		if (ent->ot_dest != addr)
			continue;
		return ent->ot_dist;
	}
	return RIP_INFINITY;
}

DEFUN
(void otab_print, (logtype, ent),
	int logtype AND
	otab_t *ent
)
{
	interface_t *ifp;
	char *cp1, *cp2, *cp3;
	int i;

	struct
	{
		int oti_flags;
		int oti_poison_cnt;
		ipaddr_t oti_ifno;
		ipaddr_t oti_gateway;
	} ot_if[INTERFACE_MAX];
	struct otab *ot_next;
	cp1= strdup(inet_ntoa(htonl(ent->ot_dest)));
	cp2= strdup(inet_ntoa(htonl(ent->ot_subnetmask)));
	switch(ent->ot_type)
	{
	case OTT_DELETED:	cp3= "deleted"; break;
	case OTT_NORMAL:	cp3= "normal"; break;
	default:		cp3= "<unknown>"; break;
	}
	syslog(logtype, "routes: to %s[%s] distance %d type %s",
		cp1, cp2, ent->ot_dist, cp3);
	free(cp1);
	free(cp2);
	for (i= 0, ifp= interface_table; i<interface_no; i++, ifp++)
	{
		cp1= strdup(inet_ntoa(htonl(ent->ot_if[i].oti_gateway)));
		if (ent->ot_if[i].oti_ifno == i)
		{
			syslog(logtype,
				"\tif %d via %d[%s] poison %d flags:%s%s",
				i, ent->ot_if[i].oti_ifno, cp1,
				ent->ot_if[i].oti_poison_cnt,
				(ent->ot_if[i].oti_flags & OTIF_TRIGGER) ?
					" trigger" : "",
				(ent->ot_if[i].oti_flags & OTIF_INVALID) ?
					" invalid" : "");
		}
		else
		{
			syslog(logtype,
				"\tif %d via %d[%s] flags:%s%s",
				i, ent->ot_if[i].oti_ifno, cp1,
				(ent->ot_if[i].oti_flags & OTIF_TRIGGER) ?
					" trigger" : "",
				(ent->ot_if[i].oti_flags & OTIF_INVALID) ?
					" invalid" : "");
		}
		free(cp1);
	}
}

DEFUN
(void otab_print_all, (logtype),
	int logtype
)
{
	otab_t *otab_p;

	for (otab_p= o_table; otab_p; otab_p= otab_p->ot_next)
		otab_print(logtype, otab_p);
}
