/*
ip_misc.c

Miscellaneous ip routines

Created:	Aug 13, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/syslog.h>
#include <net/hton.h>
#include <net/gen/in.h>
#include <net/gen/route.h>
#include <net/gen/rip.h>

#include "ansi.h"
#include "interface.h"
#include "ip_misc.h"

DEFUN
(nettype_t ip_nettype, (ipaddr),
	ipaddr_t ipaddr
)
{
	u8_t highbyte;
	nettype_t nettype;

	highbyte= (ipaddr >> 24) & 0xff;
	if (highbyte == 0)
	{
		if (ipaddr == 0)
			nettype= IPNT_ZERO;
		else
			nettype= IPNT_MARSIAN;
	}
	else if (highbyte < 127)
		nettype= IPNT_CLASS_A;
	else if (highbyte == 127)
		nettype= IPNT_LOCAL;
	else if (highbyte < 192)
		nettype= IPNT_CLASS_B;
	else if (highbyte < 224)
		nettype= IPNT_CLASS_C;
	else if (highbyte < 240)
		nettype= IPNT_CLASS_D;
	else if (highbyte < 248)
		nettype= IPNT_CLASS_E;
	else if (highbyte < 255)
		nettype= IPNT_MARSIAN;
	else
	{
		if (ipaddr == (ipaddr_t)-1)
			nettype= IPNT_BROADCAST;
		else
			nettype= IPNT_MARSIAN;
	}
	return nettype;
}

DEFUN
(ipaddr_t ip_netmask, (nettype),
	nettype_t nettype
)
{
	switch(nettype)
	{
	case IPNT_ZERO:		return 0x00000000;
	case IPNT_CLASS_A:
	case IPNT_LOCAL:	return 0xff000000;
	case IPNT_CLASS_B:	return 0xffff0000;
	case IPNT_CLASS_C:	return 0xffffff00;
	default:		return 0xffffffff;
	}
}

DEFUN
(char *ip_nettoa, (nettype),
	nettype_t nettype
)
{
	switch(nettype)
	{
	case IPNT_ZERO:		return "zero";
	case IPNT_CLASS_A:	return "class A";
	case IPNT_LOCAL:	return "local";
	case IPNT_CLASS_B:	return "class B";
	case IPNT_CLASS_C:	return "class C";
	case IPNT_CLASS_D:	return "class D";
	case IPNT_CLASS_E:	return "class E";
	case IPNT_MARSIAN:	return "marsian";
	case IPNT_BROADCAST:	return "broadcast";
	default:		return "<unknown>";
	}
}

DEFUN
(int ip_add_route, (dest, subnetmask, ifno, gateway, dist),
	ipaddr_t dest AND
	ipaddr_t subnetmask AND
	int ifno AND
	ipaddr_t gateway AND
	int dist
)
{
	nwio_route_t route;
	int r;

	route.nwr_dest= htonl(dest);
	route.nwr_netmask= htonl(subnetmask);
	route.nwr_gateway= htonl(gateway);
	route.nwr_dist= dist;
	route.nwr_flags= 0;
	if (dist == RIP_INFINITY)
		route.nwr_flags |= NWRF_UNREACHABLE;
	route.nwr_pref= 0;
	r= ioctl(interface_table[ifno].if_ipfd, NWIOSIPIROUTE, &route);
	if (r == -1)
		syslog(LOG_ERR, "ip_add_route: NWIOSIPIROUTE failed: %m");
	return r;
}

DEFUN
(int ip_del_route, (dest, subnetmask, ifno, gateway),
	ipaddr_t dest AND
	ipaddr_t subnetmask AND
	int ifno AND
	ipaddr_t gateway
)
{
	nwio_route_t route;
	int r;

	route.nwr_dest= htonl(dest);
	route.nwr_netmask= htonl(subnetmask);
	route.nwr_gateway= htonl(gateway);
	route.nwr_dist= 0;
	route.nwr_flags= 0;
	route.nwr_pref= 0;
	r= ioctl(interface_table[ifno].if_ipfd, NWIODIPIROUTE, &route);
	if (r == -1)
		syslog(LOG_ERR, "ip_del_route: NWIODIPIROUTE failed: %m");
	return r;
}
