/*
interface.c

Created:	Aug 12, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/syslog.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <net/hton.h>
#include <net/gen/in.h>
#include <net/gen/inet.h>
#include <net/gen/ip_io.h>
#include <net/gen/rip.h>
#include <net/gen/udp.h>
#include <net/gen/udp_hdr.h>
#include <net/gen/udp_io.h>

#include "ansi.h"
#include "interface.h"
#include "ip_misc.h"
#include "itab.h"
#include "otab.h"
#include "rip.h"
#include "update.h"

int interface_no;
interface_t *interface_table;

static int ioctl_nonbl ARGS(( int fd, unsigned long req, void *arg ));
static void if_init ARGS(( interface_t *ifp ));
static void if_cleanup ARGS(( interface_t *ifp ));
static void process_request ARGS(( interface_t *ifp, udp_io_hdr_t *udp_hdr,
				rip_hdr_t *rip_hdr, unsigned data_len ));
static void process_response ARGS(( interface_t *ifp, udp_io_hdr_t *udp_hdr,
				rip_hdr_t *rip_hdr, unsigned data_len ));

DEFUN_VOID (void interface_init)
{
	syslog(LOG_DEBUG, "interface_init()");

	interface_no= 0;
	interface_table= NULL;
}

DEFUN
(void interface_add, (ip_dev, udp_dev),
	char *ip_dev AND
	char *udp_dev
)
{
	interface_t interface, *ifp;
	nwio_ipconf_t ipconf;
	nwio_udpopt_t udpopt;
	ipaddr_t ipaddr, netmask, subnetmask, udp_ipaddr;
	nettype_t nettype;
	char *cp;
	int i, flags;

	syslog(LOG_DEBUG, "interface_add(ip_dev= '%s', udp_dev= '%s')",
		ip_dev, udp_dev);

	if (interface_no == INTERFACE_MAX)
	{
		syslog(LOG_ERR,
			"interface_add: too manu interfaces (max = %d)",
				INTERFACE_MAX);
		return;
	}

	if_init(&interface);
	if ((interface.if_ipfd= open(ip_dev, O_RDWR)) == -1)
	{
		syslog(LOG_ERR, "interface_add: unable to open '%s': %m",
			ip_dev);
		if_cleanup(&interface);
		return;
	}
	if (ioctl_nonbl(interface.if_ipfd, NWIOGIPCONF, &ipconf) == -1)
	{
		if (errno == EAGAIN)
		{
			syslog(LOG_ERR,
			"interface_add: ip device '%s' is not initialized",
				ip_dev);
		}
		else
		{
			syslog(LOG_ERR,
			"interface_add: NWIOGIPCONF failed on '%s': %m",
				ip_dev);
		}
		if_cleanup(&interface);
		return;
	}
	if ((interface.if_udpfd= open(udp_dev, O_RDWR)) == -1)
	{
		syslog(LOG_ERR, "interface_add: unable to open '%s': %m",
			udp_dev);
		if_cleanup(&interface);
		return;
	}
	if (ioctl_nonbl(interface.if_udpfd, NWIOGUDPOPT, &udpopt) == -1)
	{
		if (errno == EAGAIN)
		{
			syslog(LOG_ERR,
			"interface_add: udp device '%s' is not initialized",
				udp_dev);
		}
		else
		{
			syslog(LOG_ERR,
			"interface_add: NWIOGUDPOPT failed on '%s': %m",
				udp_dev);
		}
		if_cleanup(&interface);
		return;
	}
	ipaddr= ntohl(ipconf.nwic_ipaddr);
	interface.if_addr= ipaddr;
	nettype= ip_nettype(ipaddr);
	if (nettype != IPNT_CLASS_A && nettype != IPNT_CLASS_B && 
		nettype != IPNT_CLASS_C)
	{
		syslog(LOG_ERR,
		"interface_add: wrong ip address for device '%s': %s",
				ip_dev, inet_ntoa(ipaddr));
		if_cleanup(&interface);
		return;
	}
	subnetmask= ntohl(ipconf.nwic_netmask);
	interface.if_subnetmask= subnetmask;
	netmask= ip_netmask(nettype);
	interface.if_netmask= netmask;
	syslog(LOG_DEBUG, "interface_add: ipaddr= %s",
		inet_ntoa(htonl(ipaddr)));
	syslog(LOG_DEBUG, "interface_add: nettype= %s", ip_nettoa(nettype));
	syslog(LOG_DEBUG, "interface_add: netmask= %s", 
						inet_ntoa(htonl(netmask)));
	syslog(LOG_DEBUG, "interface_add: subnetmask= %s", 
						inet_ntoa(htonl(subnetmask)));
	if (ntohl(subnetmask) < ntohl(netmask))
	{
		syslog(LOG_ERR, 
			"interface_add: wrong netmask for device '%s': %s",
				ip_dev, inet_ntoa(htonl(subnetmask)));
		if_cleanup(&interface);
		return;
	}
	udp_ipaddr= ntohl(udpopt.nwuo_locaddr);
	if (udp_ipaddr != ipaddr)
	{
		cp= strdup(inet_ntoa(htonl(ipaddr)));
		syslog(LOG_ERR,
	"interface_add: wrong udp device '%s' for ip device '%s': %s != %s",
			udp_dev, ip_dev, cp, inet_ntoa(htonl(udp_ipaddr)));
		free(cp);
		if_cleanup(&interface);
		return;
	}
	for (i= 0, ifp= interface_table; i<interface_no; i++, ifp++)
	{
		if (((ifp->if_addr ^ ipaddr) & netmask) != 0)
		{
			/* Different network */
			continue;
		}
		if (ifp->if_subnetmask != subnetmask)
		{
			cp= strdup(inet_ntoa(htonl(subnetmask)));
			syslog(LOG_ERR,
"interface_add: subnetmask %s inconsistent with subnetmask %s for if %d",
				cp, inet_ntoa(htonl(ifp->if_subnetmask)), i);
			free(cp);
			if_cleanup(&interface);
			return;
		}
		if (ifp->if_addr == ipaddr)
		{
			syslog(LOG_ERR,
		"interface_add: ip address %s is also used in interface %d",
					inet_ntoa(htonl(ipaddr)), i);
			if_cleanup(&interface);
			return;
		}
		if (((ifp->if_addr ^ ipaddr) & subnetmask) == 0)
		{
			cp= strdup(inet_ntoa(htonl(ifp->if_addr)));
			syslog(LOG_INFO,
"interface_add: strange, two interfaces on the same subnet: %s and %s",
					cp, inet_ntoa(htonl(ipaddr)));
			free(cp);
		}
	}

	/* Configure udp fd */
	udpopt.nwuo_flags= NWUO_COPY | NWUO_LP_SET | NWUO_EN_LOC |
		NWUO_EN_BROAD | NWUO_RP_ANY | NWUO_RA_ANY | NWUO_RWDATALL |
		NWUO_DI_IPOPT;
	udpopt.nwuo_locport= HTONS(RIP_UDP_PORT);
	if (ioctl(interface.if_udpfd, NWIOSUDPOPT, &udpopt) == -1)
	{
		syslog(LOG_ERR,
			"interface_add: NWIOSUDPOPT failed on '%s': %m",
				udp_dev);
		if_cleanup(&interface);
		return;
	}

	/* turn on async I/O on the udp fd */
	if ((flags= fcntl(interface.if_udpfd, F_GETFD)) == -1)
	{
		syslog(LOG_ERR, "interface_add: F_GETFD failed on '%s': %m",
			udp_dev);
		if_cleanup(&interface);
		return;
	}
	flags |= FD_ASYNCHIO;
	if (fcntl(interface.if_udpfd, F_SETFD, flags) == -1)
	{
		syslog(LOG_ERR,
			"interface_add: F_SETFD 0x%x failed on '%s': %m",
			flags, udp_dev);
		if_cleanup(&interface);
		return;
	}

	interface_no++;
	ifp= realloc(interface_table, interface_no*sizeof(interface_t));
	if (ifp == NULL)
	{
		interface_no--;
		syslog(LOG_ERR, "interface_add: out of memory");
		if_cleanup(&interface);
		return;
	}
	interface_table= ifp;
	ifp[interface_no-1]= interface;
}

DEFUN
(static int ioctl_nonbl, (fd, req, arg),
	int fd AND
	unsigned long req AND
	void *arg
)
{
	int flags;
	int r, saved_errno;
	asio_fd_set_t fd_set;
	fwait_t fw;

	/* turn on async I/O */
	if ((flags= fcntl(fd, F_GETFD)) == -1)
		return -1;
	if (fcntl(fd, F_SETFD, flags | FD_ASYNCHIO) == -1)
		return -1;
	r= ioctl(fd, req, arg);
	saved_errno= errno;
	fcntl(fd, F_SETFD, flags);
	errno= saved_errno;
	if (r != -1 || errno != EINPROGRESS)
		return r;
	if (fcancel(fd, ASIO_IOCTL) == -1)
		return -1;
	ASIO_FD_ZERO(&fd_set);
	ASIO_FD_SET(fd, ASIO_IOCTL, &fd_set);
	fw.fw_flags= FWF_NONBLOCK;
	fw.fw_bits= fd_set.afds_bits;
	fw.fw_maxfd= ASIO_FD_SETSIZE;
	if (fwait(&fw) == -1)
		return -1;
	if (fw.fw_result == -1)
	{
		errno= fw.fw_errno;
		if (errno == EINTR)
			errno= EAGAIN;
		return -1;
	}
	return fw.fw_result;
}

DEFUN
(static void if_init, (ifp),
	interface_t *ifp
)
{
	if (ifp == NULL)
		return;
	ifp->if_addr= 0;
	ifp->if_subnetmask= 0;
	ifp->if_ipfd= -1;
	ifp->if_udpfd= -1;
	ifp->if_inprogress= 0;
}

DEFUN
(static void if_cleanup, (ifp),
	interface_t *ifp
)
{
	if (ifp == NULL)
		return;
	if (ifp->if_ipfd != -1)
	{
		if (close(ifp->if_ipfd) == -1)
		{
			syslog(LOG_WARNING,
				"if_cleanup: unable to close ipfd %d: %m",
					ifp->if_ipfd);
		}
	}
	if (ifp->if_udpfd != -1)
	{
		if (close(ifp->if_udpfd) == -1)
		{
			syslog(LOG_WARNING,
				"if_cleanup: unable to close udpfd %d: %m",
					ifp->if_udpfd);
		}
	}
	if_init(ifp);
}

DEFUN
(void interface_send, (ifno, packet),
	int ifno AND
	char *packet
)
{
	pkt_hdr_t *pkt_hdr;
	char *data;
	size_t size;
	int r;

	pkt_hdr= (pkt_hdr_t *)packet;
	data= (char *)(&pkt_hdr[1]);
	size= pkt_hdr->ph_datasize;

	r= write(interface_table[ifno].if_udpfd, data, size);
	if (r == -1)
		syslog(LOG_ERR, "interface_send: write failed: %m");
	else if (r != size)
	{
		syslog(LOG_ERR,
			"interface_send: write failed: %d (expected %d)",
				r, size);
	}
	free(packet);
}

DEFUN
(void interface_read, (fdset_p),
	asio_fd_set_t *fdset_p
)
{
	int i, r;
	interface_t *ifp;

	ASIO_FD_ZERO(fdset_p);

	for (i= 0, ifp= interface_table; i< interface_no; i++, ifp++)
	{
		if (ifp->if_inprogress)
		{
			ASIO_FD_SET(ifp->if_udpfd, ASIO_READ, fdset_p);
			continue;
		}
		for (;;)
		{
			r= read(ifp->if_udpfd, ifp->if_buffer, IF_BUF_SIZE);
			if (r == -1 && errno == EINPROGRESS)
			{
				ifp->if_inprogress= 1;
				ASIO_FD_SET(ifp->if_udpfd, ASIO_READ, fdset_p);
				break;
			}
			if (r == -1 || r == 0)
			{
				syslog(LOG_ERR,
		"interface_read: read failed for interface %d, fd %d: %s",
					i, ifp->if_udpfd, (r == 0) ? "eof" :
					strerror(errno));
				break;
			}
			interface_arrived(ifp->if_udpfd, ASIO_READ, r, errno);
		}
	}
}

DEFUN
(void interface_arrived, (fd, operation, result, err),
	int fd AND
	int operation AND
	int result AND
	int err
)
{
	interface_t *ifp;
	udp_io_hdr_t *udp_hdr;
	rip_hdr_t *rip_hdr;
	unsigned ip_opt_len, data_len;
	int i;

	syslog(LOG_DEBUG, "interface_arrived(%d, %d, %d, %d)",
		fd, operation, result, err);

	/* Find the interface, the fd belongs to */
	for (i= 0, ifp= interface_table; i<interface_no; i++, ifp++)
	{
		if (ifp->if_udpfd == fd)
			break;
	}
	if (i == interface_no)
	{
		syslog(LOG_ERR, "interface_arrived: no interface for fd %d",
			fd);
		return;
	}
	if (operation != ASIO_READ)
	{
		syslog(LOG_ERR, "interface_arrived: strange operation %d",
			operation);
		return;
	}
	ifp->if_inprogress= 0;
	if (result == -1)
	{
		syslog(LOG_ERR, "interface_arrived: read failed: %s",
			strerror(err));
		return;
	}
	if (result < sizeof(*udp_hdr))
	{
		syslog(LOG_ERR, "interface_arrived: read too small: %d bytes",
			result);
		return;
	}
	udp_hdr= (udp_io_hdr_t *)(ifp->if_buffer);
	ip_opt_len= udp_hdr->uih_ip_opt_len;
	data_len= udp_hdr->uih_data_len;
	if (udp_hdr->uih_dst_port != HTONS(RIP_UDP_PORT))
	{
		syslog(LOG_ERR,
			"interface_arrived: packet for wrong port: %u",
			ntohs(udp_hdr->uih_dst_port));
		return;
	}
	if (result < sizeof(*udp_hdr) + ip_opt_len + data_len)
	{
		syslog(LOG_WARNING,
		"interface_arrived: truncated packet, needed %d, got %d",
			sizeof(*udp_hdr) + ip_opt_len + data_len, result);
		return;
	}
	if (result != sizeof(*udp_hdr) + ip_opt_len + data_len)
	{
		syslog(LOG_ERR,
		"interface_arrived: strange packet, expected %d, got %d",
			sizeof(*udp_hdr) + ip_opt_len + data_len, result);
		return;
	}
	if (data_len < sizeof(*rip_hdr))
	{
		syslog(LOG_WARNING,
		"interface_arrived: not enough udp data for rip header: %d",
			data_len);
		return;
	}
	rip_hdr= (rip_hdr_t *)((char *)&udp_hdr[1] + ip_opt_len);
	data_len -= sizeof(*rip_hdr);

	switch (rip_hdr->rh_command)
	{
	case RHC_REQUEST:
		process_request(ifp, udp_hdr, rip_hdr, data_len);
		break;
	case RHC_RESPONSE:
		process_response(ifp, udp_hdr, rip_hdr, data_len);
		break;
	default:
		syslog(LOG_WARNING,
			"interface_arrived: strange rip command: %d",
			rip_hdr->rh_command);
		break;
	}
}

DEFUN
(static void process_request, (ifp, udp_hdr, rip_hdr, data_len),
	interface_t *ifp AND
	udp_io_hdr_t *udp_hdr AND
	rip_hdr_t *rip_hdr AND
	unsigned data_len
)
{
	rip_entry_t *ent;
	update_t *update;
	ipaddr_t ipaddr, subnetmask, gateway;
	int ifno, metric, ent_no;
	int i, j;

	ipaddr_t src_addr;
	udpport_t src_port;

	ifno= ifp-interface_table;

	src_port= udp_hdr->uih_src_port;
	src_addr= udp_hdr->uih_src_addr;
	udp_hdr->uih_dst_port= udp_hdr->uih_src_port;
	udp_hdr->uih_dst_addr= udp_hdr->uih_src_addr;
	ent_no= data_len / sizeof(*ent);
	if (ent_no == 0)
		return;
	for (i= 0, ent= (rip_entry_t *)&rip_hdr[1]; i < ent_no; i++, ent++)
	{
		if (i == 0 && ent_no == 1 && ent->re_family == 0
			&& ent->re_metric == htonl(RIP_INFINITY))
		{
			otab_sendto(src_addr, src_port, 
				/* no trigger */ 0,
				/* request */ 1);
			return;
		}

		if (ent->re_family != HTONS(RIP_FAMILY_IP))
		{
			syslog(LOG_NOTICE, "process_reponse: got family %d",
				ntohs(ent->re_family));
			continue;
		}
		if (rip_hdr->rh_version == RH_VERSION &&
			(ent->re_zero0 != 0 || ent->re_zero1 != 0 ||
			ent->re_zero2 != 0))
		{
			syslog(LOG_WARNING,
			"process_reponse: zero fields not zero");
			continue;
		}
		metric= otab_lookup(ifno, ntohl(ent->re_address));
		ent->re_metric= htonl(metric);
	}
	interface_send(ifno, (char *)udp_hdr);
}

DEFUN
(static void process_response, (ifp, udp_hdr, rip_hdr, data_len),
	interface_t *ifp AND
	udp_io_hdr_t *udp_hdr AND
	rip_hdr_t *rip_hdr AND
	unsigned data_len
)
{
	rip_entry_t *ent;
	update_t *update;
	ipaddr_t ipaddr, subnetmask, gateway;
	int ifno, metric, ent_no;
	int i, j;

	ifno= ifp-interface_table;

	/* src port should be equal to RIP_UDP_PORT, and the src
	 * address should on our network. Packets send by us are ignored.
	 */
	if (udp_hdr->uih_src_port != HTONS(RIP_UDP_PORT))
	{
		syslog(LOG_WARNING,
			"process_response: packet from wrong port: %u",
			ntohs(udp_hdr->uih_src_port));
		return;
	}
	gateway= ntohl(udp_hdr->uih_src_addr);
	if (gateway == ifp->if_addr)
		return;
	if (((gateway ^ ifp->if_addr) & ifp->if_subnetmask) != 0)
	{
		syslog(LOG_WARNING,
			"process_response: if %d: packet from wrong host: %s",
			ifno, inet_ntoa(gateway));
		return;
	}
	ent_no= data_len / sizeof(*ent);
	if (ent_no == 0)
		return;
	update= malloc(ent_no * sizeof(*update));
	if (update == NULL)
	{
		syslog(LOG_ERR, "process_response: out of memory");
		return;
	}
	j= 0;
	for (i= 0, ent= (rip_entry_t *)&rip_hdr[1]; i < ent_no; i++, ent++)
	{
		if (ent->re_family != HTONS(RIP_FAMILY_IP))
		{
			syslog(LOG_NOTICE, "process_reponse: got family %d",
				ntohs(ent->re_family));
			continue;
		}
		if (rip_hdr->rh_version == RH_VERSION &&
			(ent->re_zero0 != 0 || ent->re_zero1 != 0 ||
			ent->re_zero2 != 0))
		{
			syslog(LOG_WARNING,
			"process_reponse: zero fields not zero");
			continue;
		}
		metric= ntohl(ent->re_metric);
		if (metric > 0)
			metric += rip_ifdist;
		if (metric > RIP_INFINITY &&
				ntohl(ent->re_metric) <= RIP_INFINITY)
		{
			metric= RIP_INFINITY;
		}
		ipaddr= ntohl(ent->re_address);
		update[j].u_dest= ipaddr;
		if (((ipaddr ^ ifp->if_addr) & ifp->if_netmask) == 0)
			subnetmask= ifp->if_subnetmask;
		else
			subnetmask= ip_netmask(ip_nettype(ipaddr));
		if ((ipaddr & ~subnetmask) != 0)
			subnetmask= (ipaddr_t)-1;
		update[j].u_subnetmask= subnetmask;
		update[j].u_ifno= ifno;
		update[j].u_gateway= gateway;
		update[j].u_dist= metric;
		j++;
	}
	if (j != 0)
	{
		qsort(update, j, sizeof(update_t), update_compare);
		itab_update(update, j);
	}
	free(update);
}
