/*
profile.c

Write a profile of the current kernel activity to stdout.

Created:	Aug 12, 1992 by Philip Homburg
*/

#define _POSIX_C_SOURCE	2
#include <sys/types.h>
#include <sys/ioctl.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#define NAMELIST_CMD "nm -n %s | sed -n '/ [tT] /s/ [tT] / /p'"
#define PROFILE_DEV "/dev/profile"

typedef struct node
{
	unsigned long n_value;
	char *n_string;
	struct node **n_nodelist;
	int n_nextind;
	int n_indhi;
	unsigned long n_cum;
	unsigned long n_self;
	int n_printed;
} node_t;

node_t *kernel_nl, *mm_nl, *fs_nl;

#define NEW_LISTNODES	32
#define NEW_NODES	16

node_t *new_nodes;
int node_nr;

typedef struct task
{
	char *t_name;
	node_t *t_root;
	node_t *t_namelist;
} task_t;

#define TASK_MIN	(-64)
#define TASK_MAX	64

task_t task_table[TASK_MAX-TASK_MIN+3];
task_t *task_base;
int merge;

char *prog_name;

void main _ARGS(( int argc, char *argv[] ));
static node_t *read_namelist _ARGS(( char *exec_name, unsigned long min_val, 
					unsigned long max_val ));
static node_t *make_node _ARGS(( unsigned long value, char *string ));
static void add_node _ARGS(( node_t *node_list, node_t *node ));
static void build_tree _ARGS(( void ));
static char *process_trace _ARGS(( char *ptr ));
static void build_tasks _ARGS(( node_t *kernel_nl ));
static node_t *find_node _ARGS(( node_t *namelist, unsigned long pc ));
static void print_tree _ARGS(( void ));
static void print_node _ARGS(( int level, node_t *node ));
static void usage _ARGS(( void ));

void main(argc, argv)
int argc;
char *argv[];
{
	int c;
	int m_opt;
	
	prog_name= argv[0];

	m_opt= 0;
	while((c= getopt(argc, argv, "?m")) != -1)
	{
		switch(c)
		{
		case '?':
			usage();
		case 'm':
			m_opt= 1;
			break;
		}
	}
	if (optind != argc && optind != argc - 3)
		usage();
	merge= m_opt;	
	
	kernel_nl= read_namelist(optind == argc ? "kernel" : argv[optind],
								0L, 0x20000L);
	mm_nl= read_namelist(optind == argc ? "mm" : argv[optind+1],
								0L, 0x20000L);
	fs_nl= read_namelist(optind == argc ? "fs" : argv[optind+2],
								0L, 0x20000L);
	build_tasks(kernel_nl);
	task_base[0].t_namelist= mm_nl;
	task_base[1].t_namelist= fs_nl;
	build_tree();
	print_tree();
	exit(0);
}

static node_t *read_namelist(exec_name, min_val, max_val)
char *exec_name;
unsigned long min_val;
unsigned long max_val;
{
	FILE *f;
	char *cmd, *s;
	char name[1024];
	unsigned long value, prev;
	int r;
	node_t *n, *sym_tab;
	int firsttime;
	
	cmd= malloc(strlen(exec_name) + sizeof(NAMELIST_CMD));
	if (cmd == NULL)
	{
		fprintf(stderr, "%s: malloc failed\n", prog_name);
		exit(1);
	}
	sprintf(cmd, NAMELIST_CMD, exec_name);
	f= popen(cmd, "r");
	if (f == NULL)
	{
		fprintf(stderr, "%s: popen failed: %s", prog_name,
							strerror(errno));
		exit(1);
	}
	
	sym_tab= make_node(0, NULL);
	prev= min_val;
	for(firsttime= 1;;firsttime= 0)
	{
		r= fscanf(f, "%lx %1024s\n", &value, name);
		if (r != 2)
			break;
		if (firsttime && value > min_val)
		{
			n= make_node(min_val, "wrong pc (too low)");
			add_node(sym_tab, n);
			prev= min_val;
		}
		if (value < prev)
		{
			fprintf(stderr, "%s: symbol table not sorted\n",
								 prog_name);
			exit(1);
		}
		prev= value;
		s= strdup(name);
		if (s == NULL)
		{
			fprintf(stderr, "%s: strdup failed\n", prog_name);
			exit(1);
		}
		n= make_node(value, s);
		add_node(sym_tab, n);
	}
	if (r == EOF && feof(f))
	{
		if (firsttime)
		{
			fprintf(stderr, "%s: empty name list\n", prog_name);
			exit(1);
		}
		if (prev >= max_val)
		{
			fprintf(stderr, 
			"%s: symbol table doesn't match text segment size\n",
					prog_name);
#if 0
			exit(1);
#endif
		}
		n= make_node(max_val, "wrong pc (too high)");
		add_node(sym_tab, n);
		r= pclose(f);
		if (r == 0)
			return sym_tab;
		if (r == -1)
		{
			fprintf(stderr, "%s: pclose failed: %s\n", prog_name,
				strerror(errno));
			exit(1);
		}
		fprintf(stderr, "%s: nm failed, exit code %d\n", prog_name, r);
		exit(1);
	}
	if (ferror(f))
	{
		fprintf(stderr, "%s: error while reading symbol table: %s\n",
			prog_name, strerror(ferror(f)));
		exit(1);
	}
	fprintf(stderr, "%s: input from nm garbled\n", prog_name);
	exit(1);
	/* NOTREACHED */
}


static node_t *make_node (value, string)
unsigned long value;
char *string;
{
	node_t *n;
	
	if (new_nodes == NULL)
	{
		new_nodes= malloc(NEW_NODES * sizeof(*new_nodes));
		if (new_nodes == NULL)
		{
			fprintf(stderr, "%s: malloc failed\n", prog_name);
			exit(1);
		}
		node_nr= 0;
	}
	n= &new_nodes[node_nr++];
	if (node_nr == NEW_NODES)
		new_nodes= NULL;
	n->n_value= value;
	n->n_string= string;
	n->n_nodelist= NULL;
	n->n_nextind= 0;
	n->n_indhi= 0;
	n->n_cum= 0;
	n->n_self= 0;
	n->n_printed= 0;
	return n;
}


static void add_node(node_list, node)
node_t *node_list;
node_t *node;
{
	node_t **new_list;
	
	if (node_list->n_nextind == node_list->n_indhi)
	{
		node_list->n_indhi += NEW_LISTNODES;
		new_list= realloc(node_list->n_nodelist,
			node_list->n_indhi * sizeof(*node_list->n_nodelist));
		if (new_list == NULL)
		{
			fprintf(stderr, "%s: realloc failed\n", prog_name);
			exit(1);
		}
		node_list->n_nodelist= new_list;
	}
	node_list->n_nodelist[node_list->n_nextind++]= node;
}

static void build_tree()
{
	int fd, r;
	char buffer[10240];
	char *ptr, *end;
	int i;

	fd= open(PROFILE_DEV, O_RDWR);
	if (fd == -1)
	{
		fprintf(stderr, "%s: unable to open '%s'\n", prog_name,
			PROFILE_DEV);
		exit(1);
	}
	r= ioctl(fd, PIOCSTART, NULL);
	if (r == -1)
	{
		fprintf(stderr, "%s: unable to start profiling: %s\n",
			prog_name, strerror(errno));
		exit(1);
	}
	for(i= 0;; i++)
	{
		r= read(fd, buffer, sizeof(buffer));
		if (r <= 0)
			break;
		for (ptr= buffer, end= buffer+r; ptr < end; )
		{
			ptr= process_trace(ptr);
			assert(ptr <= end);
		}
	}
	if (r == 0 || i>0)
		return;
	fprintf(stderr, "%s: unable to read traces: %s\n", prog_name, 
		strerror(errno));
	exit(1);
}

static char *process_trace(ptr)
char *ptr;
{
	u32_t *desc, *pc_ptr, pc;
	int task, n;
	task_t *t;
	node_t *node, *namelist, *pc_node, **nl;
	int nls, i;

	desc= (u32_t *)ptr;
	task= (*(i32_t *)desc) >> 16;
	if (task < TASK_MIN)
		task= TASK_MIN-1;
	else if (task > TASK_MAX)
		task= TASK_MAX+1;
	t= &task_base[task];
	node= t->t_root;
	if (node == NULL)
	{
		node= make_node(0L, "");
		t->t_root= node;
	}
	n= (*desc) & 0xffff;
	node->n_cum++;
	namelist= t->t_namelist;
	if (namelist != NULL)
	{
		pc_ptr= &desc[n-1];
		for (; pc_ptr > desc; pc_ptr--)
		{
			pc= *pc_ptr;
			pc_node= find_node(namelist, pc);
			nl= node->n_nodelist;
			nls= node->n_nextind;
			if (merge)
			{
				for (i= 0; i< nls; i++, nl++)
				{
					if (*nl == pc_node)
						break;
				}
				if (i == nls)
					add_node(node, pc_node);
			}
			else
			{
				pc= pc_node->n_value;
				for (i= 0; i< nls; i++, nl++)
				{
					if ((*nl)->n_value == pc)
						break;
				}
				if (i == nls)
				{
					pc_node= make_node(pc_node->n_value,
						pc_node->n_string);
					add_node(node, pc_node);
				}
				else
					pc_node= *nl;
			}
			node= pc_node;
			node->n_cum++;
		}
		node->n_self++;	
	}
	return (char *)&desc[n];
}

static void build_tasks (kernel_nl)
node_t *kernel_nl;
{
	task_t *t;
	int i;
	
	task_base= &task_table[-TASK_MIN+1];	/* Shift the index. */
	
	for (i= TASK_MIN-1, t= &task_base[i]; i<=TASK_MAX+1; i++, t++)
	{
		t->t_name= NULL;
		t->t_root= NULL;
		t->t_namelist= NULL;
	}
	task_base[TASK_MIN-1].t_name= "task number too low";
	for (i= TASK_MIN, t= &task_base[i]; i<0; i++, t++)
	{
		t->t_namelist= kernel_nl;
	}
	task_base[TASK_MAX+1].t_name= "task number too high";
}

static node_t *find_node(namelist, pc)
node_t *namelist;
unsigned long pc;
{
	node_t **nl;
	int nli;

	nli= namelist->n_nextind-1;
	for (nl= &namelist->n_nodelist[nli]; nli >= 0; nli--, nl--)
	{
		if (pc >= (*nl)->n_value)
			return *nl;
	}
	assert(0);
	/* NOTREACHED */
}

static void print_tree()
{
	int i;
	task_t *t;
	node_t *node, **nl;
	int nls;
	
	for (i= TASK_MIN-1, t= &task_base[i]; i <= TASK_MAX+1; i++, t++)
	{
		node= t->t_root;
		if (node == NULL)
			continue;
		if (t->t_name)
			printf("%s: ", t->t_name);
		else
			printf("task %d: ", i);
		printf("%d\n", node->n_cum);
		nl= node->n_nodelist;
		nls= node->n_nextind;
		for (; nls>0; nls--,nl++)
		{
			print_node(1, *nl);
		}
	}
}

static void print_node(level, node)
int level;
node_t *node;
{
	int i, nls;
	node_t **nl;
	
	for (i= 0; i<level; i++)
		printf("  *  ");
	printf("%s %d(%d)%s\n", node->n_string, node->n_cum, node->n_self,
		node->n_printed ? "*" : "");
	if (node->n_printed)
	{
		assert(merge);
		return;
	}
	node->n_printed= 1;
	nl= node->n_nodelist;
	nls= node->n_nextind;
	for (; nls>0; nls--, nl++)
		print_node(level+1, *nl);
}

static void usage()
{
	fprintf(stderr, "USAGE: %s [-m] [kernel mm fs]\n", prog_name);
	exit(1);
}
