/*
sm.c

State machine for lcp and ipcp

Created:	May 28, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#include <sys/types.h>
#include <assert.h>
#include <stdlib.h>
#include <time.h>

#include "ansi.h"
#include "clock.h"
#include "debug.h"
#include "sm.h"

#define SM_EV_UP		 1
#define SM_EV_DOWN		 2
#define SM_EV_OPEN		 3
#define SM_EV_CLOSE		 4
#define SM_EV_TO_PLUS		 5
#define SM_EV_TO_MINUS		 6
#define SM_EV_RCR_PLUS		 7
#define SM_EV_RCR_MINUS		 8
#define SM_EV_RCA		 9
#define SM_EV_RCN		10
#define SM_EV_RTR		11
#define SM_EV_RTA		12
#define SM_EV_RUC		13
#define SM_EV_RXJ_PLUS		14
#define SM_EV_RXJ_MINUS		15
#define SM_EV_RXR		16

static void sm_initial ARGS(( sm_state_ut *sm_state, int event ));
static void sm_starting ARGS(( sm_state_ut *sm_state, int event ));
static void sm_closed ARGS(( sm_state_ut *sm_state, int event ));
static void sm_stopped ARGS(( sm_state_ut *sm_state, int event ));
static void sm_closing ARGS(( sm_state_ut *sm_state, int event ));
static void sm_stopping ARGS(( sm_state_ut *sm_state, int event ));
static void sm_req_sent ARGS(( sm_state_ut *sm_state, int event ));
static void sm_ack_rcvd ARGS(( sm_state_ut *sm_state, int event ));
static void sm_ack_sent ARGS(( sm_state_ut *sm_state, int event ));
static void sm_opened ARGS(( sm_state_ut *sm_state, int event ));

static void sm_timer_func ARGS(( clck_timer_ut *timer, void *ref ));

DEFUN
(void sm_init, (sm_state, name, action),
	sm_state_ut *sm_state AND
	char *name AND
	sm_action_ut *action
)
{
	sm_state->sms_func= sm_initial;
	sm_state->sms_name= name;
	sm_state->sms_action= *action;

	assert(sm_state->sms_action.sma_tlu != 0);
	assert(sm_state->sms_action.sma_tld != 0);
	assert(sm_state->sms_action.sma_tls != 0);
	assert(sm_state->sms_action.sma_tlf != 0);
	assert(sm_state->sms_action.sma_irc != 0);
	assert(sm_state->sms_action.sma_zrc != 0);
	assert(sm_state->sms_action.sma_scr != 0);
	assert(sm_state->sms_action.sma_sca != 0);
	assert(sm_state->sms_action.sma_scn != 0);
	assert(sm_state->sms_action.sma_str != 0);
	assert(sm_state->sms_action.sma_sta != 0);
	assert(sm_state->sms_action.sma_scj != 0);

	sm_state->sms_isup= 0;
	sm_state->sms_max_configure= SM_DFLT_MAX_CONFIGURE;
	sm_state->sms_max_terminate= SM_DFLT_MAX_TERMINATE;
	sm_state->sms_restart_counter= 0;
	sm_state->sms_restart_timer= SM_DFLT_RESTART_TIMER;
	sm_state->sms_options_inited= 0;
	clck_init_timer(&sm_state->sms_timer, sm_timer_func, sm_state);
}


DEFUN
(void sm_up, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_up()\n", sm_state->sms_name));
	if (sm_state->sms_isup)
	{
		DPRINTF(1, ("%s_up: already up\n", sm_state->sms_name));
		return;
	}
	(*sm_state->sms_func)(sm_state, SM_EV_UP);
}


DEFUN
(void sm_down, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_down()\n", sm_state->sms_name));
	if (!sm_state->sms_isup)
	{
		DPRINTF(1, ("%s_up: already down\n", sm_state->sms_name));
		return;
	}
	(*sm_state->sms_func)(sm_state, SM_EV_DOWN);
}


DEFUN
(void sm_open, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_open()\n", sm_state->sms_name));
	if (sm_state->sms_func == sm_opened || 
		sm_state->sms_func == sm_closing || 
		sm_state->sms_func == sm_stopping || 
		sm_state->sms_func == sm_stopped)
	{
		DPRINTF(1, ("%s_open: already open\n", sm_state->sms_name));
		sm_down(sm_state);
		sm_up(sm_state);
		return;
	}
	else
		(*sm_state->sms_func)(sm_state, SM_EV_OPEN);
}


DEFUN
(void sm_close, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_close()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_CLOSE);
}


DEFUN
(void sm_to_plus, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_to_plus()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_TO_PLUS);
}


DEFUN
(void sm_to_minus, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_to_minus()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_TO_MINUS);
}


DEFUN
(void sm_rcr_plus, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_rcr_plus()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_RCR_PLUS);
}


DEFUN
(void sm_rcr_minus, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_rcr_minus()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_RCR_MINUS);
}


DEFUN
(void sm_rca, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_rca()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_RCA);
}


DEFUN
(void sm_rcn, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_rcn()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_RCN);
}


DEFUN
(void sm_rtr, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_rtr()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_RTR);
}


DEFUN
(void sm_rta, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_rta()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_RTA);
}


DEFUN
(void sm_ruc, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_ruc()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_RUC);
}


DEFUN
(void sm_rxj_plus, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_rxj_plus()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_RXJ_PLUS);
}


DEFUN
(void sm_rxj_minus, (sm_state),
	sm_state_ut *sm_state
)
{
	DPRINTF(1, ("%s_rxj_minus()\n", sm_state->sms_name));
	(*sm_state->sms_func)(sm_state, SM_EV_RXJ_MINUS);
}


DEFUN
(static void sm_initial, (sm_state, event),
	sm_state_ut *sm_state AND
	int event
)
{
	DPRINTF(1, ("%s_initial, ", sm_state->sms_name));
	switch(event)
	{
	case SM_EV_UP:
		DPRINTF_CONT(1, ("Up\n"));
		sm_state->sms_isup= 1;
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_DOWN:
		DPRINTF_CONT(1, ("Down, illegal event\n"));
		break;
	case SM_EV_OPEN:
		DPRINTF_CONT(1, ("Open\n"));
		sm_state->sms_action.sma_tls();
		sm_state->sms_func= sm_starting;
		break;
	case SM_EV_CLOSE:
		DPRINTF_CONT(1, ("Close\n"));
		sm_state->sms_func= sm_initial;
		break;
	case SM_EV_TO_PLUS:
	case SM_EV_TO_MINUS:
	case SM_EV_RCR_PLUS:
	case SM_EV_RCR_MINUS:
	case SM_EV_RCA:
	case SM_EV_RCN:
	case SM_EV_RTR:
	case SM_EV_RTA:
	case SM_EV_RUC:
	case SM_EV_RXJ_PLUS:
	case SM_EV_RXJ_MINUS:
	case SM_EV_RXR:
		DPRINTF_CONT(1, ("illegal event (%d)\n", event));
		break;
	default:
		fatal("unknown event (%d)\n", event);
	}
}


DEFUN
(static void sm_starting, (sm_state, event),
	sm_state_ut *sm_state AND
	int event
)
{
	DPRINTF(1, ("%s_starting, ", sm_state->sms_name));
	switch(event)
	{
	case SM_EV_UP:
		DPRINTF_CONT(1, ("Up\n"));
		sm_state->sms_isup= 1;
		sm_state->sms_action.sma_irc(sm_state->sms_max_configure);
		sm_state->sms_action.sma_scr();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_DOWN:
		DPRINTF_CONT(1, ("Down, illegal event\n"));
		break;
	case SM_EV_OPEN:
		DPRINTF_CONT(1, ("Open\n"));
		sm_state->sms_func= sm_starting;
		break;
	case SM_EV_CLOSE:
		DPRINTF_CONT(1, ("Close\n"));
		sm_state->sms_func= sm_initial;
		break;
	case SM_EV_TO_PLUS:
	case SM_EV_TO_MINUS:
	case SM_EV_RCR_PLUS:
	case SM_EV_RCR_MINUS:
	case SM_EV_RCA:
	case SM_EV_RCN:
	case SM_EV_RTR:
	case SM_EV_RTA:
	case SM_EV_RUC:
	case SM_EV_RXJ_PLUS:
	case SM_EV_RXJ_MINUS:
	case SM_EV_RXR:
		DPRINTF_CONT(1, ("illegal event (%d)\n", event));
		break;
	default:
		fatal("unknown event (%d)\n", event);
	}
}


DEFUN
(static void sm_closed, (sm_state, event),
	sm_state_ut *sm_state AND
	int event
)
{
	DPRINTF(1, ("%s_closed, ", sm_state->sms_name));
	switch(event)
	{
	case SM_EV_UP:
		DPRINTF_CONT(1, ("Up, illegal event\n"));
		break;
	case SM_EV_DOWN:
		DPRINTF_CONT(1, ("Down\n"));
		sm_state->sms_isup= 0;
		sm_state->sms_func= sm_initial;
		break;
	case SM_EV_OPEN:
		DPRINTF_CONT(1, ("Open\n"));
		sm_state->sms_action.sma_irc(sm_state->sms_max_terminate);
		sm_state->sms_action.sma_scr();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_CLOSE:
		DPRINTF_CONT(1, ("Close\n"));
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_TO_PLUS:
	case SM_EV_TO_MINUS:
		DPRINTF_CONT(1, ("illegal event (%d)\n", event));
		break;
	case SM_EV_RCR_PLUS:
		DPRINTF_CONT(1, ("RCR+\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RCR_MINUS:
		DPRINTF_CONT(1, ("RCR-\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RCA:
		DPRINTF_CONT(1, ("RCA\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RCN:
		DPRINTF_CONT(1, ("RCN\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RTR:
		DPRINTF_CONT(1, ("RTR\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RTA:
		DPRINTF_CONT(1, ("RTA\n"));
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RUC:
		DPRINTF_CONT(1, ("RUC\n"));
		sm_state->sms_action.sma_scj();
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RXJ_PLUS:
		DPRINTF_CONT(1, ("RXJ+\n"));
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RXJ_MINUS:
		DPRINTF_CONT(1, ("RXJ-\n"));
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RXR:
		DPRINTF_CONT(1, ("RXR\n"));
		sm_state->sms_func= sm_closed;
		break;
	default:
		fatal("unknown event (%d)\n", event);
	}
}


DEFUN
(static void sm_stopped, (sm_state, event),
	sm_state_ut *sm_state AND
	int event
)
{
	DPRINTF(1, ("%s_stopped, ", sm_state->sms_name));
	switch(event)
	{
	case SM_EV_UP:
		DPRINTF_CONT(1, ("Up, illegal event\n"));
		break;
	case SM_EV_DOWN:
		DPRINTF_CONT(1, ("Down\n"));
		sm_state->sms_action.sma_tls();
		sm_state->sms_func= sm_starting;
		break;
	case SM_EV_OPEN:
		DPRINTF_CONT(1, ("Open\n"));
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_CLOSE:
		DPRINTF_CONT(1, ("Close\n"));
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_TO_PLUS:
	case SM_EV_TO_MINUS:
		DPRINTF_CONT(1, ("illegal event (%d)\n", event));
		break;
	case SM_EV_RCR_PLUS:
		DPRINTF_CONT(1, ("RCR+\n"));
		sm_state->sms_action.sma_irc(sm_state->sms_max_terminate);
		sm_state->sms_action.sma_scr();
		sm_state->sms_action.sma_sca();
		sm_state->sms_func= sm_ack_sent;
	case SM_EV_RCR_MINUS:
		DPRINTF_CONT(1, ("RCR-\n"));
		sm_state->sms_action.sma_irc(sm_state->sms_max_terminate);
		sm_state->sms_action.sma_scr();
		sm_state->sms_action.sma_scn();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_RCA:
		DPRINTF_CONT(1, ("RCA\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_RCN:
		DPRINTF_CONT(1, ("RCN\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_RTR:
		DPRINTF_CONT(1, ("RTR\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_RTA:
		DPRINTF_CONT(1, ("RTA\n"));
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_RUC:
		DPRINTF_CONT(1, ("RUC\n"));
		sm_state->sms_action.sma_scj();
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_RXJ_PLUS:
		DPRINTF_CONT(1, ("RXJ+\n"));
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_RXJ_MINUS:
		DPRINTF_CONT(1, ("RXJ-\n"));
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_RXR:
		DPRINTF_CONT(1, ("RXR\n"));
		sm_state->sms_func= sm_stopped;
		break;
	default:
		fatal("unknown event (%d)\n", event);
	}
}


DEFUN
(static void sm_closing, (sm_state, event),
	sm_state_ut *sm_state AND
	int event
)
{
	DPRINTF(1, ("%s_closing, ", sm_state->sms_name));
	switch(event)
	{
	case SM_EV_UP:
		DPRINTF_CONT(1, ("Up, illegal event\n"));
		break;
	case SM_EV_DOWN:
		DPRINTF_CONT(1, ("Down\n"));
		sm_state->sms_isup= 0;
		sm_state->sms_func= sm_initial;
		break;
	case SM_EV_OPEN:
		DPRINTF_CONT(1, ("Open\n"));
		sm_state->sms_func= sm_stopping;
		break;
	case SM_EV_CLOSE:
		DPRINTF_CONT(1, ("Close\n"));
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_TO_PLUS:
		DPRINTF_CONT(1, ("TO+\n"));
		sm_state->sms_action.sma_str();
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_TO_MINUS:
		DPRINTF_CONT(1, ("TO-\n"));
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RCR_PLUS:
		DPRINTF_CONT(1, ("RCR+\n"));
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_RCR_MINUS:
		DPRINTF_CONT(1, ("RCR-\n"));
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_RCA:
		DPRINTF_CONT(1, ("RCA\n"));
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_RCN:
		DPRINTF_CONT(1, ("RCN\n"));
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_RTR:
		DPRINTF_CONT(1, ("RTR\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_RTA:
		DPRINTF_CONT(1, ("RTA\n"));
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RUC:
		DPRINTF_CONT(1, ("RUC\n"));
		sm_state->sms_action.sma_scj();
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_RXJ_PLUS:
		DPRINTF_CONT(1, ("RXJ+\n"));
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_RXJ_MINUS:
		DPRINTF_CONT(1, ("RXJ-\n"));
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_closed;
		break;
	case SM_EV_RXR:
		DPRINTF_CONT(1, ("RXR\n"));
		sm_state->sms_func= sm_closing;
		break;
	default:
		DPRINTF_CONT(1, ("illegal event (%d)\n", event));
		abort();
	}
}


DEFUN
(static void sm_stopping, (sm_state, event),
	sm_state_ut *sm_state AND
	int event
)
{
	DPRINTF(1, ("%s_stopping, ", sm_state->sms_name));
	switch(event)
	{
	case SM_EV_CLOSE:
		DPRINTF_CONT(1, ("Close\n"));
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_RTA:
		DPRINTF_CONT(1, ("RTA\n"));
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_RXJ_MINUS:
		DPRINTF_CONT(1, ("RJX-\n"));
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_stopped;
		break;
	default:
		DPRINTF_CONT(1, ("illegal event (%d)\n", event));
		abort();
	}
}


DEFUN
(static void sm_req_sent, (sm_state, event),
	sm_state_ut *sm_state AND
	int event
)
{
	DPRINTF(1, ("%s_req_sent, ", sm_state->sms_name));
	switch(event)
	{
	case SM_EV_DOWN:
		DPRINTF_CONT(1, ("Down\n"));
		sm_state->sms_isup= 0;
		sm_state->sms_options_inited= 0;
		clck_cancel_timer(&sm_state->sms_timer);
		sm_state->sms_func= sm_starting;
		break;
	case SM_EV_CLOSE:
		DPRINTF_CONT(1, ("Close\n"));
		sm_state->sms_options_inited= 0;
		sm_state->sms_action.sma_irc(sm_state->sms_max_terminate);
		sm_state->sms_action.sma_str();
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_TO_PLUS:
		DPRINTF_CONT(1, ("TO+\n"));
		sm_state->sms_action.sma_scr();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_TO_MINUS:
		DPRINTF_CONT(1, ("TO-\n"));
		sm_state->sms_options_inited= 0;
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_RCR_PLUS:
		DPRINTF_CONT(1, ("RCR+\n"));
		sm_state->sms_action.sma_sca();
		sm_state->sms_func= sm_ack_sent;
		break;
	case SM_EV_RCR_MINUS:
		DPRINTF_CONT(1, ("RCR-\n"));
		sm_state->sms_action.sma_scn();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_RCA:
		DPRINTF_CONT(1, ("RCA\n"));
		sm_state->sms_action.sma_irc(sm_state->sms_max_configure);
		sm_state->sms_func= sm_ack_rcvd;
		break;
	case SM_EV_RCN:
		DPRINTF_CONT(1, ("RCN\n"));
		sm_state->sms_action.sma_irc(sm_state->sms_max_configure);
		sm_state->sms_action.sma_scr();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_RTR:
		DPRINTF_CONT(1, ("RTR\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_RTA:
		DPRINTF_CONT(1, ("RTA\n"));
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_RUC:
		DPRINTF_CONT(1, ("RUC\n"));
		sm_state->sms_action.sma_scj();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_RXJ_MINUS:
		DPRINTF_CONT(1, ("RXJ-\n"));
		sm_state->sms_options_inited= 0;
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_stopped;
		break;
	default:
		DPRINTF_CONT(1, ("illegal event (%d)\n", event));
		abort();
	}
}


DEFUN
(static void sm_ack_rcvd, (sm_state, event),
	sm_state_ut *sm_state AND
	int event
)
{
	DPRINTF(1, ("%s_ack_rcvd, ", sm_state->sms_name));
	switch(event)
	{
	case SM_EV_CLOSE:
		DPRINTF_CONT(1, ("Close\n"));
		sm_state->sms_options_inited= 0;
		sm_state->sms_action.sma_irc(sm_state->sms_max_terminate);
		sm_state->sms_action.sma_str();
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_TO_PLUS:
		DPRINTF_CONT(1, ("TO+\n"));
		sm_state->sms_action.sma_scr();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_RCR_PLUS:
		DPRINTF_CONT(1, ("RCR+\n"));
		sm_state->sms_options_inited= 0;
		sm_state->sms_action.sma_sca();
		sm_state->sms_action.sma_tlu();
		clck_cancel_timer(&sm_state->sms_timer);
		sm_state->sms_func= sm_opened;
		break;
	case SM_EV_RCR_MINUS:
		DPRINTF_CONT(1, ("RCR-\n"));
		sm_state->sms_action.sma_scn();
		sm_state->sms_func= sm_ack_rcvd;
		break;
	case SM_EV_RTR:
		DPRINTF_CONT(1, ("RTR\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_req_sent;
		break;
	default:
		DPRINTF_CONT(1, ("illegal event (%d)\n", event));
		abort();
	}
}


DEFUN
(static void sm_ack_sent, (sm_state, event),
	sm_state_ut *sm_state AND
	int event
)
{
	DPRINTF(1, ("%s_ack_sent, ", sm_state->sms_name));
	switch(event)
	{
	case SM_EV_CLOSE:
		DPRINTF_CONT(1, ("Close\n"));
		sm_state->sms_options_inited= 0;
		sm_state->sms_action.sma_irc(sm_state->sms_max_terminate);
		sm_state->sms_action.sma_str();
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_TO_PLUS:
		DPRINTF_CONT(1, ("TO+\n"));
		sm_state->sms_action.sma_scr();
		sm_state->sms_func= sm_ack_sent;
		break;
	case SM_EV_TO_MINUS:
		DPRINTF_CONT(1, ("TO-\n"));
		sm_state->sms_options_inited= 0;
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_stopped;
		break;
	case SM_EV_RCR_PLUS:
		DPRINTF_CONT(1, ("RCR+\n"));
		sm_state->sms_action.sma_sca();
		sm_state->sms_func= sm_ack_sent;
		break;
	case SM_EV_RCA:
		DPRINTF_CONT(1, ("RCA\n"));
		sm_state->sms_options_inited= 0;
		sm_state->sms_action.sma_irc(sm_state->sms_max_configure);
		sm_state->sms_action.sma_tlu();
		clck_cancel_timer(&sm_state->sms_timer);
		sm_state->sms_func= sm_opened;
		break;
	case SM_EV_RCN:
		DPRINTF_CONT(1, ("RCN\n"));
		sm_state->sms_action.sma_irc(sm_state->sms_max_configure);
		sm_state->sms_action.sma_scr();
		sm_state->sms_func= sm_ack_sent;
		break;
	case SM_EV_RTR:
		DPRINTF_CONT(1, ("RTR\n"));
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_RUC:
		DPRINTF_CONT(1, ("RUC\n"));
		sm_state->sms_action.sma_scj();
		sm_state->sms_func= sm_ack_sent;
		break;
	case SM_EV_RXJ_MINUS:
		DPRINTF_CONT(1, ("RXJ-\n"));
		sm_state->sms_options_inited= 0;
		sm_state->sms_action.sma_tlf();
		sm_state->sms_func= sm_stopped;
		break;
	default:
		DPRINTF_CONT(1, ("illegal event (%d)\n", event));
		abort();
	}
}


DEFUN
(static void sm_opened, (sm_state, event),
	sm_state_ut *sm_state AND
	int event
)
{
	DPRINTF(1, ("%s_opened, ", sm_state->sms_name));
	switch(event)
	{
	case SM_EV_DOWN:
		DPRINTF_CONT(1, ("Down\n"));
		sm_state->sms_action.sma_tld();
		sm_state->sms_func= sm_starting;
		break;
	case SM_EV_CLOSE:
		DPRINTF_CONT(1, ("Close\n"));
		sm_state->sms_action.sma_tld();
		sm_state->sms_action.sma_irc(sm_state->sms_max_terminate);
		sm_state->sms_action.sma_str();
		sm_state->sms_func= sm_closing;
		break;
	case SM_EV_RCR_PLUS:
		DPRINTF_CONT(1, ("RCR+\n"));
		sm_state->sms_action.sma_tld();
		sm_state->sms_action.sma_scr();
		sm_state->sms_action.sma_sca();
		sm_state->sms_func= sm_ack_sent;
		break;
	case SM_EV_RCR_MINUS:
		DPRINTF_CONT(1, ("RCR-\n"));
		sm_state->sms_action.sma_tld();
		sm_state->sms_action.sma_scr();
		sm_state->sms_action.sma_scn();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_RTR:
		DPRINTF_CONT(1, ("RTR\n"));
		sm_state->sms_action.sma_tld();
		sm_state->sms_action.sma_zrc();
		sm_state->sms_action.sma_sta();
		sm_state->sms_func= sm_stopping;
		break;
	case SM_EV_RUC:
		DPRINTF_CONT(1, ("RUC\n"));
		sm_state->sms_action.sma_tld();
		sm_state->sms_action.sma_scj();
		sm_state->sms_action.sma_scr();
		sm_state->sms_func= sm_req_sent;
		break;
	case SM_EV_RXJ_PLUS:
		DPRINTF_CONT(1, ("RXJ+\n"));
		sm_state->sms_func= sm_opened;
		break;
	case SM_EV_RXJ_MINUS:
		DPRINTF_CONT(1, ("RXJ-\n"));
		sm_state->sms_action.sma_tld();
		sm_state->sms_action.sma_irc(sm_state->sms_max_terminate);
		sm_state->sms_action.sma_str();
		sm_state->sms_func= sm_stopping;
		break;
	default:
		DPRINTF_CONT(1, ("illegal event (%d)\n", event));
		abort();
	}
}


DEFUN
(static void sm_timer_func, (timer, ref),
	clck_timer_ut *timer AND
	void *ref
)
{
	sm_state_ut *sm_state;

	sm_state= ref;

	DPRINTF(1, ("%s_timer_func()\n", sm_state->sms_name));
	if (sm_state->sms_restart_counter > 0)
		sm_to_plus(sm_state);
	else
		sm_to_minus(sm_state);

}

/*
 * $PchHeader: /mount/hd2/minix/local/cmd/ppp/RCS/sm.c,v 1.2 1995/05/23 08:12:13 philip Exp $
 */
