/*
ppp_snd.c

Created:	May 25, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#define _MINIX_SOURCE

#include <sys/types.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "ansi.h"
#include "debug.h"
#include "fcs.h"
#include "pkt.h"
#include "ppp.h"

#define PPP_SND_BLOCK		1024
#define PPP_SND_THRESHOLD	 128

int ppp_snd_more2send;
int ppp_snd_inprogress;
int ppp_snd_maxsize;
int ppp_snd_hc_proto;
int ppp_snd_hc_address;

static u8_t ppp_snd_special_byte[UCHAR_MAX+1];
static pkt_ut *ppp_snd_pack, *ppp_snd_q;
static ppp_snd_callback_ut *ppp_cb_q_head, *ppp_cb_q_tail;

DEFUN_VOID (void ppp_snd_init)
{
	int i, r, v;

	ppp_snd_more2send= 0;
	ppp_snd_inprogress= 0;
	ppp_snd_maxsize= PPP_DFLT_MAXSIZE;

#if 0
	DPRINTF(0,
	("ppp_snd_init: enabling address-and-control-field-compression\n"));
	ppp_snd_hc_address= 1;
#endif

	for (i= 0; i<UCHAR_MAX+1; i++)
		ppp_snd_special_byte[i]= 0;
	for (i= 0; i<0x20; i++)
		ppp_snd_special_byte[i]= 1;
	ppp_snd_special_byte[PPP_BYTE_FLAG]= 1;
	ppp_snd_special_byte[PPP_BYTE_ESCAPE]= 1;
ppp_snd_special_byte[255]= 1;

	ppp_snd_pack= NULL;
	ppp_snd_q= NULL;
	ppp_cb_q_head= NULL;

#if 0
	r= fcntl(1, F_GETFD);
	if (r == -1)
	{
		DPRINTF(1, ("ppp_snd_init: fcntl(1, GETFD) failed (%s)\n",
			strerror(errno)));
		exit(1);
	}
	v= r | FD_ASYNCHIO;
	r= fcntl(1, F_SETFD, v);
	if (r == -1)
	{
		DPRINTF(1, ("ppp_snd_init: fcntl(1, SETFD, %d) failed (%s)\n",
			v, strerror(errno)));
		exit(1);
	}
#endif
}

DEFUN
(void ppp_snd_init_callback, (callback, cb_f, cb_arg),
	ppp_snd_callback_ut *callback AND
	psc_cbf_ut cb_f AND
	int cb_arg
)
{
	callback->psc_set= 0;
	callback->psc_cb_f= cb_f;
	callback->psc_cb_arg= cb_arg;
}

DEFUN
(void ppp_snd_sethdr, (pkt, proto),
	pkt_ut *pkt AND
	u16_t proto
)
{
	u8_t *dptr;

	assert(pkt->p_size <= PPP_DFLT_MAXSIZE);

	assert(pkt->p_offset >= 4);

	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	if (ppp_snd_hc_proto && proto < 256)
	{
		pkt->p_offset--;
		pkt->p_size++;
		dptr--;
		*dptr= (proto & 0xFF);		/* protocol LSB */
	}
	else
	{
		pkt->p_offset -= 2;
		pkt->p_size += 2;
		dptr -= 2;
		dptr[0]= (proto >> 8);		/* protocol MSB */
		dptr[1]= (proto & 0xFF);	/* protocol LSB */
	}
	if (!ppp_snd_hc_address || proto == PPP_TYPE_LCP)
	{
		pkt->p_offset -= 2;
		pkt->p_size += 2;
		dptr -= 2;
		dptr[0]= PPP_HDR_ALLSTATIONS;
		dptr[1]= PPP_HDR_UI;
	}
}

DEFUN
(int ppp_snd, (pkt, callback),
	pkt_ut *pkt AND
	ppp_snd_callback_ut *callback
)
{
	u8_t byte;
	u16_t fcs;
	int state;
	u8_t *block_dptr, *dptr;
	size_t block_size, size;
	pkt_ut *block;
	int i;

	if (ppp_cb_q_head != NULL || 
		(ppp_snd_q != NULL && ppp_snd_q->p_size >= PPP_SND_THRESHOLD))
	{
		if (!callback->psc_set)
		{
			callback->psc_set= 1;
			callback->psc_next= NULL;

			if (ppp_cb_q_head == NULL)
				ppp_cb_q_head= callback;
			else
				ppp_cb_q_tail->psc_next= callback;
			ppp_cb_q_tail= callback;
		}
		return EAGAIN;
	}

	ppp_snd_more2send= 1;

	if (ppp_snd_q == NULL)
	{
		block= pkt_alloc(PPP_SND_BLOCK);
		block->p_next= NULL;
		ppp_snd_q= block;
		if (ppp_snd_pack == NULL)
		{
			block->p_size++;
			block->p_data[0]= PPP_BYTE_FLAG;
		}
	}
	else
		block= ppp_snd_q;

	block_size= block->p_size;
	block_dptr= (u8_t *)(block->p_data + block_size);
	size= pkt->p_size;
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);

	DBLOCK(0, 
	{
		DPRINTF(6, ("ppp_snd: sending "));
		for (i= 0; i< size; i++)
			DPRINTF_CONT(6, ("%02x", dptr[i]));
		DPRINTF_CONT(6, ("\n"));
	});

	state= 0;
	while (state != -1)
	{
		if (block_size == PPP_SND_BLOCK)
		{
			block->p_size= block_size;
			block->p_next= pkt_alloc(PPP_SND_BLOCK);
			block= block->p_next;
			block->p_next= NULL;
			block_size= 0;
			block_dptr= (u8_t *)block->p_data;
		}

		if (state == 0)
		{
			/* next byte */
			if (size == 0)
			{
				state= 2;	/* start of checksum */
				continue;
			}
			byte= *dptr;
			dptr++;
			size--;

			if (ppp_snd_special_byte[byte])
			{
				*block_dptr= PPP_BYTE_ESCAPE;
				block_dptr++;
				block_size++;
				state= 1;	/* escaped byte */
				continue;
			}

			*block_dptr= byte;
			block_dptr++;
			block_size++;
			state= 0;
			continue;
		}
		if (state == 1)
		{
			/* escaped byte */
			byte ^= PPP_BYTE_BIT6;
			*block_dptr= byte;
			block_dptr++;
			block_size++;
			state= 0;		/* next byte */
			continue;
		}
		if (state == 2)
		{
			/* start of checksum */
			fcs= pppfcs(PPPINITFCS, 
				(u8_t *)(pkt->p_data+pkt->p_offset),
				pkt->p_size);
			fcs ^= 0xFFFF;
			byte= (fcs & 0xff);
			if (ppp_snd_special_byte[byte])
			{
				*block_dptr= PPP_BYTE_ESCAPE;
				block_dptr++;
				block_size++;
				state= 3;	/* escaped first byte of 
						 * checksum
						 */
				continue;
			}

			*block_dptr= byte;
			block_dptr++;
			block_size++;
			state= 4;		/* second byte of checksum */
			continue;
		}
		if (state == 3)
		{
			/* escaped first byte of checksum */
			byte ^= PPP_BYTE_BIT6;
			*block_dptr= byte;
			block_dptr++;
			block_size++;
			state= 4;		/* second byte of checksum */
			continue;
		}
		if (state == 4)
		{
			/* second byte of checksum */
			byte= (fcs >> 8);
			if (ppp_snd_special_byte[byte])
			{
				*block_dptr= PPP_BYTE_ESCAPE;
				block_dptr++;
				block_size++;
				state= 5;	/* escaped second byte of 
						 * checksum
						 */
				continue;
			}

			*block_dptr= byte;
			block_dptr++;
			block_size++;
			state= 6;		/* flag byte */
			continue;
		}
		if (state == 5)
		{
			/* escaped second byte of checksum */
			byte ^= PPP_BYTE_BIT6;
			*block_dptr= byte;
			block_dptr++;
			block_size++;
			state= 6;		/* second byte of checksum */
			continue;
		}
		if (state == 6)
		{
			/* flag byte */
			*block_dptr= PPP_BYTE_FLAG;
			block_dptr++;
			block_size++;
			state= -1;
			continue;
		}
		assert(0);
	}
	block->p_size= block_size;
	pkt_free(pkt);

	return 0;
}


DEFUN_VOID (void ppp_snd_restart)
{
	int result;
	size_t size;
	char *dptr;
	ppp_snd_callback_ut *tmp_q_head, *tmp_cb;

	for (;;)
	{
		if (ppp_snd_pack == NULL)
		{
			if (ppp_snd_q == NULL)
			{
				ppp_snd_more2send= 0;
				break;
			}
			ppp_snd_pack= ppp_snd_q;
			ppp_snd_q= ppp_snd_q->p_next;
		}
		dptr= ppp_snd_pack->p_data + ppp_snd_pack->p_offset;
		size= ppp_snd_pack->p_size;
		while (size > 0)
		{
			result= write(1, dptr, size);
			if (result > 0)
			{
				assert(result <= size);
				dptr += result;
				size -= result;
				continue;
			}
			if (result == -1 && errno == EINPROGRESS)
			{
				ppp_snd_inprogress= 1;
				return;
			}
			if (result == -1)
			{
				DPRINTF(1, (
					"ppp_snd_restart: write error (%s)\n",
						strerror(errno)));
				exit(1);
			}
			DPRINTF(1, (
		"ppp_snd_restart: strange return value from write (%d)\n",
					result));
			exit(2);
		}
		assert(size == 0);
		pkt_free(ppp_snd_pack);
		ppp_snd_pack= NULL;
	}
	if (ppp_cb_q_head != NULL)
	{
		tmp_q_head= ppp_cb_q_head;
		ppp_cb_q_head= NULL;
		while(tmp_q_head != NULL)
		{
			tmp_cb= tmp_q_head;
			tmp_q_head= tmp_q_head->psc_next;
			assert(tmp_cb->psc_set);
			tmp_cb->psc_set= 0;
			(*tmp_cb->psc_cb_f)(tmp_cb, tmp_cb->psc_cb_arg);
		}
	}
}

DEFUN
(void ppp_snd_completed, (result, error),
	int result AND
	int error
)
{
	ppp_snd_inprogress= 0;

	if (result > 0)
	{
		assert(result <= ppp_snd_pack->p_size);
		ppp_snd_pack->p_offset += result;
		ppp_snd_pack->p_size -= result;
		return;
	}
	if (result == -1)
	{
		DPRINTF(1, ("ppp_snd_completed: write error (%s)\n",
				strerror(error)));
		exit(1);
	}
	DPRINTF(1, (
		"ppp_snd_completed: strange return value from write (%d)\n",
			result));
	exit(2);
}

/*
 * $PchHeader: /mount/hd2/minix/local/cmd/ppp/RCS/ppp_snd.c,v 1.2 1995/04/28 08:03:05 philip Exp philip $
 */
