/*
ppp_rcv.c

Created:	May 26, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#define _MINIX_SOURCE

#include <sys/types.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "ansi.h"
#include "debug.h"
#include "fcs.h"
#include "ip.h"
#include "ipcp.h"
#include "lcp.h"
#include "pkt.h"
#include "ppp.h"
#include "vjhc.h"

int ppp_rcv_inprogress;
int ppp_rcv_hc_address;
int ppp_rcv_hc_proto;

#define PPP_RCV_BUF_SIZE	4096
#define PPP_RCV_BLOCK		64
#define PPP_RCV_OFFSET		120

static char ppp_rcv_buf[PPP_RCV_BUF_SIZE];
static size_t ppp_rcv_buf_offset;
static u8_t ppp_rcv_special_byte[UCHAR_MAX+1];
static pkt_ut *ppp_rcv_pkt;

static void ppp_rcv_err_pack ARGS(( pkt_ut *pkt ));
static void ppp_rcv_pack ARGS(( pkt_ut *pkt ));

DEFUN_VOID (void ppp_rcv_init)
{
	int result;
	int i, v;

	DPRINTF(0,
	("ppp_rcv_init: enabling address-and-control-field-compression\n"));
	ppp_rcv_hc_address= 1;

	ppp_rcv_inprogress= 0;

	ppp_rcv_buf_offset= 0;

	for (i= 0; i<UCHAR_MAX+1; i++)
		ppp_rcv_special_byte[i]= 0;
	for (i= 0; i<0x20; i++)
		ppp_rcv_special_byte[i]= 1;
	ppp_rcv_special_byte[PPP_BYTE_FLAG]= 1;
	ppp_rcv_special_byte[PPP_BYTE_ESCAPE]= 1;

	ppp_rcv_pkt= NULL;

	result= fcntl(0, F_GETFD);
	if (result == -1)
	{
		DPRINTF(1, ("ppp_rcv_init: fcntl(0, GETFD) failed (%s)\n",
			strerror(errno)));
		exit(1);
	}
	v= result | FD_ASYNCHIO;
	result= fcntl(0, F_SETFD, v);
	if (result == -1)
	{
		DPRINTF(1, ("ppp_rcv_init: fcntl(0, SETFD, %d) failed (%s)\n",
			v, strerror(errno)));
		exit(1);
	}
}


DEFUN_VOID (void ppp_rcv_restart)
{
	size_t offset;
	int result;
	int error;

	offset= 0;
	ppp_rcv_buf_offset= 0;
	while (offset < PPP_RCV_BUF_SIZE)
	{
		result= read(0, ppp_rcv_buf+offset, PPP_RCV_BUF_SIZE-offset);
		if (result > 0)
		{
			assert(result <= PPP_RCV_BUF_SIZE-offset);
			offset += result;
			continue;
		}
		error= errno;
		if (offset != 0)
		{
			ppp_rcv_completed(offset, 0);
		}
		if (result == -1 && error == EINPROGRESS)
		{
			ppp_rcv_buf_offset= offset;
			ppp_rcv_inprogress= 1;
			return;
		}
		offset= 0;
		ppp_rcv_completed(result, error);
	}
	ppp_rcv_completed(offset, 0);
}


DEFUN
(void ppp_rcv_completed, (result, error),
	int result AND
	int error
)
{
	static int escape_seen= 0;

	u8_t *block_dptr, *dptr;
	u8_t byte;
	pkt_ut *pkt;
	size_t size;

	ppp_rcv_inprogress= 0;

	if (result < 0 || result == 0)
	{
		if (result == 0 || result == -1)
		{
			DPRINTF(1, ("ppp_rcv_completed: read error (%s)\n",
				result == 0 ? "EOF" : strerror(error)));
		}
		else 
		{
			DPRINTF(1, (
		"ppp_rcv_completed: read returned strange value (%d)\n",
					result));
		}
		exit(1);
	}
	assert(ppp_rcv_buf_offset + result <= PPP_RCV_BUF_SIZE);

	dptr= (u8_t *)ppp_rcv_buf + ppp_rcv_buf_offset;
	size= result;
	while (size > 0)
	{
		if (ppp_rcv_pkt == NULL)
		{
			if (size < PPP_RCV_BLOCK)
				ppp_rcv_pkt= pkt_alloc(PPP_RCV_OFFSET +
					PPP_RCV_BLOCK);
			else
				ppp_rcv_pkt= pkt_alloc(PPP_RCV_OFFSET + size);
			pkt= ppp_rcv_pkt;
			pkt->p_offset += PPP_RCV_OFFSET;
		}
		else
		{
			pkt= ppp_rcv_pkt;
			if (pkt->p_offset + pkt->p_size + size > 
								pkt->p_maxsize)
			{
				if (size < PPP_RCV_BLOCK)
				{
					pkt_realloc(pkt, pkt->p_offset + 
						pkt->p_size + PPP_RCV_BLOCK);
				}
				else
				{
					pkt_realloc(pkt, pkt->p_offset + 
						pkt->p_size + size);
				}
			}
		}
		block_dptr= (u8_t *)(pkt->p_data + pkt->p_offset + 
								pkt->p_size);
		while (size > 0)
		{
			byte= *dptr;
			dptr++;
			size--;
			if (escape_seen)
			{
				escape_seen= 0;
				if (byte == PPP_BYTE_FLAG)
				{
					ppp_rcv_pkt= NULL;
					assert((char *)block_dptr <= 
						pkt->p_data + pkt->p_maxsize);
					pkt->p_size= (char *)block_dptr - 
						pkt->p_data - pkt->p_offset;
					ppp_rcv_err_pack(pkt);
					break;
				}
				byte ^= PPP_BYTE_BIT6;
				*block_dptr= byte;
				block_dptr++;
				continue;
			}
			if (!ppp_rcv_special_byte[byte])
			{
				*block_dptr= byte;
				block_dptr++;
				continue;
			}
			if (byte == PPP_BYTE_ESCAPE)
			{
				escape_seen= 1;
				continue;
			}
			if (byte == PPP_BYTE_FLAG)
			{
				assert((char *)block_dptr <= pkt->p_data + 
						pkt->p_maxsize);
				pkt->p_size= (char *)block_dptr - pkt->p_data -
					pkt->p_offset;
				if (pkt->p_size == 0)
					break;
				ppp_rcv_pkt= NULL;
				ppp_rcv_pack(pkt);
				break;
			}
			DPRINTF(1, ("ppp_rcv_completed: ignoring byte %d\n",
				byte));
		}
		if (ppp_rcv_pkt != NULL && size == 0)
		{
			ppp_rcv_pkt->p_size= (char *)block_dptr-
				ppp_rcv_pkt->p_data - ppp_rcv_pkt->p_offset;
		}
	}
}


DEFUN
(static void ppp_rcv_err_pack, (pkt),
	pkt_ut *pkt
)
{
	DPRINTF(2, ("ppp_rcv_err_pack: got erronous packet\n"));
	pkt_free(pkt);
}


DEFUN
(static void ppp_rcv_pack, (pkt),
	pkt_ut *pkt
)
{
	u16_t fcs;
	u8_t *dptr;
	size_t size;
	int i;
	u16_t protocol;

	size= pkt->p_size;
	if (size < 3)
	{
		DPRINTF(1, ("ppp_rcv_pack: packet too short (%d)\n", size));
		ppp_rcv_err_pack(pkt);
		return;
	}
	if (pkt->p_offset + size + PPP_RCV_BLOCK < pkt->p_maxsize)
		pkt_realloc(pkt, pkt->p_offset + size);

	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);

	DBLOCK(0,
	{
		DPRINTF(6, ("ppp_rcv_pack: received "));
		for (i= 0; i< size; i++)
			DPRINTF_CONT(6, ("%02x", dptr[i]));
		DPRINTF_CONT(6, ("\n"));
	});

	fcs= pppfcs(PPPINITFCS, dptr, size);
	if (fcs != PPPGOODFCS)
	{
		DPRINTF(1, (
			"ppp_rcv_pack: packet with checksum error (0x%04x)\n",
				fcs));
		ppp_rcv_err_pack(pkt);
		return;
	}
	size -= 2;

	/* Assume that data bytes make up for compressed header bytes */
	if (size < 4)
	{
		DPRINTF(1, ("ppp_rcv_pack: ppp packet too short\n"));
		ppp_rcv_err_pack(pkt);
		return;
	}

	if (!ppp_rcv_hc_address)
	{
		if (dptr[0] != PPP_HDR_ALLSTATIONS)
		{
			DPRINTF(1,
			("ppp_rcv_pack: strange ppp destination (0x%x)\n",
				dptr[0]));
			ppp_rcv_err_pack(pkt);
			return;
		}

		if (dptr[1] != PPP_HDR_UI)
		{
			DPRINTF(1,
			("ppp_rcv_pack: strange ppp control field (0x%x)\n",
				dptr[1]));
			ppp_rcv_err_pack(pkt);
			return;
		}
		dptr += 2;
		pkt->p_offset += 2;
		size -= 2;
	}
	else if (dptr[0] == PPP_HDR_ALLSTATIONS && dptr[1] == PPP_HDR_UI)
	{
		dptr += 2;
		pkt->p_offset += 2;
		size -= 2;
	}
	if (dptr[0] & 1)
	{
		/* Compressed protocol */
		protocol= dptr[0];
		dptr++;
		pkt->p_offset++;
		size--;
	}
	else
	{
		protocol= (dptr[0] << 8) | dptr[1];
		dptr += 2;
		pkt->p_offset += 2;
		size -= 2;
	}
	pkt->p_size= size;
	switch(protocol)
	{
	case PPP_TYPE_IP:
		ip_arrived(pkt);
		break;
	case PPP_TYPE_VJHC_COMPR:
		ip_vjhc_arr_compr(pkt);
		break;
	case PPP_TYPE_VJHC_UNCOMPR:
		ip_vjhc_arr_uncompr(pkt);
		break;
	case PPP_TYPE_IPCP:
		ipcp_arrived(pkt);
		break;
	case PPP_TYPE_LCP:
		lcp_arrived(pkt);
		break;
	default:
		lcp_reject_proto(pkt, protocol);
		break;
	}
}

/*
 * $PchHeader: /mount/hd2/minix/local/cmd/ppp/RCS/ppp_rcv.c,v 1.3 1995/05/23 08:10:43 philip Exp $
 */
