/*
ppp.c

Created:	May 24, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#define _MINIX_SOURCE
#define _POSIX_C_SOURCE	2

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/syslog.h>
#include <sys/types.h>

#include "ansi.h"
#include "clock.h"
#include "debug.h"
#include "fcs.h"
#include "ip.h"
#include "ipcp.h"
#include "lcp.h"
#include "ncp.h"
#include "pkt.h"
#include "ppp.h"
#include "sm.h"

char *prog_name;

static void fwait_mainloop ARGS(( void ));
static void trydown ARGS(( int sig ));
static void usage ARGS(( void ));
static void ppp_option ARGS(( char *option ));

DEFUN
(int main, (argc, argv),
	int argc AND
	char *argv[]
)
{
	char *ipdev_name;
	int c;

	prog_name= argv[0];

	openlog(prog_name, LOG_PID|LOG_PERROR, LOG_DAEMON);

	clck_init();
	lcp_init();
	ncp_init();
	ppp_snd_init();
	ppp_rcv_init();
	fcs_init();
	ip_init();

	while ((c= getopt(argc, argv, "o:?")) != -1)
	{
		switch(c)
		{
		case 'o':
			ppp_option(optarg);
			break;
		case '?':
			usage();
		default:
			fprintf(stderr, "%s: getopt failed: '%c'\n",
				prog_name, c);
			exit(1);
		}
	}

	if (optind >= argc)
		usage();
	ipdev_name= argv[optind++];

	if (optind != argc)
		usage();

	ip_device(ipdev_name);

	signal(SIGINT, trydown);

	sm_open(&ipcp_state);
	sm_up(&lcp_state);

	fwait_mainloop();
	for(;;)
	{
		pause();
		clck_chk_timers();
	}
}

DEFUN_VOID (static void fwait_mainloop)
{
	asio_fd_set_t fd_set;
	fwait_t fw;
	int result;

	for (;;)
	{
		ASIO_FD_ZERO(&fd_set);
		fw.fw_flags= 0;
		fw.fw_bits= fd_set.afds_bits;
		fw.fw_maxfd= ASIO_FD_SETSIZE;

		if (!ppp_rcv_inprogress)
		{
			ppp_rcv_restart();
			fw.fw_flags |= FWF_NONBLOCK;
		}
		if (ppp_rcv_inprogress)
			ASIO_FD_SET(0, ASIO_READ, &fd_set);

		if (ppp_snd_more2send && !ppp_snd_inprogress)
		{
			ppp_snd_restart();
			fw.fw_flags |= FWF_NONBLOCK;
		}
		if (ppp_snd_inprogress)
			ASIO_FD_SET(1, ASIO_WRITE, &fd_set);

		if (ip_rcv_more2recv && !ip_rcv_inprogress)
		{
			ip_rcv_restart();
			fw.fw_flags |= FWF_NONBLOCK;
		}
		if (ip_rcv_inprogress)
			ASIO_FD_SET(ip_rcv_fd, ASIO_READ, &fd_set);

		if (ip_snd_more2send && !ip_snd_inprogress)
		{
			ip_snd_restart();
			fw.fw_flags |= FWF_NONBLOCK;
		}
		if (ip_snd_inprogress)
			ASIO_FD_SET(ip_snd_fd, ASIO_WRITE, &fd_set);

		if (clck_timer_expired)
		{
			clck_chk_timers();
			fw.fw_flags |= FWF_NONBLOCK;
		}

		for (;;)
		{
			result= fwait(&fw);
			if (result == -1 && (errno == EAGAIN || 
							errno == EINTR))
			{
				break;
			}
			if (result == -1)
			{
				DPRINTF(1, ("fwait failed (%s)\n", 
							strerror(errno)));
				exit(1);
			}
			assert(result == 0);
			if (fw.fw_fd == 0 && fw.fw_operation == ASIO_READ)
			{
				ppp_rcv_completed(fw.fw_result, fw.fw_errno);
			}
			else if (fw.fw_fd == 1 && 
					fw.fw_operation == ASIO_WRITE)
			{
				ppp_snd_completed(fw.fw_result, fw.fw_errno);
			}
			else if (fw.fw_fd == ip_rcv_fd && 
					fw.fw_operation == ASIO_READ)
			{
				ip_rcv_completed(fw.fw_result, fw.fw_errno);
			}
			else if (fw.fw_fd == ip_snd_fd && 
					fw.fw_operation == ASIO_WRITE)
			{
				ip_snd_completed(fw.fw_result, fw.fw_errno);
			}
			else
			{
				DPRINTF(1, (
			"strange result from fwait: fd= %d, operation= %d\n",
					fw.fw_fd, fw.fw_operation));
				exit(1);
			}
			if (!(fw.fw_flags & FWF_MORE))
				break;
		}
	}
}


DEFUN
(static void trydown, (sig),
	int sig
)
{
	sm_close(&lcp_state);
}

DEFUN_VOID (static void usage)
{
	fprintf(stderr, "USAGE: %s [-o option]... <psip device>\n",
		prog_name);
	exit(1);
}

DEFUN
(static void ppp_option, (option),
	char *option
)
{
	int plus, min, action;
	char *modp, *emod;
	int modl;

	DPRINTF(0, ("ppp_option('%s')\n", option));
	for (plus= 0, min= 0, modp= option; ;modp++)
	{
		if (*modp == '+')
		{
			plus++;
			continue;
		}
		if (*modp == '-')
		{
			min--;
			continue;
		}
		break;
	}
	action= plus + min;
	if (plus * min != 0 || action < -2 || action > 2)
	{
		fprintf(stderr, "%s: invalid option '%s'\n",
			prog_name, option);
		exit(1);
	}
	if ((emod= strpbrk(modp, ".=")) != NULL)
		modl= emod-modp;
	else
		modl= strlen(modp);

	if (modl == strlen(MOD_IP) && strncmp(modp, MOD_IP, modl) == 0)
	{
		ipcpo_parse_arg(action, modp);
		return;
	}
	if (modl == strlen(MOD_LCP) && strncmp(modp, MOD_LCP, modl) == 0)
	{
		lcpo_parse_arg(action, modp);
		return;
	}
	if (modl != 0)
	{
		fprintf(stderr, "%s: unkown module '%.*s'\n",
			prog_name, modl, modp);
		exit(1);
	}
	else
	{
		fprintf(stderr, "%s: invalid option '%s'\n",
			prog_name, option);
		exit(1);
	}
}

/*
 * $PchHeader: /mount/hd2/minix/local/cmd/ppp/RCS/ppp.c,v 1.2 1995/04/28 07:58:53 philip Exp $
 */
