/*
optlib.c

Created:	Dec 3, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "ansi.h"
#include "debug.h"
#include "clock.h"
#include "optlib.h"
#include "pkt.h"
#include "ppp.h"
#include "sm.h"

#define CHUNK_SIZE	64

static void ol_reject_truncated_option ARGS(( sm_state_ut *state_p,
	void *data ));

DEFUN
(void ol_send_cr, (state_p),
	sm_state_ut *state_p
)
{
	pkt_ut *pkt, *optpkt;
	size_t option_size;

	DPRINTF(1, ("ol_send_cr(%s)\n", state_p->sms_name));

	assert(state_p->sms_restart_counter > 0);
	state_p->sms_restart_counter--;
	clck_set_timer(&state_p->sms_timer, clck_get_time() + 
		state_p->sms_restart_timer);

	if (state_p->sms_scr_pkt != NULL)
		pkt_free(state_p->sms_scr_pkt);

	if (!state_p->sms_options_inited)
	{
		state_p->sms_option_init();
		state_p->sms_options_inited= 1;
	}

	optpkt= state_p->sms_mk_request();
	state_p->sms_scr_pkt= optpkt;
	option_size= optpkt->p_size;
	state_p->sms_send_data(state_p->sms_type_req,
		optpkt->p_data+optpkt->p_offset, option_size);
}


DEFUN
(void ol_init, (table),
	option_ent_ut *table
)
{
	option_ent_ut *ent_p;

	DPRINTF(1, ("ol_init()\n"));

	for (ent_p= table; ent_p->oe_func != 0; ent_p++)
	{
		(void)ent_p->oe_func(OEC_INIT, NULL);
	}
}


DEFUN
(pkt_ut *ol_mk_request, (table),
	option_ent_ut *table
)
{
	pkt_ut *opt_pkt;
	option_ent_ut *ent_p;

	DPRINTF(1, ("ol_mk_request()\n"));

	opt_pkt= pkt_alloc(CHUNK_SIZE);
	for (ent_p= table; ent_p->oe_func != 0; ent_p++)
	{
		(void)ent_p->oe_func(OEC_SND_REQ, opt_pkt);
	}
	return opt_pkt;
}


DEFUN
(void ol_configure_req, (state_p, identifier, length, data, pkt),
	sm_state_ut *state_p AND
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	pkt_ut *ack, *nak, *rej;
	int res;

	DPRINTF(1, ("ol_configure_req(%s)\n", state_p->sms_name));

	if (!state_p->sms_options_inited)
	{
		state_p->sms_option_init();
		state_p->sms_options_inited= 1;
	}

	state_p->sms_last_rcvd_seq= identifier;
	length -= 4;
	data += 4;

	if (state_p->sms_sca_pkt != NULL)
	{
		pkt_free(state_p->sms_sca_pkt);
		state_p->sms_sca_pkt= NULL;
	}
	if (state_p->sms_scn_pkt != NULL)
	{
		pkt_free(state_p->sms_scn_pkt);
		state_p->sms_scn_pkt= NULL;
	}
	ack= NULL;
	nak= NULL;
	rej= NULL;
	res= state_p->sms_got_request(length, data, &ack, &nak, &rej);
	pkt_free(pkt);

	if (res == -1)
	{
		if (ack)
			pkt_free(ack);
		if (nak)
			pkt_free(nak);
		if (rej)
			pkt_free(rej);
		DPRINTF(0,
		("ol_configure_req(%s): options do not converge, closing\n",
			state_p->sms_name));
		sm_close(state_p);
		return;
	}
	if (rej != NULL)
	{
		state_p->sms_scn_pkt= rej;
		state_p->sms_scn_type= state_p->sms_type_reject;
		if (ack != NULL)
			pkt_free(ack);
		if (nak != NULL)
			pkt_free(nak);
		sm_rcr_minus(state_p);
		return;
	}
	if (nak != NULL)
	{
		state_p->sms_scn_pkt= nak;
		state_p->sms_scn_type= state_p->sms_type_nak;
		if (ack != NULL)
			pkt_free(ack);
		sm_rcr_minus(state_p);
		return;
	}
	if (ack == NULL)
		ack= pkt_alloc(1);
	state_p->sms_sca_pkt= ack;
	sm_rcr_plus(state_p);
}


DEFUN
(int ol_got_request, (state_p, table, length, data, ack_p, nak_p, rej_p),
	sm_state_ut *state_p AND
	option_ent_ut *table AND
	size_t length AND
	u8_t *data AND
	pkt_ut **ack_p AND
	pkt_ut **nak_p AND
	pkt_ut **rej_p
)
{
	option_ent_ut *ent_p;
	u8_t *dptr;
	size_t l;
	int optcnt;
	int res;

	DPRINTF(1, ("ol_got_request(%s)\n", state_p->sms_name));

	state_p->sms_ack_pkt_p= ack_p;
	state_p->sms_nak_pkt_p= nak_p;
	state_p->sms_rej_pkt_p= rej_p;

	for (ent_p= table; ent_p->oe_func != 0; ent_p++)
	{
		(void)ent_p->oe_func(OEC_RCV_REQ, NULL);
	}

	while (length > 0)
	{
		optcnt= 0;
		for (ent_p= table; length > 0; ent_p++)
		{
			if (*data != ent_p->oe_code && ent_p->oe_func != 0)
			{
				continue;
			}
			if (length == 1)
			{
				ol_reject_truncated_option(state_p, data);
				return 0;
			}
			l= data[1];
			if (l > length)
			{
				ol_reject_truncated_option(state_p, data);
				return 0;
			}
			if (ent_p->oe_func == 0)
			{
				if (optcnt == 0)
				{
					ol_reject_option(state_p, data, l);
					length -= l;
					data += l;
				}
				break;
			}
			res= ent_p->oe_func(OEC_RCV_OPTION, data);
			if (res == -1)
				return -1;
			optcnt++;
			length -= l;
			data += l;
		}
	}
	for (ent_p= table; ent_p->oe_func != 0; ent_p++)
	{
		res= ent_p->oe_func(OEC_RCV_REQ_END, NULL);
		if (res == -1)
			return -1;
	}

	assert(length == 0);
	return 0;
}


DEFUN
(void ol_ack_option, (state_p, data, lenght),
	sm_state_ut *state_p AND
	void *data AND
	size_t length
)
{
	int i;

	DPRINTF(1, ("ol_ack_option(%s): ack option type %d, data: ",
		state_p->sms_name, *(u8_t *)data ));
	for (i= 0; i < length; i++)
		DPRINTF_CONT(1, ( "%02x", ((u8_t *)data)[i] ));
	DPRINTF_CONT(1, ( "\n" ));
	if (*(state_p->sms_ack_pkt_p) == NULL)
		*(state_p->sms_ack_pkt_p)= pkt_alloc(CHUNK_SIZE);
	pkt_add(*(state_p->sms_ack_pkt_p), data, length, CHUNK_SIZE);
}


DEFUN
(void ol_nak_option, (state_p, data, lenght),
	sm_state_ut *state_p AND
	void *data AND
	size_t length
)
{
	int i;

	DPRINTF(1, ("ol_nak_option(%s): nak option type %d, data: ",
		state_p->sms_name, *(u8_t *)data ));
	for (i= 0; i < length; i++)
		DPRINTF_CONT(1, ( "%02x", ((u8_t *)data)[i] ));
	DPRINTF_CONT(1, ( "\n" ));
	if (*(state_p->sms_nak_pkt_p) == NULL)
		*(state_p->sms_nak_pkt_p)= pkt_alloc(CHUNK_SIZE);
	pkt_add(*(state_p->sms_nak_pkt_p), data, length, CHUNK_SIZE);
}


DEFUN
(void ol_reject_option, (state_p, data, lenght),
	sm_state_ut *state_p AND
	void *data AND
	size_t length
)
{
	int i;

	DPRINTF(1, ("ol_reject_option(%s): reject option type %d, data: ",
		state_p->sms_name, *(u8_t *)data ));
	for (i= 0; i < length; i++)
		DPRINTF_CONT(1, ( "%02x", ((u8_t *)data)[i] ));
	DPRINTF_CONT(1, ( "\n" ));
	if (*state_p->sms_rej_pkt_p == NULL)
		*state_p->sms_rej_pkt_p= pkt_alloc(CHUNK_SIZE);
	pkt_add(*state_p->sms_rej_pkt_p, data, length, CHUNK_SIZE);
}


DEFUN
(static void ol_reject_truncated_option, (state_p, data),
	sm_state_ut *state_p AND
	void *data
)
{
	u8_t bytes[2];

	DPRINTF(1, ("ol_reject_truncated_option(%s): option %d\n",
		state_p->sms_name, *(u8_t *)data));
	bytes[0]= *(u8_t *)data;
	bytes[1]= 2;
	ol_reject_option(state_p, bytes, 2);
}

/* XXX */

DEFUN
(void ol_configure_ack, (state_p, identifier, length, data, pkt),
	sm_state_ut *state_p AND
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	if (identifier != state_p->sms_last_snd_seq)
	{
		DPRINTF(1, ( 
"ol_configure_ack: got packet with invalid sequence number (%d != %d)\n",
				identifier, state_p->sms_last_snd_seq));
		pkt_free(pkt);
		return;
	}

	if (length != 4+state_p->sms_scr_pkt->p_size ||
		(state_p->sms_scr_pkt->p_size != 0 && 
		memcmp(data+4, state_p->sms_scr_pkt->p_data +
		state_p->sms_scr_pkt->p_offset, length-4) != 0))
	{
		DPRINTF(1, ( 
		"ol_configure_ack: packet didn't ack the right options\n" ));
		pkt_free(pkt);
		return;
	}
	state_p->sms_got_ack();
	pkt_free(pkt);
	sm_rca(state_p);
}


DEFUN
(void ol_configure_nak, (state_p, identifier, length, data, pkt),
	sm_state_ut *state_p AND
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	int res;

	if (identifier != state_p->sms_last_snd_seq)
	{
		DPRINTF(1, (
"ol_configure_reject: got packet with invalid sequence number (%d != %d)\n",
				identifier, state_p->sms_last_snd_seq));
		pkt_free(pkt);
		return;
	}

	length -= 4;
	data += 4;

	res= state_p->sms_got_nak(length, data, state_p->sms_scr_pkt);
	pkt_free(pkt);

	if (res == -1)
	{
		DPRINTF(0,
		("ol_configure_nak: options do not converge, closing\n"));
		sm_close(state_p);
	}
	else
		sm_rcn(state_p);
}


DEFUN
(void ol_configure_reject, (state_p, identifier, length, data, pkt),
	sm_state_ut *state_p AND
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	int res;

	if (identifier != state_p->sms_last_snd_seq)
	{
		DPRINTF(1, (
"ol_configure_reject: got packet with invalid sequence number (%d != %d)\n",
				identifier, state_p->sms_last_snd_seq));
		pkt_free(pkt);
		return;
	}

	length -= 4;
	data += 4;

	res= state_p->sms_got_reject(length, data, state_p->sms_scr_pkt);
	pkt_free(pkt);

	if (res == -1)
	{
		DPRINTF(0,
		("ol_configure_reject: options do not converge, closing\n"));
		sm_close(state_p);
	}
	else
		sm_rcn(state_p);
}


DEFUN
(void ol_sca, (state_p),
	sm_state_ut *state_p
)
{
	pkt_ut *pkt;

	DPRINTF(1, ("ol_sca()\n"));

	state_p->sms_sending_ack();

	pkt= state_p->sms_sca_pkt;

	state_p->sms_send_data(state_p->sms_type_ack,
		pkt->p_data + pkt->p_offset, pkt->p_size);
}


DEFUN
(void ol_scn, (state_p),
	sm_state_ut *state_p
)
{
	pkt_ut *pkt;

	DPRINTF(1, ("ol_scn()\n"));

	pkt= state_p->sms_scn_pkt;

	state_p->sms_send_data(state_p->sms_scn_type,
		pkt->p_data + pkt->p_offset, pkt->p_size);
}


DEFUN
(void ol_sending_ack, (table),
	option_ent_ut *table
)
{
	option_ent_ut *ent_p;

	for (ent_p= table; ent_p->oe_func != 0; ent_p++)
	{
		(void)ent_p->oe_func(OEC_SENT_ACK, NULL);
	}
}


DEFUN
(void ol_got_ack, (table),
	option_ent_ut *table
)
{
	option_ent_ut *ent_p;

	for (ent_p= table; ent_p->oe_func != 0; ent_p++)
	{
		(void)ent_p->oe_func(OEC_GOT_ACK, NULL);
	}
}


DEFUN
(int ol_got_nak, (table, length, data, req_pkt),
	option_ent_ut *table AND
	size_t length AND
	u8_t *data AND
	pkt_ut *req_pkt
)
{
	option_ent_ut *ent_p;
	u8_t *dptr;
	size_t l;
	int optcnt;
	int res;

	while (length > 0)
	{
		optcnt= 0;
		for (ent_p= table; length > 0; ent_p++)
		{
			if (*data != ent_p->oe_code && ent_p->oe_func != 0)
			{
				continue;
			}
			if (length == 1)
			{
				DPRINTF(0,
			("ol_got_nak: illegal option length: 1\n"));
				return 0;
			}
			l= data[1];
			if (l > length)
			{
				DPRINTF(0,
			("ol_got_nak: truncated option: %d > %d\n",
					l, length));
				return 0;
			}
			if (ent_p->oe_func == 0)
			{
				if (optcnt == 0)
				{
					DPRINTF(0,
			("ol_got_nak: nak for unknown option: %d\n",
						*data));
					length -= l;
					data += l;
				}
				break;
			}
			res= ent_p->oe_func(OEC_GOT_NAK, data);
			if (res == -1)
				return -1;
			optcnt++;
			length -= l;
			data += l;
		}
	}
	assert(length == 0);
	return 0;
}


DEFUN
(int ol_got_reject, (table, length, data, req_pkt),
	option_ent_ut *table AND
	size_t length AND
	u8_t *data AND
	pkt_ut *req_pkt
)
{
	u8_t *rdata;
	size_t rlength, rl;
	int i, res;
	int rop;
	option_ent_ut *ent_p;

	rlength= req_pkt->p_size;
	rdata= (u8_t *)(req_pkt->p_data + req_pkt->p_offset);
	ent_p= table;

	while (length > 0 && rlength > 0)
	{
		rop= rdata[0];
		rl= rdata[1];
		assert(rl >= 2 && rl <= rlength);

		if (rop != data[0])
		{
			/* This option is not rejected */
			rlength -= rl;
			rdata += rl;
			continue;
		}

		/* Option should be a copy of the original option in the
		 * configuration request.
		 */
		if (length < rl ||memcmp(data, rdata, rl) != 0)
			break;

		for (; ent_p->oe_code != rop && ent_p->oe_func != 0; ent_p++)
		{
			;	/* Do nothing */
		}
		if (ent_p->oe_func == 0)
			break;
		res= ent_p->oe_func(OEC_GOT_REJ, NULL);
		if (res == -1)
			return -1;
		rlength -= rl;
		rdata += rl;
		length -= rl;
		data += rl;
	}
	if (length != 0)
	{
		DPRINTF(1, ( "ipcpo_got_rej: bad data, req = " ));
		for (i= 0; i<rlength; i++)
			DPRINTF_CONT(1, ( "%02x", rdata[i] ));
		DPRINTF_CONT(1, ( " repl= " ));
		for (i= 0; i<length; i++)
			DPRINTF_CONT(1, ( "%02x", data[i] ));
		DPRINTF_CONT(1, ( "\n" ));
	}
	return 0;
}


DEFUN
(int ol_on_off_option, (state_p, name, type, snd_p, rcv_p, action, data),
	sm_state_ut *state_p AND
	char *name AND
	int type AND
	opt_on_off_ut *snd_p AND
	opt_on_off_ut *rcv_p AND
	int action AND
	void *data
)
{
	pkt_ut *pkt;
	u8_t bytes[2];
	u8_t *option;

	DPRINTF(1, ("ol_on_off_option(%s): %d, %d\n", name,
		snd_p->curr_action, rcv_p->curr_action));

	switch(action)
	{
	case OEC_INIT:
		DPRINTF(1, ("ol_on_off_option(%s, init)\n", name));
		snd_p->curr_action= snd_p->action;
		rcv_p->curr_action= rcv_p->action;
		return 0;
	case OEC_SND_REQ:
		DPRINTF(1, ("ol_on_off_option(%s, snd_req)\n", name));
		snd_p->curr_opt= NULL;
		pkt= data;
		action= snd_p->curr_action;
		if (action == PPP_OPT_FORCED_OFF || action == PPP_OPT_OFF)
			return 0;
		assert (action == PPP_OPT_FORCED_ON || action == PPP_OPT_ON);
		bytes[0]= type;
		bytes[1]= 2;
		pkt_add(pkt, bytes, 2, CHUNK_SIZE);
		return 0;
	case OEC_RCV_REQ:
		DPRINTF(1, ("ol_on_off_option(%s, rcv_req)\n", name));
		rcv_p->curr_opt= NULL;
		return 0;
	case OEC_RCV_OPTION:
		DPRINTF(1, ("ol_on_off_option(%s, rcv_option)\n", name));
		option= data;
		action= rcv_p->curr_action;
		if (action == PPP_OPT_FORCED_OFF)
		{
			ol_reject_option(state_p, option, option[1]);
			return 0;
		}
		if (action == PPP_OPT_OFF)
		{
			action= rcv_p->curr_action= PPP_OPT_ON;
		}
		if (rcv_p->curr_opt != NULL)
		{
			if (memcmp(option, rcv_p->curr_opt, option[1]) != 0)
			{
				DPRINTF(0, ("ol_on_off_option(%s):\n", name));
				DPRINTF(0,
		("\trejecting duplicate option with different values\n"));
				ol_reject_option(state_p, option, option[1]);
				return 0;
			}
			DPRINTF(1,
			("ol_on_off_option(%s): got duplicate option\n",
				name));
		}
		else
			rcv_p->curr_opt= option;

		/* Try the ACK case first */
		if (option[1] == 2)
		{
			ol_ack_option(state_p, option, option[1]);
			return 0;
		}

		/* Check the retry counter */
		if (rcv_p->nak_retry > 0)
			rcv_p->nak_retry--;
		else
		{
			DPRINTF(0, ("ol_on_off_option(%s): value conflict\n",
				name));
			if (action == PPP_OPT_FORCED_ON)
				return -1;
			assert(action == PPP_OPT_ON);
			action= rcv_p->curr_action= PPP_OPT_OFF;
			ol_reject_option(state_p, option, option[1]);
			return 0;
		}

		/* Standard NAK */
		bytes[0]= type;
		bytes[1]= 2;
		if (option[1] != 2)
		{
			DPRINTF(1, ("ol_on_off_option(%s): strange length %d\n",
				name, option[1]));
		}
		ol_nak_option(state_p, bytes, bytes[1]);
		return 0;
	case OEC_RCV_REQ_END:
		DPRINTF(1, ("ol_on_off_option(%s, rcv_req_end)\n", name));
		if (rcv_p->curr_opt != NULL)
		{
			/* Option received and processed. */
			return 0;
		}
		action= rcv_p->curr_action;
		if (action == PPP_OPT_FORCED_OFF || action == PPP_OPT_OFF)
			return 0;	/* We do not care */

		assert (action == PPP_OPT_FORCED_ON || action == PPP_OPT_ON);

		/* Standard NAK */
		bytes[0]= type;
		bytes[1]= 2;

		if (rcv_p->nak_retry > 0)
		{
			rcv_p->nak_retry--;
			ol_nak_option(state_p, bytes, bytes[1]);
			DPRINTF(1,
			("ol_on_off_option(%s, rcv_req_end): %d, %d\n",
				name, snd_p->curr_action, rcv_p->curr_action));
			return 0;
		}
		if (action == PPP_OPT_ON)
		{
			rcv_p->curr_action= PPP_OPT_OFF;
			DPRINTF(1,
			("ol_on_off_option(%s, rcv_req_end): %d, %d\n",
				name, snd_p->curr_action, rcv_p->curr_action));
			return 0;
		}
		DPRINTF(0, ("ol_on_off_option(%s): on/off conflict\n", name));
		return -1;
	case OEC_GOT_NAK:
		DPRINTF(1, ("ol_on_off_option(%s, got_nak)\n", name));
		option= data;
		action= snd_p->curr_action;
		if (action == PPP_OPT_FORCED_OFF)
		{
			return 0;
		}
		if (action == PPP_OPT_OFF)
		{
			action= snd_p->curr_action= PPP_OPT_ON;
		}
		if (snd_p->curr_opt == NULL)
			snd_p->curr_opt= option;

		/* Try the acceptable case first */
		if (option[1] == 2)
			return 0;

		if (option == snd_p->curr_opt)
		{
			/* Check the retry counter */
			if (snd_p->nak_retry > 0)
				snd_p->nak_retry--;
			else
			{
				DPRINTF(0,
				("ol_on_off_option(%s): value conflict\n",
					name));
				if (action == PPP_OPT_FORCED_ON)
					return -1;
				assert(action == PPP_OPT_ON);
				action= snd_p->curr_action= PPP_OPT_OFF;
				return 0;
			}

			if (option[1] != 2)
			{
				DPRINTF(1,
				("ol_on_off_option(%s): strange length %d\n",
					name, option[1]));
				return 0;
			}
		}
		return 0;
	case OEC_GOT_REJ:
		DPRINTF(1, ("ol_on_off_option(%s, got_rej)\n", name));
		if (snd_p->curr_action == PPP_OPT_FORCED_ON)
		{
			if (snd_p->rej_retry > 0)
			{
				snd_p->rej_retry--;
				return 0;
			}
			DPRINTF(0, ("ol_on_off_option(%s): on/off conflict\n",
				name));
			return -1;
		}
		assert(snd_p->curr_action == PPP_OPT_ON);
		snd_p->curr_action= PPP_OPT_OFF;
		return 0;
	default:
		DPRINTF(0, ("ol_on_off_option(%s): unknown action %d\n",
			name, action));
		abort();
	}
}

/*
 * $PchHeader: /mount/hd2/minix/local/cmd/ppp/RCS/optlib.c,v 1.1 1995/02/14 07:52:41 philip Exp $
 */
