/*
lcp.c

Implementation of the link control protocol.

Created:	May 19, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#include <sys/types.h>
#include <assert.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "ansi.h"
#include "clock.h"
#include "debug.h"
#include "lcp.h"
#include "ncp.h"
#include "optlib.h"
#include "pkt.h"
#include "ppp.h"
#include "sm.h"

sm_state_ut lcp_state;
sm_action_ut lcp_action;

#define LCP_PKT_Q_SIZE	64

static pkt_ut *lcp_pkt_q[LCP_PKT_Q_SIZE];
static pkt_ut **lcp_pkt_nq_ptr;

static ppp_snd_callback_ut lcp_snd_callback;
static pkt_ut *lcp_code_rej_pkt;

static int lcp_debug_option= 0;

static void lcp_snd_restart ARGS(( struct ppp_snd_callback *cb, int arg ));
static void lcp_send_data ARGS(( int type, void *data, size_t size ));

static void lcp_action_tlu ARGS(( void ));
static void lcp_action_tld ARGS(( void ));
static void lcp_action_tls ARGS(( void ));
static void lcp_action_tlf ARGS(( void ));

static void lcp_action_irc ARGS(( int retry_cnt ));
static void lcp_action_zrc ARGS(( void ));

static void lcp_action_scr ARGS(( void ));

static void lcp_action_sca ARGS(( void ));
static void lcp_action_scn ARGS(( void ));

static void lcp_action_str ARGS(( void ));
static void lcp_action_sta ARGS(( void ));

static void lcp_action_scj ARGS(( void ));

static void lcp_configure_req ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void lcp_configure_ack ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void lcp_configure_nak ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void lcp_configure_reject ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void lcp_terminate_req ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void lcp_terminate_ack ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void lcp_code_reject ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));
static void lcp_protocol_reject ARGS(( int identifier, size_t length,
						u8_t *data, pkt_ut *pkt ));

DEFUN_VOID (void lcp_init)
{
	int i;

	DPRINTF(1, ("lcp_init()\n"));

	lcp_action.sma_tlu= lcp_action_tlu;
	lcp_action.sma_tld= lcp_action_tld;
	lcp_action.sma_tls= lcp_action_tls;
	lcp_action.sma_tlf= lcp_action_tlf;
	lcp_action.sma_irc= lcp_action_irc;
	lcp_action.sma_zrc= lcp_action_zrc;
	lcp_action.sma_scr= lcp_action_scr;
	lcp_action.sma_sca= lcp_action_sca;
	lcp_action.sma_scn= lcp_action_scn;
	lcp_action.sma_str= lcp_action_str;
	lcp_action.sma_sta= lcp_action_sta;
	lcp_action.sma_scj= lcp_action_scj;

	sm_init(&lcp_state, "lcp", &lcp_action);
	lcp_state.sms_max_configure= LCP_DFLT_MAX_CONFIGURE;
	lcp_state.sms_max_terminate= LCP_DFLT_MAX_TERMINATE;
	lcp_state.sms_restart_timer= LCP_DFLT_RESTART_TIMER;
	lcp_state.sms_type_req= LCP_TYPE_CONFIGURE_REQ;
	lcp_state.sms_type_ack= LCP_TYPE_CONFIGURE_ACK;
	lcp_state.sms_type_nak= LCP_TYPE_CONFIGURE_NAK;
	lcp_state.sms_type_reject= LCP_TYPE_CONFIGURE_REJECT;
	lcp_state.sms_last_snd_seq= 0;
	lcp_state.sms_option_init= lcpo_init;
	lcp_state.sms_mk_request= lcpo_mk_request;
	lcp_state.sms_send_data= lcp_send_data;
	lcp_state.sms_got_request= lcpo_got_request;
	lcp_state.sms_sending_ack= lcpo_sending_ack;
	lcp_state.sms_got_ack= lcpo_got_ack;
	lcp_state.sms_got_nak= lcpo_got_nak;
	lcp_state.sms_got_reject= lcpo_got_reject;

	for (i= 0; i<LCP_PKT_Q_SIZE; i++)
		lcp_pkt_q[i]= NULL;
	lcp_pkt_nq_ptr= lcp_pkt_q;

	ppp_snd_init_callback(&lcp_snd_callback, lcp_snd_restart, 0);
}


DEFUN_VOID (static void lcp_action_tlu)
{
	DPRINTF(1, ("lcp_action_tlu()\n"));
	ncp_up();
}


DEFUN_VOID (static void lcp_action_tld)
{
	DPRINTF(1, ("lcp_action_tld()\n"));
	ncp_down();
}


DEFUN_VOID (static void lcp_action_tls)
{
	DPRINTF(1, ("lcp_action_tls()\n"));
	DPRINTF(1, ("lcp_action_tls: should call physical layer\n"));
}


DEFUN_VOID (static void lcp_action_tlf)
{
	DPRINTF(1, ("lcp_action_tlf()\n"));
	DPRINTF(1, ("lcp_action_tlf: should call physical layer\n"));
	exit(2);
}


DEFUN
(static void lcp_action_irc, (retry_cnt),
	int retry_cnt
)
{
	DPRINTF(1, ("lcp_action_irc()\n"));
	lcp_state.sms_restart_counter= retry_cnt;
}


DEFUN_VOID (static void lcp_action_zrc)
{
	DPRINTF(1, ("lcp_action_zrc()\n"));
	lcp_state.sms_restart_counter= 0;
}


DEFUN_VOID (static void lcp_action_scr)
{
	DPRINTF(1, ("lcp_action_scr()\n"));

	ol_send_cr(&lcp_state);
}


DEFUN_VOID (static void lcp_action_sca)
{
	ol_sca(&lcp_state);
}


DEFUN_VOID (static void lcp_action_scn)
{
	ol_scn(&lcp_state);
}


DEFUN_VOID (static void lcp_action_str)
{
	pkt_ut *pkt;
	u8_t *dptr;
	int error;

	DPRINTF(1, ("lcp_action_str()\n"));

	assert(lcp_state.sms_restart_counter > 0);
	lcp_state.sms_restart_counter--;
	clck_set_timer(&lcp_state.sms_timer, clck_get_time() + 
						lcp_state.sms_restart_timer);

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	dptr[0]= LCP_TYPE_TERMINATE_REQ;
	dptr[1]= ++lcp_state.sms_last_snd_seq;
	dptr[2]= 0;
	dptr[3]= 4;
	ppp_snd_sethdr(pkt, PPP_TYPE_LCP);
	error= ppp_snd(pkt, &lcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*lcp_pkt_nq_ptr != NULL)
		pkt_free(*lcp_pkt_nq_ptr);
	*lcp_pkt_nq_ptr= pkt;
	lcp_pkt_nq_ptr++;
	if (lcp_pkt_nq_ptr == &lcp_pkt_q[LCP_PKT_Q_SIZE])
		lcp_pkt_nq_ptr= lcp_pkt_q;
}


DEFUN_VOID (static void lcp_action_sta)
{
	pkt_ut *pkt;
	u8_t *dptr;
	int error;

	DPRINTF(1, ("lcp_action_sta()\n"));

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	dptr[0]= LCP_TYPE_TERMINATE_ACK;
	dptr[1]= lcp_state.sms_last_rcvd_seq;
	dptr[2]= 0;
	dptr[3]= 4;
	ppp_snd_sethdr(pkt, PPP_TYPE_LCP);
	error= ppp_snd(pkt, &lcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*lcp_pkt_nq_ptr != NULL)
		pkt_free(*lcp_pkt_nq_ptr);
	*lcp_pkt_nq_ptr= pkt;
	lcp_pkt_nq_ptr++;
	if (lcp_pkt_nq_ptr == &lcp_pkt_q[LCP_PKT_Q_SIZE])
		lcp_pkt_nq_ptr= lcp_pkt_q;
}


DEFUN_VOID (static void lcp_action_scj)
{
	size_t length, sendlength;
	u8_t *dptr;
	int error;
	pkt_ut *pkt, *rej_pkt;

	rej_pkt= lcp_code_rej_pkt;
	length= rej_pkt->p_size;

	DPRINTF(1, ("lcp_action_scj\n"));

	if (4 + length > ppp_snd_maxsize)
		sendlength= ppp_snd_maxsize-4;
	else
		sendlength= length;

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4+sendlength);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4+sendlength;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	dptr[0]= LCP_TYPE_CODE_REJECT;
	dptr[1]= ++lcp_state.sms_last_snd_seq;
	dptr[2]= (4+sendlength) >> 8;
	dptr[3]= 4+sendlength;
	memcpy(dptr+4, rej_pkt->p_data + rej_pkt->p_offset, sendlength);

	pkt_free(lcp_code_rej_pkt);
	lcp_code_rej_pkt= NULL;

	ppp_snd_sethdr(pkt, PPP_TYPE_LCP);
	error= ppp_snd(pkt, &lcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*lcp_pkt_nq_ptr != NULL)
		pkt_free(*lcp_pkt_nq_ptr);
	*lcp_pkt_nq_ptr= pkt;
	lcp_pkt_nq_ptr++;
	if (lcp_pkt_nq_ptr == &lcp_pkt_q[LCP_PKT_Q_SIZE])
		lcp_pkt_nq_ptr= lcp_pkt_q;
}


DEFUN
(void lcp_reject_proto, (pkt, protocol),
	pkt_ut *pkt AND
	u16_t protocol
)
{
	size_t length, sendlength;
	u8_t *dptr;
	int error;
	pkt_ut *new_pkt;

	DPRINTF(1, ("lcp_reject_proto: rejecting protocol (0x%04x)\n",
		protocol));

	length= pkt->p_size;

	if (6 + length > ppp_snd_maxsize)
		sendlength= ppp_snd_maxsize-6;
	else
		sendlength= length;

	new_pkt= pkt_alloc(PPP_MAX_HDR_SIZE+6+sendlength);
	new_pkt->p_offset= PPP_MAX_HDR_SIZE;
	new_pkt->p_size= 6+sendlength;
	assert(new_pkt->p_offset + new_pkt->p_size <= new_pkt->p_maxsize);
	dptr= (u8_t *)(new_pkt->p_data + new_pkt->p_offset);
DPRINTF(1, ( "lcp_reject_proto: dptr= %p\n", dptr ));
DPRINTF(1, ( "lcp_reject_proto: new_pkt->p_data= %p\n", new_pkt->p_data ));
	dptr[0]= LCP_TYPE_PROTOCOL_REJECT;
	dptr[1]= ++lcp_state.sms_last_snd_seq;
	dptr[2]= (6+sendlength) >> 8;
	dptr[3]= 6+sendlength;
	dptr[4]= protocol >> 8;
	dptr[5]= protocol;
DPRINTF(1, ( "lcp_reject_proto: new_pkt->p_data= %p\n", new_pkt->p_data ));
	if (sendlength != 0)
		memcpy(dptr+6, pkt->p_data + pkt->p_offset, sendlength);
DPRINTF(1, ( "lcp_reject_proto: new_pkt->p_data= %p\n", new_pkt->p_data ));

	pkt_free(pkt);

DPRINTF(1, ( "lcp_reject_proto: new_pkt->p_data= %p\n", new_pkt->p_data ));
	ppp_snd_sethdr(new_pkt, PPP_TYPE_LCP);
	error= ppp_snd(new_pkt, &lcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*lcp_pkt_nq_ptr != NULL)
		pkt_free(*lcp_pkt_nq_ptr);
	*lcp_pkt_nq_ptr= new_pkt;
	lcp_pkt_nq_ptr++;
	if (lcp_pkt_nq_ptr == &lcp_pkt_q[LCP_PKT_Q_SIZE])
		lcp_pkt_nq_ptr= lcp_pkt_q;
}


DEFUN
(void lcp_arrived, (pkt),
	pkt_ut *pkt
)
{
	int code, identifier;
	size_t length;
	u8_t *dptr;

	if (pkt->p_size < 4)
	{
		DPRINTF(1, ("lcp_arrived: packet with invalid length (%d)\n",
			pkt->p_size));
		pkt_free(pkt);
		return;
	}
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	code= dptr[0];
	identifier= dptr[1];
	length= (dptr[2] << 8) | dptr[3];
	if (length < 4 || length > pkt->p_size)
	{
		DPRINTF(1, (
			"lcp_arrived: packet with invalid length field (%d)\n",
				length));
		pkt_free(pkt);
		return;
	}
	pkt->p_size= length;

	switch(code)
	{
	case LCP_TYPE_CONFIGURE_REQ:
		lcp_configure_req(identifier, length, dptr, pkt);
		break;
	case LCP_TYPE_CONFIGURE_ACK:
		lcp_configure_ack(identifier, length, dptr, pkt);
		break;
	case LCP_TYPE_CONFIGURE_NAK:
		lcp_configure_nak(identifier, length, dptr, pkt);
		break;
	case LCP_TYPE_CONFIGURE_REJECT:
		lcp_configure_reject(identifier, length, dptr, pkt);
		break;
	case LCP_TYPE_TERMINATE_REQ:
		lcp_terminate_req(identifier, length, dptr, pkt);
		break;
	case LCP_TYPE_TERMINATE_ACK:
		lcp_terminate_ack(identifier, length, dptr, pkt);
		break;
	case LCP_TYPE_CODE_REJECT:
		lcp_code_reject(identifier, length, dptr, pkt);
		break;
	case LCP_TYPE_PROTOCOL_REJECT:
		lcp_protocol_reject(identifier, length, dptr, pkt);
		break;
	default:
		if (lcp_code_rej_pkt)
		{
			DPRINTF(0,
	("lcp_arrived: warning unprocessed packet with unkown code"));
			pkt_free(lcp_code_rej_pkt);
			lcp_code_rej_pkt= NULL;
		}
		lcp_code_rej_pkt= pkt;
		sm_ruc(&lcp_state);
		break;
	}
}


DEFUN
(static void lcp_configure_req, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	ol_configure_req(&lcp_state, identifier, length, data, pkt);
}


DEFUN
(static void lcp_configure_ack, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	ol_configure_ack(&lcp_state, identifier, length, data, pkt);
}


DEFUN
(static void lcp_configure_nak, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	ol_configure_nak(&lcp_state, identifier, length, data, pkt);
}


DEFUN
(static void lcp_configure_reject, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	ol_configure_reject(&lcp_state, identifier, length, data, pkt);
}


DEFUN
(static void lcp_terminate_req, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	DPRINTF(1, ("lcp_terminate_req\n"));
	lcp_state.sms_last_rcvd_seq= identifier;

	pkt_free(pkt);
	sm_rtr(&lcp_state);
}


DEFUN
(static void lcp_terminate_ack, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	DPRINTF(1, ("lcp_terminate_ack\n"));
	if (identifier != lcp_state.sms_last_snd_seq)
	{
		DPRINTF(1, (
"lcp_terminate_ack: got packet with invalid sequence number (%d != %d)\n",
				identifier, lcp_state.sms_last_snd_seq));
		pkt_free(pkt);
		return;
	}

	pkt_free(pkt);
	sm_rta(&lcp_state);
}


DEFUN
(static void lcp_code_reject, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	int i;

	DPRINTF(1, ("lcp_code_reject: length= %d, data=", length-4));
	for (i= 4; i<length && i<20; i++)
		DPRINTF_CONT(1, (" %02x", data[i]));
	if (i<length)
		DPRINTF_CONT(1, (" ..."));
	DPRINTF_CONT(1, ("\n"));
	pkt_free(pkt);
	sm_rxj_minus(&lcp_state);
}


DEFUN
(static void lcp_protocol_reject, (identifier, length, data, pkt),
	int identifier AND
	size_t length AND
	u8_t *data AND
	pkt_ut *pkt
)
{
	int i;
	u16_t protocol;

	if (length < 6)
	{
		DPRINTF(1, ("lcp_protocol_reject: packet to short (%d)\n",
			length));
		pkt_free(pkt);
		return;
	}
	protocol= (data[4] << 8) | data[5];

	DPRINTF(1, ("lcp_protocol_reject: protocol= 0x%04x, length= %d, data=", 
		protocol, length-6));
	for (i= 6; i<length && i<22; i++)
		DPRINTF_CONT(1, (" %02x", data[i]));
	if (i<length)
		DPRINTF_CONT(1, (" ..."));
	DPRINTF_CONT(1, ("\n"));
	pkt_free(pkt);
	if (protocol >= 0x8000 && protocol < 0xc000)
	{
		/* One of our NCPs is not supported by the remote site, tell
		 * ncp.
		 */
		ncp_protocol_reject(protocol);
		sm_rxj_plus(&lcp_state);
		return;
	}

	sm_rxj_minus(&lcp_state);
}


DEFUN
(static void lcp_snd_restart, (cb, arg),
	ppp_snd_callback_ut *cb AND
	int arg

)
{
	int error, i;

	assert(cb == &lcp_snd_callback);
	assert(arg == 0);

	for (i= 0; i<LCP_PKT_Q_SIZE; i++)
	{
		if (*lcp_pkt_nq_ptr != NULL)
		{
			DPRINTF(0, ("lcp_snd_restart: sending packet\n"));
			error= ppp_snd(*lcp_pkt_nq_ptr, &lcp_snd_callback);
			if (error == EAGAIN)
				return;
		}
		*lcp_pkt_nq_ptr= NULL;
		lcp_pkt_nq_ptr++;
		if (lcp_pkt_nq_ptr == &lcp_pkt_q[LCP_PKT_Q_SIZE])
			lcp_pkt_nq_ptr= lcp_pkt_q;
	}
}

DEFUN
(static void lcp_send_data, (type, data, size),
	int type AND
	void *data AND
	size_t size
)
{
	pkt_ut *pkt;
	int error;
	u8_t *dptr;

	pkt= pkt_alloc(PPP_MAX_HDR_SIZE+4+size);
	pkt->p_offset= PPP_MAX_HDR_SIZE;
	pkt->p_size= 4+size;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	dptr= (u8_t *)(pkt->p_data + pkt->p_offset);
	switch (type)
	{
	case LCP_TYPE_CONFIGURE_REQ:
		dptr[0]= type;
		dptr[1]= ++lcp_state.sms_last_snd_seq;
		dptr[2]= (4 + size) >> 8;
		dptr[3]= 4+size;
		if (size != 0)
			memcpy(dptr+4, data, size);
		break;
	case LCP_TYPE_CONFIGURE_ACK:
	case LCP_TYPE_CONFIGURE_NAK:
	case LCP_TYPE_CONFIGURE_REJECT:
		dptr[0]= type;
		dptr[1]= lcp_state.sms_last_rcvd_seq;
		dptr[2]= (4 + size) >> 8;
		dptr[3]= 4+size;
		if (size != 0)
			memcpy(dptr+4, data, size);
		break;
	default:
		DPRINTF(0, ("lcp_send_data: illegal type %d\n", type));
		abort();
	}

	ppp_snd_sethdr(pkt, PPP_TYPE_LCP);
	error= ppp_snd(pkt, &lcp_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	if (*lcp_pkt_nq_ptr != NULL)
		pkt_free(*lcp_pkt_nq_ptr);
	*lcp_pkt_nq_ptr= pkt;
	lcp_pkt_nq_ptr++;
	if (lcp_pkt_nq_ptr == &lcp_pkt_q[LCP_PKT_Q_SIZE])
		lcp_pkt_nq_ptr= lcp_pkt_q;
}

/*
 * $PchHeader: /mount/hd2/minix/local/cmd/ppp/RCS/lcp.c,v 1.3 1995/05/23 08:09:11 philip Exp $
 */
