/*
ip.c

send and receive IP packets

Created:	June 3, 1993 by Philip Homburg <philip@cs.vu.nl>
*/

#define _MINIX_SOURCE

#include <sys/types.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <net/gen/in.h>
#include <net/gen/ip_hdr.h>

#include "ansi.h"
#include "debug.h"
#include "ip.h"
#include "pkt.h"
#include "ppp.h"
#include "vjhc.h"

#define IP_SND_PKTS_NR	10	/* Buffer this many arrived ip packets */

int ip_rcv_more2recv;
int ip_rcv_inprogress;
int ip_rcv_fd;
int ip_snd_more2send;
int ip_snd_inprogress;
int ip_snd_fd;

static pkt_ut *ip_rcv_pkt;
static pkt_ut *ip_snd_pkt;
static pkt_ut *ip_snd_pkts[IP_SND_PKTS_NR];
static char *ip_devname;
static ppp_snd_callback_ut ip_snd_callback;
static pkt_ut *ip_pppsnd_pkt;

static void ip_pppsnd_restart ARGS(( struct ppp_snd_callback *cb, int arg ));

DEFUN_VOID (void ip_init)
{
	int i, v;
	int result;

	DPRINTF(1, ("ip_init()\n"));

	ip_rcv_more2recv= 0;
	ip_rcv_inprogress= 0;
	ip_snd_more2send= 0;
	ip_snd_inprogress= 0;

	ip_vjhc_init();

	ip_rcv_pkt= NULL;
	ip_snd_pkt= NULL;
	for (i= 0; i<IP_SND_PKTS_NR; i++)
		ip_snd_pkts[i]= NULL;

	ppp_snd_init_callback(&ip_snd_callback, ip_pppsnd_restart, 0);
	ip_pppsnd_pkt= NULL;
}

DEFUN
(void ip_device, (ipdev_name),
	char *ipdev_name
)
{
	int v;
	int result;

	DPRINTF(1, ("ip_device('%s')\n", ipdev_name));

	ip_devname= ipdev_name;

	ip_rcv_fd= open(ip_devname, O_RDWR);
	if (ip_rcv_fd == -1)
	{
		fatal("ip_init: unable to open '%s' (%s)\n",
			ip_devname, strerror(errno));
	}

	result= fcntl(ip_rcv_fd, F_GETFD);
	if (result == -1)
	{
		DPRINTF(1, ("ip_init: fcntl(%d, GETFD) failed (%s)\n",
			ip_rcv_fd, strerror(errno)));
		exit(1);
	}
	v= result | FD_ASYNCHIO;
	result= fcntl(ip_rcv_fd, F_SETFD, v);
	if (result == -1)
	{
		DPRINTF(1, ("ip_init: fcntl(%d, SETFD, %d) failed (%s)\n",
			ip_rcv_fd, v, strerror(errno)));
		exit(1);
	}
	ip_snd_fd= ip_rcv_fd;
}

DEFUN_VOID (void ip_rcv_restart)
{
	int result;

	assert (!ip_rcv_inprogress);
	while (ip_rcv_more2recv)
	{
		assert(ip_rcv_pkt == NULL);
		ip_rcv_pkt= pkt_alloc(PPP_MAX_HDR_SIZE+1500);
		ip_rcv_pkt->p_offset= PPP_MAX_HDR_SIZE;

		result= read(ip_rcv_fd, 
			ip_rcv_pkt->p_data+ip_rcv_pkt->p_offset, 1500);
		if (result > 0)
		{
			ip_rcv_completed(result, 0);
			continue;
		}
		if (result == -1 && errno == EINPROGRESS)
		{
			ip_rcv_inprogress= 1;
			return;
		}
		ip_rcv_completed(-1, errno);
	}
}


DEFUN
(void ip_rcv_completed, (result, error),
	int result AND
	int error
)
{
	pkt_ut *pkt;
	int pkt_type;
	ip_hdr_t *ip_hdr;

	ip_rcv_inprogress= 0;

	if (result == 0 || result == -1)
	{
		fatal("ip_rcv_completed: read failed (%s)\n",
			result == 0 ? "EOF" : strerror(error));
	}
	assert(ip_pppsnd_pkt == NULL);
	DPRINTF(2, ("got ip packet from inet\n"));

	pkt= ip_rcv_pkt;
	pkt->p_size= result;
	assert(pkt->p_offset + pkt->p_size <= pkt->p_maxsize);
	ip_rcv_pkt= NULL;

	pkt_type= PPP_TYPE_IP;
	ip_hdr= (ip_hdr_t *)(pkt->p_data + pkt->p_offset);
	if (ip_snd_vjhc && (((ip_hdr->ih_vers_ihl >> 4) &
		IH_VERSION_MASK) == IP_VERSION) &&
		((ip_hdr->ih_vers_ihl & IH_IHL_MASK) >= 5) &&
		(ip_hdr->ih_proto == IPPROTO_TCP))
	{
		pkt_type= ip_vjhc_compress(pkt);
	}
	ppp_snd_sethdr(pkt, pkt_type);
	error= ppp_snd(pkt, &ip_snd_callback);
	if (error == 0)
		return;
	assert(error == EAGAIN);
	ip_rcv_more2recv= 0;
	ip_pppsnd_pkt= pkt;
}


DEFUN
(void ip_snd_completed, (result, error),
	int result AND
	int error
)
{
	if (result != ip_snd_pkt->p_size)
		fatal("ip: write failure");
	pkt_free(ip_snd_pkt);
	ip_snd_pkt= NULL;
	if (ip_snd_pkts[0] != NULL)
		ip_snd_more2send= 1;
	ip_snd_inprogress= 0;
	DPRINTF(2, ("sent ip packet to inet\n"));
}

DEFUN_VOID (void ip_snd_restart)
{
	pkt_ut *pkt;
	int i, result;

	while(ip_snd_pkts[0] != NULL)
	{
		pkt= ip_snd_pkts[0];
		for (i= 0; i<IP_SND_PKTS_NR-1; i++)
			ip_snd_pkts[i]= ip_snd_pkts[i+1];
		ip_snd_pkts[i]= NULL;

		result= write(ip_snd_fd, pkt->p_data + pkt->p_offset,
			pkt->p_size);
		if (result == -1 && errno == EINPROGRESS)
		{
			/* write blocked */
			DPRINTF(1, ("ip_snd_restart: write blocked\n"));
			assert(ip_snd_pkt == NULL);
			ip_snd_pkt= pkt;
			ip_snd_inprogress= 1;
			return;
		}
		if (result != pkt->p_size)
			fatal("ip: write failure");
		pkt_free(pkt);
		DPRINTF(2, ("sent ip packet to inet\n"));
	}
	ip_snd_more2send= 0;
}

DEFUN_VOID (void ip_up)
{
	ip_rcv_more2recv= 1;
	ip_vjhc_init();
}

DEFUN_VOID (void ip_down)
{
	ip_rcv_more2recv= 0;
}

DEFUN
(void ip_arrived, (pkt),
	pkt_ut *pkt
)
{
	int result;
	int i;

	if (ip_snd_inprogress || ip_snd_more2send)
	{
		/* send to ip is busy, try to buffer the packet. */
		DPRINTF(1, ("ip_arrived: buffering packet\n"));
		if (ip_snd_pkts[IP_SND_PKTS_NR-1] != NULL)
		{
			/* Buffer is full. */
			DPRINTF(1, ("ip_arrived: send buffer is full\n"));
			pkt_free(ip_snd_pkts[0]);
			for (i= 0; i<IP_SND_PKTS_NR-1; i++)
				ip_snd_pkts[i]= ip_snd_pkts[i+1];
			ip_snd_pkts[i]= pkt;
			return;
		}
		for (i= 0; i<IP_SND_PKTS_NR && ip_snd_pkts[i] != NULL; i++)
			;	/* do nothing */
		assert(i<IP_SND_PKTS_NR);
		ip_snd_pkts[i]= pkt;
		return;
	}
	result= write(ip_snd_fd, pkt->p_data + pkt->p_offset, pkt->p_size);
	if (result == -1 && errno == EINPROGRESS)
	{
		/* write blocked */
		DPRINTF(1, ("ip_arrived: write blocked\n"));
		assert(ip_snd_pkt == NULL);
		ip_snd_pkt= pkt;
		ip_snd_inprogress= 1;
		return;
	}
	if (result != pkt->p_size)
		fatal("ip: write failure");
	pkt_free(pkt);
	DPRINTF(2, ("sent ip packet to inet\n"));
}


DEFUN
(static void ip_pppsnd_restart, (cb, arg),
	ppp_snd_callback_ut *cb AND
	int arg

)
{
	pkt_ut *pkt;
	int error;

	DPRINTF(1, ("ip_pppsnd_restart\n" ));
	assert(cb == &ip_snd_callback);
	assert(arg == 0);
	assert(ip_pppsnd_pkt != NULL);
	pkt= ip_pppsnd_pkt;
	ip_pppsnd_pkt= NULL;
	error= ppp_snd(pkt, &ip_snd_callback);
	assert(error == 0);
	ip_rcv_more2recv= 1;
}

/*
 * $PchHeader: /mount/hd2/minix/local/cmd/ppp/RCS/ip.c,v 1.3 1995/05/23 08:09:11 philip Exp $
 */
