/*	patchtree 1.6 - patch up a directory tree	Author: Kees J. Bot
 *								23 Jan 1994
 */
#define nil 0
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <dirent.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <utime.h>
#include <time.h>
#include <ctype.h>
#include "patchtree.h"
#include "maptree.h"
#include "inspect.h"

void report(const char *label)
{
	fprintf(stderr, "patchtree: %s: %s\n", label, strerror(errno));
}

void fatal(const char *label)
{
	report(label);
	quit(1);
}

void *allocate(void *mem, size_t len)
/* Checked malloc/realloc. */
{
	if ((mem= mem == nil ? malloc(len) : realloc(mem, len)) == nil)
		fatal("malloc()");

	return mem;
}

void (deallocate)(void *mem)
{
	free(mem);
}

int isdots(const char *name)
/* True for "." and "..". */
{
	if (name[0] != '.') return 0;
	if (name[1] == 0) return 1;
	if (name[1] != '.') return 0;
	if (name[2] == 0) return 2;
	return 0;
}

/* Make a temporary directory in /tmp or in $TMPDIR.  This directory allows
 * us to create a file under any name, no need to be careful if stomping on
 * something.  It also makes it easy to get rid of all the junk that patch
 * leaves behind.
 */

static char *mytmpdir;		/* My own directory to play in. */
static int need_quit= 0;

void maketmpdir(void)
{
	char *tmpdir;

	if ((tmpdir= getenv("TMPDIR")) == nil) tmpdir= "/tmp";

	mytmpdir= allocate(nil, (strlen(tmpdir) + 1 + 12 + 1)
						* sizeof(mytmpdir[0]));
	strcpy(mytmpdir, tmpdir);
	strcat(mytmpdir, "/ptree.XXXXXX");
	if (mktemp(mytmpdir) == nil) fatal(mytmpdir);
	need_quit= 1;
	if (mkdir(mytmpdir, 0700) < 0) { need_quit= 0; fatal(mytmpdir); }
}

char *tmpdirfile(char *name)
/* Return a path name in the temp dir. */
{
	char *path;
	size_t len;

	len= strlen(mytmpdir);
	path= allocate(nil, (len + 1 + strlen(name) + 1) * sizeof(path[0]));
	strcpy(path, mytmpdir);
	path[len++]= '/';
	strcpy(path + len, name);
	return path;
}

void wipetmpdir(void)
/* Clean out the temp dir, but leave the dir itself. */
{
	DIR *dp;
	struct dirent *entry;
	char *file;

	need_quit--;

	if ((dp= opendir(mytmpdir)) == nil) fatal(mytmpdir);

	while ((entry= readdir(dp)) != nil) {
		if (isdots(entry->d_name)) continue;
		file= tmpdirfile(entry->d_name);
		if (unlink(file) < 0 && errno != ENOENT) fatal(file);
		deallocate(file);
	}
	(void) closedir(dp);

	need_quit++;
}

void quit(int excode)
/* First get rid of the temporary dir, then exit. */
{
	if (need_quit > 0) {
		need_quit= 0;
		map_release();
		wipetmpdir();
		if (rmdir(mytmpdir) < 0) fatal(mytmpdir);
	}
	exit(excode);
}

static int not_now= 0;		/* Can't take an interrupt now. */
static int then_later= 0;	/* Then die later. */

static void catch(int sig)
/* Catch a signal, put out the garbage, die horribly. */
{
	if (not_now) {
		signal(sig, catch);
		then_later= sig;
		return;
	}

	if (need_quit > 0) {
		need_quit= 0;
		wipetmpdir();
		if (rmdir(mytmpdir) < 0) report(mytmpdir);
	}
	signal(sig, SIG_DFL);
	raise(sig);
	exit(1);
}

void copyfile(const char *src, const char *dst)
/* Copy a file at high warp. */
{
	char buf[(sizeof(char *) == 2 ? 1 : 32) * 1024];
	int sfd, dfd;
	ssize_t n;

	if ((sfd= open(src, O_RDONLY)) < 0) fatal(src);
	if ((dfd= open(dst, O_WRONLY|O_CREAT|O_TRUNC, 0600)) < 0) fatal(dst);

	while ((n= read(sfd, buf, sizeof(buf))) > 0) {
		if (write(dfd, buf, n) < 0) fatal(dst);
	}
	if (n < 0) fatal(src);
	(void) close(sfd);
	(void) close(dfd);
}

static void setmodes(const char *file, struct stat *stp)
/* Set the attributes of a file to be the same as those in *stp. */
{
	struct utimbuf times;

	if (chown(file, stp->st_uid, stp->st_gid) < 0 && errno != EPERM)
		fatal(file);

	if (chmod(file, stp->st_mode & 07777) < 0) fatal(file);

	times.actime= stp->st_atime;
	times.modtime= stp->st_mtime;
	if (utime(file, &times) < 0) fatal(file);
}

char *basename(const char *path)
/* Return the last component of a pathname. */
{
	char *wpath= (char *) path;	/* Usual declassification. */
	char *base;

	while ((base= strrchr(wpath, '/')) != nil) {
		if (base[1] == 0 && base != wpath) {
			/* Remove trailing /. */
			*base= 0;
		} else {
			return base + 1;
		}
	}
	return wpath;
}

static void usage(void)
{
	fprintf(stderr,
"Usage: patchtree [-znf] [-d] [-c<n>] [-u<n>] [-x file] original target ...\n");
	quit(1);
}

typedef struct entry {
	struct entry	*next;
	char		*name;
} entry_t;

static entry_t *collectdir(const char *dir)
/* Make a list of the names in a directory. */
{
	DIR *dp;
	struct dirent *entry;
	entry_t *list, **plp= &list;

	if ((dp= opendir(dir)) == nil) fatal(dir);

	while ((entry= readdir(dp)) != nil) {
		if (isdots(entry->d_name)) continue;

		*plp= allocate(nil, sizeof(**plp));
		(*plp)->name= allocate(nil, (strlen(entry->d_name) + 1)
						* sizeof((*plp)->name[0]));
		strcpy((*plp)->name, entry->d_name);
		plp= &(*plp)->next;
	}
	(void) closedir(dp);
	*plp= nil;
	return list;
}

action_t action= PATCH;		/* Patch or diff */
unsigned context= DEFCONTEXT;	/* Amount of diff context */
int nflag= 0, zflag= 0;		/* Playact or remove unchanged files. */
int fflag= 0;			/* Force diffs even if larger. */

static char *original;		/* Original files. */
static char **targets;		/* File to patch or make patches of */
static int onefile;		/* Is there just one original file? */

static char *target;		/* Current target to work on. */
static size_t targ_len= 0;	/* Current length (excl null). */
static size_t targ_size= 0;	/* Allocated length + null. */

#define TOP	0		/* Toplevel base value. */

static void patchtree(size_t base)
/* Patch one target file, or recursively patch a target dir. */
{
	static struct stat st;

	if ((base == TOP ? stat : lstat)(target, &st) < 0) {
		if (errno == ENOENT) return;
		fatal(target);
	}

	if (S_ISDIR(st.st_mode)) {
		/* Patch each file in this directory. */
		entry_t *files= collectdir(target);

		base= targ_len;
		target[base++]= '/';

		while (files != nil) {
			entry_t *junk;

			targ_len= base + strlen(files->name);
			if (targ_len >= targ_size) {
				targ_size= 2 * targ_len;
				target= allocate(target, targ_size
							* sizeof(target[0]));
			}
			strcpy(target + base, files->name);
			junk= files;
			files= files->next;
			deallocate(junk->name);
			deallocate(junk);

			patchtree(base);
		}
		return;
	}

	if (!S_ISREG(st.st_mode)) {
		/* skip_weird_file(); */
		return;
	}

	/* Patch or diff a single file. */ {
	char *origbase, *origfile;
	char *result;
	unsigned ocrc, ncrc;
	off_t osize, nsize, rsize;
	char *(*do_one)(char *, char *,
			unsigned, off_t, unsigned, off_t, off_t *);

	if (find_magic(target, &ncrc, &nsize, &ocrc, &osize, &origfile))
	{
		/* This file is a patch, so don't diff. */
		if (action != PATCH) return;

		/* Use the basename of the original filename in the
		 * patch by default.
		 */
		origbase= basename(origfile);
		do_one= patchone;
	} else {
		/* This file is not a patch, so don't patch. */
		if (action == PATCH) return;

		/* Look for a file with the same basename. */
		origbase= base == TOP ? basename(target) : target + base;
		do_one= diffone;
	}

	result = nil;

	if (onefile) {
		/* Special case: Original is just one file. */
		origfile= original;
		result= (*do_one)(origfile, target,
					ncrc, nsize, ocrc, osize, &rsize);
	} else {
		unsigned long rmin= -1;
		int use_this= 0;

		if (action != PATCH) {
			/* Try all possible diffs looking for the smallest. */
			map_search(origbase);

			while (map_next_file(&origfile)) {
				if (result != nil) deallocate(result);

				result= (*do_one)(origfile, target,
					ncrc, nsize, ocrc, osize, &rsize);

				use_this= 0;
				if (result != nil && rsize < rmin) {
					rmin= rsize;
					use_this= 1;
				}
			}
			if (rmin == -1) use_this= 1;	/* Use nothing. */
		}

		if (!use_this) {
			/* Look for a good patch or the best diff. */
			map_search(origbase);

			while (map_next_file(&origfile)) {
				if (result != nil) deallocate(result);

				result= (*do_one)(origfile, target,
					ncrc, nsize, ocrc, osize, &rsize);

				if (result != nil && rsize <= rmin) break;
			}
		}
	}

	if (zflag && result != nil && rsize == 0) {
		/* Files diff'd were found to be the same, and zflag is set
		 * asking for removal.
		 */
		if (!nflag) {
			if (unlink(target) < 0) fatal(target);
		}
		printf("%s == %s\n", target, origfile);
		deallocate(result);
	} else
	if (result != nil) {
		/* A result has been obtained, i.e. a patch has been generated,
		 * or one has been successfully applied.  Replace the target
		 * file.
		 */
		if (!nflag) {
			not_now= 1;

			copyfile(result, target);
			setmodes(target, &st);

			not_now= 0;
			if (then_later != 0) raise(then_later);
		}
		printf("%s %c= %s\n", target,
				action == PATCH ? '+' : '-', origfile);
		deallocate(result);
	} else {
		printf("%s ?\n", target);
	}
	wipetmpdir();
}}

int main(int argc, char **argv)
{
	int i;
	struct stat st;

	i= 1;
	while (i < argc && argv[i][0] == '-') {
		char *p= argv[i++] + 1;

		if (p[0] == '-' && p[1] == 0) break;	/* -- */

		while (*p != 0) {
			switch (*p++) {
			case 'd':
				if (action != PATCH) usage();
				action= DIFF;
				break;
			case 'c':
				if (action != PATCH) usage();
				action= CDIFF;
				if (isdigit(*p)) context= strtoul(p, &p, 10);
				break;
			case 'u':
				if (action != PATCH) usage();
				action= UDIFF;
				if (isdigit(*p)) context= strtoul(p, &p, 10);
				break;
			case 'n':
				nflag= 1;
				break;
			case 'z':
				zflag= 1;
				break;
			case 'f':
				fflag= 1;
				break;
			case 'x':
				if (*p == 0) {
					if (i == argc) usage();
					p= argv[i++];
				}
				if (strchr(p, '/') != nil) {
					fprintf(stderr,
			"patchtree: %s: Full paths can't be excluded\n", p);
					exit(1);
				}
				map_exclude(p);
				p= "";
				break;
			default:
				usage();
			}
		}
	}

	if ((zflag || fflag) && action == PATCH) usage();
	if ((argc - i) < 2) usage();
	original= argv[i++];
	targets= argv + i;

	if (signal(SIGHUP, SIG_IGN) != SIG_IGN) signal(SIGHUP, catch);
	if (signal(SIGINT, SIG_IGN) != SIG_IGN) signal(SIGINT, catch);
	if (signal(SIGTERM, SIG_IGN) != SIG_IGN) signal(SIGTERM, catch);
	maketmpdir();

	if (lstat(original, &st) < 0) fatal(original);
	onefile= S_ISREG(st.st_mode);

	if (!onefile) map_tree(original);

	while (*targets != nil) {
		targ_len= strlen(*targets);
		if (targ_len >= targ_size) {
			targ_size= 2 * targ_len;
			target= allocate(target, targ_size * sizeof(target[0]));
		}
		strcpy(target, *targets++);

		patchtree(TOP);
	}
	quit(0);
}
