/*
mnx.c

Created:	Aug 21, 1992 by Philip Homburg <philip@cs.vu.nl>
*/

#include <assert.h>
#include <errno.h>
#include <nlist.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "ansi.h"
#include "types.h"

#include "ack.h"
#include "dis386.h"

#define USE16	0

typedef unsigned char header_t[32];	/* A minix i86 or i386 header is 32 
					 * bytes */
#define MAGIC0_OFF	0
#define MAGIC1_OFF	1
#define FLAGS_OFF	2
#define CPU_OFF		3
#define HDRLEN_OFF	4
#define UNUSED_OFF	5
#define VERSION_OFF	6
#define TEXT_OFF	8
#define DATA_OFF	12
#define BSS_OFF		16
#define ENTRY_OFF	20
#define TOTAL_OFF	24
#define SYMS_OFF	28

#define A_MAGIC0	0x1
#define A_MAGIC1	0x3

#define A_UZP		0x01
#define A_PAL		0x02
#define A_NSYM		0x04
#define A_EXEC		0x10
#define A_SEP		0x20

#define A_I8086		0x04
#define A_I80386 	0x10

char block[1024];

char *prog_name;

struct nlist *text_label_first, *data_label_first, *const_label_first,
	*other_label_first;
int text_syms, data_syms, const_syms;
unsigned long text_start, text_end, text_offset;
unsigned long data_start;
int word_size;
FILE *objf;

static void usage ARGS(( void ));
static u32_t b2u32 ARGS(( unsigned char *b ));
static int cmp_nlist_ent ARGS(( const void *e1, const void *e2 ));
static void text_func ARGS(( unsigned long value, char **l_name, 
	unsigned long *l_value, char **h_name, unsigned long *h_value ));
static void data_func ARGS(( unsigned long value, char **l_name, 
	unsigned long *l_value, char **h_name, unsigned long *h_value ));
static void const_func ARGS(( unsigned long value, char **l_name, 
	unsigned long *l_value, char **h_name, unsigned long *h_value ));
static struct nlist *find_text_label ARGS(( char *str ));
static int text_cmp ARGS(( const void *e1, const void *e2 ));	
static void parse_text_range ARGS(( char *str, unsigned long *start_p,
	unsigned long *end_p ));
static void dis_text_range ARGS(( unsigned long start, unsigned long end ));
static char *get_text ARGS(( unsigned long text_addr, size_t size ));

DEFUN
(void main, (argc, argv),
	int argc AND
	char *argv[]
)
{
	char *fname;
	header_t hdr;
	int c;
	int a_flags, a_uzp, a_pal, a_nsym, a_exec, a_sep, a_cpu;
	u32_t a_text, a_data, a_bss, a_entry;
	unsigned long start, end;
	int nsyms;
	int n;
	struct nlist *nlist, *nlist_end;
	char *t_arg, *w_arg;
	int c_flag;
	char *check;

	prog_name= argv[0];
	t_arg= NULL;
	w_arg= NULL;
	c_flag= 0;
	while ((c= getopt(argc, argv, "t:w:c?")) != -1)
	{
		switch(c)
		{
		case 't':
			if (t_arg)
				usage();
			t_arg= optarg;
			break;
		case 'w':
			if (w_arg)
				usage();
			w_arg= optarg;
			break;
		case 'c':
			c_flag= 1;
			break;
		case '?':
			usage();
		default:
			fprintf(stderr, "%s: getopt failed: '%c'\n",
				prog_name, c);
			exit(1);
		}
	}

	if (w_arg)
	{
		word_size= strtol(w_arg, &check, 0);
		if (check[0] != '\0' || (word_size != 16 && word_size != 32))
		{
			fprintf(stderr, "%s: illegal word size '%s'\n",
				prog_name, w_arg);
			exit(1);
		}
	}
	else
		word_size= 0;	/* 0 means unknown */

	if (optind >= argc)
		usage();
	fname= argv[optind++];

	objf= fopen(fname, "r");
	if (objf == NULL)
	{
		fprintf(stderr, "%s: unable to open '%s': %s\n", prog_name,
			fname, strerror(errno));
		exit(1);
	}
	n= fread((char *)hdr, sizeof(hdr), 1, objf);
	if (n != 1)
	{
		fprintf(stderr, "%s: unable to read header: %s\n", prog_name,
			feof(objf) ? "at eof" : strerror(ferror(objf)));
		exit(1);
	}

	/* Parse the header, first check the magic number */
	if (hdr[MAGIC0_OFF] != A_MAGIC0 || hdr[MAGIC1_OFF] != A_MAGIC1)
	{
		fprintf(stderr, "%s: not a Minix executable\n",
			prog_name);
		exit(1);
	}
	
	/* Extract the flags */
	a_flags= hdr[FLAGS_OFF];
	a_uzp= a_flags & A_UZP;		/* unmapped zero page */
	a_pal= a_flags & A_PAL;		/* page aligned */
	a_nsym= a_flags & A_NSYM;	/* symbol table with separate strings 
					 * table */
	a_exec= a_flags & A_EXEC;	/* common I&D executable */
	a_sep= a_flags & A_SEP;		/* separate I&D executable */
	
	/* Check the cpu, and set the word size */
	a_cpu= hdr[CPU_OFF];
	if (a_cpu != A_I8086 && a_cpu != A_I80386)
	{
		fprintf(stderr, "%s: not a Minix i86 or i386 executable\n",
			prog_name);
		exit(1);
	}
	if (word_size == 0)
	{
		if (a_cpu == A_I8086)
			word_size= 16;
		else
			word_size= 32;
	}
	if (hdr[HDRLEN_OFF] != sizeof(hdr))
	{
		fprintf(stderr, "%s: wrong header len (%d)\n",
			prog_name, hdr[HDRLEN_OFF]);
		exit(1);
	}
	a_text= b2u32(&hdr[TEXT_OFF]);
	a_data= b2u32(&hdr[DATA_OFF]);
	a_bss= b2u32(&hdr[BSS_OFF]);
	a_entry= b2u32(&hdr[ENTRY_OFF]);

	/* Check if a_entry is consistent with a_pal, a_uzp and a_cpu */
	text_start= 0;
	data_start= 0;
	if (a_pal)
		text_start += sizeof(hdr);
	if (a_uzp)
	{
		if (a_cpu == A_I8086)	/* Ridiculous, but possible */
		{
			text_start += 0x100;	/* 1 click, 256 bytes */
			data_start += 0x100;
		}
		else
		{
			text_start += 0x1000;	/* 1 click, is 1 page is 4096
						 * bytes */
			data_start += 0x1000;
		}
	}
	if (text_start != a_entry)
	{
		fprintf(stderr, 
		"%s: calculated entry point differs from a_entry: %d != %d\n",
			prog_name, text_start, a_entry);
	}
	text_end= text_start+a_text;
	text_offset= sizeof(hdr);

	/* Fetch the namelist */
	nsyms= read_nlist(fname, &nlist);
	if (nsyms == -1)
	{
		fprintf(stderr, "%s: unable to read namelist: %s\n",
			prog_name, strerror(errno));
		exit(1);
	}
	/* Let's sort the namelist */
	qsort(nlist, nsyms, sizeof(*nlist), cmp_nlist_ent);
	
	/* Let's look for the number and location of the text, data and
	 * const symbols */
	nlist_end= nlist+nsyms; 
	text_label_first= nlist;
	for (data_label_first= text_label_first; data_label_first<nlist_end;
		data_label_first++)
	{
		if ((data_label_first->n_type & N_TYPE) == N_TEXT)
			continue;
		break;
	}
	for (const_label_first= data_label_first; const_label_first<nlist_end;
		const_label_first++)
	{
		if ((const_label_first->n_type & N_TYPE) == N_DATA ||
			(const_label_first->n_type & N_TYPE) == N_BSS ||
			(const_label_first->n_type & N_TYPE) == N_COMM)
		{	
			continue;
		}	
		break;
	}
	for (other_label_first= const_label_first; other_label_first<nlist_end;
		other_label_first++)
	{
		if ((other_label_first->n_type & N_TYPE) == N_ABS)
			continue;
		break;
	}
	text_syms= data_label_first-text_label_first;
	data_syms= const_label_first-data_label_first;
	const_syms= other_label_first-const_label_first;
	
	if (t_arg)
	{
		parse_text_range(t_arg, &start, &end);
		if (c_flag) end= text_end;
	}
	else
	{
		start= text_start;
		end= text_end;
	}

	ack_label_funcs(text_func, data_func, const_func);	

	printf(".sect .text; .sect .rom; .sect .data; .sect bss; .sect .end\n");
	printf(".sect .text\n");
	if (word_size == 16)
		printf(".use16\n");

	dis_text_range(start, end);
	exit(0);
}

DEFUN_VOID(static void usage )
{
	fprintf(stderr,
		"Usage: %s [-c] [-t label] [-w wordsize] <executable>\n",
		prog_name);
	exit(1);
}

dump(char *s, char *e)
{
	if (e>s+16) e=s+16;
	for (; s<e; s++)
		printf("0x%x ", *(unsigned char *)s);
	printf("\n");
}

DEFUN
(static u32_t b2u32, (b),
	unsigned char *b
)
{
	u32_t v;

	v= b[0] | (b[1] << 8) | (b[2] << 16) | (b[3] << 24);

	return v;
}

DEFUN
(static int cmp_nlist_ent, (e1, e2),
	const void *e1 AND
	const void *e2
)
{
	/* 
	 * The order in which the symbol table is sorted:
	 * - text symbols, in numerical order
	 * - data, bss, common in numerical order
	 * - constantant (ABS) in numerical order
	 * - other symbols in any order
	 */
	 
	const struct nlist *n1, *n2;
	int nt1, nt2;
	int dot1, dot2;

	n1= e1;
	n2= e2;

	nt1= n1->n_type & N_TYPE;
	nt2= n2->n_type & N_TYPE;
	if (nt1 == N_BSS || nt1 == N_COMM)
		nt1= N_DATA;
	if (nt2 == N_BSS || nt2 == N_COMM)
		nt2= N_DATA;
	if (nt1 == nt2)
	{
		if (n1->n_value < n2->n_value)
			return -1;
		if (n1->n_value > n2->n_value)
			return 1;
		dot1= !!strchr(n1->n_name, '.');
		dot2= !!strchr(n2->n_name, '.');
		if (!dot1 && dot2)
			return -1;
		else if (dot1 && !dot2)
			return 1;
		return 0;
	}
	if (nt1 == N_TEXT)
		return -1;
	if (nt2 == N_TEXT)
		return 1;
	if (nt1 == N_DATA)
		return -1;
	if (nt2 == N_DATA)
		return 1;
	if (nt1 == N_ABS)
		return -1;
	if (nt2 == N_ABS)
		return 1;
	return 0;	
}

DEFUN
(static void text_func, (value, l_name, l_value, h_name, h_value),
	unsigned long value AND
	char **l_name AND
	unsigned long *l_value AND
	char **h_name AND
	unsigned long *h_value
)
{
	struct nlist *symbol;
	struct nlist dummy;

	dummy.n_value= value;
	symbol= bsearch(&dummy, text_label_first, text_syms,
		sizeof(*text_label_first), text_cmp);
	if (symbol == 0)
	{
		symbol= &dummy;
		dummy.n_name= NULL;
	}
	if (l_name)
		*l_name= symbol->n_name;
	if (*l_value)
		*l_value= symbol->n_value;
	if (symbol != &dummy && symbol < text_label_first+text_syms)
	{	
		if (h_name)
			*h_name= (symbol+1)->n_name;
		if (h_value)
			*h_value= (symbol+1)->n_value;
	}
	else
	{	
		if (h_name)
			*h_name= NULL;
	}		
}

DEFUN
(static void data_func, (value, l_name, l_value, h_name, h_value),
	unsigned long value AND
	char **l_name AND
	unsigned long *l_value AND
	char **h_name AND
	unsigned long *h_value
)
{
	printf("data_func, not yet implemented\n");
	if (l_name)
		*l_name= NULL;
	if (h_name)
		*h_name= NULL;
}

DEFUN
(static void const_func, (value, l_name, l_value, h_name, h_value),
	unsigned long value AND
	char **l_name AND
	unsigned long *l_value AND
	char **h_name AND
	unsigned long *h_value
)
{
	printf("const_func, not yet implemented\n");
	if (l_name)
		*l_name= NULL;
	if (h_name)
		*h_name= NULL;
}

DEFUN
(static int text_cmp, (e1, e2),
	const void *e1 AND
	const void *e2
)
{
	struct nlist *n1, *n2;

	n1= e1;
	n2= e2;

	if (n2->n_name == NULL)
	{
		if (n1->n_name == NULL)
			return 0;
		return -text_cmp(e2, e1);
	}
	if (n1->n_value < n2->n_value)
		return -1;
	if (n2->n_value < n1->n_value)
	{
		if (n2+1 == text_label_first+text_syms ||
			n2[1].n_value > n1->n_value)
		{
			return 0;
		}
		return 1;
	}
	if (n2 == text_label_first || n2[-1].n_value < n1->n_value)
		return 0;
	return -1;
}

DEFUN
(static void parse_text_range, (str, start_p, end_p),
	char *str AND
	unsigned long *start_p AND
	unsigned long *end_p
)
{
	struct nlist *start_label, *end_label;

	/* Try to parse str as a single text label first. In that case
	 * the range is taken from that label to the next.
	 */
	start_label= find_text_label(str);
	if (start_label != NULL)
	{
		*start_p= start_label->n_value;
		for (end_label= start_label+1; end_label < data_label_first &&
			end_label->n_value == start_label->n_value; end_label++)
		{
			; /* Do nothing */
		}
		if (end_label == data_label_first)
			*end_p= text_end;
		else
			*end_p= end_label->n_value;
		return;
	}
	fprintf(stderr, "dis386: label '%s' not found\n", str);
	exit(1);
}

DEFUN
(static void dis_text_range, (start, end),
unsigned long start AND
unsigned long end
)
{
	struct nlist *text_label;
	struct nlist dummy;
	char *str;
	unsigned long pc, pc_hi;
	unsigned size;
	unsigned actsize;
	char string[16];
	char *data;

	if (start < text_start)
	{
		fprintf(stderr,
"starting point (0x%x) before start of text segment (0x%x) (adjusted)\n",
				start, text_start);
		start= text_start;
	}

	if (end > text_end)
	{
		fprintf(stderr,
"end point (0x%x) after start of text segment (0x%x) (adjusted)\n",
				end, text_end);
		end= text_end;
	}
	if (end < start)
	{
		fprintf(stderr, "end (0x%x) before start (0x%x)\n",
			end, start);
		return;
	}

	/* Fetch the first text label */
	dummy.n_value= start;
	text_label= bsearch(&dummy, text_label_first, text_syms,
		sizeof(*text_label_first), text_cmp);

	if (text_label != NULL)
	{
		for (; text_label < data_label_first &&
			text_label->n_value < start; text_label++)
		{
			; /* Do nothing */
		}
		if (text_label == data_label_first)
			text_label= NULL;
	}

	dis386_wordsize(word_size);

	if (text_label == NULL || text_label->n_value > start)
	{
		sprintf(string, "L_0x%x", start);
		str= ack_text_label(string);
		puts(str);
		free(str);
	}	
	
	pc= start;
	pc_hi= end;
	for(;;)
	{
		/* Let's check if we should print a text label */

		if (text_label && text_label->n_value <= pc)
		{
			if (text_label->n_value < pc)
			{
				fprintf(stderr,
"%s: inconsistent symbol table, text label ('%s', 0x%x) < 0x%x\n",
					prog_name, text_label->n_name,
					text_label->n_value, pc);
				exit(1);
			}
			str= ack_text_label(text_label->n_name);
			puts(str);
			free(str);
			text_label++;
			if (text_label-text_label_first >= text_syms)
				text_label= NULL;
			continue;
		}	
		size= 32;
		if (text_label && pc+size>text_label->n_value)
			size= text_label->n_value-pc;
		if (pc+size > pc_hi)
			size= pc_hi-pc;
		if (size == 0)
			break;	
		data= get_text(pc, size);
		str= ack_dis386(pc, data, size, &actsize);
		if (str == 0)
		{
			dump(data, data+size);
			actsize= size;
		}
		else
		{
			putchar('\t');
			puts(str);
			free(str);
		}	
		pc += actsize;
	}
}

DEFUN
(static struct nlist *find_text_label, (str),
char *str
)
{
	struct nlist *label, *extra_label;

	fprintf(stderr, "looking for label '%s'\n", str);

	extra_label= NULL;
	for (label= text_label_first; label<data_label_first; label++)
	{
		if (strcmp(label->n_name, str) == 0)
			break;
		if (label->n_name[0]== '_' && 
			strcmp(label->n_name+1, str) == 0)
		{
			extra_label= label;
		}
	}
	if (label == data_label_first)
		label= extra_label;
	return label;
}

DEFUN
(static char *get_text, (text_addr, size),
unsigned long text_addr AND
size_t size
)
{
	static char block[1024];
	static unsigned long first= -1;
	static size_t data_size;

	if (first != -1 && text_addr >= first &&
		text_addr + size <= first + data_size)
	{
		return block + (text_addr-first);
	}

	if (text_addr < text_start || size > sizeof(block))
		return NULL;

	first= text_addr;
	text_addr= text_addr - text_start + text_offset;
	if (fseek(objf, text_addr, SEEK_SET) == -1)
	{
		fprintf(stderr, "%s: seek to 0x%lx failed: %s\n",
			prog_name, text_addr, strerror(ferror(objf)));
		exit(1);
	}
	data_size= fread(block, sizeof(char), sizeof(block), objf);
	if (data_size == 0)
	{
		fprintf(stderr, "%s: read failed: %s\n", prog_name,
			feof(objf) ? "end of file" : strerror(ferror(objf)));
		exit(1);
	}
	return block;
}
