
/* This file contains some utility routines for MM.
 *
 * The entry points are:
 *   allowed:	see if an access is permitted
 *   no_sys:	this routine is called for invalid system call numbers
 *   panic:	MM has run aground of a fatal error and cannot continue
 *   tell_fs:	interface to FS
 */

#include "mm.h"
#include <sys/stat.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include <fcntl.h>
#include <signal.h>		/* needed only because mproc.h needs it */
#include <unistd.h>
#include "assert.h"
INIT_ASSERT
#include "mproc.h"

/*===========================================================================*
 *				allowed					     *
 *===========================================================================*/
PUBLIC int allowed(name_buf, s_buf, mask)
char *name_buf;			/* pointer to file name to be EXECed */
struct stat *s_buf;		/* buffer for doing and returning stat struct*/
int mask;			/* R_BIT, W_BIT, or X_BIT */
{
/* Check to see if file can be accessed.  Return EACCES or ENOENT if the access
 * is prohibited.  If it is legal open the file and return a file descriptor.
 */

  int fd;
  int save_errno;

  /* Use the fact that mask for access() is the same as the permissions mask.
   * E.g., X_BIT in <minix/const.h> is the same as X_OK in <unistd.h> and
   * S_IXOTH in <sys/stat.h>.  tell_fs(DO_CHDIR, ...) has set MM's real ids
   * to the user's effective ids, so access() works right for setuid programs.
   */
  if (access(name_buf, mask) < 0) return(-errno);

  /* The file is accessible but might not be readable.  Make it readable. */
  tell_fs(SETUID, MM_PROC_NR, (int) SUPER_USER, (int) SUPER_USER);

  /* Open the file and fstat it.  Restore the ids early to handle errors. */
  fd = open(name_buf, O_RDONLY);
  save_errno = errno;		/* open might fail, e.g. from ENFILE */
  tell_fs(SETUID, MM_PROC_NR, (int) mp->mp_effuid, (int) mp->mp_effuid);
  if (fd < 0) return(-save_errno);
  if (fstat(fd, s_buf) < 0) panic("allowed: fstat failed", NO_NUM);

  /* Only regular files can be executed. */
  if (mask == X_BIT && (s_buf->st_mode & I_TYPE) != I_REGULAR) {
	close(fd);
	return(EACCES);
  }
  return(fd);
}


/*===========================================================================*
 *				find_proc  				     *
 *===========================================================================*/
PUBLIC struct mproc *find_proc(pid)
pid_t pid;
{
  register struct mproc *rmp;

  for (rmp = mproc_inuse; rmp; rmp= rmp->mp_next)
  {
  	assert(rmp->mp_flags & IN_USE);
  	if (rmp->mp_pid == pid)
  		return rmp;
  }

  return(NIL_MPROC);
}


/*===========================================================================*
 *				no_sys					     *
 *===========================================================================*/
PUBLIC int no_sys()
{
/* A system call number not implemented by MM has been requested. */

#if PCH_DEBUG
  printf("mm: no_sys(): process %d sends %d\n", who, mm_call);
#endif /* PCH_DEBUG */
  if (who <= INIT_PROC_NR)
  {
  	panic(NULL, NO_NUM);
  }
  return(ENOSYS);
}


/*===========================================================================*
 *				panic					     *
 *===========================================================================*/
PUBLIC void panic(format, num)
char *format;			/* format string */
int num;			/* number to go with format string */
{
/* Something awful has happened.  Panics are caused when an internal
 * inconsistency is detected, e.g., a programming error or illegal value of a
 * defined constant.
 */

  if (format)
  {
	  printf("Memory manager panic: %s ", format);
	  if (num != NO_NUM) printf("%d",num); 
	  printf("\n");
  }
  tell_fs(SYNC, 0, 0, 0);	/* flush the cache to the disk */
  sys_abort(RBT_PANIC);
}


/*===========================================================================*
 *				get_time				     *
 *===========================================================================*/
PUBLIC time_t get_time()
{
	int r;
	message m;

	/* Clear the message */
	m.m1_i1= 0;
	m.m1_i2= 0;
	m.m1_i3= 0;
	m.m1_p1= NULL;
	m.m1_p2= NULL;
	m.m1_p3= NULL;

	r= _taskcall(clck_tasknr, GET_TIME, &m);
	if (r != OK)
		panic("GET_TIME failure ", r);

	return (time_t) m.NEW_TIME;
}


/*===========================================================================*
 *				tell_fs					     *
 *===========================================================================*/
PUBLIC void tell_fs(what, p1, p2, p3)
int what, p1, p2, p3;
{
/* This routine is only used by MM to inform FS of certain events:
 *      tell_fs(CHDIR, slot, dir, 0)
 *      tell_fs(EXEC, proc, 0, 0)
 *      tell_fs(EXIT, proc, 0, 0)
 *      tell_fs(FORK, parent, child, pid)
 *      tell_fs(SETGID, proc, realgid, effgid)
 *      tell_fs(SETUID, proc, realuid, effuid)
 *      tell_fs(SYNC, 0, 0, 0)
 *      tell_fs(UNPAUSE, proc, signr, 0)
 *      tell_fs(SETPGRP, proc, 0, 0)
 */
/*
 *      tell_fs(SETSID, proc, 0, 0)
 */

  message m;

  m.m1_i1 = p1;
  m.m1_i2 = p2;
  m.m1_i3 = p3;

 /* clear the other fields */
  m.m1_p1= 0;
  m.m1_p2= 0;
  m.m1_p3= 0;

  _taskcall(FS_PROC_NR, what, &m);
}

#if DEBUG
#if (CHIP == INTEL && WORD_SIZE == 4)
PUBLIC void stacktrace()
{
	typedef unsigned int reg_t;
	reg_t bp, pc, hbp;

	bp= get_bp();
	while(bp)
	{
		pc= ((reg_t *)bp)[1];
		hbp= ((reg_t *)bp)[0];
		printf("0x%lx ", (unsigned long)pc);
		if (hbp != 0 && hbp <= bp)
		{
			printf("???");
			break;
		}
		bp= hbp;
	}
	printf("\n");
}
#endif /* CHIP == INTEL && WORD_SIZE == 4 */
#endif

/*
 * $PchId: utility.c,v 1.6 1996/02/29 23:28:59 philip Exp $
 */
