/* This file deals with creating processes (via FORK) and deleting them (via
 * EXIT/WAIT).  When a process forks, a new slot in the 'mproc' table is
 * allocated for it, and a copy of the parent's core image is made for the
 * child.  Then the kernel and file system are informed.  A process is removed
 * from the 'mproc' table when two events have occurred: (1) it has exited or
 * been killed by a signal, and (2) the parent has done a WAIT.  If the process
 * exits first, it continues to occupy a slot until the parent does a WAIT.
 *
 * The entry points into this file are:
 *   do_fork:	 perform the FORK system call
 *   do_mm_exit: perform the EXIT system call (by calling mm_exit())
 *   mm_exit:	 actually do the exiting
 *   do_wait:	 perform the WAITPID or WAIT system call
 */


#include "mm.h"
#include <sys/wait.h>
#include <minix/callnr.h>
#include <signal.h>
#include <string.h>
#include "assert.h"
INIT_ASSERT
#include "mproc.h"
#include "param.h"

/* force run of find_pidrange */
PRIVATE pid_t next_pid = 0, bad_pid = 0;

FORWARD _PROTOTYPE (void cleanup, (register struct mproc *child) );
FORWARD _PROTOTYPE (void find_pidrange, (void) );

/*===========================================================================*
 *				do_fork					     *
 *===========================================================================*/
PUBLIC int do_fork()
{
/* The process pointed to by 'mp' has forked.  Create a child process. */

  register struct mproc *rmp;	/* pointer to parent */
  register struct mproc *rmc;	/* pointer to child */
  struct mproc *saved_next;
  int i, child_nr;
#if VIRT_MEM
  int result;
#endif /* VIRT_MEM */
  char *sptr, *dptr;
  phys_clicks prog_clicks, child_base = 0;
  phys_bytes prog_bytes;
#if !(CHIP == INTEL && VIRT_MEM)
  phys_bytes parent_abs, child_abs;	/* Intel only */
#endif

  assert(mm_call == FORK);

 /* If the process table is almost full, try to unload some cached
  * executables.
  */
  if (mproc_freelist == NULL || mproc_freelist->mp_next == NULL ||
  	mproc_freelist->mp_next->mp_next == NULL)
  {
  	ec_free_pslot();
  	ec_free_pslot();
  	ec_free_pslot();
  }

 /* If tables might fill up during FORK, don't even start since recovery half
  * way through is such a nuisance.
  */
  rmp = mp;
  if (mproc_freelist == NULL) return(EAGAIN);
  if (rmp->mp_effuid != 0 &&
  	(mproc_freelist->mp_next == NULL ||
  	mproc_freelist->mp_next->mp_next == NULL))
  {
	return(EAGAIN);
  }

  prog_bytes= rmp->mp_virsize;
  assert(prog_bytes != 0);
  prog_clicks= prog_bytes >> CLICK_SHIFT;

  if ( (child_base = alloc_mem(prog_clicks)) == NO_MEM) return(EAGAIN);

#if !(CHIP == INTEL && VIRT_MEM)
  /* Create a copy of the parent's core image for the child. */
  child_abs = (phys_bytes) child_base << CLICK_SHIFT;
  parent_abs = (phys_bytes) rmp->mp_seg[SEG_T].mem_phys << CLICK_SHIFT;
  i = sys_copy(HARDWARE, 0, parent_abs, HARDWARE, 0, child_abs, prog_bytes);
  if ( i < 0) panic("do_fork can't copy", i);
#endif

  /* Find a slot in 'mproc' for the child process.  A slot must exist. */
  rmc= mproc_freelist;
  assert(!(rmc->mp_flags & IN_USE));
  assert(rmc != &mproc[NR_PROCS]);

  /* Set up the child and its memory map; copy its 'mproc' slot from parent. */
  child_nr = (int)(rmc - mproc);	/* slot number of the child */

  /* copy from parent slot to child's */
  saved_next= rmc->mp_next;	/* save next ptr */
  sptr = (char *) rmp;		/* pointer to parent's 'mproc' slot */
  dptr = (char *) rmc;		/* pointer to child's 'mproc' slot */
  i = sizeof(struct mproc);	/* number of bytes in a proc slot. */
  memmove(dptr, sptr, sizeof(struct mproc));
  rmc->mp_next= saved_next;	/* restore next ptr */

  rmc->mp_parent = who;		/* record child's parent */
  rmc->mp_flags &= ~TRACED;	/* child does not inherit trace status */

  /* No need to calculate the map, we leave that to the kernel. We have
   * to store a new mp_virstart though.
   */
  rmc->mp_virstart= (child_base << CLICK_SHIFT);
  rmc->mp_exitstatus = 0;
  rmc->mp_sigstatus = 0;

  /* Find a allocate a pid for the child and put it in the table. */
  if (next_pid >= bad_pid)
  {
	find_pidrange();
	assert(next_pid < bad_pid);
  }
  rmc->mp_pid = next_pid++;

  /* Tell kernel and file system about the (now successful) FORK. */
#if (CHIP == INTEL) && VIRT_MEM
  /* Ask the kernel to fork too, on a vm system it might fail. */
  result= sys_fork(who, child_nr, rmc->mp_pid, child_base);
  if (result != OK)	/* Now we have a problem, backup */
  {
	rmc->mp_flags = 0;
	free_mem (child_base, prog_clicks);
  	return result;
  }
#else /* !(CHIP == INTEL) && VIRT_MEM */
  sys_fork(who, child_nr, rmc->mp_pid, child_base); /* child_base is 68K only*/
#endif /* (CHIP == INTEL) && VIRT_MEM */
  tell_fs(FORK, who, child_nr, rmc->mp_pid);

#if !(CHIP == INTEL && VIRT_MEM)
  /* Report child's memory map to kernel. */
  sys_newmap(child_nr, rmc->mp_seg);
#endif

  /* Move rmc from the freelist to the inuse list */
  assert(rmc == mproc_freelist);
  if (rmc->mp_next != NULL)
  	rmc->mp_next->mp_prev= NULL;
  mproc_freelist= rmc->mp_next;

  rmc->mp_prev= NULL;
  rmc->mp_next= mproc_inuse;
  if (rmc->mp_next != NULL)
  	rmc->mp_next->mp_prev= rmc;
  mproc_inuse= rmc;

  /* Reply to child to wake it up. */
  reply(child_nr, 0, 0, NIL_PTR);
  return(rmc->mp_pid);		 /* child's pid */
}


/*===========================================================================*
 *				do_mm_exit				     *
 *===========================================================================*/
PUBLIC int do_mm_exit()
{
/* Perform the exit(status) system call. The real work is done by mm_exit(),
 * which is also called when a process is killed by a signal.
 */

  assert(mm_call == EXIT);

  mm_exit(mp, exit_status);
  dont_reply = TRUE;		/* don't reply to newly terminated process */
  return(OK);			/* pro forma return code */
}


/*===========================================================================*
 *				mm_exit					     *
 *===========================================================================*/
PUBLIC void mm_exit(rmp, status)
register struct mproc *rmp;	/* pointer to the process to be terminated */
int status;			/* the process' exit status (for parent) */
{
/* A process is done.  Release most of the process' possessions.  If its
 * parent is waiting, release the rest, else hang.
 */

  register int proc_nr;
  int parent_waiting, right_child;
  pid_t pidarg;
  struct mproc *s_mp, *smp_next;
  struct mproc *p_mp;

  proc_nr = (int) (rmp - mproc);	/* get process slot number */

  /* Tell the kernel and FS that the process is no longer runnable. */
  tell_fs(EXIT, proc_nr, 0, 0);	/* file system can free the proc slot */

  /* If the process is a session leader, send a SIGHUP to all other processes
   * in the same session. */
  if (rmp->mp_procgrp == rmp->mp_pid)
  {
	for (s_mp = mproc_inuse; s_mp; s_mp= smp_next) {
		/* If the process dies because of the SIGHUP it will be moved
		 * to the deadlist. Save the successor first.
		 */
		smp_next= s_mp->mp_next;

		if (s_mp == rmp) continue;
		assert (s_mp->mp_flags & IN_USE);
		assert (!(s_mp->mp_flags & HANGING));
		if (s_mp->mp_procgrp != rmp->mp_procgrp) continue;
		sig_proc(s_mp, SIGHUP);
	}
  }

  if (rmp->mp_pid == 1)
  {
  	/* Init died */
  	panic("Init died", NO_NUM);
  }

  sys_xit(rmp->mp_parent, proc_nr);

  /* Release the memory occupied by the child. */

  assert (rmp->mp_virstart != 0);
  free_mem(rmp->mp_virstart >> CLICK_SHIFT, rmp->mp_virsize >> CLICK_SHIFT);
  rmp->mp_virstart= rmp->mp_virsize= 0;

  /* The process slot can only be freed if the parent has done a WAIT. */
  p_mp= &mproc[rmp->mp_parent];

  /* Try to send a SIGCHLD to the parent. */
  if (sigismember(&p_mp->mp_catch, SIGCHLD))
	sig_proc(p_mp, SIGCHLD);

  rmp->mp_exitstatus = (char) status;
  pidarg = p_mp->mp_wpid;	/* who's being waited for? */
  parent_waiting = p_mp->mp_flags & WAITING;
  if (pidarg == -1 || pidarg == rmp->mp_pid || -pidarg == rmp->mp_procgrp)
	right_child = TRUE;		/* child meets one of the 3 tests */
  else
	right_child = FALSE;		/* child fails all 3 tests */

  /* Move the process to the deadlist. The cleanup code depends on that. */
  rmp->mp_flags |= HANGING;
  if (rmp->mp_next != NULL)
  	rmp->mp_next->mp_prev= rmp->mp_prev;
  if (rmp->mp_prev != NULL)
  	rmp->mp_prev->mp_next= rmp->mp_next;
  else
  {
  	assert(mproc_inuse == rmp);
  	mproc_inuse= rmp->mp_next;
  }
  rmp->mp_prev= NULL;
  rmp->mp_next= mproc_deadlist;
  if (mproc_deadlist)
  	rmp->mp_next->mp_prev= rmp;
  mproc_deadlist= rmp;

  if (parent_waiting && right_child)
	cleanup(rmp);		/* tell parent and release child slot */

  /* If the process has children, disinherit them.  INIT is the new parent. */
  rmp= mproc_inuse;
  while (rmp)
  {
  	assert(rmp->mp_flags & IN_USE);
  	assert(!(rmp->mp_flags & HANGING));
	if (rmp->mp_parent == proc_nr) {
		/* 'rmp' now points to a child to be disinherited. */
		rmp->mp_parent = INIT_PROC_NR;
		if (rmp->mp_flags & TRACED)
		{
			rmp->mp_flags &= ~TRACED;
			sig_proc(rmp, SIGKILL);

			/* sig_proc changes the mproc_inuse list, restart
			 * from the beginning.
			 */
			rmp= mproc_inuse;
			continue;
		}
	}
	rmp= rmp->mp_next;
  }
  for (rmp = mproc_deadlist; rmp; rmp= smp_next)
  {
  	smp_next= rmp->mp_next;

  	assert(rmp->mp_flags & IN_USE);
  	assert(rmp->mp_flags & HANGING);
	if (rmp->mp_parent == proc_nr) {
		/* 'rmp' now points to a child to be disinherited. */
		rmp->mp_parent = INIT_PROC_NR;
		parent_waiting = mproc[INIT_PROC_NR].mp_flags & WAITING;
		if (parent_waiting) cleanup(rmp);
	}
  }
}


/*===========================================================================*
 *				do_waitpid				     *
 *===========================================================================*/
PUBLIC int do_waitpid()
{
/* A process wants to wait for a child to terminate. If one is already waiting,
 * go clean it up and let this WAIT call terminate.  Otherwise, really wait.
 * Both WAIT and WAITPID are handled by this code.
 */

  register struct mproc *rp;
  int pidarg, options, children, res2;

  assert(mm_call == WAIT || mm_call == WAITPID);

  /* A process calling WAIT never gets a reply in the usual way via the
   * reply() in the main loop (unless WNOHANG is set or no qualifying child
   * exists).  If a child has already exited, the routine cleanup() sends 
   * the reply to awaken the caller.
   */

  /* Set internal variables, depending on whether this is WAIT or WAITPID. */
  pidarg  = (mm_call == WAIT ? -1 : waitpid_pid); /* first param of waitpid */
  options = (mm_call == WAIT ?  0 : waitpid_flags); /* third param of waitpid */
  if (pidarg == 0) pidarg = -mp->mp_procgrp;	/* pidarg < 0 ==> proc grp */

  /* Is there a child waiting to be collected? At this point, pidarg != 0:
   *	pidarg  >  0 means pidarg is pid of a specific process to wait for
   *	pidarg == -1 means wait for any child
   *	pidarg  < -1 means wait for any child whose process group = -pidarg
   */
  children = 0;

  /* Check dead children first */
  for (rp = mproc_deadlist; rp; rp= rp->mp_next)
  {
  	assert(rp->mp_flags & IN_USE);
  	assert(rp->mp_flags & HANGING);

	if (rp->mp_parent == who) {
		/* The value of pidarg determines which children qualify. */
		if (pidarg  > 0 && pidarg != rp->mp_pid) continue;
		if (pidarg < -1 && -pidarg != rp->mp_procgrp) continue;

		children++;		/* this child is acceptable */

		/* This child meets the pid test and has exited. */
		cleanup(rp);		/* this child has already exited */
		dont_reply = TRUE;
		return(OK);
	}
  }

  /* Continue with the living */
  for (rp = mproc_inuse; rp; rp= rp->mp_next)
  {
  	assert(rp->mp_flags & IN_USE);
  	assert(!(rp->mp_flags & HANGING));

	if (rp->mp_parent == who) {
		/* The value of pidarg determines which children qualify. */
		if (pidarg  > 0 && pidarg != rp->mp_pid) continue;
		if (pidarg < -1 && -pidarg != rp->mp_procgrp) continue;

		children++;		/* this child is acceptable */

		if ((rp->mp_flags & STOPPED) && rp->mp_sigstatus) {
			/* This child meets the pid test and is being traced.*/
			res2 =  0177 | (rp->mp_sigstatus << 8);
			reply(who, rp->mp_pid, res2, NIL_PTR);
			dont_reply = TRUE;
			rp->mp_sigstatus = 0;
			return(OK);
		}
	}
  }

  /* No qualifying child has exited.  Wait for one, unless none exists. */
  if (children > 0) {
	/* At least 1 child meets the pid test exists, but has not exited. */
	if (options & WNOHANG) return(0);    /* parent does not want to wait */
	mp->mp_flags |= WAITING;	     /* parent wants to wait */
	mp->mp_wpid = (pid_t) pidarg;	     /* save pid for later */
	dont_reply = TRUE;		     /* do not reply now though */
	return(OK);			     /* yes - wait for one to exit */
  } else {
	/* No child even meets the pid test.  Return error immediately. */
	return(ECHILD);			     /* no - parent has no children */
  }
}


/*===========================================================================*
 *				cleanup					     *
 *===========================================================================*/
PRIVATE void cleanup(child)
register struct mproc *child;	/* tells which process is exiting */
{
/* Finish off the exit of a process.  The process has exited or been killed
 * by a signal, and its parent is waiting.
 */

  int exitstatus;

  /* Wake up the parent. */
  exitstatus = (child->mp_exitstatus << 8) | (child->mp_sigstatus & 0377);
  reply(child->mp_parent, child->mp_pid, exitstatus, NIL_PTR);
  mproc[child->mp_parent].mp_flags &= ~WAITING;	/* parent no longer waiting */

  /* Release the process table entry. */
  child->mp_flags = 0;

  if (child->mp_next != NULL)
  	child->mp_next->mp_prev= child->mp_prev;
  if (child->mp_prev != NULL)
  	child->mp_prev->mp_next= child->mp_next;
  else
  {
  	assert(mproc_deadlist == child);
  	mproc_deadlist= child->mp_next;
  }
  child->mp_next= mproc_freelist;
  mproc_freelist= child;
}

/*===========================================================================*
 *				find_pidrange				     *
 *===========================================================================*/
PRIVATE void find_pidrange()
{
	struct mproc *rmp;
	int once_more;

	if (next_pid < INIT_PID + 1)
		next_pid= INIT_PID + 1;

	once_more= 1;
	while (once_more)
	{
		bad_pid= 30001;
		once_more= 0;
		if (next_pid >= bad_pid)
			next_pid= INIT_PID+1;
		for (rmp= mproc_inuse; rmp; rmp= rmp->mp_next)
		{
			if (next_pid == rmp->mp_pid || 
				next_pid == rmp->mp_procgrp)
			{
				next_pid++;
				once_more= 1;
				continue;
			}
			if (rmp->mp_pid > next_pid && rmp->mp_pid < bad_pid)
				bad_pid= rmp->mp_pid;
			if (rmp->mp_procgrp > next_pid &&
				rmp->mp_procgrp < bad_pid)
			{
				bad_pid= rmp->mp_procgrp;
			}
		}
	}
}

/*
 * $PchId: forkexit.c,v 1.5 1995/11/28 07:18:29 philip Exp $
 */
