/*
exec.c

Created:	Jan 25, 1994 by Philip Homburg <philip@cs.vu.nl>
*/

#include "mm.h"
#include <a.out.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include "assert.h"
INIT_ASSERT
#include "mproc.h"
#include "param.h"

#define MAX_ITER	8	/* Max 8 levels of nested scripts,
				 * race conditions, etc.
				 */
#define READ_CHUNK	0x10000 /* read segments in chunks to allow other
				 * processes access to FS while loading large
				 * executables.
				 */ 
struct execstate
{
	char pathname[PATH_MAX];
	struct stat stat;
	uid_t eff_uid;
	gid_t eff_gid;
	int hdr_size;
	int extra_argc;
	char extra_args_block[1024];
	char *extra_argp;
	int ign_first_arg;
	struct exec exechdr;
	vir_bytes initial_sp;
	int sepID;

	/* Calculations about arguments. */
	int user_argc;
	vir_bytes user_vecoff;
	vir_bytes user_vecsize;
	vir_bytes user_stroff;
	vir_bytes user_strsize;
	vir_bytes user_bias;
	vir_bytes extra_vecsize;
	vir_bytes extra_strsize;
	vir_bytes args_total;

	/* Calculations about the executable */
	vir_bytes entry_point;
	off_t text_filestart;
	unsigned text_filesize;
	vir_bytes text_virstart;
	vir_bytes text_virsize;
	vir_bytes text_loadstart;
	vir_bytes text_loadsize;
	int text_loadseg;
	off_t data_filestart;
	unsigned data_filesize;
	vir_bytes data_virstart;
	vir_bytes data_virsize;
	vir_bytes data_loadstart;
	vir_bytes data_loadsize;
	vir_bytes stack_loadstart;
	vir_bytes stack_loadsize;
	struct mem_map map[NR_SEGS];

	/* Resources */
	int fd;
	phys_bytes mem_start;
	phys_bytes mem_size;
	int got_map;
} state;

typedef struct ec_ent
{
	struct mproc *ece_proc;
	struct ec_ent *ece_next;
	dev_t ece_dev;
	ino_t ece_ino;
	time_t ece_ctime;
	phys_bytes ece_size;
	long ece_cost;
	long ece_score;
	unsigned long ece_overflow;
	long ece_minscore;
} ec_ent_t;
static ec_ent_t ec_ent_table[EC_ENT_NO];
static ec_ent_t *ec_hash_table[EC_HASH_NO];

static long ec_bias;
static long ec_maxdiff;
static long ec_minscore;
#define EC_MAX_BIAS		0x10000000

char block[1024];

_PROTOTYPE( static void init, (void)					);
_PROTOTYPE( static void cleanup, (void)					);
_PROTOTYPE( static int open4exec, (void)				);
_PROTOTYPE( static int is_script, (void)				);
_PROTOTYPE( static int parse_script, (void)				);
_PROTOTYPE( static int is_aout, (void)					);
_PROTOTYPE( static int exec_aout, (void)				);
_PROTOTYPE( static int calculate_args, (void)				);
_PROTOTYPE( static int calculate_aout, (void)				);
_PROTOTYPE( static int load_args, (void)				);
_PROTOTYPE( static int load_text, (struct mproc *proc)			);
_PROTOTYPE( static int load_data, (struct mproc *proc)			);
_PROTOTYPE( static int load_segment, (struct mproc *proc,
				off_t filestart, unsigned filesize,
				int loadseg, vir_bytes loadstart)	);
_PROTOTYPE( static void update_mproc, (void)				);
_PROTOTYPE( static void fs_setuid, (Uid_t realuid, Uid_t effuid)	);
_PROTOTYPE( static void fs_exec, (void)				);
_PROTOTYPE( static int fs_read, (int fd, unsigned size, int proc,
						int seg, vir_bytes ptr)	);
_PROTOTYPE( static void chdir2user, (int procnr)			);
_PROTOTYPE( static int push_extra_arg, (char *arg)			);
_PROTOTYPE( static void shtxt_register, (void)				);
_PROTOTYPE( static int shtxt_load, (struct mproc *proc)			);
_PROTOTYPE( static int valid_time, (time_t time)			);

_PROTOTYPE( static void ec_unload, (ec_ent_t *ec_ent)			);
_PROTOTYPE( static ec_ent_t *ec_get_ent, (Dev_t dev, Ino_t ino,
							time_t ctime)	);
_PROTOTYPE( static int ec_load, (void)					);
_PROTOTYPE( static void ec_load_cache, (ec_ent_t *ec_ent)		);
_PROTOTYPE( static void ec_alloc_cache, (ec_ent_t *ec_ent)		);
_PROTOTYPE( static void ec_update_bias, (void)				);
#if DEBUG
_PROTOTYPE( static int ec_check_cache, (void)				);
#endif

int do_exec()
{
	int iter;
	int r, s_errno;

	init();

	/* Fetch the initial pathname of the executable. */
	if (exec_namelen < 1 || exec_namelen > PATH_MAX)
	{
		cleanup();
		if (exec_namelen < 1)
			return EINVAL;
		else
			return ENAMETOOLONG;
	}
	if (sys_copy(who, SEG_D, (phys_bytes) exec_name,
		MM_PROC_NR, SEG_D, (phys_bytes) state.pathname,
		(phys_bytes) exec_namelen) != OK)
	{
		cleanup();
		return EFAULT;
	}
	state.pathname[exec_namelen-1]= '\0';	/* just in case */
#if DEBUG & 256
 { where(); printf("got pathname '%s'\n", state.pathname); }
#endif

	for (iter= 0; iter<MAX_ITER; iter++)
	{
		r= open4exec();
		if (r != OK)
		{
			cleanup();
			if (r == EAGAIN)
				continue;
			return r;
		}

		/* Assume that setuid handling is generic. If this is not
		 * the case, the following code should be moved to a function
		 * which should be called by the implementations of the
		 * executable types that implement setuid.
		 */
		if (state.stat.st_mode & S_ISUID)
			state.eff_uid= state.stat.st_uid;
		if (state.stat.st_mode & S_ISGID)
			state.eff_gid= state.stat.st_gid;

		state.hdr_size= read(state.fd, block, sizeof(block));
		s_errno= errno;
		if (state.hdr_size == -1)
		{
			cleanup();
			return -s_errno;
		}
		if (is_script())
		{
			r= parse_script();
			cleanup();
			if (r != OK)
				return r;
			continue;
		}
		if (is_aout())
		{
			r= exec_aout();
			cleanup();
			return r;
		}
		break;
	}

	cleanup();
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
	return ENOEXEC;
}


static void init()
{
	state.eff_uid= mp->mp_effuid;
	state.eff_gid= mp->mp_effgid;
	state.extra_argc= 0;
	state.extra_argp= &state.extra_args_block
		[sizeof(state.extra_args_block)];
	state.ign_first_arg= 0;
	state.fd= -1;
	state.mem_start= 0;
	state.got_map= 0;
}


static void cleanup()
{
	int r;

	if (state.fd != -1)
	{
		close(state.fd);
		state.fd= -1;
	}
	if (state.mem_start != 0)
	{
#if DEBUG & 256
 { where(); printf("free_mem() <- 0x%x, 0x%x\n",
		state.mem_start, state.mem_size); }
#endif
		free_mem(state.mem_start >> CLICK_SHIFT,
			state.mem_size >> CLICK_SHIFT);
		state.mem_start= 0;
	}
	if (state.got_map != 0)
	{
		r= sys_delmap(mproc_exec-mproc);
		assert(r == OK);
		state.got_map= 0;
	}
}


static int is_script()
{
	return (state.hdr_size >= 2 && block[0] == '#' && block[1] == '!');
}


static int parse_script()
{
	int argc;
	char *eol, *argp;

	block[sizeof(block)-1]= '\0';	/* Just in case */
	eol= strchr(block, '\n');
	if (eol == NULL)
		return ENOEXEC;
	*eol= '\0';

	/* Get rid of argv[0] */
	if (state.extra_argc > 0)
	{
#if DEBUG
 { where(); printf("popping '%s'\n", state.extra_argp); }
#endif
		state.extra_argc--;
		state.extra_argp += strlen(state.extra_argp)+1;
		assert(state.extra_argp <= &state.extra_args_block
			[sizeof(state.extra_args_block)]);
	}
	else
	{
#if DEBUG & 256
 { where(); printf("ignoring first argument\n"); }
#endif
		assert(state.extra_argp == &state.extra_args_block
			[sizeof(state.extra_args_block)]);
		assert(state.ign_first_arg == 0);
		state.ign_first_arg= 1;
	}

	/* Push the file name */
	if (push_extra_arg(state.pathname) != OK)
		return E2BIG;

	/* Parse the first line of the script and push the arguments. Assume
	 * that arguments are separated by tabs or spaces.
	 */
	argc= 0;
	argp= &block[strlen(block)-1];
	assert(argp >= &block[1]);
	for (; argp > &block[1]; argp--)
	{
		if (*argp != ' ' && *argp != '\t')
			continue;
		if (argp[1] != '\0')
		{
			argc++;
			if (push_extra_arg(&argp[1]) != OK)
				return E2BIG;
		}
		*argp= '\0';
	}
	if (argp[1] != '\0')
	{
		argc++;
		if (push_extra_arg(&argp[1]) != OK)
			return E2BIG;
	}
	if (argc == 0)
		return ENOEXEC;
	if (strlen(state.extra_argp)+1 > PATH_MAX)
		return ENAMETOOLONG;
	strcpy(state.pathname, state.extra_argp);
	return OK;
}


static int is_aout()
{
	int this_cpu;

	if (state.hdr_size < A_MINHDR)
	{
#if DEBUG
 { printf("is_aout: wrong hdr_size\n"); }
#endif
		return 0;
	}
	state.exechdr= *(struct exec *)block;
	if (BADMAG(state.exechdr))
	{
		return 0;
	}
	if (state.exechdr.a_hdrlen < A_MINHDR ||
		state.exechdr.a_hdrlen >= state.hdr_size)
	{
#if DEBUG
 { printf("is_aout: wrong a_hdrlen\n"); }
#endif
		return 0;
	}
	this_cpu= 0;
#if _INTEL_CPU && _WORD_SIZE == 4
	this_cpu= A_I80386;
#endif
	if (state.exechdr.a_cpu != this_cpu)
	{
#if DEBUG
 { printf("is_aout: wrong a_cpu\n"); }
#endif
		return 0;
	}
	return 1;
}


/*
exec_aout

Execute a Minix a.out binary
*/

static int exec_aout()
{
	int r, loaded;
	phys_clicks mem_clicks;
	char *basename;

	r= calculate_args();
	if (r != OK)
		return r;
	r= calculate_aout();
	if (r != OK)
		return r;

	/* Allocate memory. On a non-VM system a failure of alloc_mem indicates
	 * that all real memory is used. On a VM system we don't expect a
	 * failure.
	 */
	mem_clicks= alloc_mem(state.mem_size >> CLICK_SHIFT);
	if (mem_clicks == 0)
		return ENOMEM;
	state.mem_start= mem_clicks << CLICK_SHIFT;
#if DEBUG & 256
 { where(); printf("alloc_mem() -> 0x%x, 0x%x\n", state.mem_start,
	state.mem_size); }
#endif

	/* Adjust map[] and try to convince the kernel to use this memory.
	 * On a VM system this may fail because the kernel has it own ideas
	 * about free memory.
	 */
	state.map[SEG_T].mem_phys += mem_clicks;
	state.map[SEG_D].mem_phys += mem_clicks;
	state.map[SEG_S].mem_phys += mem_clicks;
	r= sys_newmap(mproc_exec-mproc, state.map);
	if (r != OK)
		return r;
	state.got_map= 1;

	r= load_args();
	if (r != OK)
		return r;

	loaded= 0;

	/* Only do complicated lookups if there a chance of success */
	if (ec_limit != 0)
	{
		r= ec_load();
		loaded= (r == OK);
	}

	if (!loaded)
	{
		r= load_text(mproc_exec);
		if (r != OK)
			return r;

		r= load_data(mproc_exec);
		if (r != OK)
			return r;
	}

	/* This is the point of no return. Tell FS about the exec, this gives
	 * FS the chance to stop any asynchronous I/O that might be in
	 * progress. After that we inform kernel.
	 */
	fs_exec();

	basename= strrchr(state.pathname, '/');
	if (basename == NULL)
		basename= state.pathname;
	else
		basename++;
	r= sys_exec(mp-mproc, mproc_exec-mproc, state.initial_sp,
		mp->mp_flags & TRACED, basename, state.entry_point);
	assert(r == OK);

	update_mproc();
	return OK;
}


/*
calculate_args

Calculate the inital stack size, but also where the various pieces should
be fetched, and where they should be stored in memory. 
*/
static int calculate_args()
{
	vir_bytes words[4];
	int over_flow;
	int r;

	/* Fetch the first four words of the argument vectors. In the worst
	 * case that would be argc, argv[0], NULL, NULL. Normally we expect
	 * argc, argv[0], NULL, env[0] or argc, argv[0], argv[1], xxx
	 */
	if (exec_stacklen < sizeof(words))
	{
#if DEBUG
 { where(); printf("Test1 failed\n"); }
#endif
		return EINVAL;
	}
	r= sys_copy(who, SEG_D, (phys_bytes) exec_stack,
		MM_PROC_NR, SEG_D, (phys_bytes) words,
		(phys_bytes) sizeof(words));
	if (r != OK)
		return r;

	state.user_argc= words[0];
	state.user_vecoff= sizeof(vir_bytes);

	/* Enforce at least argv[0] */
	if (state.user_argc < 1)
	{
#if DEBUG
 { where(); printf("Test2 failed\n"); }
#endif
		return EINVAL;
	}

	/* argv[0] points to the begin of the strings section, which is right
	 * after the environment vector.
	 */
	state.user_vecsize= words[1] - state.user_vecoff;
	if (words[1] < state.user_vecoff ||
		state.user_vecsize < 3*sizeof(vir_bytes) ||
		state.user_vecsize % sizeof(vir_bytes) != 0)
	{
#if DEBUG
 { where(); printf("Test3 failed\n"); }
#endif
		return EINVAL;
	}
	state.user_stroff= words[1];
	state.user_strsize= exec_stacklen - state.user_stroff;
	if (state.user_stroff > exec_stacklen)
	{
#if DEBUG
 { where(); printf("Test4 failed\n"); }
#endif
		return EINVAL;
	}

	/* Later, when the vectors are patched, we calculate the offset
	 * relative to the begin of the string section. This assumes that the
	 * argv[0] the user passed is 0. Since this is not the case, we record
	 * a bias.
	 */
	state.user_bias= state.user_stroff;

	/* Now the tricky part. It is possible that the executable is a
	 * script. This causes the first argument to be ignored. Three
	 * possible cases are:
	 * - there is an argv[1]. Simply repeat most of the calculations with
	 *   argv[1]
	 * - argv[1] == NULL (ie. there was only one argument but there is an 
	 *   envp[0]). In this case the string section starts at envp[0].
	 * - Both argv[1] and envp[0] are NULL. In this case the string
	 *   section is empty.
	 */
	if (state.ign_first_arg)
	{
		state.user_argc--;
		state.user_vecoff += sizeof(vir_bytes);
		state.user_vecsize -= sizeof(vir_bytes);
		if (words[2] != 0 || words[3] != 0)
		{
			if (words[2] != 0)
				state.user_stroff= words[2];
			else
				state.user_stroff= words[3];
			state.user_strsize= exec_stacklen - state.user_stroff;
			if (state.user_stroff > exec_stacklen)
			{
#if DEBUG
 { where(); printf("Test5 failed\n"); }
#endif
				return EINVAL;
			}
		}
		else
			state.user_strsize= 0;

		/* Reset user_bias */
		state.user_bias= state.user_stroff;
	}

	over_flow= 0;
	state.args_total= 2*sizeof(vir_bytes);		/* argc + padding */
	if (state.args_total + state.user_vecsize < state.args_total)
		over_flow= 1;
	state.args_total += state.user_vecsize;
	if (state.args_total + state.user_strsize < state.args_total)
		over_flow= 1;
	state.args_total += state.user_strsize;

	/* Now we have to total size and the individual pieces of the
	 * arguments supplied by the user. Next come the extra arguments
	 * (if any).
	 */
	state.extra_vecsize= state.extra_argc * sizeof(vir_bytes);
	state.extra_strsize= &state.extra_args_block
		[sizeof(state.extra_args_block)]- state.extra_argp;
	if (state.args_total + state.extra_vecsize < state.args_total)
		over_flow= 1;
	state.args_total += state.extra_vecsize;
	if (state.args_total + state.extra_strsize < state.args_total)
		over_flow= 1;
	state.args_total += state.extra_strsize;

	if (over_flow)
		return E2BIG;
#if 0
	where(); printf("\nu: argc= %d, vecoff= %d, vecsize= %d\n",
		state.user_argc, state.user_vecoff, state.user_vecsize);
	printf("\tstroff= %d, strsize= %d, bias= %d\n",
		state.user_stroff, state.user_strsize, state.user_bias);
	printf("e: vecsize= %d, strsize= %d\n",
		state.extra_vecsize, state.extra_strsize);
	printf("args_total= %d\n", state.args_total);
#endif
	return OK;
}


/*
calculate_aout

Calculate the location and size of the text, data, bss and stack segments.
*/
static int calculate_aout()
{
	int sep, uzp, pal;
	vir_bytes stack_virstart, stack_virsize;
	vir_bytes bss_size;

	sep= !!((state.exechdr.a_flags & 0xF0) == A_SEP);
	uzp= !!(state.exechdr.a_flags & A_UZP);
	pal= !!(state.exechdr.a_flags & A_PAL);

	state.sepID= sep;

	/* Text segment */
	state.entry_point= 0;
	state.text_filestart= state.exechdr.a_hdrlen;
	state.text_filesize= state.exechdr.a_text;
	state.text_virstart= 0;
	state.text_virsize= state.exechdr.a_text;
	if (state.text_virsize != state.exechdr.a_text)
	{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
		return ENOEXEC;		/* Overflow */
	}
	
	/* Adjust for the UZP and PAL flags. */
	if (uzp)
	{
		state.entry_point += CLICK_SIZE;
		state.text_virstart += CLICK_SIZE;
	}
	if (pal)
	{
		state.entry_point += state.exechdr.a_hdrlen;
		state.text_filestart= 0;
		state.text_filesize += state.exechdr.a_hdrlen;
		state.text_virsize += state.exechdr.a_hdrlen;

#if DEBUG & 256
		if ((state.text_virsize & (CLICK_SIZE-1)) != 0)
		{
			where();
			printf("warning inefficient PAL executable '%s'\n",
				state.pathname);
		}
#endif
	}

	/* Check a_entry, overflow, and set map info, and load info. */
	if (state.entry_point != state.exechdr.a_entry)
	{
#if DEBUG
 { printf("executable '%s' has an invalid value in a_entry\n",
			state.pathname); }
#endif
		return ENOEXEC;
	}
	if (state.text_virstart + state.text_virsize < state.text_virsize)
	{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
		return ENOEXEC;		/* Overflow */
	}
	state.map[SEG_T].mem_phys= 0;
	state.map[SEG_T].mem_vir= state.text_virstart >> CLICK_SHIFT;
	state.map[SEG_T].mem_len= (state.text_virsize+CLICK_SIZE-1)
							>> CLICK_SHIFT;
	if (((state.map[SEG_T].mem_vir + state.map[SEG_T].mem_len)
						<< CLICK_SHIFT) <
		state.text_virstart + state.text_virsize)
	{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
		return ENOEXEC;	/* Overflow */
	}
	state.text_loadstart= state.text_virstart;
	state.text_loadsize= state.text_virsize;
	state.text_loadseg= SEG_T;	/* Text sement */

	/* Data segment */
	state.data_filestart= state.text_filestart + state.text_filesize;
	state.data_filesize= state.exechdr.a_data;
	if (!sep)
	{
		/* Common I&D, data segment includes text segment. */
		state.data_virstart= state.text_virstart;
		state.data_virsize= state.text_virsize + state.data_filesize;
		if (state.data_virsize != state.text_virsize +
			state.data_filesize)
		{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
			return ENOEXEC;		/* Overflow */
		}
		state.data_loadstart= state.data_virstart + state.text_virsize;
		state.data_loadsize= state.data_virsize;

		/* Adjust text segment */
		state.text_loadseg= SEG_D;  /* Load text in data segment also */
		state.text_virsize= 0;	/* Empty text segment */
		state.map[SEG_T].mem_len= 0;	/* Also empty in clicks */
		state.map[SEG_D].mem_phys= 0;	/* Start at the same place as
						 * the text segment */
		state.map[SEG_D].mem_vir= state.map[SEG_T].mem_vir;

		/* Defer calculation of mem_len until bss is added */
	}
	else
	{
		/* Separate I&D */
		if (uzp)
			state.data_virstart= CLICK_SIZE;
		else
			state.data_virstart= 0;
		state.data_virsize= state.data_filesize;
		if (state.data_virsize != state.data_filesize)
		{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
			return ENOEXEC;		/* Overflow */
		}
		state.data_loadstart= state.data_virstart;
		state.data_loadsize= state.data_virsize;

		/* Start right after the text segment */
		state.map[SEG_D].mem_phys= state.map[SEG_T].mem_vir +
			state.map[SEG_T].mem_len;
		state.map[SEG_D].mem_vir= state.data_virstart >> CLICK_SHIFT;

		/* Defer calculation of mem_len until bss is added */
	}

	/* Bss */
	bss_size= state.exechdr.a_bss;
	if (bss_size != state.exechdr.a_bss ||
		state.data_virsize + bss_size < state.data_virsize)
	{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
		return ENOEXEC;		/* Overflow */
	}
	state.data_virsize += bss_size;
	state.map[SEG_D].mem_len= (state.data_virsize + CLICK_SIZE-1) >>
		CLICK_SHIFT;
	if (((state.map[SEG_D].mem_vir + state.map[SEG_D].mem_len)
						<< CLICK_SHIFT) <
		state.data_virstart + state.data_virsize)
	{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
		return ENOEXEC;	/* Overflow */
	}

	/* Correct a design error: a_total should have included PAL header
	 * and UZP unmapped page.
	 */
	if (uzp) state.exechdr.a_total+= CLICK_SIZE;
	if (pal && !sep) state.exechdr.a_total+= state.exechdr.a_hdrlen;

	/* Calculate the start of the stack segment */
	stack_virstart= state.exechdr.a_total;
	if (stack_virstart != state.exechdr.a_total)
	{
		/* Also try a_total-1 */
		stack_virstart= state.exechdr.a_total-1;
		if (stack_virstart != state.exechdr.a_total-1)
		{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
			return ENOEXEC;		/* Overflow */
		}
	}
	if (stack_virstart < state.data_virstart + state.data_virsize)
	{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
		return ENOEXEC;		/*  Strange value for a_total */
	}
#ifdef MIN_GAP_SIZE
	if (stack_virstart < state.data_virstart + state.data_virsize +
		MIN_GAP_SIZE)
	{
		stack_virstart= state.data_virstart + state.data_virsize +
		MIN_GAP_SIZE;
	}
#endif
	/* Try to start the argumets at stack_virstart. In case of overflow,
	 * try to end arguments at stack_virstart
	 */
	stack_virsize= state.args_total;
	if (stack_virstart + stack_virsize < stack_virstart)
	{
		/* Overflow */
		if (stack_virstart < stack_virsize)
		{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
			return ENOEXEC;		/* Overflow */
		}
		stack_virstart -= stack_virsize;
	}
	state.stack_loadstart= stack_virstart;
	state.stack_loadsize= stack_virsize;

	state.map[SEG_S].mem_phys= state.map[SEG_D].mem_phys;
	state.map[SEG_S].mem_vir= stack_virstart >> CLICK_SHIFT;
	state.map[SEG_S].mem_len= ((stack_virstart + stack_virsize
		+ CLICK_SIZE-1) >> CLICK_SHIFT) - state.map[SEG_S].mem_vir;
	
	if (state.map[SEG_S].mem_vir < state.map[SEG_D].mem_vir +
		state.map[SEG_D].mem_len + 1)
	{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
		return ENOEXEC;		/* Stack too small */
	}
	if (((state.map[SEG_S].mem_vir + state.map[SEG_S].mem_len)
						<< CLICK_SHIFT) <
		state.data_virstart + state.data_virsize)
	{
#if DEBUG
 { where(); printf("ENOEXEC\n"); }
#endif
		return ENOEXEC;	/* Overflow */
	}

#if 0
	where(); printf("\ntest:\n\t");
	printf("entry_point= 0x%x, ", state.entry_point);
	printf("filestart= 0x%lx, ", state.text_filestart);
	printf("filesize= 0x%x, ", state.text_filesize);
	printf("virstart= 0x%x, ", state.text_virstart);
	printf("virsize= 0x%x, ", state.text_virsize);
	printf("loadstart= 0x%x, ", state.text_loadstart);
	printf("loadsize= 0x%x, ", state.text_loadsize);
	printf("loadseg= %d, ", state.text_loadseg);
	printf("\ndata:\n\t");
	printf("filestart= 0x%lx, ", state.data_filestart);
	printf("filesize= 0x%x, ", state.data_filesize);
	printf("virstart= 0x%x, ", state.data_virstart);
	printf("virsize= 0x%x, ", state.data_virsize);
	printf("loadstart= 0x%x, ", state.data_loadstart);
	printf("loadsize= 0x%x, ", state.data_loadsize);
	printf("\nstack:\n\t");
	printf("virstart= 0x%x, ", stack_virstart);
	printf("virsize= 0x%x, ", stack_virsize);
	printf("\n");
	printf("T: p= 0x%x, v= 0x%x, l= 0x%x\n", state.map[SEG_T].mem_phys, 
		state.map[SEG_T].mem_vir, state.map[SEG_T].mem_len);
	printf("D: p= 0x%x, v= 0x%x, l= 0x%x\n", state.map[SEG_D].mem_phys, 
		state.map[SEG_D].mem_vir, state.map[SEG_D].mem_len);
	printf("S: p= 0x%x, v= 0x%x, l= 0x%x\n", state.map[SEG_S].mem_phys, 
		state.map[SEG_S].mem_vir, state.map[SEG_S].mem_len);
#endif

	/* Calculate the memory requirements */
	state.mem_size= (state.map[SEG_S].mem_phys + state.map[SEG_S].mem_vir +
		state.map[SEG_S].mem_len) << CLICK_SHIFT;
	return OK;
}


/*
load_args

Load the arguments and the environment at the top of the stack segment.
*/
static int load_args()
{
	int args, env, args_seen, env_seen, cnt;
	vir_bytes *vp, *evp;
	char *cp, *ep;
	vir_bytes stack_top;
	vir_bytes extra_vecstart, user_vecstart;
	vir_bytes extra_strstart, user_strstart;
	vir_bytes user_vecsize, size, putsize, src_vecstart;
	size_t l;
	int r;

	stack_top= state.stack_loadstart + state.stack_loadsize;
	user_strstart= stack_top - state.user_strsize;
	extra_strstart= user_strstart - state.extra_strsize;
	user_vecstart= extra_strstart - state.user_vecsize;
	user_vecstart &= ~(sizeof(vir_bytes)-1); /* Round down for alignment */
	extra_vecstart= user_vecstart - state.extra_vecsize;
	extra_vecstart -= sizeof(vir_bytes);		/* argc */

	state.initial_sp= extra_vecstart;
	assert(extra_vecstart >= state.stack_loadstart);

	if (state.user_strsize != 0)
	{
		r= sys_copy(mp-mproc, SEG_D,
			(phys_bytes) exec_stack + state.user_stroff,
			mproc_exec-mproc, SEG_D, (phys_bytes) user_strstart,
			(phys_bytes) state.user_strsize);
		if (r != OK)
		{
#if DEBUG
 { where(); printf("failure\n"); }
#endif
			return r;
		}
	}
	if (state.extra_strsize != 0)
	{
		r= sys_copy(MM_PROC_NR, SEG_D, (phys_bytes) state.extra_argp,
			mproc_exec - mproc, SEG_D, (phys_bytes) extra_strstart,
			(phys_bytes) state.extra_strsize);
		assert(r == OK);
	}

	/* Create a vector with argc and the extra arg pointers */
	vp= (vir_bytes *)state.extra_args_block;
	cp= state.extra_argp;
	ep= &state.extra_args_block[sizeof(state.extra_args_block)];
	if (cp < (char *)&vp[1])
	{
#if DEBUG
 { where(); printf("failure\n"); }
#endif
		return E2BIG;
	}
	*vp= state.extra_argc + state.user_argc;
	vp++;
	while (cp < ep)
	{
		if (cp < (char *)&vp[1])
		{
#if DEBUG
 { where(); printf("failure\n"); }
#endif
			return E2BIG;
		}
		*vp= extra_strstart + (cp - state.extra_argp);
		vp++;
		l= strlen(cp)+1;
		cp += l;
	}
	assert(cp == ep);
	user_vecsize= state.user_vecsize;
	src_vecstart= (vir_bytes)exec_stack + state.user_vecoff;
	cnt= 0;
	args_seen= env_seen= 0;
	while (user_vecsize != 0)
	{
		size= user_vecsize;
		if ((char *)vp + size > ep)
			size= ep - (char *)vp;
		r= sys_copy(who, SEG_D, (phys_bytes) src_vecstart,
			MM_PROC_NR, SEG_D, (phys_bytes) vp, (phys_bytes) size);
		if (r != OK)
		{
#if DEBUG
 { where(); printf("failure\n"); }
#endif
			return r;
		}
		evp= (vir_bytes *)((char *)vp + size);
		assert((evp-vp) * sizeof(*vp) == size);
		for (; vp< evp; vp++)
		{
			cnt++;
			if (*vp == 0)
			{
				if (!args_seen)
				{
					args= cnt;
					args_seen= 1;
					cnt= 0;
				}
				else if (!env_seen)
				{
					env= cnt;
					env_seen= 1;
					cnt= 0;
				}
				else
				{
#if DEBUG
 { where(); printf("failure\n"); }
#endif
					return EINVAL;
				}
				continue;
			}
			*vp += user_strstart - state.user_bias;
		}
		putsize= (char *)vp - state.extra_args_block;
		r= sys_copy(MM_PROC_NR,SEG_D,(phys_bytes)state.extra_args_block,
			mproc_exec - mproc, SEG_D, (phys_bytes) extra_vecstart,
			(phys_bytes) putsize);
		assert(r == OK);
		vp= (vir_bytes *)state.extra_args_block;
		extra_vecstart += putsize;
		src_vecstart += size;
		user_vecsize -= size;
	}
	putsize= (char *)vp - state.extra_args_block;
	r= sys_copy(MM_PROC_NR, SEG_D, (phys_bytes) state.extra_args_block,
		mproc_exec - mproc, SEG_D, (phys_bytes) extra_vecstart,
		(phys_bytes) putsize);
	assert(r == OK);
	extra_vecstart += putsize;
	assert(extra_vecstart == user_vecstart + state.user_vecsize);
	assert(user_vecsize == 0);

	if (!args_seen || !env_seen || args != state.user_argc+1 ||
		(args+env)*sizeof(vir_bytes) != state.user_vecsize)
	{
#if DEBUG
 { where(); printf("failure\n");
	printf(
"as= %d, es= %d, args= %d, env= %d, user_argc= %d, user_vecsize= %d\n",
		args_seen, env_seen, args, env, state.user_argc,
		state.user_vecsize); }
#endif
		return EINVAL;
	}
	return OK;
}


/*
load_text

Load the text segment.
*/
static int load_text(proc)
struct mproc *proc;
{
	int r;

	r= shtxt_load(proc);
	if (r == OK)
		return r;
	return load_segment(proc, state.text_filestart, state.text_filesize,
		state.text_loadseg, state.text_loadstart);
}


/*
load_data

Load the data segment.
*/
static int load_data(proc)
struct mproc *proc;
{
	return load_segment(proc, state.data_filestart, state.data_filesize,
		SEG_D, state.data_loadstart);
}


/*
load_segment

Load the text or the data segment.
*/
static int load_segment(proc, filestart, filesize, loadseg, loadstart)
struct mproc *proc;
off_t filestart;
unsigned filesize;
int loadseg;
vir_bytes loadstart;
{
	off_t o;
	unsigned s;
	int r;

	o= lseek(state.fd, filestart, SEEK_SET);
	if (o == -1)
		return -errno;

	o= 0;
	while(o < filesize)
	{
		s= filesize-o;
		if (s > READ_CHUNK)
			s= READ_CHUNK;
		r= fs_read(state.fd, s, proc-mproc, loadseg,
			(vir_bytes)(loadstart+o));
		if (r != OK)
			return r;
		o += s;
	}
	return OK;
}


/*
update_mproc

Update fields in the process structure to reflect the now succesful exec.
*/
static void update_mproc()
{
	state.got_map= FALSE;
	if (mp->mp_virstart != 0)
	{
		free_mem(mp->mp_virstart >> CLICK_SHIFT,
			mp->mp_virsize >> CLICK_SHIFT);
	}
	mp->mp_virstart= state.mem_start;
	mp->mp_virsize= state.mem_size;
	state.mem_start= 0;

	if (state.eff_uid != mp->mp_effuid)
	{
		mp->mp_effuid= state.eff_uid;
		tell_fs(SETUID, mp-mproc, (int) mp->mp_realuid,
			(int) mp->mp_effuid);
	}
	if (state.eff_gid != mp->mp_effgid)
	{
		mp->mp_effgid= state.eff_gid;
		tell_fs(SETGID, mp-mproc, (int) mp->mp_realgid,
			(int) mp->mp_effgid);
	}

	/* Remember the real and effective uid's and gid's. */
	mp->mp_saveuid[0] = mp->mp_realuid;
	mp->mp_saveuid[1] = mp->mp_effuid;
	mp->mp_savegid[0] = mp->mp_realgid;
	mp->mp_savegid[1] = mp->mp_effgid;

	sigemptyset(&mp->mp_catch);	/* reset caught signals to default */

	shtxt_register();
}

static int open4exec()
{
	struct stat userstat;
	int s1, s2, fd, s1_errno, fd_errno, i;

	assert(state.fd == -1);

	/* Change the root and current working directory of MM to the
	 * user's. This will also change the real and effective uid of MM
	 * to the user's effective uid.
	 */
	chdir2user(mp-mproc);
	s1= stat(state.pathname, &userstat);
	s1_errno= errno;

	/* Only open regular files. */
	if (s1 != -1 && !S_ISREG(userstat.st_mode))
	{
		s1= -1;
		s1_errno= EACCES;
	}
	
	/* Switch to root privileges to open the executable no matter what. */
	fs_setuid(SUPER_USER, SUPER_USER);

	/* Open the file if the stat succeeded. */
	if (s1 != -1)
	{
		fd= open(state.pathname, O_RDONLY | O_NONBLOCK);
		fd_errno= errno;
	}
	else
		fd= -1;

	if (fd != -1)
		state.fd= fd;

	/* Switch to the root and working directory of FS to return to normal
	 * operation.
	 */
	chdir2user(FS_PROC_NR);

	if (s1 == -1)
	{
		/* The stat failed, no need to continue. */
		return -s1_errno;
	}
	if (fd == -1)
	{
#if DEBUG
 { printf("exec: open failed: %s\n", strerror(fd_errno)); }
#endif
		return -fd_errno;
	}

	if (userstat.st_uid == state.eff_uid)
	{
		if (!(userstat.st_mode & S_IXUSR))
			return EACCES;
	}
	else if (userstat.st_gid == state.eff_gid)
	{
		if (!(userstat.st_mode & S_IXGRP))
			return EACCES;
	}
	else
	{
		for (i= 0; i < mp->mp_ngroups; i++)
		{
			if (userstat.st_gid == mp->mp_groups[i])
				break;
		}

		if (i < mp->mp_ngroups)
		{
			if (!(userstat.st_mode & S_IXGRP))
				return EACCES;
		}
		else if (!(userstat.st_mode & S_IXOTH))
			return EACCES;
	}

	/* Everything looks alright, a regular file with the appropriate
	 * permissions. We only have to ensure that the file we just opened
	 * as root is in fact the same file as the one in the stat buffer.
	 * An fstat and a compare of the major/minor and inodes will take
	 * care of this.
	 */
	s2= fstat(fd, &state.stat);
	assert(s2 != -1);

	if (state.stat.st_dev != userstat.st_dev ||
		state.stat.st_ino != userstat.st_ino)
	{
#if DEBUG & 256
 { where(); printf("We have been fooled!\n"); }
#endif
		return EAGAIN;
	}
	return OK;
}


static void fs_setuid(realuid, effuid)
uid_t realuid;
uid_t effuid;
{
	message m;
	int r;

	m.m1_i1= MM_PROC_NR;
	m.m1_i2= realuid;
	m.m1_i3= effuid;
	m.m1_p1= 0;
	m.m1_p2= 0;
	m.m1_p3= 0;

	r= _taskcall(FS_PROC_NR, SETUID, &m);
	assert(r == OK);
}


static void fs_exec()
{
	message m;
	int r;

	m.m1_i1= mp-mproc;
	m.m1_i2= 0;
	m.m1_i3= 0;
	m.m1_p1= 0;
	m.m1_p2= 0;
	m.m1_p3= 0;

	r= _taskcall(FS_PROC_NR, EXEC, &m);
	assert(r == OK);
}


static int fs_read(fd, nbytes, proc, seg, ptr)
int fd;
unsigned nbytes;
int proc;
int seg;
vir_bytes ptr;
{
	message m;
	int r;

	assert(seg >= 0 && seg < NR_SEGS);

	m.m1_i1 = fd;
	m.m1_i2 = nbytes;
	m.m1_i3= (proc << 4) | seg;
	m.m1_p1 = (char *) ptr;

	/* Clear unused fields */
	m.m1_p2= 0;
	m.m1_p3= 0;

	r= _taskcall(FS_PROC_NR, READ, &m);
	if (r < 0)
		return r;
	if (r != nbytes)
		return EIO;
	return OK;
}


static void chdir2user(procnr)
int procnr;
{
	message m;
	int r;

	m.m1_i1= procnr;
	m.m1_i2= 0;
	m.m1_i3= 0;
	m.m1_p1= 0;
	m.m1_p2= 0;
	m.m1_p3= 0;

	r= _taskcall(FS_PROC_NR, CHDIR, &m);
	assert(r == OK);
}


static int push_extra_arg(arg)
char *arg;
{
	size_t l;

	l= strlen(arg)+1;
	if (state.extra_argp - state.extra_args_block < l)
		return ENOMEM;
	state.extra_argp -= l;
	strcpy(state.extra_argp, arg);
	state.extra_argc++;
#if DEBUG & 256
 { where(); printf("pushed '%s'\n", arg); }
#endif
	return OK;
}

/*
Shared Text.

Here starts the code for shared text. Shared text means that when an
executable is started and some process is already running that executable,
then the new text segment will be loaded from the existing text segment
(or in the case of a VM system, share the pages with the exiting segment).
This is only done for executable with separate I&D that or not traced.
*/

/*
shtxt_register

Store the major/minor, inode number and ctime in mproc if the executable is
separate I&D, not traced, and the ctime is valid.
*/

static void shtxt_register()
{
	if ((mp->mp_flags & TRACED) || !state.sepID ||
		!valid_time(state.stat.st_ctime))
	{
		mp->mp_device= mp->mp_ino= mp->mp_ctime= 0;
		return;
	}
	else
	{
		mp->mp_device= state.stat.st_dev;
		mp->mp_ino= state.stat.st_ino;
		mp->mp_ctime= state.stat.st_ctime;
	}
}


/*
shtxt_load

Check if the executable is separate I&D and the process is not traced,
locate a suitable segment and tell the kernel to share this segment.
*/
static int shtxt_load(proc)
struct mproc *proc;
{
	struct mproc *pp;

	if ((mp->mp_flags & TRACED) || !state.sepID)
		return -1;
	for (pp= mproc_inuse; pp; pp= pp->mp_next)
	{
		if (pp->mp_device == state.stat.st_dev &&
			pp->mp_ino == state.stat.st_ino &&
			pp->mp_ctime == state.stat.st_ctime)
		{
			break;
		}
	}
	if (pp == NULL)
		return -1;

#if DEBUG & 256
 { where(); printf("found cache for '%s': %d\n", state.pathname, pp-mproc); }
#endif

	return sys_dupseg(pp-mproc, proc-mproc, (1 << SEG_T));
}


/*
valid_time

Determine if a ctime value is valid. Valid times lay in the past!
*/
static int valid_time(time)
time_t time;
{
	static time_t recent_time= 0;
	static time_t future_time= 0;

	if (time < recent_time)
		return 1;

	recent_time= get_time();

	if (time == recent_time)
		return 0;		/* This is the case we want to trap */

	if (time < recent_time)
		return 1;

	if (recent_time < 60 || time < recent_time + 24 * 3600L)
	{
		/* Don't say anything if the time is in the first minute of
		 * January 1, 1970. We assume that the clocks has not been
		 * set yet.  Also don't mind if the time is a day in the
		 * future, someone may have adjusted the clock.
		 */
		return 0;
	}

	/* Time is in the future! Sometimes we say something about it. */
	if (time > future_time)
	{
		printf("valid_time: time %d is in the future, now is %d\n",
			time, get_time());
		future_time= time;
	}
	return 0;
}

/*
 * The executable cache code starts here.
 */

/*
ec_adjust_usage

Make sure that the memory usage doesn't exceed the limit.
*/
void ec_adjust_usage()
{
	while (ec_usage > ec_limit)
	{
		assert(mproc_eclist != NULL);
		ec_free_pslot();
	}
}

/*
ec_free_pslot

Unload one cached executable (if there is one).
*/
void ec_free_pslot()
{
	ec_ent_t *ec_ent;
	struct mproc *pslot;

	pslot= mproc_eclist;
	if (pslot == NULL)
		return;

	ec_ent= ec_get_ent(pslot->mp_device, pslot->mp_ino, pslot->mp_ctime);
	assert(ec_ent->ece_proc == pslot);
	ec_unload(ec_ent);
	assert(mproc_eclist != pslot);
}


/*
ec_unload

Destroy the cached image of an executable and remove the process slot entry
from the mproc_eclist.
*/
static void ec_unload(ec_ent)
ec_ent_t *ec_ent;
{
	struct mproc *proc;
	int r;

#if DEBUG & 256
 { where(); printf("ec_unload()\n"); }
#endif

	proc= ec_ent->ece_proc;
	assert(proc != NULL);
	r= sys_delmap(proc-mproc);
	assert(r == OK);
	free_mem(proc->mp_virstart >> CLICK_SHIFT,
		proc->mp_virsize >> CLICK_SHIFT);
	proc->mp_virstart= proc->mp_virsize= 0;
	ec_ent->ece_proc= NULL;
	if (proc->mp_prev != NULL)
		proc->mp_prev->mp_next= proc->mp_next;
	else
		mproc_eclist= proc->mp_next;
	if (proc->mp_next != NULL)
		proc->mp_next->mp_prev= proc->mp_prev;
	proc->mp_device= 0;
	ec_usage -= ec_ent->ece_size;

	proc->mp_next= mproc_freelist;
	mproc_freelist= proc;
	if (proc->mp_next != NULL)
		proc->mp_next->mp_prev= proc;
}


/*
ec_get_ent

Locate and possibly create an entry in the executable cache table for 
this executable.
*/
static ec_ent_t *ec_get_ent(dev, ino, ctime)
dev_t dev;
ino_t ino;
time_t ctime;
{
	static ec_ent_t *next_ent= ec_ent_table;
	ec_ent_t *prev, *curr;
	int hash, hash2;

	hash= ec_hash(dev, ino, ctime);

	for (curr= ec_hash_table[hash]; curr; curr= curr->ece_next)
	{
		if (curr->ece_dev == dev && curr->ece_ino == ino &&
			curr->ece_ctime == ctime)
		{
			return curr;
		}
	}

#if DEBUG & 256
 { where(); printf("ec_get_ent: creating entry for 0x%x,%d,%d: %s\n",
	dev, ino, ctime, state.pathname); }
#endif

	for(;;)
	{
		if (next_ent == &ec_ent_table[EC_ENT_NO])
			next_ent= ec_ent_table;
		if (next_ent->ece_proc == NULL)
			break;
		next_ent++;
	}
	hash2= ec_hash(next_ent->ece_dev, next_ent->ece_ino,
		next_ent->ece_ctime);
	for (prev= NULL, curr= ec_hash_table[hash2]; curr; prev= curr,
		curr= curr->ece_next)
	{
		if (curr == next_ent)
			break;
	}

	if (curr != NULL)
	{
		if (prev != NULL)
			prev->ece_next= curr->ece_next;
		else
			ec_hash_table[hash2]= curr->ece_next;
	}
	else
	{
		assert(next_ent->ece_dev == 0 && next_ent->ece_ino == 0 &&
			next_ent->ece_ctime == 0);
		curr= next_ent;
	}

	next_ent++;

	curr->ece_next= ec_hash_table[hash];
	ec_hash_table[hash]= curr;
	curr->ece_proc= NULL;
	curr->ece_dev= dev;
	curr->ece_ino= ino;
	curr->ece_ctime= ctime;
	curr->ece_size= 0;
	curr->ece_cost= 0;
	curr->ece_score= ec_bias;
	curr->ece_overflow= 0;
	curr->ece_minscore= 0;

#if DEBUG & 256
 { where(); printf("ec_get_ent: newent %d, hash %d\n",
 	curr-ec_ent_table, hash); }
#endif
	return curr;
}


/*
ec_load

Try to load a cached executable, possibly by creating the cache first.
*/
static int ec_load()
{
	ec_ent_t *ec_ent;
	long overflow;
	struct mproc *proc;
	int r;

	if (ec_bias > EC_MAX_BIAS)
		ec_update_bias();

	ec_ent= ec_get_ent(state.stat.st_dev, state.stat.st_ino,
		state.stat.st_ctime);

	if (ec_ent->ece_size == 0)
	{
		ec_ent->ece_size= (state.map[SEG_T].mem_len +
			state.map[SEG_D].mem_len) << CLICK_SHIFT;
		ec_ent->ece_cost= ec_ent->ece_size >> 16;
		if (ec_ent->ece_cost == 0)
			ec_ent->ece_cost= 1;
	}

	if (ec_ent->ece_proc == NULL)
	{
		ec_ent->ece_proc= mproc_exec;	/* Lock this entry */

		ec_alloc_cache(ec_ent);
		ec_load_cache(ec_ent);

		if (ec_ent->ece_proc == NULL)
		{
			ec_ent->ece_score += ec_ent->ece_cost;
			if (ec_ent->ece_score > ec_bias + ec_maxdiff)
				ec_bias= ec_ent->ece_score-ec_maxdiff;
			else if (ec_ent->ece_score < ec_bias - ec_maxdiff)
				ec_ent->ece_score= ec_bias - ec_maxdiff;

			return EGENERIC;
		}
	}
	if (mp->mp_flags & TRACED)
		return EGENERIC;

#if DEBUG & 256
 { where(); printf("ec_load: using executable cache for %s\n",
 	state.pathname); }
#endif
	proc= ec_ent->ece_proc;
	if (state.sepID)
	{
		r= sys_dupseg(proc-mproc, mproc_exec-mproc,
			(1 << SEG_T) | (1 << SEG_D));
	}
	else
		r= sys_dupseg(proc-mproc, mproc_exec-mproc, (1 << SEG_D));
	if (r != OK)
		return r;
	ec_ent->ece_score += ec_ent->ece_cost;
	if (ec_ent->ece_score < ec_bias - ec_maxdiff)
		ec_ent->ece_score= ec_bias - ec_maxdiff;
	else if (ec_ent->ece_score > ec_bias + ec_maxdiff)
	{
		overflow= ec_ent->ece_score - (ec_bias + ec_maxdiff);
		ec_ent->ece_overflow += overflow;
		ec_ent->ece_score= ec_bias + ec_maxdiff;
	}
	return OK;
}


/*
ec_load_cache

Try to cache an executable.
*/
static void ec_load_cache(ec_ent)
ec_ent_t *ec_ent;
{
	struct mproc *proc;
	int r;
	phys_clicks vir_start, vir_size, phys_bias;
	struct mem_map map[NR_SEGS];

	assert(ec_ent->ece_proc == mproc_exec);
	ec_ent->ece_proc= NULL;					/* Unlock */

	if (mproc_freelist == NULL || ec_usage + ec_ent->ece_size > ec_limit)
		return;
	if (!valid_time(ec_ent->ece_ctime))
		return;

	proc= mproc_freelist;
	mproc_freelist= proc->mp_next;

	map[SEG_T]= state.map[SEG_T];
	map[SEG_D]= state.map[SEG_D];
	map[SEG_S].mem_phys= map[SEG_D].mem_phys;
	map[SEG_S].mem_vir= map[SEG_D].mem_vir + map[SEG_D].mem_len + 2;
	map[SEG_S].mem_len= 0;
	phys_bias= map[SEG_T].mem_phys;
	vir_size= map[SEG_S].mem_phys + map[SEG_S].mem_vir
		+ map[SEG_S].mem_len - map[SEG_T].mem_phys;
	vir_start= alloc_mem(vir_size);
	if (vir_start == 0)
	{
		mproc_freelist= proc;
		return;
	}
	map[SEG_T].mem_phys= vir_start;
	map[SEG_D].mem_phys += vir_start - phys_bias;
	map[SEG_S].mem_phys= map[SEG_D].mem_phys;

	r= sys_newmap(proc-mproc, map);
	if (r != OK)
	{
		free_mem(vir_start, vir_size);
		mproc_freelist= proc;
		return;
	}
	r= load_text(proc);
	if (r == OK)
		r= load_data(proc);
	if (r != OK)
	{
		r= sys_delmap(proc-mproc);
		assert(r == OK);
		free_mem(vir_start, vir_size);
		mproc_freelist= proc;
		return;
	}
	assert(proc->mp_virstart == 0);
	proc->mp_virstart= (vir_start << CLICK_SHIFT);
	proc->mp_virsize= (vir_size << CLICK_SHIFT);
	proc->mp_next= mproc_eclist;
	proc->mp_device= ec_ent->ece_dev;
	proc->mp_ino= ec_ent->ece_ino;
	proc->mp_ctime= ec_ent->ece_ctime;
	mproc_eclist= proc;
	proc->mp_prev= NULL;
	if (proc->mp_next != NULL)
		proc->mp_next->mp_prev= proc;
	ec_ent->ece_proc= proc;
	ec_usage += ec_ent->ece_size;
	if (ec_ent->ece_cost > ec_maxdiff)
		ec_maxdiff= ec_ent->ece_cost;
	ec_ent->ece_overflow= 0;
#if DEBUG & 256
 { where(); printf("ec_load_cache: loaded '%s'\n", state.pathname); }
#endif
}


/*
ec_alloc_cache

Try to create enough free space in the cache to load an executable.
*/
static void ec_alloc_cache(ec_ent)
ec_ent_t *ec_ent;
{
	struct mproc *list, *curr, *head, *tail, *vhead, *vtail;
	ec_ent_t *curr_ent;
	long add, c_score, score, maxdiff, minscore, myminscore;
	long overflow;
	phys_bytes freesize;

	/* Update the score in case ec_maxdiff changed. */
	if (ec_ent->ece_score > ec_bias + ec_maxdiff)
		ec_bias= ec_ent->ece_score-ec_maxdiff;
	else if (ec_ent->ece_score < ec_bias - ec_maxdiff)
		ec_ent->ece_score= ec_bias - ec_maxdiff;
	score= ec_ent->ece_score;

	/* First a few simple checks */
	if (score <= ec_minscore || score <= ec_ent->ece_minscore)
		return;
	if (ec_ent->ece_size > ec_limit)
		return;

	/* Now select the victims to be trown out. Also calculate three bounds:
	 * the lowest score among the cached entries, the lowest score higher
	 * than then score of the current executable and the maximum cost.
	 */
	list= mproc_eclist;
	mproc_eclist= NULL;
	vhead= NULL;
	vtail= NULL;
	head= NULL;
	tail= NULL;
	minscore= ec_bias + ec_maxdiff+1;
	myminscore= ec_bias + ec_maxdiff+1;
	maxdiff= 0;
	freesize= 0;
	while(list)
	{
		curr= list;
		list= list->mp_next;

		curr_ent= ec_get_ent(curr->mp_device, curr->mp_ino,
			curr->mp_ctime);
		assert(curr_ent->ece_proc == curr);

		if (curr_ent->ece_score < ec_bias - ec_maxdiff)
			curr_ent->ece_score= ec_bias - ec_maxdiff;
		else if (curr_ent->ece_score > ec_bias + ec_maxdiff)
		{
			overflow= curr_ent->ece_score - (ec_bias + ec_maxdiff);
			curr_ent->ece_overflow += overflow;
			curr_ent->ece_score= ec_bias + ec_maxdiff;
		}
		c_score= curr_ent->ece_score;
		if (c_score < score)
		{

			if (c_score < ec_bias)
				add= ec_bias + ec_maxdiff - c_score;
			else
				add= ec_maxdiff;
			if (add > curr_ent->ece_overflow)
			{
				add= curr_ent->ece_overflow;
				curr_ent->ece_overflow= 0;
			}
			else
			{
				curr_ent->ece_overflow=
					(curr_ent->ece_overflow - add) >> 1;
			}
			curr_ent->ece_score += add;
			c_score= curr_ent->ece_score;
			if (c_score > ec_bias + ec_maxdiff)
				ec_bias= c_score - ec_maxdiff;
		}
		if (c_score < minscore)
			minscore= c_score;
		if (curr_ent->ece_cost > maxdiff)
			maxdiff= curr_ent->ece_cost;
		if (c_score >= score)
		{
			if (!head)
				head= curr;
			else
				tail->mp_next= curr;
			tail= curr;
			curr->mp_next= NULL;

			if (c_score < myminscore)
				myminscore= c_score;
			continue;
		}
		if (!vhead)
			vhead= curr;
		else
			vtail->mp_next= curr;
		vtail= curr;
		curr->mp_next= NULL;
		freesize += curr_ent->ece_size;
	}
	if (ec_usage - freesize + ec_ent->ece_size <= ec_limit &&
		vhead != NULL)
	{
		while(ec_usage + ec_ent->ece_size > ec_limit ||
			mproc_freelist == NULL)
		{
			curr= vhead;
			assert(curr != NULL);
			vhead= curr->mp_next;
			mproc_eclist= curr;
			curr->mp_next= curr->mp_prev= NULL;
			curr_ent= ec_get_ent(curr->mp_device,
				curr->mp_ino, curr->mp_ctime);
			ec_unload(curr_ent);
			assert(mproc_eclist == NULL);
		}
	}

	/* Copy both lists in mproc_eclist */
	mproc_eclist= NULL;
	tail= NULL;
	while(vhead || head)
	{
		if (vhead)
		{
			curr= vhead;
			vhead= vhead->mp_next;
		}
		else
		{
			curr= head;
			head= head->mp_next;
		}

		if (!mproc_eclist)
		{
			mproc_eclist= curr;
			curr->mp_prev= NULL;
		}
		else
		{
			tail->mp_next= curr;
			curr->mp_prev= tail;
		}
		tail= curr;
		curr->mp_next= NULL;
	}

	/* Update statistics */
	if (minscore < ec_bias + ec_maxdiff + 1)
		ec_minscore= minscore;
	else
	{
		assert(mproc_eclist == NULL);
		ec_minscore= 0;
	}
	if (ec_usage + ec_ent->ece_size > ec_limit || mproc_freelist == NULL)
	{
		if (myminscore < ec_bias + ec_maxdiff + 1)
			ec_ent->ece_minscore= myminscore;
		else
		{
			assert ((mproc_eclist == NULL &&
				mproc_freelist == NULL) ||
			(
				printf(
		"ec_alloc_cache: ec_usage= %d, ece_size= %d, ec_limit= %d\n",
					ec_usage, ec_ent->ece_size, ec_limit),
				printf(
			"myminscore= %d, ec_bias= %d, ec_maxdiff= %d\n",
					myminscore, ec_bias, ec_maxdiff),
				printf(
		"mproc_eclist= 0x%x, mproc_freelist= 0x%x, maxdiff= %d\n",
					mproc_eclist, mproc_freelist,
					maxdiff),
				0));
		}
	}
	ec_maxdiff= maxdiff;
}


static void ec_update_bias()
{
	ec_ent_t *ec_ent;
	int i;

	for (i= 0, ec_ent= ec_ent_table; i<EC_ENT_NO; i++, ec_ent++)
	{
		if ((ec_ent->ece_score -= ec_bias) < -ec_maxdiff)
			ec_ent->ece_score= -ec_maxdiff;
		if ((ec_ent->ece_minscore -= ec_bias) < -ec_maxdiff)
			ec_ent->ece_minscore= -ec_maxdiff;
	}
	if ((ec_minscore -= ec_bias) < -ec_maxdiff)
		ec_minscore= -ec_maxdiff;
	ec_bias= 0;
}


#if DEBUG
static int ec_check_cache()
{
	static int recursive= 0;

	struct mproc *proc;
	ec_ent_t *ec_ent;

	return 1;

	if (recursive)
		return 1;
	recursive= 1;

#if DEBUG & 256
 { where(); printf("ec_check_cache:\n\t"); }
#endif
	for (proc= mproc_eclist; proc; proc= proc->mp_next)
	{
		if (proc->mp_prev == NULL)
		{
			if (proc != mproc_eclist)
			{
#if DEBUG
 { printf("ec_check_cache: proc[%d] has NULL prev\n", proc-mproc); }
#endif
				recursive= 0;
				return 0;
			}
		}
		else
		{
			if (proc->mp_prev->mp_next != proc)
			{
#if DEBUG
 { printf(
			"ec_check_cache: proc[%d] has wrong prev %d->%d\n",
					proc-mproc, proc->mp_prev-mproc,
					proc->mp_prev->mp_next == NULL ?
					-1 : proc->mp_prev->mp_next-mproc); }
#endif
				recursive= 0;
				return 0;
			}
		}
		ec_ent= ec_get_ent(proc->mp_device, proc->mp_ino,
			proc->mp_ctime);
		if (ec_ent->ece_proc != proc)
		{
#if DEBUG
 { printf(
"ec_check_cache: proc in ec_ent for 0x%x,%d,%d is %d and should be proc[%d]\n",
				proc->mp_device, proc->mp_ino, proc->mp_ctime,
				ec_ent->ece_proc == NULL ?
					-1 : ec_ent->ece_proc-mproc,
				proc-mproc); }
#endif
			recursive= 0;
			return 0;
		}
#if DEBUG & 256
 { printf("%d[0x%x,%d,%d:%d] ", proc-mproc, proc->mp_device, proc->mp_ino,
 	proc->mp_ctime, ec_hash(proc->mp_device, proc->mp_ino,
 	proc->mp_ctime)); }
#endif
	}
#if DEBUG & 256
 { printf("\n"); }
#endif
	recursive= 0;
	return 1;
}
#endif

/*
 * $PchId: exec.c,v 1.5 1996/02/29 23:28:59 philip Exp $
 */
