/* This task handles the interface between file system and kernel as well as
 * between memory manager and kernel.  System services are obtained by sending
 * sys_task() a message specifying what is needed.  To make life easier for
 * MM and FS, a library is provided with routines whose names are of the
 * form sys_xxx, e.g. sys_xit sends the SYS_XIT message to sys_task.  The
 * message types and parameters are:
 *
 *   SYS_FORK	 informs kernel that a process has forked
 *   SYS_NEWMAP	 allows MM to set up a process memory map
 *   SYS_EXEC	 sets program counter and stack pointer after EXEC
 *   SYS_XIT	 informs kernel that a process has exited
 *   SYS_GETSP	 caller wants to read out some process' stack pointer
 *   SYS_TIMES	 caller wants to get accounting times for a process
 *   SYS_ABORT	 MM or FS cannot go on; abort MINIX
 *   SYS_FRESH	 start with a fresh process image during EXEC (68000 only)
 *   SYS_OLDSIG	 send a signal to a process
 *   SYS_KILL	 cause a signal to be sent via MM
 *   SYS_COPY	 request a block of data to be copied between processes
 *   SYS_VCOPY   request a series of data blocks to be copied between procs
 *   SYS_NICE	 adjust the scheduling priority of a process
 *   SYS_REV	 tells the kernel revision number
 *   SYS_UMAP	 compute the physical address for a given virtual address
 *   SYS_MEM	 returns the next free chunk of physical memory
 *   SYS_TRACE	 request a trace operation
 *   SYS_CORE    request a core header
 *   SYS_ADJDATA  allows MM to change a map for a brk
 *   SYS_ADJSTACK  allows MM to change a map for a signal
 *   SYS_GETMAP	 allows MM to get a process' memory map
 *   SYS_GETSIG	 allows MM to get a signal for a user process
 *   SYS_PUTS    a server (MM, FS, ...) wants to issue a diagnostic
 *   SYS_SYSENV  get a kernel environment variable
#if PAGING_VM
 *   SYS_SWAPOFF turn swapping off (partly)
 *   SYS_VM_LOCK lock a part of a user process into memory
 *   SYS_VM_UNLOCK unlock a locked part.
#endif
 *   SYS_DELMAP delete the address map of some process
 *   SYS_DUPSEG dup segments of some process to another proess
 *   SYS_FINDPROC find a process' task number given it's names
 *   SYS_SYSCTL	handles miscelleneous kernel control functions
 *
 * Message type m1 is used for all except SYS_OLDSIG and SYS_COPY, both of
 * which need special parameter types.
 *
 *    m_type       PROC1     PROC2      PID     MEM_PTR
 * ------------------------------------------------------
 * | SYS_FORK   | parent  |  child  |   pid   |chld_base|
 * |------------+---------+---------+---------+---------|
 * | SYS_NEWMAP | proc nr |         |         | map ptr |
 * |------------+---------+---------+---------+---------|
 * | SYS_EXEC   | proc nr | traced  | new sp  |         |
 * |------------+---------+---------+---------+---------|
 * | SYS_XIT    | parent  | exitee  |         |         |
 * |------------+---------+---------+---------+---------|
 * | SYS_GETSP  | proc nr |         |         |         |
 * |------------+---------+---------+---------+---------|
 * | SYS_TIMES  | proc nr |         | buf ptr |         |
 * |------------+---------+---------+---------+---------|
 * | SYS_ABORT  |         |         |         |         |
 * |------------+---------+---------+---------+---------|
 * | SYS_FRESH  | proc nr | data_cl |         |         |
 * |------------+---------+---------+---------+---------|
 * | SYS_REV    |         |         |         |         |
 * ------------------------------------------------------
 *
 *    m_type          m1_i1     m1_i2     m1_i3       m1_p1
 * ----------------+---------+---------+---------+--------------
 * | SYS_VCOPY     |  src p  |  dst p  | vec siz | vc addr     |
 * |---------------+---------+---------+---------+-------------|
 * | SYS_NICE      | proc nr |  incr   |         |             |
 * |---------------+---------+---------+---------+-------------|
 * | SYS_SENDSIG   | proc nr |         |         | smp         |
 * |---------------+---------+---------+---------+-------------|
 * | SYS_SIGRETURN | proc nr |         |         | scp         |
 * -------------------------------------------------------------
 *    m_type       PROC      m1_i2     m1_i3    MEM_PTR
 * ------------------------------------------------------
 * | SYS_GETMAP | proc nr |         |         | map ptr |
 * |------------+---------+---------+---------+---------|
 * | SYS_ADJDATA| proc nr |         |         |break_ptr|
 * |------------+---------+---------+---------+---------|
 * |SYS_ADJSTACK| proc nr |         |         |stack_ptr|
 * -------------+---------+---------+---------+----------
 *
 *    m_type       m1_i1     m1_i2     m1_i3     m1_p1	   m1_p2
 * -------------+---------+---------+---------+---------+----------
 * | SYS_SYSENV |  proc   | keylen  | vallen  |  key	|  val    |
 * -------------+---------+---------+---------+---------+----------
 * | SYS_PUTS   |  count  |         |         |  buf	|         |
 * -------------+---------+---------+---------+---------+----------
 * | SYS_CORE   |  proc   |  len    |         |  addr   |	  |
 * -------------+---------+---------+---------+---------+----------
 * | SYS_DELMAP | proc nr |         |         |         |         |
 * |------------+---------+---------+---------+---------+----------
 * | SYS_DUPSEG | srcproc | dstproc |segmbits |         |         |
 * |------------+---------+---------+---------+---------+----------
 *
 *    m_type       m2_i1     m2_i2     m2_l1      m2_l2     m2_p1
 * ----------------------------------------------------------------
 * | SYS_TRACE  | proc_nr | request |  addr    |  data   |        |
 * ----------------------------------------------------------------
 * | SYS_SYSCTL | proc_nr |         |  request |         |  argp  |
 * ----------------------------------------------------------------
 *
 *
 *    m_type       m6_i1     m6_i2     m6_i3     m6_f1
 * ------------------------------------------------------
 * | SYS_OLDSIG | proc_nr  |  sig    |         | handler |
 * ------------------------------------------------------
 * | SYS_KILL   | proc_nr  |  sig    |         |         |
 * ------------------------------------------------------
 *
 *
 *    m_type        m3_i1   m3_i2    m3_p1   m3_ca1
 * --------------------------------------------------
 * | SYS_FINDPROC | flags |        |       | name   |
 * --------------------------------------------------
 *
 *    m_type      m5_c1   m5_i1    m5_l1   m5_c2   m5_i2    m5_l2   m5_l3
 * --------------------------------------------------------------------------
 * | SYS_COPY   |src seg|src proc|src vir|dst seg|dst proc|dst vir| byte ct |
 * --------------------------------------------------------------------------
 * | SYS_UMAP   |  seg  |proc nr |vir adr|       |        |       | byte ct |
 * --------------------------------------------------------------------------
 *
 *
 *    m_type      m1_i1      m1_i2
 * |------------+----------+-----------
 * | SYS_MEM    | mem base | mem size |
 * -------------+----------+-----------
 *
 * In addition to the main sys_task() entry point, there are 5 other minor
 * entry points:
 *   cause_sig:	take action to cause a signal to occur, sooner or later
 *   numap:	umap SEG_D starting from process number instead of pointer
 *   umap:	compute the physical address for a given virtual address
 *   alloc_segments: allocate segments for 8088 or higher processor
 */

#include "kernel.h"
#include <stddef.h>
#include <signal.h>
#include <unistd.h>
#include <sys/sigcontext.h>
#include <sys/core.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/queryparam.h>
#include <sys/svrctl.h>
#include "assert.h"
INIT_ASSERT
#include "proc.h"
#if (CHIP == INTEL)
#include "i386/protect.h"
#if VIRT_MEM && !PAGING_VM
#include "i386/vm386.h"
#endif /* VIRT_MEM && !PAGING_VM */
#endif

PRIVATE char sig_stuff[SIG_PUSH_BYTES];	/* used to send signals to processes */
#if CHIP == INTEL && WORD_SIZE == 2
PRIVATE core_86_t core;			/* Header for i86 core file */
#endif
#if CHIP == INTEL && WORD_SIZE == 4
PRIVATE core_386_t core;		/* Header for i386 core file */
#endif
PRIVATE int synal_tasknr= ANY;
PRIVATE int pager_tasknr= ANY;
#if CHIP == INTEL
int irq_owner[NR_IRQ_VECTORS];
unsigned irq_pending_set;
#endif

/* Variables that may have to be surrendered to an outside force. */
PRIVATE struct export_param_list sys_ex_param_list[] = {
	QP_VARIABLE(proc),
	QP_ARRAY(proc),
	QP_FIELD(p_flags, struct proc),
	QP_FIELD(p_pid, struct proc),
	QP_FIELD(user_time, struct proc),
	QP_FIELD(sys_time, struct proc),
	QP_FIELD(p_tofrom, struct proc),
	QP_FIELD(p_name, struct proc),
	QP_FIELD(p_map, struct proc),
	QP_ARRAY(proc[0].p_map),
	QP_FIELD(mem_vir, struct mem_map),
	QP_FIELD(mem_phys, struct mem_map),
	QP_FIELD(mem_len, struct mem_map),
	QP_VARIABLE(loadav),
	QP_ARRAY(loadav),
	QP_END()
};
PRIVATE struct export_params sys_ex_params = { sys_ex_param_list };

FORWARD _PROTOTYPE( int do_abort, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_copy, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_exec, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_fork, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_revision, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_getsp, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_kill, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_mem, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_newmap, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_sendsig, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_oldsig, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_sigreturn, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_times, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_trace, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_umap, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_xit, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_vcopy, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_nice, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_core, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_adjdata, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_adjstack, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_getmap, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_getsig, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_puts, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_sysenv, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_delmap, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_dupseg, (message *m_ptr) );
FORWARD _PROTOTYPE( void remap, (struct proc *sp, struct proc *dp) );
FORWARD _PROTOTYPE( void unmap, (struct proc *pp) );
FORWARD _PROTOTYPE( void update_stack, (struct proc *pp) );
#if PAGING_VM
FORWARD _PROTOTYPE( int do_swapoff, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_vm_lock, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_vm_unlock, (message *m_ptr) );
#endif /* PAGING_VM */
FORWARD _PROTOTYPE( int do_findproc, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_sysctl, (message *m_ptr) );
FORWARD _PROTOTYPE( int sysqueryparam, (int proc_nr, vir_bytes argp) );
FORWARD _PROTOTYPE( int sq_getc, (void) );
FORWARD _PROTOTYPE( void sq_flush, (void) );

#if (CHIP == M68000)
FORWARD _PROTOTYPE( void build_sig, (char *sig_stuff,struct proc *rp,int sig));
#endif

#if (SHADOWING == 1)
FORWARD _PROTOTYPE( int do_fresh, (message *m_ptr) );
#endif

#if CHIP == INTEL
FORWARD _PROTOTYPE( int sys_handler, (int irq) );
#endif

/*===========================================================================*
 *				sys_task				     *
 *===========================================================================*/
PUBLIC void sys_task()
{
/* Main entry point of sys_task.  Get the message and dispatch on type. */

  register int i, r;
  message m;

  qp_export(&sys_ex_params);

  synal_tasknr= findproc(SYN_AL_NAME);
  pager_tasknr= findproc(PAGER_NAME);

  while (TRUE) {
	receive(ANY, &m);

	switch (m.m_type) {	/* which system call */
	    case SYS_FORK:	r = do_fork(&m);	break;
	    case SYS_NEWMAP:	r = do_newmap(&m);	break;
	    case SYS_EXEC:	r = do_exec(&m);	break;
	    case SYS_XIT:	r = do_xit(&m);		break;
	    case SYS_GETSP:	r = do_getsp(&m);	break;
	    case SYS_TIMES:	r = do_times(&m);	break;
	    case SYS_ABORT:	r = do_abort(&m);	break;
#if (SHADOWING == 1)
	    case SYS_FRESH:	r = do_fresh(&m);	break;
#endif
	    case SYS_SENDSIG:	r = do_sendsig(&m);	break;
	    case SYS_SIGRETURN: r = do_sigreturn(&m);	break;
	    case SYS_OLDSIG:	r = do_oldsig(&m);	break;
	    case SYS_KILL:	r = do_kill(&m);	break;
	    case SYS_COPY:	r = do_copy(&m);	break;
            case SYS_VCOPY:	r = do_vcopy(&m);	break;
	    case SYS_NICE:	r = do_nice(&m);	break;
	    case SYS_REV:	r = do_revision(&m);	break;
	    case SYS_UMAP:	r = do_umap(&m);	break;
	    case SYS_MEM:	r = do_mem(&m);		break;
	    case SYS_TRACE:	r = do_trace(&m);	break;
	    case SYS_CORE:	r = do_core(&m);	break;
	    case SYS_ADJDATA:	r = do_adjdata(&m);	break;
	    case SYS_ADJSTACK:	r = do_adjstack(&m);	break;
	    case SYS_GETMAP:	r = do_getmap(&m);	break;
	    case SYS_GETSIG:	r = do_getsig(&m);	break;
	    case SYS_PUTS:	r = do_puts(&m);	break;
	    case SYS_SYSENV:	r = do_sysenv(&m);	break;
#if PAGING_VM
	    case SYS_SWAPOFF:	r = do_swapoff(&m);	break;
	    case SYS_VM_LOCK:	r = do_vm_lock(&m);	break;
	    case SYS_VM_UNLOCK:	r = do_vm_unlock(&m);	break;
#endif
	    case SYS_DELMAP:	r = do_delmap(&m);	break;
	    case SYS_DUPSEG:	r = do_dupseg(&m);	break;
	    case SYS_FINDPROC:	r = do_findproc(&m);	break;
	    case SYS_SYSCTL:	r = do_sysctl(&m);	break;
	    case HARD_INT:
		assert (m.m_source == HARDWARE);
		/* Pass a hardware interrupt on to a user process as SIGINT. */

		for (i= 0; i<NR_IRQ_VECTORS; i++)
		{
			if (!(irq_pending_set & (1 << i)))
				continue;
			assert(irq_owner[i] != 0);
			cause_sig(irq_owner[i], SIGINT);
			lock();
			irq_pending_set &= ~(1 << i);
			unlock();
		}
		continue;
	    default:		r = E_BAD_FCN;
			panic("SYSTASK got invalid request: ", m.m_type);
	}

	m.m_type = r;		/* 'r' reports status of call */
	send(m.m_source, &m);	/* send reply to caller */
  }
}


/*===========================================================================*
 *				do_fork					     *
 *===========================================================================*/
PRIVATE int do_fork(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Handle sys_fork().  m_ptr->PROC1 has forked.  The child is m_ptr->PROC2. */

#if (CHIP == INTEL)
  reg_t old_ldt_sel;
#if PAGING_VM
  phys_clicks base_clicks, top_clicks, size_clicks;
#endif /* PAGING_VM */
#endif
  register struct proc *rpc;
  struct proc *rpp;

  phys_clicks child_base;

  /* Calculate the pointers before we check. */
  rpp = proc_addr(m_ptr->PROC1);
  rpc = proc_addr(m_ptr->PROC2);

  assert (isuserp(rpp) && isemptyp(rpc));
  if (!isuserp(rpp) || !isemptyp(rpc))
  {
	return(E_BAD_PROC);
  }

  child_base= (phys_clicks)m_ptr->m1_p1;

#if CHIP == INTEL && VIRT_MEM
  /* On a vm system we have to check available memory first. */
  if (vm_not_alloc < rpp->p_map[SEG_T].mem_len + rpp->p_map[SEG_D].mem_len +
  	rpp->p_map[SEG_S].mem_len)
  {
  	return ENOMEM;	/* Bad luck */
  }
#endif /* CHIP == INTEL && VIRT_MEM */

  sched_ptr= NULL;		/* Avoid further updates to p_ticksleft of the
				 * current process.
				 */

  /* Copy parent 'proc' struct to child. */
#if (CHIP == INTEL)
  old_ldt_sel = rpc->p_ldt_sel;	/* stop this being obliterated by copy */
  *rpc = *rpp;
  rpc->p_ldt_sel = old_ldt_sel;
#if PAGING_VM
  rpc->p_nr = m_ptr->PROC2;	/* this was obliterated by copy */
#endif /* PAGING_VM */

  rpc->p_map[SEG_T].mem_phys = child_base;
  if (rpc->p_map[SEG_T].mem_len)	/* Separate I&D */
  {
	rpc->p_map[SEG_D].mem_phys = rpc->p_map[SEG_T].mem_phys +
			rpc->p_map[SEG_T].mem_vir + rpc->p_map[SEG_T].mem_len;
  }
  else
	rpc->p_map[SEG_D].mem_phys = child_base;
  rpc->p_map[SEG_S].mem_phys = rpc->p_map[SEG_D].mem_phys;

  alloc_segments(rpc);

#if VIRT_MEM			/* Make pages shared or copy on access */
  vm_not_alloc -= rpc->p_map[SEG_T].mem_len + rpc->p_map[SEG_D].mem_len +
  						rpc->p_map[SEG_S].mem_len;
#if PAGING_VM
  base_clicks= rpc->p_map[SEG_T].mem_phys;
  top_clicks= rpc->p_map[SEG_S].mem_phys + rpc->p_map[SEG_S].mem_vir +
						rpc->p_map[SEG_S].mem_len;
  size_clicks= top_clicks-base_clicks;
#if DEBUG
  vm_check_unmapped(base_clicks, size_clicks);
#endif /* DEBUG */
assert(vm_check_stats());
  vm_allocmap(rpc, base_clicks, size_clicks);

  /* Fork the text segment */
  if (rpc->p_map[SEG_T].mem_len)
  {
	if (rpp->p_status & P_ST_TRACED)
	{
		/* If the parent is traced then the text segment can be
		 * modified so we have to map copy on write. We also have
		 * to pass this flag to the child. */
		assert (rpc->p_status & P_ST_TRACED);
		vm_fork(rpp->p_map[SEG_T].mem_phys+rpp->p_map[SEG_T].mem_vir,
			rpc->p_map[SEG_T].mem_phys+rpc->p_map[SEG_T].mem_vir,
					rpc->p_map[SEG_T].mem_len, FALSE);
	}
	else
	{
		/* Writing is protected by the segments so we can make the
		 * pages read/only shared.
		 */
		vm_fork(rpp->p_map[SEG_T].mem_phys+rpp->p_map[SEG_T].mem_vir,
			rpc->p_map[SEG_T].mem_phys+rpc->p_map[SEG_T].mem_vir,
					rpc->p_map[SEG_T].mem_len, TRUE);
	}
  }
  /* else we treat text as data */

  /* Fork the data segment */
  vm_fork(rpp->p_map[SEG_D].mem_phys+rpp->p_map[SEG_D].mem_vir,
			rpc->p_map[SEG_D].mem_phys+rpc->p_map[SEG_D].mem_vir,
					rpc->p_map[SEG_D].mem_len, FALSE);

  /* Fork the stack segment */
  vm_fork(rpp->p_map[SEG_S].mem_phys+rpp->p_map[SEG_S].mem_vir,
			rpc->p_map[SEG_S].mem_phys+rpc->p_map[SEG_S].mem_vir,
					rpc->p_map[SEG_S].mem_len, FALSE);
  level0(vm_reload);
#else /* !PAGING_VM */
#if DEBUG
  vm_check_unmapped((rpc->p_map[SEG_T].mem_phys) <<
  	CLICK_SHIFT, (rpc->p_map[SEG_S].mem_phys+rpc->p_map[SEG_S].mem_vir+
	rpc->p_map[SEG_S].mem_len) << CLICK_SHIFT);
#endif /* DEBUG */
  vm_fork(rpp, child_base);
#endif /* PAGING_VM */
assert(vm_check_stats() ? 1: (vm_dump(), 0));
#endif /* VIRT_MEM */

#else /* !INTEL */
  /* HACK because structure copy is or was slow. */
  phys_copy( (phys_bytes)rpp, (phys_bytes)proc_addr(m_ptr->PROC2),
	     (phys_bytes)sizeof(struct proc));
#endif /* INTEL */
#if !PAGING_VM
  rpc->p_nr = m_ptr->PROC2;	/* this was obliterated by copy */
#endif /* !PAGING_VM */

  rpc->p_flags &= ~(SIG_PENDING | P_STOP);

  /* Only 1 in group should have SIG_PENDING, child does not inherit trace
   * status. The child gets the current schedule round, no cumulative
   * quantum and half of the parents ticks.
   */
  sigemptyset(&rpc->p_pending);
  rpc->p_signal = 0;
  rpc->p_pid = m_ptr->PID;	/* install child's pid */
  rpc->p_reg.sf_retreg = 0;	/* child sees pid = 0 to know it is child */
  rpc->p_alarm = 0;		/* child does not inherit pending alarms */

  rpc->p_priority= PPRI_USER;	/* Mark this slot as a user process */
  rpc->p_flags &= ~NO_MAP;
  rpc->p_status &= ~P_ST_INRUNQ;
  rpc->p_cumquant= 0;
  rpc->p_schedr= schedule_round;
  rpc->p_ticksleft /= 2;

  lock_ready(rpc);

  rpp->p_ticksleft -= rpc->p_ticksleft; /* The parent pays for the child */
  rpc->user_time = 0;		/* set all the accounting times to 0 */
  rpc->sys_time = 0;
  rpc->child_utime = 0;
  rpc->child_stime = 0;

#if (SHADOWING == 1)
  rpc->p_nflips = 0;
  mkshadow(rpp, (phys_clicks)m_ptr->m1_p1);	/* run child first */
#endif

  return(OK);
}


/*===========================================================================*
 *				do_newmap				     *
 *===========================================================================*/
PRIVATE int do_newmap(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Handle sys_newmap().  Fetch new memory map from MM. */

  register struct proc *rp;
  phys_bytes src_phys;
  int caller;			/* whose space has the new map (usually MM) */
  int k;			/* process whose map is to be loaded */
  int i;
  phys_clicks base_clicks, top_clicks;
  struct mem_map *map_ptr;	/* virtual address of map inside caller (MM) */
  struct mem_map new_map[NR_SEGS];

  /* Extract message parameters and copy new memory map from MM. */
  caller = m_ptr->m_source;	/* MM's proc number */
  k = m_ptr->PROC1;
  map_ptr = (struct mem_map *) m_ptr->MEM_PTR;
  if (!isokprocn(k)) return(E_BAD_PROC);
  rp = proc_addr(k);		/* ptr to entry of user getting new map */

  /* Copy the map from MM. */
  src_phys = umap(proc_addr(caller), SEG_D, (vir_bytes) map_ptr,
							sizeof(new_map));
  if (src_phys == 0) panic("bad call to sys_newmap", NO_NUM);
  phys_copy(src_phys, vir2phys(new_map), (phys_bytes) sizeof(new_map));

  /* Make sure that the process that gets the map doesn't have a map
   * already.
   */
  if ((rp->p_flags & (P_SLOT_FREE|NO_MAP)) != (P_SLOT_FREE|NO_MAP))
  	panic("do_newmap: 2nd map for process ", k);

  /* Is there enough physical memory ? */
  if (vm_not_alloc < new_map[SEG_T].mem_len + new_map[SEG_D].mem_len +
							new_map[SEG_S].mem_len)
  {
  	return ENOMEM;
  }

  /* Copy new map */
  for (i= 0; i<NR_SEGS; i++)
  	rp->p_map[i]= new_map[i];

  /* Set the stack pointer */
  rp->p_reg.sf_sp= (rp->p_map[SEG_S].mem_vir << CLICK_SHIFT)-sizeof(char *);

#if (CHIP == INTEL)
  if (rp->p_map[SEG_D].mem_phys != rp->p_map[SEG_S].mem_phys)
	panic("do_newmap: invalid map for process ", proc_number(rp));
#endif

  base_clicks= rp->p_map[SEG_T].mem_phys;
  top_clicks= rp->p_map[SEG_S].mem_phys + rp->p_map[SEG_S].mem_vir +
  						rp->p_map[SEG_S].mem_len;

  /* Allocate physical memory */

  vm_allocmap(rp, base_clicks, top_clicks-base_clicks);
  vm_not_alloc -= rp->p_map[SEG_T].mem_len + rp->p_map[SEG_D].mem_len +
						rp->p_map[SEG_S].mem_len;
  rp->p_flags &= ~NO_MAP;

assert(vm_check_stats());
  alloc_segments(rp);

  return(OK);
}


/*===========================================================================*
 *				do_exec					     *
 *===========================================================================*/
PRIVATE int do_exec(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Handle sys_exec().  A process has done a successful EXEC. Patch it up. */

  register struct proc *rp, *srcp;
  int *sp;			/* new sp */
  int srcproc;
  phys_bytes phys_name;
  char *np;
#define NLEN (sizeof(rp->p_name)-1)

  sp = (int *) m_ptr->STACK_PTR;	/* bad ptr type */
  rp = proc_addr(m_ptr->PROC1);
  assert(isuserp(rp) || isservp(rp));
  if (!isuserp(rp) && !isservp(rp)) return E_BAD_PROC;
  assert(!(rp->p_flags & NO_MAP));
  unmap(rp);
  srcproc= m_ptr->m1_i3;
  assert(srcproc >= 0 && srcproc < NR_PROCS);
  srcp= proc_addr(srcproc);
  assert((srcp->p_flags & (P_SLOT_FREE|NO_MAP)) == P_SLOT_FREE);
  remap(srcp, rp);
  srcp->p_flags |= NO_MAP;
  if (m_ptr->PROC2)
  {
  	cause_sig(m_ptr->PROC1, SIGTRAP);

  	/* Special hack for system call tracing: we need to make a
  	 * distiction between a sucessfull, and an unsucessfull exec 
  	 * call.
  	 */
  	if (rp->p_status & P_ST_TRSYS_E)
  		rp->p_status |= P_ST_TRSYS_B;
  }
  rp->p_reg.sf_sp = (reg_t) sp;	/* set the stack pointer (bad type) */
#if (CHIP == INTEL) && _VMD_EXT
  rp->p_reg.sf_psw = INIT_PSW;
  if (processor >= 486) {
	/* Enable the 486 alignment check.  Both CR0_AM in the MSW and EF_AC
	 * must be set to have alignment checking.  Setting EF_AC is up to
	 * the process.
	 */
	rp->p_reg.sf_msw |= CR0_AM;
  }
#endif /* CHIP == INTEL */

#if (CHIP == M68000)
  rp->p_splow = (reg_t) sp;	/* set the stack pointer low water */
  rp->p_reg.pc = (reg_t) ((vir_bytes)rp->p_map[SEG_T].mem_vir << CLICK_SHIFT);
#ifdef FPP
  /* Initialize fpp for this process */
  fpp_new_state(rp);
#endif
#else
  rp->p_reg.sf_pc = (reg_t) m_ptr->IP_PTR;	/* reset pc */
#endif
  if (m_ptr->PROC2)
  	rp->p_status |= P_ST_TRACED;	/* This process is traced */
  rp->p_flags &= ~RECEIVING;	/* MM does not reply to EXEC call */
  lock_ready(rp);

  /* Save command name for debugging, ps(1) output, etc. */
  phys_name = numap(m_ptr->m_source, (vir_bytes) m_ptr->NAME_PTR,
  							(vir_bytes) NLEN);
  if (phys_name != 0) {
	phys_copy(phys_name, vir2phys(rp->p_name), (phys_bytes) NLEN);
	for (np = rp->p_name; (*np & BYTE) >= ' '; np++) {}
	*np = 0;
  }
  return(OK);
}


/*===========================================================================*
 *				do_xit					     *
 *===========================================================================*/
PRIVATE int do_xit(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Handle sys_xit().  A process has exited. */

	struct proc *rp, *rc;
	struct proc *dest_p;
	struct proc *curr_p, *prev_p;
	int parent;			/* number of exiting proc's parent */
	int proc_nr;			/* number of process doing the exit */
	int r;
	message m;
#if (SHADOWING == 1)
	phys_clicks base, size;
#endif

	parent = m_ptr->PROC1;	/* slot number of parent process */
	proc_nr = m_ptr->PROC2;	/* slot number of exiting process */
	rp = proc_addr(parent);
	rc = proc_addr(proc_nr);

	assert(isuserp(rp) && isuserp(rc));
	if (!isuserp(rp) || !isuserp(rc))
		panic("do_xit: bad procs", NO_NUM);

	/* accumulate child times */
	lock();
	rp->child_utime += rc->user_time + rc->child_utime;
	rp->child_stime += rc->sys_time + rc->child_stime;
	unlock();
	rp= NULL;

	rc->p_alarm = 0;		/* turn off alarm timer */

	/* Mark the process non-ready */
	rc->p_flags |= NO_MAP;
	lock_unready(rc);

	/* If the process being terminated happens to be queued trying to send
	 * a message (i.e., the process was killed by a signal, rather than it
	 * doing an EXIT), then it must be removed from the message queue.
	 */
	if (rc->p_flags & SENDING) {
		dest_p= proc_addr(rc->p_tofrom);
		for (prev_p= NULL, curr_p= dest_p->p_senderq; curr_p != rc;
			prev_p= curr_p, curr_p= curr_p->p_sendlink)
		{
			;	/* do nothing */
		}
		if (prev_p == NULL)
		{
#if DEBUG
 { printf("do_xit: proc %d was sending to %d\n",
				proc_nr, proc_number(dest_p)); }
#endif

			dest_p->p_senderq= curr_p->p_sendlink;
			if (dest_p->p_flags & P_PAGEFAULT)
			{
				/* Pagefault during copy mess, force handling
				 * before unmapping this process.
				 */
				m.m_type= PAGER_SYNC;
				r= sendrec(pager_tasknr, &m);
				assert(r == OK && m.m1_i1 == OK);
#if DEBUG
 { printf("do_xit: pagefault for dest\n"); }
#endif
			}
		}
		else
			prev_p->p_sendlink= curr_p->p_sendlink;
	}

	if (rc->p_flags & P_PAGEFAULT)
	{
		m.m_type= PAGER_SYNC;
		r= sendrec(pager_tasknr, &m);
		assert(r == OK && m.m1_i1 == OK);
		assert(!(rc->p_flags & P_PAGEFAULT));
#if DEBUG
 { printf("do_xit: proc %d has pagefault\n", proc_nr); }
#endif
	}

	sigemptyset(&rc->p_pending);
	if (rc->p_flags & SIG_PENDING)
	{
		/* Remove this process from sig_proclist */
		for (curr_p= sig_proclist, prev_p= NULL;
			curr_p && curr_p != rc;
			prev_p= curr_p, curr_p= curr_p->p_signal)
		{
			/* do nothing */;
		}
		assert(curr_p);
		if (!prev_p)
			sig_proclist= curr_p->p_signal;
		else
			prev_p->p_signal= curr_p->p_signal;
		curr_p->p_signal = NULL;
	}

#if CHIP == INTEL
	/* Remove any irq subscriptions. */
	if (rc->p_status & P_ST_SIGIRQ)
	{
		int irq;
		for (irq = 0; irq < NR_IRQ_VECTORS; irq++)
		{
			if (irq_owner[irq] == proc_nr) {
				irq_owner[irq] = 0;
				disable_irq(irq);
				put_irq_handler(irq, (irq_handler_t) 0);
			}
		}
	}
#endif

#if (SHADOWING == 1)
	rmshadow(rc, &base, &size);
	m_ptr->m1_i1 = (int)base;
	m_ptr->m1_i2 = (int)size;
#endif

#if VIRT_MEM
#if PAGING_VM
	assert(!(rc->p_flags & P_PAGEFAULT));
#endif /* PAGING_VM */
	unmap(rc);
#endif /* VIRT_MEM */

	strcpy(rc->p_name, "<noname>");	/* process no longer has a name */

#if (CHIP == M68000) && (SHADOWING == 0)
	pmmu_delete(rc);	/* we're done remove tables */
#endif

	/* Finish stopping the process */
	rc->p_flags = P_SLOT_FREE | NO_MAP;
	lock_unready(rc);
	lock_dequeue(rc);
	rc->p_priority= PPRI_NONE;		/* Mark the slot as empty */
	return(OK);
}


/*===========================================================================*
 *				do_getsp				     *
 *===========================================================================*/
PRIVATE int do_getsp(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Handle sys_getsp().  MM wants to know what sp is. */

  register struct proc *rp;

  rp = proc_addr(m_ptr->PROC1);
  assert(isuserp(rp) || isservp(rp));
  if (!isuserp(rp) && !isservp(rp)) return E_BAD_PROC;
  m_ptr->STACK_PTR = (char *) rp->p_reg.sf_sp;	/* return sp here (bad type) */
  return(OK);
}


/*===========================================================================*
 *				do_times				     *
 *===========================================================================*/
PRIVATE int do_times(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Handle sys_times().  Retrieve the accounting information. */

  register struct proc *rp;

  rp = proc_addr(m_ptr->PROC1);
  assert(isuserp(rp));
  if (!isuserp(rp)) return E_BAD_PROC;

  /* Insert the times needed by the TIMES system call in the message. */
  lock();			/* halt the volatile time counters in rp */
  m_ptr->USER_TIME   = rp->user_time;
  m_ptr->SYSTEM_TIME = rp->sys_time;
  unlock();
  m_ptr->CHILD_UTIME = rp->child_utime;
  m_ptr->CHILD_STIME = rp->child_stime;
  m_ptr->BOOT_TICKS  = get_uptime();
  return(OK);
}


/*===========================================================================*
 *				do_abort				     *
 *===========================================================================*/
PRIVATE int do_abort(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Handle sys_abort.  MINIX is unable to continue.  Terminate operation. */
  char monitor_code[256];
  phys_bytes src_phys;

  if (m_ptr->m1_i1 == RBT_MONITOR) {
	/* The monitor is to run user specified instructions. */
	src_phys = numap(m_ptr->m_source, (vir_bytes) m_ptr->m1_p1,
					(vir_bytes) sizeof(monitor_code));
	if (src_phys == 0) panic("bad monitor code from", m_ptr->m_source);
	phys_copy(src_phys, vir2phys(monitor_code),
					(phys_bytes) sizeof(monitor_code));
	reboot_code = vir2phys(monitor_code);
  }
  wreboot(m_ptr->m1_i1);
  return(OK);			/* pro-forma (really EDISASTER) */
}


#if (SHADOWING == 1)
/*===========================================================================*
 *				do_fresh				     *
 *===========================================================================*/
PRIVATE int do_fresh(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Handle sys_fresh.  Start with fresh process image during EXEC. */

  register struct proc *p;
  int proc_nr;			/* number of process doing the exec */
  phys_clicks base, size;
  phys_clicks c1, nc;

  proc_nr = m_ptr->PROC1;	/* slot number of exec-ing process */
  if (!isokprocn(proc_nr)) return(E_BAD_PROC);
  p = proc_addr(proc_nr);
  rmshadow(p, &base, &size);
  do_newmap(m_ptr);
  c1 = p->p_map[SEG_D].mem_phys;
  nc = p->p_map[SEG_S].mem_phys - p->p_map[SEG_D].mem_phys
  						+ p->p_map[SEG_S].mem_len;
  c1 += m_ptr->m1_i2;
  nc -= m_ptr->m1_i2;
  zeroclicks(c1, nc);
  m_ptr->m1_i1 = (int)base;
  m_ptr->m1_i2 = (int)size;
  return(OK);
}
#endif /* (SHADOWING == 1) */


/*===========================================================================*
 *			      do_sendsig				     *
 *===========================================================================*/
PRIVATE int do_sendsig(m_ptr)
message *m_ptr;			/* pointer to request message */
{
  struct sigmsg smsg;
  register struct proc *rp;
  phys_bytes src_phys, dst_phys;
  struct sigcontext *scp, *u_scp;
  struct sigframe sf, *u_sfp;

  rp = proc_addr(m_ptr->PROC1);
  assert(isuserp(rp));
  if (!isuserp(rp))  return(E_BAD_PROC);

  /* Get the sigmsg structure into our address space.  */
  src_phys = umap(proc_addr(MM_PROC_NR), SEG_D, (vir_bytes) m_ptr->SIG_CTXT_PTR,
		  (vir_bytes) sizeof(struct sigmsg));
  if (src_phys == 0)
	panic("do_sendsig can't signal: bad sigmsg address from MM", NO_NUM);
  phys_copy(src_phys, vir2phys(&smsg), (phys_bytes) sizeof(struct sigmsg));

  /* Compute the usr stack pointer value where the sigframe will be stored. */
  u_sfp = (struct sigframe *) smsg.sm_stkptr - 1;

  /* Fill in the signal context. */
  scp= &sf.sf_sc;
  scp->sc_size= sizeof(sf.sf_sc);
  scp->sc_version= SC_VERSION;
  scp->sc_flags = SC_SIGCONTEXT;
  scp->sc_mask = smsg.sm_mask;

#if (CHIP == INTEL)
#if INTEL_32BITS
  scp->sc_regs.sr_gs= rp->p_reg.sf_gs;
  scp->sc_regs.sr_fs= rp->p_reg.sf_fs;
#endif /* INTEL_32BITS */
  scp->sc_regs.sr_ss= rp->p_reg.sf_ss;
  scp->sc_regs.sr_cs= rp->p_reg.sf_cs;
  scp->sc_regs.sr_es= rp->p_reg.sf_es;
  scp->sc_regs.sr_ds= rp->p_reg.sf_ds;
  scp->sc_regs.sr_di= rp->p_reg.sf_di;
  scp->sc_regs.sr_si= rp->p_reg.sf_si;
  scp->sc_regs.sr_bp= rp->p_reg.sf_bp;
  scp->sc_regs.sr_sp= rp->p_reg.sf_sp;
  scp->sc_regs.sr_bx= rp->p_reg.sf_bx;
  scp->sc_regs.sr_dx= rp->p_reg.sf_dx;
  scp->sc_regs.sr_cx= rp->p_reg.sf_cx;
  scp->sc_regs.sr_ax= rp->p_reg.sf_ax;
  scp->sc_regs.sr_pc= rp->p_reg.sf_pc;
  scp->sc_regs.sr_flags= rp->p_reg.sf_flags;

  /* Check floating point. */
  if (!(rp->p_status & P_ST_FPU_USED))
  {
  	scp->sc_flags |= SC_FPU_RESTORE | SC_FPU_INIT;
  }
  else
  {
  	/* Make sure that the context is not in the fpu. */
  	if (fpu_proc_ptr == rp)
  	{
  		level0(fpu_save);	/* call fpu_save at level 0 */
  		assert(fpu_proc_ptr == NULL);
  	}
  	if (rp->p_status & P_ST_FPU_EX)
  	{
#if DEBUG
 { printW(); printf("fpu exception pending\n"); }
#endif
		rp->p_status &= ~P_ST_FPU_EX;
		rp->p_status |= P_ST_FPU_INVAL;
		cause_sig(proc_number(rp), SIGFPE);
	}
	scp->sc_flags |= SC_FPU_CONTEXT;
	if (rp->p_status & P_ST_FPU_INVAL)
		scp->sc_flags |= SC_FPU_INVALID;
	scp->sc_flags |= SC_FPU_RESTORE;
	if ((rp->p_status & P_ST_FPU_INVAL) && smsg.sm_signo == SIGFPE)
		scp->sc_flags |= SC_FPU_INIT;
	scp->sc_fpregs.fpr_cw= rp->p_reg.sf_fp_cw;
	scp->sc_fpregs.fpr_res0= rp->p_reg.sf_fp_res0;
	scp->sc_fpregs.fpr_sw= rp->p_reg.sf_fp_sw;
	scp->sc_fpregs.fpr_res1= rp->p_reg.sf_fp_res1;
	scp->sc_fpregs.fpr_tw= rp->p_reg.sf_fp_tw;
	scp->sc_fpregs.fpr_res2= rp->p_reg.sf_fp_res2;
	scp->sc_fpregs.fpr_ip= rp->p_reg.sf_fp_ip;
	scp->sc_fpregs.fpr_cs= rp->p_reg.sf_fp_cs;
	scp->sc_fpregs.fpr_res3= rp->p_reg.sf_fp_res3;
	scp->sc_fpregs.fpr_dp= rp->p_reg.sf_fp_dp;
	scp->sc_fpregs.fpr_ds= rp->p_reg.sf_fp_ds;
	scp->sc_fpregs.fpr_res4= rp->p_reg.sf_fp_res4;
	assert(sizeof(scp->sc_fpregs.fpr_reg) == sizeof(rp->p_reg.sf_fp_reg));
	memcpy (scp->sc_fpregs.fpr_reg, rp->p_reg.sf_fp_reg,
					sizeof(scp->sc_fpregs.fpr_reg));
	rp->p_status &= ~P_ST_FPU_USED;
#if DEBUG & 0
 { printW(); printf("0x%x\n",
 	rp->p_status & (P_ST_FPU_USED|P_ST_FPU_INVAL|P_ST_FPU_EX)); }
#endif
  }
#endif

  /* Initialize the sigframe structure. */
  u_scp= &u_sfp->sf_sc;
  sf.sf_scpcopy= u_scp;
  sf.sf_retadr2= (_PROTOTYPE(void, (*)(void))) rp->p_reg.sf_pc;
  sf.sf_bp= rp->p_reg.sf_bp;
  rp->p_reg.sf_bp= (reg_t)&u_sfp->sf_bp;
  sf.sf_scp= u_scp;
  sf.sf_code= 0;
  sf.sf_signo= smsg.sm_signo;
  sf.sf_retadr= (_PROTOTYPE(void, (*)(void))) smsg.sm_sigreturn;

  /* Copy the sigframe structure to the user's stack. */
  dst_phys = umap(rp, SEG_D, (vir_bytes) u_sfp,
  					(vir_bytes) sizeof(struct sigframe));
  if (dst_phys == 0) return(EFAULT);
  phys_copy(vir2phys(&sf), dst_phys, (phys_bytes) sizeof(struct sigframe));

  /* Reset user registers to execute the signal handler. */
  rp->p_reg.sf_sp = (reg_t) u_sfp;
  rp->p_reg.sf_pc = (reg_t) smsg.sm_sighandler;

  return(OK);
}

/*===========================================================================*
 *			      do_sigreturn				     *
 *===========================================================================*/
PRIVATE int do_sigreturn(m_ptr)
register message *m_ptr;
{
  struct sigcontext sc;
  register struct proc *rp;
  phys_bytes src_phys;

  rp = proc_addr(m_ptr->PROC1);
  assert(isuserp(rp));
  if (!isuserp(rp))  return(E_BAD_PROC);

  /* Copy in the sigcontext structure. */
  src_phys = umap(rp, SEG_D, (vir_bytes) m_ptr->SIG_CTXT_PTR,
		  (vir_bytes) sizeof(struct sigcontext));
  if (src_phys == 0) return(EFAULT);
  phys_copy(src_phys, vir2phys(&sc), (phys_bytes) sizeof(struct sigcontext));

  /* Check size and version. The simplistic assumption is that there will
   * be an exact match.
   */
  if (sc.sc_size != sizeof(sc))
  {
#if DEBUG
  	printW();
#endif
	printf("do_sigreturn: got invalid size in signal context: %d\n",
		sc.sc_size);
	return EINVAL;
  }
  if (sc.sc_version != SC_VERSION)
  {
#if DEBUG
  	printW();
#endif
	printf("do_sigreturn: got wrong version in signal context: %d\n",
		sc.sc_version);
	return EINVAL;
  }

  /* Make sure that this is not just a jmp_buf. */
  if ((sc.sc_flags & SC_SIGCONTEXT) == 0) return(EINVAL);

  /* Fix up only certain key registers if the compiler doesn't use
   * register variables within functions containing setjmp.
   */
  if (sc.sc_flags & SC_NOREGLOCALS) {
  	printW(); printf("got noreglocals\n");
	rp->p_reg.sf_retreg = sc.sc_regs.sr_retreg;
	rp->p_reg.sf_fp = sc.sc_regs.sr_fp;
	rp->p_reg.sf_pc = sc.sc_regs.sr_pc;
	rp->p_reg.sf_sp = sc.sc_regs.sr_sp;
	return (OK);
  }
  else
  {
#if (CHIP == INTEL)
	/* Ignore selectors for now (actually the zero selector would be legal). */
#if INTEL_32BITS
	/* rp->p_reg.sf_gs= sc.sc_regs.sr_gs; */
	/* rp->p_reg.sf_fs= sc.sc_regs.sr_fs; */
#endif /* INTEL_32BITS */
	/* rp->p_reg.sf_ss= sc.sc_regs.sr_ss; */
	/* rp->p_reg.sf_cs= sc.sc_regs.sr_cs; */
	/* rp->p_reg.sf_es= sc.sc_regs.sr_es; */
	/* rp->p_reg.sf_ds= sc.sc_regs.sr_ds; */

	rp->p_reg.sf_di= sc.sc_regs.sr_di;
	rp->p_reg.sf_si= sc.sc_regs.sr_si;
	rp->p_reg.sf_bp= sc.sc_regs.sr_bp;
	rp->p_reg.sf_sp= sc.sc_regs.sr_sp;
	rp->p_reg.sf_bx= sc.sc_regs.sr_bx;
	rp->p_reg.sf_dx= sc.sc_regs.sr_dx;
	rp->p_reg.sf_cx= sc.sc_regs.sr_cx;
	rp->p_reg.sf_ax= sc.sc_regs.sr_ax;
	rp->p_reg.sf_pc= sc.sc_regs.sr_pc;
	rp->p_reg.sf_flags= sc.sc_regs.sr_flags;

	if (sc.sc_flags & SC_FPU_RESTORE)
	{
		/* make sure that the current context of the user process is
		 * not in the fpu.
		 */
		if (fpu_proc_ptr == rp)
		{
			level0(fpu_save);	/* call fpu_save at level 0 */
			assert(fpu_proc_ptr == NULL);
		}
		rp->p_status &= ~(P_ST_FPU_USED|P_ST_FPU_INVAL|P_ST_FPU_EX);
		if (sc.sc_flags & SC_FPU_INIT)
			; /* nothing to do */
		else if (sc.sc_flags & SC_FPU_CONTEXT)
		{
			rp->p_reg.sf_fp_cw= sc.sc_fpregs.fpr_cw;
			rp->p_reg.sf_fp_res0= sc.sc_fpregs.fpr_res0;
			rp->p_reg.sf_fp_sw= sc.sc_fpregs.fpr_sw;
			rp->p_reg.sf_fp_res1= sc.sc_fpregs.fpr_res1;
			rp->p_reg.sf_fp_tw= sc.sc_fpregs.fpr_tw;
			rp->p_reg.sf_fp_res2= sc.sc_fpregs.fpr_res2;
			rp->p_reg.sf_fp_ip= sc.sc_fpregs.fpr_ip;
			rp->p_reg.sf_fp_cs= sc.sc_fpregs.fpr_cs;
			rp->p_reg.sf_fp_res3= sc.sc_fpregs.fpr_res3;
			rp->p_reg.sf_fp_dp= sc.sc_fpregs.fpr_dp;
			rp->p_reg.sf_fp_ds= sc.sc_fpregs.fpr_ds;
			rp->p_reg.sf_fp_res4= sc.sc_fpregs.fpr_res4;
			assert(sizeof(sc.sc_fpregs.fpr_reg) ==
				sizeof(rp->p_reg.sf_fp_reg));
			memcpy (rp->p_reg.sf_fp_reg, sc.sc_fpregs.fpr_reg,
					sizeof(rp->p_reg.sf_fp_reg));
			rp->p_status |= P_ST_FPU_USED;
			if (sc.sc_flags & SC_FPU_INVALID)
				rp->p_status |= P_ST_FPU_INVAL;
		}
		else
		{
			printf(
		"do_sigreturn: SC_FPU_RESTORE but no SC_FPU_CONTEXT\n");
		}
	}
#endif

  }
  return(OK);
}

/*===========================================================================*
 *				do_oldsig				     *
 *===========================================================================*/
PRIVATE int do_oldsig(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Handle sys_sig(). Signal a process.  The stack is known to be big enough. */

  register struct proc *rp;
  phys_bytes dst_phys;
  vir_bytes new_sp;
  int sig;			/* signal number, 1 to _NSIG */
  sighandler_t sig_handler;	/* pointer to the signal handler */

  /* Extract parameters and prepare to build the words that get pushed. */
  rp = proc_addr(m_ptr->PR);
  assert(isuserp(rp));
  if (!isuserp(rp))  return(E_BAD_PROC);
  sig = m_ptr->SIGNUM;

  assert(sig != -1);

  sig_handler = m_ptr->FUNC;	/* run time system addr for catching sigs */
  new_sp = (vir_bytes) rp->p_reg.sf_sp;

  /* Actually build the block of words to push onto the stack. */
  build_sig(sig_stuff, rp, sig);	/* build up the info to be pushed */

  /* Prepare to do the push, and do it. */
  new_sp -= SIG_PUSH_BYTES;
  dst_phys = umap(rp, SEG_D, new_sp, (vir_bytes) SIG_PUSH_BYTES);
  if (dst_phys == 0) return(EFAULT);

  /* Push pc, psw from sig_stuff. */
  phys_copy(vir2phys(sig_stuff), dst_phys, (phys_bytes) SIG_PUSH_BYTES);

  /* Change process' sp and pc to reflect the interrupt. */
  rp->p_reg.sf_sp = new_sp;
  rp->p_reg.sf_pc = (reg_t) sig_handler;	/* bad ptr type */
  return(OK);
}

/*===========================================================================*
 *				do_kill					     *
 *===========================================================================*/
PRIVATE int do_kill(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Handle sys_kill(). Cause a signal to be sent to a process via MM. */

  int sig;
  int proc_nr;
  struct proc *rp;

  proc_nr = m_ptr->PR;
  sig = m_ptr->SIGNUM;
  rp= proc_addr(proc_nr);
  assert(isuserp(rp));
  if (!isuserp(rp)) return(E_BAD_PROC);
  cause_sig(proc_nr, sig);
  return(OK);
}


/*==========================================================================*
 *				do_trace				    *
 *==========================================================================*/

#define TR_PROCNR	(m_ptr->m2_i1)
#define TR_REQUEST	(m_ptr->m2_i2)
#define TR_ADDR		((vir_bytes) m_ptr->m2_l1)
#define TR_DATA		(m_ptr->m2_l2)
#define TR_VLSIZE	((vir_bytes) sizeof(long))

PRIVATE int do_trace(m_ptr)
register message *m_ptr;
{
  register struct proc *rp;
  phys_bytes src, dst;
  int i;

  rp = proc_addr(TR_PROCNR);
  if (rp->p_flags & P_SLOT_FREE) return(EIO);
  switch (TR_REQUEST) {
  case -1:			/* stop process */
	rp->p_flags |= P_STOP;
	lock_unready(rp);
	rp->p_reg.sf_psw &= ~TRACEBIT;	/* clear trace bit */
	return(OK);

  case 1:			/* return value from instruction space */
	if (rp->p_map[SEG_T].mem_len != 0) {
		if ((src = umap(rp, SEG_T, TR_ADDR, TR_VLSIZE)) == 0)
			return(EIO);
		phys_copy(src, vir2phys(&TR_DATA), (phys_bytes) sizeof(long));
		break;
	}
	/* Text space is actually data space - fall through. */

  case 2:			/* return value from data space */
	if ((src = umap(rp, SEG_D, TR_ADDR, TR_VLSIZE)) == 0) return(EIO);
	phys_copy(src, vir2phys(&TR_DATA), (phys_bytes) sizeof(long));
	break;

  case 3:			/* return value from process table */
	if ((TR_ADDR & (sizeof(long) - 1)) != 0 ||
	    TR_ADDR > sizeof(struct proc) - sizeof(long))
		return(EIO);
	TR_DATA = *(long *) ((char *) rp + (int) TR_ADDR);
	break;

  case 4:			/* set value from instruction space */
	if (rp->p_map[SEG_T].mem_len != 0) {
		if ((dst = umap(rp, SEG_T, TR_ADDR, TR_VLSIZE)) == 0)
			return(EIO);
		phys_copy(vir2phys(&TR_DATA), dst, (phys_bytes) sizeof(long));
		TR_DATA = 0;
		break;
	}
	/* Text space is actually data space - fall through. */

  case 5:			/* set value from data space */
	if ((dst = umap(rp, SEG_D, TR_ADDR, TR_VLSIZE)) == 0) return(EIO);
	phys_copy(vir2phys(&TR_DATA), dst, (phys_bytes) sizeof(long));
	TR_DATA = 0;
	break;

  case 6:			/* set value in process table */
	if ((TR_ADDR & (sizeof(reg_t) - 1)) != 0 ||
	     TR_ADDR > sizeof(struct stackframe_s) - sizeof(reg_t))
		return(EIO);
	i = (int) TR_ADDR;
#if (CHIP == INTEL)
	/* Altering segment registers might crash the kernel when it
	 * tries to load them prior to restarting a process, so do
	 * not allow it.
	 */
	if (i == (int) &((struct proc *) 0)->p_reg.sf_cs ||
	    i == (int) &((struct proc *) 0)->p_reg.sf_ds ||
	    i == (int) &((struct proc *) 0)->p_reg.sf_es ||
#if INTEL_32BITS
	    i == (int) &((struct proc *) 0)->p_reg.sf_gs ||
	    i == (int) &((struct proc *) 0)->p_reg.sf_fs ||
#endif
	    i == (int) &((struct proc *) 0)->p_reg.sf_ss)
		return(EIO);
#endif
	if (i == (int) &((struct proc *) 0)->p_reg.sf_psw)
		/* only selected bits are changeable */
		SETPSW(rp, TR_DATA);
	else
		*(reg_t *) ((char *) &rp->p_reg + i) = (reg_t) TR_DATA;
	TR_DATA = 0;
	break;

  case 7:			/* resume execution */
	rp->p_flags &= ~P_STOP;
	lock_ready(rp);
	TR_DATA = 0;
	break;

  case 9:			/* set trace bit */
	rp->p_reg.sf_psw |= TRACEBIT;
	rp->p_flags &= ~P_STOP;
	lock_ready(rp);
	TR_DATA = 0;
	break;

  case 10:			/* execute until a system call */
  	if ((rp->p_status & (P_ST_TRSYS_B|P_ST_TRSYS_E)) == 0)
  		rp->p_status |= P_ST_TRSYS_B|P_ST_TRSYS_E;

  	if (rp->p_flags & P_RSTSYS)
  		sys_restart(rp);

	rp->p_flags &= ~P_STOP;
	lock_ready(rp);
	TR_DATA = 0;
	break;

  default:
	return(EIO);
  }
  return(OK);
}


/*===========================================================================*
 *				do_copy					     *
 *===========================================================================*/
PRIVATE int do_copy(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Handle sys_copy().  Copy data for MM or FS. */

  int src_proc, dst_proc, src_space, dst_space;
  vir_bytes src_vir, dst_vir;
  phys_bytes src_phys, dst_phys, bytes;

  /* Dismember the command message. */
  src_proc = m_ptr->SRC_PROC_NR;
  dst_proc = m_ptr->DST_PROC_NR;
  src_space = m_ptr->SRC_SPACE;
  dst_space = m_ptr->DST_SPACE;
  src_vir = (vir_bytes) m_ptr->SRC_BUFFER;
  dst_vir = (vir_bytes) m_ptr->DST_BUFFER;
  bytes = (phys_bytes) m_ptr->COPY_BYTES;

  if (src_space == SEG_S || dst_space == SEG_S)
  {
  	printW();
  	printf("Illegal sys_copy request from %d\n", m_ptr->m_source);
  	return EINVAL;
  }

  /* Compute the source and destination addresses and do the copy. */
  src_phys = umap(proc_addr(src_proc), src_space, src_vir, (vir_bytes) bytes);
#if DEBUG
	if (src_phys == 0)
	{
		printW(); printf("umap(proc_addr(%d), %d, 0x%x, 0x%x) failed\n",
			src_proc, src_space, src_vir, bytes);
	}
#endif

  dst_phys = umap(proc_addr(dst_proc), dst_space, dst_vir, (vir_bytes) bytes);
#if DEBUG
	if (dst_phys == 0)
	{
		printW(); printf("umap(proc_addr(%d), %d, 0x%x, 0x%x) failed\n",
			dst_proc, dst_space, dst_vir, bytes);
	}
#endif

  if (src_phys == 0 || dst_phys == 0) return(EFAULT);
  phys_copy(src_phys, dst_phys, bytes);
  return(OK);
}


/*===========================================================================*
 *				cause_sig				     *
 *===========================================================================*/
PUBLIC void cause_sig(proc_nr, sig_nr)
int proc_nr;			/* process to be signalled */
int sig_nr;			/* signal to be sent, 1 to _NSIG */
{
/* A task wants to send a signal to a process.   Examples of such tasks are:
 *   TTY wanting to cause SIGINT upon getting a DEL
 *   CLOCK wanting to cause SIGALRM when timer expires
 * Signals are handled by sending a message to MM.  The tasks don't dare do
 * that directly, for fear of what would happen if MM were busy.  Instead they
 * call cause_sig, which sets bits in p_pending, and then wakes up the
 * SYN_ALRM task. The SYN_ALRM sends a message to MM.
 * The process being signaled is blocked while MM has not seen or finished
 * with all signals for it. SIG_PENDING flag is kept nonzero while there are
 * some. It is not sufficient to ready the process when MM is
 * informed, because MM can block waiting for FS to do a core dump.
 */

  register struct proc *rp;

  rp = proc_addr(proc_nr);
  assert(isuserp(rp));

  sigaddset(&rp->p_pending, sig_nr);
  if (!(rp->p_flags & SIG_PENDING))
  {
	rp->p_flags |= SIG_PENDING;
	assert(rp->p_signal == NULL);
	rp->p_signal= sig_proclist;
	sig_proclist= rp;
  }
  lock_unready(rp);

  interrupt(synal_tasknr);	/* the synchronous alarm task will take
  				 * care of the signal.
  				 */
}


/*==========================================================================*
 *				numap					    *
 *==========================================================================*/
PUBLIC phys_bytes numap(proc_nr, vir_addr, bytes)
int proc_nr;			/* process number to be mapped */
vir_bytes vir_addr;		/* virtual address in bytes within SEG_D */
vir_bytes bytes;		/* # of bytes required in segment  */
{
/* Do umap() starting from a process number instead of a pointer.  This
 * function is used by device drivers, so they need not know about the
 * process table.  To save time, there is no 'seg' parameter. The segment
 * is always SEG_D.
 */

  return(umap(proc_addr(proc_nr), SEG_D, vir_addr, bytes));
}


/*===========================================================================*
 *				umap					     *
 *===========================================================================*/
PUBLIC phys_bytes umap(rp, seg, vir_addr, bytes)
register struct proc *rp;	/* pointer to proc table entry for process */
int seg;			/* SEG_T, or SEG_D */
vir_bytes vir_addr;		/* virtual address in bytes within the seg */
vir_bytes bytes;		/* # of bytes to be copied */
{
/* Calculate the physical memory address for a given virtual address. */
  vir_clicks vl, vh;		/* low and high clicks of region */
  struct mem_map *mapptr;

  /* If 'seg' is SEG_D it could really be SEG_S. */

  /* We are using HARDWARE to denote absolute memory */
  if (rp == cproc_addr(HARDWARE)) return((phys_bytes) vir_addr);

#if (CHIP == INTEL)
  assert (rp->p_map[SEG_D].mem_phys == rp->p_map[SEG_S].mem_phys &&
	rp->p_map[SEG_D].mem_vir + rp->p_map[SEG_D].mem_len
						<= rp->p_map[SEG_S].mem_vir);
#endif /* CHIP == INTEL */

  vl= vir_addr >> CLICK_SHIFT;
  vh= (vir_addr + bytes - 1) >> CLICK_SHIFT;

  if (bytes <= 0 || (long)vir_addr + bytes < vir_addr)
  	return( (phys_bytes) 0);

  assert(seg == SEG_D || seg == SEG_T);
  mapptr= &rp->p_map[seg];
  if (seg == SEG_D && vl >= mapptr->mem_vir+mapptr->mem_len)
  {
  	/* SEG_D denotes both the data segment and the stack segment. */
  	seg= SEG_S;
  	mapptr= &rp->p_map[seg];
  	if (vl < mapptr->mem_vir)
  	{
  		/* Update the stack segment based on the stack pointer. */
		update_stack(rp);
  	}
  }

  if (vl < mapptr->mem_vir || vh >= mapptr->mem_vir+mapptr->mem_len)
	return 0;
  
#if SEGMENTED_MEMORY
  return vir_addr + (mapptr->mem_phys << CLICK_SHIFT);
#endif
  assert(0);
}


#if (CHIP == INTEL)
/*==========================================================================*
 *				alloc_segments				    *
 *==========================================================================*/
PUBLIC void alloc_segments(rp)
register struct proc *rp;
{
  phys_bytes code_bytes;
  phys_bytes data_bytes;
  int privilege;

  if (protected_mode) {
	data_bytes = (phys_bytes) (rp->p_map[SEG_S].mem_vir
				+ rp->p_map[SEG_S].mem_len) << CLICK_SHIFT;
	if (rp->p_map[SEG_T].mem_len == 0)
		code_bytes = data_bytes;	/* common I&D, poor protect */
	else
#if SEGMENTED_MEMORY
		code_bytes = ((phys_bytes) rp->p_map[SEG_T].mem_len+
			rp->p_map[SEG_T].mem_vir) << CLICK_SHIFT;
#else /* !SEGMENTED_MEMORY */
		code_bytes = (phys_bytes) rp->p_map[SEG_T].mem_len
								<< CLICK_SHIFT;
#endif /* SEGMENTED_MEMORY */
	privilege = istaskp(rp) ? TASK_PRIVILEGE : USER_PRIVILEGE;
	if (rp == cproc_addr(IDLE)) privilege= TASK_PRIVILEGE;
	init_codeseg(&rp->p_ldt[CS_LDT_INDEX],
		     (phys_bytes) rp->p_map[SEG_T].mem_phys << CLICK_SHIFT,
		     code_bytes, privilege);
	init_dataseg(&rp->p_ldt[DS_LDT_INDEX],
		     (phys_bytes) rp->p_map[SEG_D].mem_phys << CLICK_SHIFT,
		     data_bytes, privilege);
	rp->p_reg.sf_cs = (CS_LDT_INDEX * DESC_SIZE) | TI | privilege;
#if INTEL_32BITS
	rp->p_reg.sf_gs =
	rp->p_reg.sf_fs =
#endif
	rp->p_reg.sf_ss =
	rp->p_reg.sf_es =
	rp->p_reg.sf_ds = (DS_LDT_INDEX*DESC_SIZE) | TI | privilege;
  } else {
	rp->p_reg.sf_cs = click_to_hclick(rp->p_map[SEG_T].mem_phys);
	rp->p_reg.sf_ss =
	rp->p_reg.sf_es =
	rp->p_reg.sf_ds = click_to_hclick(rp->p_map[SEG_D].mem_phys);
  }
}
#endif /* (CHIP == INTEL) */


/*==========================================================================*
 *				do_revision				    *
 *==========================================================================*/
PRIVATE int do_revision(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Return the kernel revision number. */
  extern int revision;

  return(revision);
}


/*==========================================================================*
 *				do_umap					    *
 *==========================================================================*/
PRIVATE int do_umap(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Same as umap(), for non-kernel processes. */

  m_ptr->SRC_BUFFER = umap(proc_addr((int) m_ptr->SRC_PROC_NR),
                           (int) m_ptr->SRC_SPACE,
                           (vir_bytes) m_ptr->SRC_BUFFER,
                           (vir_bytes) m_ptr->COPY_BYTES);
  return(OK);
}


/*===========================================================================*
 *				do_mem					     *
 *===========================================================================*/
PRIVATE int do_mem(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Return the base and size of the next chunk of memory of a given type. */

  phys_clicks mem_base, mem_size;

  mem_base= 0;
  mem_size= 0;
  if (chunk_find(&mem_base, &mem_size))
  {
  	chunk_del(mem_base, mem_size);
  }
  m_ptr->m1_i1= mem_base;
  m_ptr->m1_i2= mem_size;
  return OK;
}


#if (CHIP == M68000)
/*===========================================================================*
 *                              build_sig                                    *
 *===========================================================================*/
PRIVATE void build_sig(sig_stuff, rp, sig)
char *sig_stuff;
register struct proc *rp;
int sig;
{
  register struct frame {
		int	f_sig;
		u16_t	f_psw;
		reg_t	f_pc;
  } *fp;

  fp = (struct frame *) sig_stuff;
  fp->f_sig = sig;
  fp->f_psw = rp->p_reg.psw;
  fp->f_pc = rp->p_reg.pc;
}
#endif /* (CHIP == M68000) */

/*===========================================================================*
 *				do_vcopy				     *
 *===========================================================================*/
PRIVATE int do_vcopy(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Handle sys_vcopy(). */

  int src_proc, dst_proc, vect_s, i;
  vir_bytes src_vir, dst_vir, vect_addr;
  struct proc *spp, *dpp;
  phys_bytes src_phys, dst_phys, bytes;
  cpvec_t cpvec_table[CPVEC_NR];

  /* Dismember the command message. */
  src_proc = m_ptr->m1_i1;
  spp= proc_addr(src_proc);
  dst_proc = m_ptr->m1_i2;
  dpp= proc_addr(dst_proc);
  vect_s = m_ptr->m1_i3;
  vect_addr = (vir_bytes)m_ptr->m1_p1;

  if (vect_s > CPVEC_NR) return EDOM;

  src_phys= numap (m_ptr->m_source, vect_addr, vect_s * sizeof(cpvec_t));
  if (!src_phys) return EFAULT;
  phys_copy(src_phys, vir2phys(cpvec_table),
  				(phys_bytes) (vect_s * sizeof(cpvec_t)));

  for (i = 0; i < vect_s; i++) {
	src_vir= cpvec_table[i].cpv_src;
	dst_vir= cpvec_table[i].cpv_dst;
	bytes= cpvec_table[i].cpv_size;
	src_phys = umap(spp, SEG_D, src_vir, (vir_bytes)bytes);
	dst_phys = umap(dpp, SEG_D, dst_vir, (vir_bytes)bytes);
	if (src_phys == 0 || dst_phys == 0) return(EFAULT);
	phys_copy(src_phys, dst_phys, bytes);
  }
  return(OK);
}


/*===========================================================================*
 *				do_nice					     *
 *===========================================================================*/
PRIVATE int do_nice(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Handle sys_nice(). */

  int proc;
  unsigned long quantum;
  struct proc *pp;

  /* Dismember the command message. */
  proc = m_ptr->m2_i1;
  quantum = m_ptr->m2_l1;

  assert(quantum > 0 && !(quantum & 0x80000000));
  pp= proc_addr(proc);
  pp->p_schedquant= quantum;

  return(OK);
}


/*===========================================================================*
 *				do_core					     *
 *===========================================================================*/
PRIVATE int do_core(m_ptr)
register message *m_ptr;	/* pointer to request message */
{
/* Handle sys_core(). */

  register struct proc *rp;	/* pointer to proc table entry */
  int req_proc, len;
  vir_bytes addr;
  phys_bytes dst_phys;

  /* Dismember the command message. */
  req_proc = m_ptr->m_source;
  rp= proc_addr(m_ptr->m1_i1);
  len = m_ptr->m1_i2;
  addr = (vir_bytes)m_ptr->m1_p1;

#if CHIP == INTEL && WORD_SIZE == 4

  core.cc.core_magic[0]= CORE_MAGIC0;
  core.cc.core_magic[1]= CORE_MAGIC1;
  core.cc.core_magic[2]= CORE_MAGIC2;
  core.cc.core_magic[3]= CORE_MAGIC3;
  strncpy(core.cc.core_name, rp->p_name, CORE_NAME_SIZE);
  core.cc.core_cpu= CORE_CPU_I80386;
  core.cc.core_flags= 0;
  core.cc.core_len= sizeof(core);
  core.cr.core_eax= rp->p_reg.sf_ax;
  core.cr.core_ebx= rp->p_reg.sf_bx;
  core.cr.core_ecx= rp->p_reg.sf_cx;
  core.cr.core_edx= rp->p_reg.sf_dx;
  core.cr.core_esp= rp->p_reg.sf_sp;
  core.cr.core_ebp= rp->p_reg.sf_bp;
  core.cr.core_esi= rp->p_reg.sf_si;
  core.cr.core_edi= rp->p_reg.sf_di;

  core.cr.core_es= rp->p_reg.sf_es;
  core.cr.core_cs= rp->p_reg.sf_cs;
  core.cr.core_ss= rp->p_reg.sf_ss;
  core.cr.core_ds= rp->p_reg.sf_ds;
  core.cr.core_fs= rp->p_reg.sf_fs;
  core.cr.core_gs= rp->p_reg.sf_gs;

  core.cr.core_eflags= rp->p_reg.sf_flags;
  core.cr.core_eip= rp->p_reg.sf_pc;

  core.c_text.core_phys= rp->p_map[SEG_T].mem_phys << CLICK_SHIFT;
  core.c_text.core_vir= rp->p_map[SEG_T].mem_vir << CLICK_SHIFT;
  core.c_text.core_len= rp->p_map[SEG_T].mem_len << CLICK_SHIFT;

  core.c_data.core_phys= rp->p_map[SEG_D].mem_phys << CLICK_SHIFT;
  core.c_data.core_vir= rp->p_map[SEG_D].mem_vir << CLICK_SHIFT;
  core.c_data.core_len= rp->p_map[SEG_D].mem_len << CLICK_SHIFT;

  core.c_stack.core_phys= rp->p_map[SEG_S].mem_phys << CLICK_SHIFT;
  core.c_stack.core_vir= rp->p_map[SEG_S].mem_vir << CLICK_SHIFT;
  core.c_stack.core_len= rp->p_map[SEG_S].mem_len << CLICK_SHIFT;

#endif /* CHIP == INTEL && WORD_SIZE == 4 */

#if CHIP == INTEL && WORD_SIZE == 2

  core.cc.core_magic[0]= CORE_MAGIC0;
  core.cc.core_magic[1]= CORE_MAGIC1;
  core.cc.core_magic[2]= CORE_MAGIC2;
  core.cc.core_magic[3]= CORE_MAGIC3;
  strncpy(core.cc.core_name, rp->p_name, CORE_NAME_SIZE);
  core.cc.core_cpu= CORE_CPU_I8086;
  core.cc.core_flags= 0;
  core.cc.core_len= sizeof(core);
  core.cr.core_ax= rp->p_reg.retreg;
  core.cr.core_bx= rp->p_reg.bx;
  core.cr.core_cx= rp->p_reg.cx;
  core.cr.core_dx= rp->p_reg.dx;
  core.cr.core_sp= rp->p_reg.sp;
  core.cr.core_bp= rp->p_reg.bp;
  core.cr.core_si= rp->p_reg.si;
  core.cr.core_di= rp->p_reg.di;

  core.cr.core_es= rp->p_reg.es;
  core.cr.core_cs= rp->p_reg.cs;
  core.cr.core_ss= rp->p_reg.ss;
  core.cr.core_ds= rp->p_reg.ds;

  core.cr.core_flags= rp->p_reg.psw;
  core.cr.core_ip= rp->p_reg.pc;

  core.c_text.core_phys= rp->p_map[SEG_T].mem_phys << CLICK_SHIFT;
  core.c_text.core_vir= rp->p_map[SEG_T].mem_vir << CLICK_SHIFT;
  core.c_text.core_len= rp->p_map[SEG_T].mem_len << CLICK_SHIFT;

  core.c_data.core_phys= rp->p_map[SEG_D].mem_phys << CLICK_SHIFT;
  core.c_data.core_vir= rp->p_map[SEG_D].mem_vir << CLICK_SHIFT;
  core.c_data.core_len= rp->p_map[SEG_D].mem_len << CLICK_SHIFT;

  core.c_stack.core_phys= rp->p_map[SEG_S].mem_phys << CLICK_SHIFT;
  core.c_stack.core_vir= rp->p_map[SEG_S].mem_vir << CLICK_SHIFT;
  core.c_stack.core_len= rp->p_map[SEG_S].mem_len << CLICK_SHIFT;

#endif /* CHIP == INTEL && WORD_SIZE == 2 */

  if (len != sizeof(core))
  			/* MM should use the same core header as the kernel */
  	panic("Invalid sys_core()", NO_NUM);

  dst_phys= numap(req_proc, addr, len);
  if (dst_phys == 0) panic ("can't umap in do_core()", 0);
  phys_copy(vir2phys(&core), dst_phys, (phys_bytes) len);

  return(OK);
}


/*===========================================================================*
 *				do_adjdata				     *
 *===========================================================================*/
PRIVATE int do_adjdata(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Handle sys_adjdata().  Change the memory map for MM. */

  int k;			/* process whose map is to be loaded */
  vir_bytes break_ptr;		/* New size of the data segment */
  vir_clicks new_clicks;	/* New size of the data segment in clicks */

  register struct proc *rp;
  vir_clicks data_change;

  /* Extract message parameters. */
  k = m_ptr->PROC1;
  break_ptr= (vir_bytes)(m_ptr->m1_p1);
  new_clicks= (break_ptr + CLICK_SIZE) >> CLICK_SHIFT;

  if (!isokprocn(k))
  	panic("bad proc in do_adjdata: ", m_ptr->PROC1);

  rp = proc_addr(k);		/* ptr to entry of the map */
  update_stack(rp);
  if (new_clicks > rp->p_map[SEG_D].mem_len)
  	data_change= new_clicks - rp->p_map[SEG_D].mem_len;
  else
  	data_change= 0;

  /* Check if the requested memory is really there. */
  if (vm_not_alloc < data_change)
  {
  	return ENOMEM;
  }

  /* check gaps, etc */
  if (rp->p_map[SEG_D].mem_vir + rp->p_map[SEG_D].mem_len + data_change +
  	STACK_SAFETY_CLICKS > rp->p_map[SEG_S].mem_vir)
  {
  	return ENOMEM;
  }

  rp->p_map[SEG_D].mem_len += data_change;
  vm_not_alloc -= data_change;
  return OK;
}


/*===========================================================================*
 *				do_adjstack				     *
 *===========================================================================*/
PRIVATE int do_adjstack(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Handle sys_adjstack().  Change the memory map for MM. */

  int k;			/* process whose map is to be loaded */
  vir_bytes stack_ptr;		/* New start of the stack segment */
  vir_clicks new_clicks;	/* New start of the stack segment in clicks */

  register struct proc *rp;
  vir_clicks stack_change;

  /* Extract message parameters. */
  k = m_ptr->PROC1;
  stack_ptr= (vir_bytes)(m_ptr->m1_p1);
  new_clicks= stack_ptr >> CLICK_SHIFT;

  if (!isokprocn(k))
  	panic("bad proc in do_adjstack: ", m_ptr->PROC1);

  rp = proc_addr(k);		/* ptr to entry of the map */
  update_stack(rp);
  if (new_clicks < rp->p_map[SEG_S].mem_vir)
  	stack_change= rp->p_map[SEG_S].mem_vir - new_clicks;
  else
  	stack_change= 0;

  /* Check if the requested memory is really there. */
  if (vm_not_alloc < stack_change)
  {
  	return ENOMEM;
  }

  /* check gaps, etc */
  if (rp->p_map[SEG_D].mem_vir + rp->p_map[SEG_D].mem_len +
	STACK_SAFETY_CLICKS > rp->p_map[SEG_S].mem_vir - stack_change)
  {
  	return ENOMEM;
  }

  rp->p_map[SEG_S].mem_vir -= stack_change;
  rp->p_map[SEG_S].mem_len += stack_change;
  vm_not_alloc -= stack_change;
  return OK;
}


/*===========================================================================*
 *				do_getmap				     *
 *===========================================================================*/
PRIVATE int do_getmap(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Handle sys_getmap().  Report the memory map to MM. */

  register struct proc *rp;
  phys_bytes dst_phys;
  int caller;			/* where the map has to be stored */
  int k;			/* process whose map is to be loaded */
  struct mem_map *map_ptr;	/* virtual address of map inside caller (MM) */

  /* Extract message parameters and copy new memory map to MM. */
  caller = m_ptr->m_source;
  k = m_ptr->PROC1;
  map_ptr = (struct mem_map *) m_ptr->MEM_PTR;

  if (!isokprocn(k))
  	panic("do_getmap got bad proc: ", m_ptr->PROC1);

  rp = proc_addr(k);		/* ptr to entry of the map */
  if (rp->p_flags & P_SLOT_FREE) return(ESRCH);

  /* Copy the map. */
  dst_phys = umap(proc_addr(caller), SEG_D, (vir_bytes) map_ptr,
  						sizeof(rp->p_map));
  if (dst_phys == 0) panic("bad call to sys_getmap", NO_NUM);
  phys_copy(vir2phys(rp->p_map), dst_phys, sizeof(rp->p_map));

  return(OK);
}


/*===========================================================================*
 *				do_getsig				     *
 *===========================================================================*/
PRIVATE int do_getsig(m_ptr)
message *m_ptr;			/* pointer to request message */
{
  register struct proc *rp;

  while((rp= sig_proclist) != NULL)
  {
	assert(rp->p_flags & SIG_PENDING);

	/* Assume that sigset_t is an integral type. */
	if (rp->p_pending == 0)
	{
		/* This process has no signals left. Remove the process from
		 * sig_proclist and make it ready.
		 */
		sig_proclist= rp->p_signal;
		rp->p_signal= NULL;
		rp->p_flags &= ~SIG_PENDING;
		lock_ready(rp);
		continue;
	}
	assert(isuserp(rp));
	m_ptr->m2_i1 = proc_number(rp);
	m_ptr->m2_l1 = rp->p_pending;
	sigemptyset(&rp->p_pending); /* the ball is now in MM's court */
	return 0;
  }
  return ESRCH;
}


/*==========================================================================*
 *				do_puts 				    *
 *==========================================================================*/
PRIVATE int do_puts(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Print a string. */

  log_print((vir_bytes)m_ptr->m1_p1, m_ptr->m1_i1, m_ptr->m_source);
  return OK;
}


/*==========================================================================*
 *				do_sysenv				    *
 *==========================================================================*/
PRIVATE int do_sysenv(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Deliver a kernel environment variable. */
  char key[256];
  char *val;
  phys_bytes src_phys, dst_phys;
  vir_bytes keylen, vallen;

  keylen= m_ptr->m1_i2;
  if (keylen <= 0 || keylen > sizeof(key))
	return EINVAL;
  src_phys = numap(m_ptr->m1_i1, (vir_bytes) m_ptr->m1_p1, keylen);
  if (src_phys == 0)
	return EFAULT;
  phys_copy(src_phys, vir2phys(key), (phys_bytes) keylen);
  key[keylen-1]= '\0';	/* Just to be sure */

  val= k_getenv(key);
  if (val == NULL)
  	return ESRCH;

  vallen= strlen(val)+1;
  if (vallen > m_ptr->m1_i3)
	return E2BIG;

  dst_phys = numap(m_ptr->m1_i1, (vir_bytes) m_ptr->m1_p2, vallen);
  if (dst_phys == 0)
  	return EFAULT;
  phys_copy(vir2phys(val), dst_phys, (phys_bytes) vallen);
  return(OK);
}


/*===========================================================================*
 *				remap					     *
 *===========================================================================*/
PRIVATE void remap(sp, dp)
struct proc *sp;
struct proc *dp;			/* Process that get's new pages */
{
  phys_bytes base, top, size;
  int snr, dnr;

  snr= proc_number(sp);
  dnr= proc_number(dp);
  dp->p_map[SEG_T]= sp->p_map[SEG_T];
  if (dp->p_map[SEG_T].mem_phys != dp->p_map[SEG_D].mem_phys)
  {
  	/* Separate I&D */
  	vm_remap_pages((dp->p_map[SEG_T].mem_phys + dp->p_map[SEG_T].mem_vir) <<
  		CLICK_SHIFT, dp->p_map[SEG_T].mem_len << CLICK_SHIFT, snr, dnr);
  }
  dp->p_map[SEG_D]= sp->p_map[SEG_D];
  vm_remap_pages((dp->p_map[SEG_D].mem_phys + dp->p_map[SEG_D].mem_vir) <<
	CLICK_SHIFT, dp->p_map[SEG_D].mem_len << CLICK_SHIFT, snr, dnr);
  dp->p_map[SEG_S]= sp->p_map[SEG_S];
  vm_remap_pages((dp->p_map[SEG_S].mem_phys + dp->p_map[SEG_S].mem_vir) <<
	CLICK_SHIFT, dp->p_map[SEG_S].mem_len << CLICK_SHIFT, snr, dnr);
  base= dp->p_map[SEG_T].mem_phys << CLICK_SHIFT;
  top= (dp->p_map[SEG_S].mem_phys + dp->p_map[SEG_S].mem_vir
				+ dp->p_map[SEG_S].mem_len) << CLICK_SHIFT;
  if (top < base)
  	panic("Stack not above text", NO_NUM);
  size= top-base;
  vm_remap_space(base, size, snr, dnr);
  alloc_segments(dp);
}


/*===========================================================================*
 *				unmap					     *
 *===========================================================================*/
PRIVATE void unmap(pp)
struct proc *pp;			/* which process needs to be unmapped */
{

#if PAGING_VM
  phys_bytes base, top, size;

  /* Determine the lowest amount of free memory.  One can tell by this number
   * how much memory was free at the worst, most memory intensive moment.
   */
  if (vm_not_alloc < vm_lowest_free) vm_lowest_free= vm_not_alloc;

  if (pp->p_map[SEG_T].mem_phys != pp->p_map[SEG_D].mem_phys)
  {
  	/* Separate I&D */
  	vm_unmap_pages((pp->p_map[SEG_T].mem_phys + pp->p_map[SEG_T].mem_vir) <<
  		CLICK_SHIFT, pp->p_map[SEG_T].mem_len << CLICK_SHIFT);
  }
  vm_unmap_pages((pp->p_map[SEG_D].mem_phys + pp->p_map[SEG_D].mem_vir) <<
	CLICK_SHIFT, pp->p_map[SEG_D].mem_len << CLICK_SHIFT);
  vm_unmap_pages((pp->p_map[SEG_S].mem_phys + pp->p_map[SEG_S].mem_vir) <<
	CLICK_SHIFT, pp->p_map[SEG_S].mem_len << CLICK_SHIFT);
  base= pp->p_map[SEG_T].mem_phys << CLICK_SHIFT;
  top= (pp->p_map[SEG_S].mem_phys + pp->p_map[SEG_S].mem_vir
				+ pp->p_map[SEG_S].mem_len) << CLICK_SHIFT;
  if (top < base)
  	panic("Stack not above text", NO_NUM);
  size= top-base;
  vm_unmap_space(base, size);
assert(vm_check_stats() ? 1: (vm_dump(), 0));
#if DEBUG
  vm_check_unmapped(base >> CLICK_SHIFT, (top-base) >> CLICK_SHIFT);
#endif

#else /* !PAGING_VM */
  phys_bytes base, top, size;

  base= pp->p_map[SEG_T].mem_phys << CLICK_SHIFT;
  top= (pp->p_map[SEG_S].mem_phys + pp->p_map[SEG_T].mem_vir +
				pp->p_map[SEG_S].mem_len) << CLICK_SHIFT;
  if (top < base)
  	panic("Stack not above text", NO_NUM);
  size= top-base;
  vm_unmap(base, size, pp->p_map[SEG_T].mem_len + pp->p_map[SEG_D].mem_len +
  	pp->p_map[SEG_S].mem_len);
#if DEBUG
  vm_check_unmapped(base, top);
#endif

#endif /* PAGING_VM */
}


/*===========================================================================*
 *				update_stack				     *
 *===========================================================================*/
PUBLIC static void update_stack(rp)
register struct proc *rp;	/* pointer to proc table entry for process */
{
/* Update the stack segment based on the current value of the stack pointer. */
  vir_clicks sp_click;		/* click where the stack pointer is. */
  vir_clicks adjust;		/* amount mem_vir has to be lowered to reach
				 * sp. */
  vir_clicks data_end;		/* end of data segment */

#if (CHIP == INTEL)
  if (rp->p_map[SEG_D].mem_phys != rp->p_map[SEG_S].mem_phys ||
	rp->p_map[SEG_D].mem_vir + rp->p_map[SEG_D].mem_len
					> rp->p_map[SEG_S].mem_vir)
  {
	panic("umap: invalid map for process ", proc_number(rp));
  }
#endif /* CHIP == INTEL */

  sp_click= rp->p_reg.sf_sp >> CLICK_SHIFT;
  if (sp_click < rp->p_map[SEG_D].mem_vir)
  	return;

  if (sp_click < rp->p_map[SEG_S].mem_vir)
  {
	data_end= rp->p_map[SEG_D].mem_vir + rp->p_map[SEG_D].mem_len;
	if (sp_click < data_end + STACK_SAFETY_CLICKS)
	{
		/* The process is running with the stack pointer in the
		 * data segment, or too close to the data segment.
		 * No need to adjust the stack segment.
		 */
		return;
	}
	adjust= rp->p_map[SEG_S].mem_vir-sp_click;
	if (adjust > vm_not_alloc)
		adjust= vm_not_alloc;
	rp->p_map[SEG_S].mem_vir -= adjust;
	rp->p_map[SEG_S].mem_len += adjust;
#if !SEGMENTED_MEMORY
	rp->p_map[SEG_S].mem_phys -= adjust;
#endif /* !SEGMENTED_MEMORY */
	vm_not_alloc -= adjust;
#if (CHIP == INTEL)
	if (rp->p_map[SEG_D].mem_phys != rp->p_map[SEG_S].mem_phys ||
		rp->p_map[SEG_D].mem_vir + rp->p_map[SEG_D].mem_len >
		rp->p_map[SEG_S].mem_vir)
	{
		panic("umap: invalid map for process ", proc_number(rp));
	}
#endif /* CHIP == INTEL */
  }
}


#if PAGING_VM
/*==========================================================================*
 *				do_swapoff				    *
 *==========================================================================*/
PRIVATE int do_swapoff(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Tell the vm system to stop using a certain segment for swapping. */

	int segm;

	segm= m_ptr->m1_i1;
	return vm_swapoff(segm);
}
#endif /* PAGING_VM */


#if PAGING_VM
/*==========================================================================*
 *				do_vm_lock				    *
 *==========================================================================*/
PRIVATE int do_vm_lock(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Tell the vm system to lock certain pages of a user process. */

	struct proc *pp;
	vir_bytes vir_addr, vir_count;
	phys_bytes phys_addr;

	pp= proc_addr(m_ptr->m2_i1);
	vir_addr= m_ptr->m2_l1;
	vir_count= m_ptr->m2_l2;
	phys_addr= umap(pp, SEG_D, vir_addr, vir_count);
	assert(phys_addr != 0);
	return vm_lock(phys_addr, vir_count, TRUE);
}
#endif /* PAGING_BM */


#if PAGING_VM
/*==========================================================================*
 *				do_vm_unlock				    *
 *==========================================================================*/
PRIVATE int do_vm_unlock(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Tell the vm system to unlock certain pages of a user process. */

	struct proc *pp;
	vir_bytes vir_addr, vir_count;
	phys_bytes phys_addr;

	pp= proc_addr(m_ptr->m2_i1);
	vir_addr= m_ptr->m2_l1;
	vir_count= m_ptr->m2_l2;
	phys_addr= umap(pp, SEG_D, vir_addr, vir_count);
	assert(phys_addr != 0);
	vm_unlock(phys_addr, vir_count);
	return OK;
}
#endif /* PAGING_VM */


/*===========================================================================*
 *				do_delmap				     *
 *===========================================================================*/
PRIVATE int do_delmap(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Handle sys_delmap().  A process loses it's map. */

  register struct proc *rp;
  int proc_nr;			/* number of process losing it's map */

  proc_nr = m_ptr->PROC1;
  rp = proc_addr(proc_nr);

  /* Test if the process infact had a map */
  if (rp->p_flags & NO_MAP)
  	panic("do_delmap: no map for process ", proc_nr);

#if VIRT_MEM
  unmap(rp);
#endif /* VIRT_MEM */
  rp->p_flags |= NO_MAP;
  return(OK);
}


/*===========================================================================*
 *				do_dupseg				     *
 *===========================================================================*/
PRIVATE int do_dupseg(m_ptr)
message *m_ptr;			/* pointer to request message */
{
	int srcproc;
	int dstproc;
	int segmbits;
	int shared, i;
	struct proc *srcp, *dstp;

	srcproc= m_ptr->m1_i1;
	dstproc= m_ptr->m1_i2;
	segmbits= m_ptr->m1_i3;

	srcp = proc_addr(srcproc);
	dstp = proc_addr(dstproc);

	/* Check bits in segmbits */
	if (segmbits & ~((1 << NR_SEGS)-1))
		panic("do_dupseg: invalid segmbits ", segmbits);

	/* Check the segment sizes */
	for (i= 0; i<NR_SEGS; i++)
	{
		if (!(segmbits & (1 << i)))
			continue;
		if (srcp->p_map[i].mem_len != dstp->p_map[i].mem_len)
		{
			printW(); printf("segment %d doesn't match\n", i);
			return EINVAL;
		}
	}

	/* Dup the segments */
	for (i= 0; i<NR_SEGS; i++)
	{
		if (!(segmbits & (1 << i)))
			continue;
		shared= (i == SEG_T);
		vm_fork(srcp->p_map[i].mem_phys+srcp->p_map[i].mem_vir,
			dstp->p_map[i].mem_phys+dstp->p_map[i].mem_vir,
			srcp->p_map[i].mem_len, shared);
	}
	return OK;
}


/*===========================================================================*
 *				do_findproc				     *
 *===========================================================================*/
PRIVATE int do_findproc(m_ptr)
message *m_ptr;			/* pointer to request message */
{
	struct proc *pp;

	m_ptr->m3_ca1[M3_STRING-1]= '\0';
	for (pp= BEG_PROC_ADDR; pp<END_PROC_ADDR; pp++)
	{
		if (!istaskp(pp) && !isservp(pp))
			continue;
		if (strcmp(pp->p_name, m_ptr->m3_ca1) != 0)
			continue;
		m_ptr->m3_i1= proc_number(pp);
		return OK;
	}
	return ESRCH;
}


/*===========================================================================*
 *				findproc				     *
 *===========================================================================*/
PUBLIC int findproc(name)
char *name;
{
	struct proc *pp;

	for (pp= BEG_PROC_ADDR; pp<END_PROC_ADDR; pp++)
	{
		if (!istaskp(pp) && !isservp(pp))
			continue;
		if (strcmp(pp->p_name, name) != 0)
			continue;
		return proc_number(pp);
	}
	return ANY;
}


/*===========================================================================*
 *				do_sysctl				     *
 *===========================================================================*/
PRIVATE int do_sysctl(m_ptr)
message *m_ptr;			/* pointer to request message */
{
	int proc_nr;
	struct proc *pp;
	int request;
	vir_bytes argp;

	proc_nr= m_ptr->m2_i1;
	pp= proc_addr(proc_nr);
	request= m_ptr->m2_l1;
	argp= (vir_bytes) m_ptr->m2_p1;

	switch (request) {
	case SYSQUERYPARAM:
		return sysqueryparam(proc_nr, argp);
	case SYSSIGNON: {
		struct systaskinfo info;
		phys_bytes phys_argp;

		/* Make this process a server. */
		if (!isuserp(pp)) return(EPERM);
		phys_argp= umap(pp, SEG_D, argp, (vir_bytes) sizeof(info));
		if (phys_argp == 0) return(EINVAL);

		lock_unready(pp);
		lock_dequeue(pp);
		pp->p_priority= PPRI_SERVER;
		lock_ready(pp);

		pp->p_pid= 0;
		info.proc_nr= proc_nr;
		phys_copy(vir2phys(&info), phys_argp,
						(phys_bytes) sizeof(info));
		return OK;	}
#if (CHIP == INTEL)
	case SYSREQIRQ: {
		phys_bytes phys_argp;
		struct sysreqirq rqirq;

		phys_argp= umap(pp, SEG_D, argp, (vir_bytes) sizeof(rqirq));
		if (phys_argp == 0) return(EFAULT);
		phys_copy(phys_argp, vir2phys(&rqirq),
						(phys_bytes) sizeof(rqirq));

		if (rqirq.irq < 0 || rqirq.irq >= NR_IRQ_VECTORS)
			return(EINVAL);
		irq_owner[rqirq.irq]= proc_nr;
		put_irq_handler(rqirq.irq, sys_handler);
		enable_irq(rqirq.irq);
		pp->p_status |= P_ST_SIGIRQ;
		return(OK);	}
	case SYSIOPRIV:
		/* Grant I/O privileges. */
		enable_iop(pp);
		return(OK);
#endif /* CHIP == INTEL */
	default:
		return EINVAL;
	}
}


/*===========================================================================*
 *				sysqueryparam				     *
 *===========================================================================*/
PRIVATE struct queryvars {
	struct svrqueryparam qpar;
	phys_bytes phys_param, phys_value;
	char parbuf[256], valbuf[256];
	char *param, *value;
	size_t vcount;
} *qvars;

PRIVATE int sysqueryparam(proc_nr, argp)
int proc_nr;
vir_bytes argp;
{
	/* Return values, sizes, or addresses of variables in kernel space. */

	struct queryvars qv;
	phys_bytes phys_argp;
	void *addr;
	size_t n, size;
	int byte;
	int more;
	static char hex[]= "0123456789ABCDEF";
#define sq_putc(c) ((void)((qv.vcount == 0 ? sq_flush() : (void) 0), \
					*qv.value++= (c), qv.vcount--))

	phys_argp= numap(proc_nr, argp, sizeof(qv.qpar));
	if (phys_argp == 0) return EFAULT;

	phys_copy(phys_argp, vir2phys(&qv.qpar),
					(phys_bytes) sizeof(qv.qpar));

	qv.phys_param= numap(proc_nr, (vir_bytes) qv.qpar.param, qv.qpar.psize);
	qv.phys_value= numap(proc_nr, (vir_bytes) qv.qpar.value, qv.qpar.vsize);
	if (qv.phys_param == 0 || qv.phys_value == 0) return EFAULT;

	/* Export these variables to sq_getc() and sq_flush(). */
	qvars= &qv;
	qv.param= qv.parbuf + sizeof(qv.parbuf);
	qv.value= qv.valbuf;
	qv.vcount= sizeof(qv.valbuf);

	do {
		more= queryparam(sq_getc, &addr, &size);
		for (n= 0; n < size; n++) {
			byte= ((u8_t *) addr)[n];
			sq_putc(hex[byte >> 4]);
			sq_putc(hex[byte & 0x0F]);
		}
		sq_putc(more ? ',' : 0);
	} while (more);
	sq_flush();
	return OK;
}


/*===========================================================================*
 *				sq_getc					     *
 *===========================================================================*/
PRIVATE int sq_getc()
{
	/* Return one character of the names to search for. */
	struct queryvars *qv= qvars;
	size_t n;

	if (qv->qpar.psize == 0) return 0;
	if (qv->param == qv->parbuf + sizeof(qv->parbuf)) {
		/* Need to fill the parameter buffer. */
		n= qv->qpar.psize;
		if (n > sizeof(qv->parbuf)) n= sizeof(qv->parbuf);
		phys_copy(qv->phys_param, vir2phys(qv->parbuf), (phys_bytes) n);
		qv->phys_param+= n;
		qv->param= qv->parbuf;
	}
	qv->qpar.psize--;
	return (u8_t) *qv->param++;
}


/*===========================================================================*
 *				sq_flush				     *
 *===========================================================================*/
PRIVATE void sq_flush()
{
	/* Send gathered characters back to the user. */
	struct queryvars *qv= qvars;
	size_t n;

	if ((n= qv->value - qv->valbuf) > qv->qpar.vsize) n= qv->qpar.vsize;
	if (n > 0) {
		/* Copy the value buffer to user space. */
		phys_copy(vir2phys(qv->valbuf), qv->phys_value, (phys_bytes) n);
		qv->phys_value+= n;
		qv->qpar.vsize-= n;
	}
	qv->value= qv->valbuf;
	qv->vcount= sizeof(qv->valbuf);
}


#if (CHIP == INTEL)
/*===========================================================================*
 *				sys_handler				     *
 *===========================================================================*/
PRIVATE int sys_handler(int irq)
{
	/* Interrupt requested by third party.  Send it on. */
	int proc_nr;

	if (irq < 0 || irq >= NR_IRQ_VECTORS)
		panic("invalid call to sys_handler", irq);

	irq_pending_set |= (1 << irq);

	/* Send an interrupt message.  The system task will send SIG_INT to
	 * a user process.
	 */
	interrupt(SYSTASK);
	return 0;
}
#endif /* CHIP == INTEL */

/*
 * $PchId: system.c,v 1.10 1996/05/07 08:12:07 philip Exp $
 */
