/*	synch.c - kernel process synchronization.	Author: Kees J. Bot
 *								15 Nov 1993
 */
#include "kernel.h"
#include <minix/com.h>
#include "proc.h"
#include "mq.h"
#include "synch.h"
#include "assert.h"
INIT_ASSERT

/* Interrupts must be disabled in critical code to keep others from
 * interfering.  (Not that it's possible, but we like to play this serious.)
 */
#define synch_lock()	lock()
#define synch_unlock()	unlock()

int mutex_lock(mutex_t *mvp)
/* Wait for a mutex variable to become available, then lock it. */
{
	struct proc *mvq;
	message sig;

	synch_lock();
	if (*mvp == NULL) {
		/* Mutex is free, grab it! */
		*mvp= proc_ptr;
		synch_unlock();
		return 0;
	}

	/* Mutex is busy, put myself on the end of the waiting list and wait
	 * for a message.
	 */
	for (mvq= *mvp; mvq->p_synchqueue != NULL; mvq= mvq->p_synchqueue) {}
	mvq->p_synchqueue= proc_ptr;
	synch_unlock();

	receive(mvq->p_nr, &sig);
	assert(sig.m_type == THREAD_SYNCH);
	assert((mutex_t *) sig.m1_p1 == mvp);
	return 0;
}

int mutex_unlock(mutex_t *mvp)
/* Unlock a mutex variable. */
{
	struct proc *mvq;
	message sig;

	/* Remove myself as mutex owner, giving it to the next in line. */
	assert(*mvp == proc_ptr);
	synch_lock();
	*mvp= mvq= proc_ptr->p_synchqueue;
	proc_ptr->p_synchqueue= NULL;
	synch_unlock();

	/* Someone may be waiting, wake 'em up. */
	if (mvq != NULL) {
		sig.m_type= THREAD_SYNCH;
		sig.m1_p1= (void *) mvp;
		send(mvq->p_nr, &sig);
	}
	return 0;
}

int mutex_trylock(mutex_t *mvp)
/* Try to lock a mutex.  Return EBUSY if you can't get it. */
{
	synch_lock();
	if (*mvp == NULL) {
		/* The mutex is free, so we can have it. */
		*mvp= proc_ptr;
		synch_unlock();
		return 0;
	}

	/* Mutex not free, bail out. */
	synch_unlock();
	return EBUSY;
}

int cond_wait(cond_t *cvp, mutex_t *mvp, mq_t **pmq)
/* Wait for a signal that a condition has changed.  Messages received while
 * waiting are put on the message queue *pmq.
 */
{
	struct proc **pcvq, *mvq;
	message sig;

	/* Release mutex. */
	assert(*mvp == proc_ptr);
	synch_lock();
	*mvp= mvq= proc_ptr->p_synchqueue;
	proc_ptr->p_synchqueue= NULL;

	/* Add myself to the queue waiting on the condition variable. */
	for (pcvq= cvp; *pcvq != NULL; pcvq= &(*pcvq)->p_synchqueue) {}
	*pcvq= proc_ptr;
	synch_unlock();

	/* Someone may be waiting on the mutex. */
	if (mvq != NULL) {
		sig.m_type= THREAD_SYNCH;
		sig.m1_p1= (void *) mvp;
		send(mvq->p_nr, &sig);
	}

	/* Find the end of message queue. */
	while (*pmq != NULL) pmq= &(*pmq)->mq_next;

	/* Wait for a signal on the condition variable. */
	for (;;) {
		receive(ANY, &sig);
		if (sig.m_type == THREAD_SYNCH) break;

		/* Wrong message, queue it. */
		*pmq= mq_get();
		(*pmq)->mq_mess= sig;
		pmq= &(*pmq)->mq_next;
	}
	assert((cond_t *) sig.m1_p1 == cvp);

	/* Condition variable released, get the mutex back. */
	return mutex_lock(mvp);
}

static int cond_Nsignal(cond_t *cvp, int bcast)
/* Signal a waiter or all waiters that a condition has changed.  Weather will
 * be foul if this is not protected by the same mutex as in cond_wait().
 */
{
	struct proc *cq;
	message sig;

	do {
		synch_lock();
		if (*cvp == NULL) {
			/* No (more) waiters. */
			synch_unlock();
			return 0;
		}

		/* Take one and send it a message. */
		cq= *cvp;
		*cvp= cq->p_synchqueue;
		synch_unlock();
		sig.m_type= THREAD_SYNCH;
		sig.m1_p1= (void *) cvp;
		send(cq->p_nr, &sig);
	} while (bcast);
	return 0;
}

int cond_signal(cond_t *cvp)
/* Signal just one waiter. */
{
	return cond_Nsignal(cvp, 0);
}

int cond_broadcast(cond_t *cvp)
/* Signal all waiters. */
{
	return cond_Nsignal(cvp, 1);
}

/*
 * $PchId: synch.c,v 1.5 1996/01/19 22:21:49 philip Exp $
 */
