/*==========================================================================*
 *		rs232.c - serial driver for 8250 and 16450 UARTs 	    *
 *		Added support for Atari ST M68901 and YM-2149	--kub	    *
 *		Standalone task by Philip Homburg			    *
 *==========================================================================*/

#include "kernel.h"

#include <stdlib.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include <termios.h>
#include "assert.h"
INIT_ASSERT
#include "mq.h"
#include "proc.h"
#include "timer.h"
#include "tty.h"

/*
 * Requests:
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_COUNT NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_OPEN    |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_CLOSE   |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_READ    |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_WRITE   |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_IOCTL NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_IOCTL3  |minor dev  | proc nr   |  fd       |  command  | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_OPERATION
 * -------------------------------------------------------------------|
 * | DEV_CANCEL  |minor dev  | proc nr   |  fd       | which operation|
 * |-------------+-----------+-----------+-----------+----------------|
 *
 * Replies:
 *
 *    m_type        REP_PROC_NR   REP_STATUS   REP_REF    REP_OPERATION
 * ----------------------------------------------------------------------|
 * | DEVICE_REPLY |   proc nr   |  status    |  fd     | which operation |
 * |--------------+-------------+------------+---------+-----------------|
 */

#if (CHIP != INTEL) && (MACHINE != ATARI)
#error				/* rs232.c only supports PC/AT and Atari ST */
#endif

#if (MACHINE == ATARI)
#include "staddr.h"
#include "stsound.h"
#include "stmfp.h"
#if (NR_RS_LINES > 1)
#error				/* Only one physical RS232 line available */
#endif /* NR_LINES > 1 */
#endif /* MACHINE == ATARI */

/* Constants. */
#define TTY_RB_SIZ	260	/* Should be at least T_RD_BUF_SIZ_MIN */
#define TTY_WB_SIZ	256	/* Should be at least T_WR_BUF_SIZ_MIN */

#define RS_RD_BUF	1024	/* Local input buffer */
#define RS_WR_BUF	1024	/* Local output buffer */

#if (CHIP == INTEL)
#define	NR_RS_LINES	   4	/* how many rs232 terminals can system handle*/  
#endif /* CHIP == INTEL */
#if (MACHINE == ATARI)
#define	NR_RS_LINES	   1	/* how many rs232 terminals can system handle*/
#endif /* MACHINE == ATARI */

#if (CHIP == INTEL)		/* PC/AT 8250/16450 chip combination */

/* 8250 constants. */
#define UART_FREQ         115200L	/* timer frequency */

/* Interrupt enable bits. */
#define IE_RECEIVER_READY       1
#define IE_TRANSMITTER_READY    2
#define IE_LINE_STATUS_CHANGE   4
#define IE_MODEM_STATUS_CHANGE  8

/* Interrupt identification bits. */
#define IIR_NO_INT		1
#define IIR_MODEM_STATUS_CHANGE	0
#define IIR_TRANSMITTER_READY	2
#define IIR_RECEIVER_READY	4
#define IIR_LINE_STATUS_CHANGE	6

/* Fifo control register (16550A) */
#define FCR_ENABLE		 0x1
#define FCR_CLEAR_RCVR		 0x2
#define FCR_CLEAR_XMIT		 0x4
#define FCR_DMA_SELECT		 0x8
#define FCR_TRIGGER_MASK	0xC0
#define FCR_TRIGGER_1		 0x0
#define FCR_TRIGGER_4		0x40
#define FCR_TRIGGER_8		0x80
#define FCR_TRIGGER_14		0xC0

/* Line control bits. */
#define LCR_CS5			 0x0
#define LCR_CS6			 0x1
#define LCR_CS7			 0x2
#define LCR_CS8			 0x3
#define LCR_STOPB		 0x4
#define LCR_PARENB		 0x8
#define LCR_PAREVN		0x10
#define LCR_STUCKP		0x20
#define LCR_BREAK		0x40
#define LCR_ADDRESS_DIVISOR	0x80

/* Line status bits. */
#define LSR_DATA_READY		 0x1
#define LSR_OVERRUN_ERR		 0x2
#define LSR_PARITY_ERR		 0x4
#define LSR_FRAMING_ERR		 0x8
#define LSR_BREAK_INTERRUPT	0x10
#define LSR_TRANSMITTER_READY	0x20

/* Modem control bits. */
#define MC_DTR                  1
#define MC_RTS                  2
#define MC_OUT2                 8	/* required for PC & AT interrupts */

/* Modem status bits. */
#define MSR_DCTS		 0x1
#define MSR_DDSR		 0x2
#define MSR_TERI		 0x4
#define MSR_DDCD		 0x8
#define MSR_CTS			0x10
#define MSR_DSR			0x20
#define MSR_RI			0x40
#define MSR_DCD			0x80

#else /* MACHINE == ATARI */		/* Atari ST 68901 USART */

/* Most of the USART constants are already defined in stmfp.h . The local
 * definitions made here are for keeping C code changes smaller.   --kub
 */

#define UART_FREQ          19200L	/* timer frequency */

/* Line control bits. */
#define LC_NO_PARITY            0
#define LC_DATA_BITS            3
#define LC_ODD_PARITY       U_PAR
#define LC_EVEN_PARITY     (U_PAR|U_EVEN)

/* Line status bits. */
#define LS_OVERRUN_ERR       R_OE
#define LS_PARITY_ERR        R_PE
#define LS_FRAMING_ERR       R_FE
#define LS_BREAK_INTERRUPT   R_BREAK

/* Modem status bits. */
#define MS_CTS               IO_SCTS	/* 0x04 */

#endif /* CHIP/MACHINE */

/* Input buffer watermarks.
 * The external device is asked to stop sending when the buffer
 * exactly reaches high water, or when TTY requests it.
 * TTY is also notified directly (rather than at a later clock tick) when
 * this watermark is reached.
 * A lower threshold could be used, but the buffer size and wakeup intervals
 * are chosen so the watermark shouldn't be hit at reasonable baud rates,
 * so this is unnecessary - the throttle is applied when TTY's buffers
 * get too full.
 * The low watermark is invisibly 0 since the buffer is always emptied all
 * at once.
 */
#define RS_IHIGHWATER (3 * RS_IBUFSIZE / 4)

#if (CHIP == INTEL)

/* Macros to handle flow control.
 * Interrupts must be off when they are used.
 * Time is critical - already the function call for out_byte() is annoying.
 * If out_byte() can be done in-line, tests to avoid it can be dropped.
 * istart() tells external device we are ready by raising RTS.
 * istop() tells external device we are not ready by dropping RTS.
 * DTR is kept high all the time (it probably should be raised by open and
 * dropped by close of the device).
 * OUT2 is also kept high all the time.
 */
#define istart(rs) \
  (out_byte( (rs)->modem_ctl_port, MC_OUT2 | MC_RTS | MC_DTR), \
   (rs)->rs_idevready = TRUE)
#define istop(rs) \
  (out_byte( (rs)->modem_ctl_port, MC_OUT2 | MC_DTR), \
  						(rs)->rs_idevready = FALSE)

/* Macro to tell if device is ready.
 * Don't require DSR, since modems drop this to indicate the line is not
 * ready even when the modem itself is ready.
 */
#define devready(rs) (in_byte(rs->rs_modem_status_reg) & MSR_CTS)

/* Macro to tell if transmitter is ready. */
#define txready(rs) (in_byte(rs->rs_line_status_reg) & LSR_TRANSMITTER_READY)

/* Macro to tell is receiver is ready. */
#define dtready(rs) (in_byte(rs->rs_line_status_reg) & LSR_DATA_READY)

#else /* MACHINE == ATARI */

/* Macros to handle flow control.
 * Time is critical - already the function call for lock()/restore() is
 * annoying.
 * istart() tells external device we are ready by raising RTS.
 * istop() tells external device we are not ready by dropping RTS.
 * DTR is kept high all the time (it probably should be raised by open and
 * dropped by close of the device). NOTE: The modem lines are active low.
 */
#define set_porta(msk,val) { register int s = lock();		\
			     SOUND->sd_selr = YM_IOA;		\
			     SOUND->sd_wdat =			\
				SOUND->sd_rdat & (msk) | (val);	\
			     restore(s);	}
#define istart(rs)         { set_porta( ~(PA_SRTS|PA_SDTR),0 ); \
			     (rs)->idevready = TRUE;	}
#define istop(rs)          { set_porta( ~PA_SDTR, PA_SRTS );	\
			     (rs)->idevready = FALSE;	}

/* Macro to tell if device is ready.
 * Don't require DSR, since modems drop this to indicate the line is not
 * ready even when the modem itself is ready.
 */
#define devready(rs)         (~MFP->mf_gpip & MS_CTS)

/* Transmitter ready test */
#define txready(rs)          (MFP->mf_tsr & (T_EMPTY | T_UE))

#endif /* CHIP/MACHINE */

/* Types. */
typedef unsigned char bool_t;	/* boolean */

/* RS232 device structure, one per device. */
typedef struct rs232
{
  int		rs_flags;
  int		rs_clocal;	/* ignore modem status lines */
  int 		rs_mode;	/* enabled or disabled */
  int		rs_type;	/* 8250, 16450, 16550, 16550A */
  char		rs_tty_readbuf[TTY_RB_SIZ];
  char		rs_tty_writebuf[TTY_WB_SIZ];
  struct tty	rs_tty;

  struct rs232	*rs_nextirq;

  bool_t rs_idevready;		/* nonzero if we are ready to receive (RTS) */

  char rs_rd_buf[RS_RD_BUF];	/* input buffer */
  int rs_rd_write;		/* index in rs_rd_buf for the next character
  				 * that arrives. */
  int rs_rd_read;		/* index in rs_rd_buf of the first character to 
  				 * give to the tty driver. */
  int rs_rd_size;		/* Number of character that can arrive before
  				 * the buffer is full. */
  int rs_rd_threshold;		/* if rs_rd_size drops below rs_rd_threshold we
  				 * negate RTS */
  int rs_rd_interrupt;		/* indicates a pending interrupt */
  int rs_rd_overflow;		/* # characters lost due to buffer overflow. */
  int rs_got_break;		/* indicates that we have to report a break */

  char rs_wr_buf[RS_WR_BUF];
  int rs_wr_write;
  int rs_wr_read;
  int rs_wr_size;
  int rs_wr_threshold;
  int rs_wr_interrupt;

#if (CHIP == INTEL)

  int rs_irq;
  port_t rs_port;

  port_t rs_trans_holding_reg;		/* i/o ports */
  port_t recv_port;
  port_t div_low_port;
  port_t div_hi_port;
  port_t int_enab_port;
  port_t rs_int_id_reg;
  port_t rs_fifo_ctl_reg;
  port_t line_ctl_port;
  port_t modem_ctl_port;
  port_t rs_line_status_reg;
  port_t rs_modem_status_reg;
  port_t rs_scratch_reg;

#endif /* CHIP == INTEL */

} rs232_t;

/* Bits in the rs_flags field */
#define RSF_READ_INITIATIVE	  0x1
#define RSF_WRITE_INITIATIVE	  0x2
#define RSF_IGNBRK		  0x8
#define RSF_BRKINT		 0x10
#define RSF_IGNPAR		 0x20
#define RSF_PARMRK		 0x40
#define RSF_INPCK		 0x80
#define RSF_ISTRIP		0x100

/* Values of rs_mode */
#define RSM_DISABLED		0
#define RSM_ENABLED		1

/* Values of rs_type */
#define RST_UNKNOWN		-1
#define RST_8250		0
#define RST_16450		1
#define RST_16550		2
#define RST_16550A		3

/* Special IRQ in rs_irq to indicate that the port is not in use. */
#define IRQ_OFF			(-1)

/* Table and macro to translate an RS232 minor device number to its
 * struct rs232_s pointer.
 */
PUBLIC rs232_t *p_rs_addr[NR_RS_LINES];

#define rs_addr(minor) (p_rs_addr[minor])

#if (CHIP == INTEL)
/* 8250 base addresses. */
typedef struct rs_conf
{
	int rc_mode;
	port_t rc_port;
	int rc_irq;
	char *rc_envvar;
} rs_conf_t;

PRIVATE rs_conf_t conf_8250[] =
{
/*	mode,		port,	irq,	envvar	*/
  {	RSM_ENABLED,	0x3F8,	4,	"SERIAL0"}, /* COM1: (line 0) */
  {	RSM_DISABLED,	0x2F8,	3,	"SERIAL1"}, /* COM2: (line 1) */
  {	RSM_DISABLED,	0x3E8,	4,	"SERIAL2"}, /* COM3: (line 2) */
  {	RSM_DISABLED,	0x2E8,	3,	"SERIAL3"}, /* COM4: (line 3) */
};
#endif

PUBLIC struct rs232 rs_lines[NR_RS_LINES];

PRIVATE struct rs232 *rs_irq[NR_IRQ_VECTORS];
PRIVATE mq_t *repl_queue, *repl_queue_tail;
PRIVATE tmrs_context_ut ser_cntxt;
PRIVATE int exp_timers= 0;
PRIVATE int ser_initialized;
PRIVATE int ser_debug_line= -1;
PRIVATE int ser_tasknr= ANY;			/* ANY to prevent accidents */
PRIVATE int hwint_count= 0;
PRIVATE volatile int dummy;

FORWARD _PROTOTYPE( void in_int, (struct rs232 *rs) );
FORWARD _PROTOTYPE( void line_int, (struct rs232 *rs) );
FORWARD _PROTOTYPE( void modem_int, (struct rs232 *rs) );
FORWARD _PROTOTYPE( void out_int, (struct rs232 *rs) );
FORWARD _PROTOTYPE( void reply, (mq_t *mq, int result, int can_enqueue ));
FORWARD _PROTOTYPE( int reply_queue, (int proc, int ref, int operation)	);
FORWARD _PROTOTYPE( void serial_init, (void)				);
FORWARD _PROTOTYPE( void serial_conf, (void)				);
FORWARD _PROTOTYPE( void update_conf, (rs232_t *serial, rs_conf_t *rcp)	);
FORWARD _PROTOTYPE( void serial_get, (int serial_line)			);
FORWARD _PROTOTYPE( int serial_put, (int serial_line, char *buf, 
							int siz)	);
FORWARD _PROTOTYPE( void serial_reply, (int serial_line, mq_t *mq)	);
FORWARD _PROTOTYPE( void serial_int, (void)				);
FORWARD _PROTOTYPE( void serial_setattr, (int serial_line, 
					struct termios *tmios_p)	);
FORWARD _PROTOTYPE( void rs_init, (int serial_line)				);
FORWARD _PROTOTYPE( int rs232handler, (int irq)				);



/*===========================================================================*
 *				serial_task				     *
 *===========================================================================*/
PUBLIC void serial_task()
{
	mq_t *mq;
	int result;
	int minor, proc;
	rs232_t *serial;

	serial_conf();
	serial_init();
	ser_initialized= TRUE;

	while(TRUE)
	{
		mq= mq_get();
		if (!mq)
			panic("out of messages", NO_NUM);
		
		result= receive(ANY, &mq->mq_mess);
		if (result < 0)
			panic("unable to receive: ", result);

		if (mq->mq_mess.m_source == HARDWARE)
		{
			assert(mq->mq_mess.m_type == HARD_INT);
			if (exp_timers)
				tmrs_exptimers(&ser_cntxt);
			serial_int();
			mq_free(mq);
			continue;
		}

		if (mq->mq_mess.m_type == DEV_IOCTL)
		{
			/* Compatibility code for old (sgtty) ioctls */
			minor= mq->mq_mess.m2_i1;
			if (minor >= 0 && minor < NR_RS_LINES)
			{
				/* This packet is handled by the tty library. */
				serial= &rs_lines[minor];
				tty_receive(&serial->rs_tty, mq);
				continue;
			}
			/* The serial task doesn't implement old ioctls */
			proc= mq->mq_mess.m2_i2;	/* PROC_NR */
			mq->mq_mess.m_type= TASK_REPLY;
			mq->mq_mess.m2_i1= proc;	/* REP_PROC_NR */
			mq->mq_mess.m2_i2= ENOTTY;
			result= send(mq->mq_mess.m_source, &mq->mq_mess);
			assert(result == OK);
			mq_free(mq);
			continue;
		}
		
		assert(mq->mq_mess.m_type != DEV_IOCTL);
		if (repl_queue)
		{
			if (mq->mq_mess.m_type == DEV_CANCEL)
			{
				result= reply_queue(mq->mq_mess.NDEV_PROC,
					mq->mq_mess.NDEV_REF, 
					mq->mq_mess.NDEV_OPERATION);
				if (result)
				{
					/* The reply for the canceled request
					 * was already in the queue.
					 */
					mq_free(mq);
					continue;
				}
			}
			else
				reply_queue(ANY, 0, 0);
		}

		minor= mq->mq_mess.NDEV_MINOR;
		if (minor >= 0 && minor < NR_RS_LINES)
		{
			/* This packet is handled by the tty library. */
			serial= &rs_lines[minor];
			if (serial->rs_mode == RSM_DISABLED)
			{
				reply(mq, ENXIO, FALSE);
				mq_free(mq);
				continue;
			}
			tty_receive(&serial->rs_tty, mq);
			continue;
		}
		reply(mq, ENXIO, FALSE);
		mq_free(mq);
		continue;
	}
}


/*===========================================================================*
 *				reply					     *
 *===========================================================================*/
PRIVATE void reply(mq, result, can_enqueue)
mq_t *mq;
int result;
int can_enqueue;
{
	int proc, ref, status, operation;
	message m, *mp;

	assert(mq->mq_allocated);
	
	proc= mq->mq_mess.NDEV_PROC;
	ref= mq->mq_mess.NDEV_REF;
	operation= mq->mq_mess.m_type;

	if (can_enqueue)
		mp= &mq->mq_mess;
	else
		mp= &m;

	mp->m_type= DEVICE_REPLY;
	mp->REP_PROC_NR= proc;
	mp->REP_STATUS= result;
	mp->REP_REF= ref;
	mp->REP_OPERATION= operation;

	status= send(mq->mq_mess.m_source, mp);
	if (status == ELOCKED && can_enqueue)
	{
		mq->mq_next= NULL;
		if (repl_queue)
			repl_queue_tail->mq_next= mq;
		else
			repl_queue= mq;
		repl_queue_tail= mq;
		return;
	}
	if (status != OK)
		panic("send failed", result);
	if (can_enqueue)
		mq_free(mq);
}


/*===========================================================================*
 *				reply_queue				     *
 *===========================================================================*/
PRIVATE int reply_queue(proc, ref, operation)
int proc;
int ref;
int operation;
{
	mq_t *mq, *cancel_reply, *tmp_m;
	int result;

	cancel_reply= NULL;
	for (mq= repl_queue; mq;)
	{
		if (cancel_reply == NULL &&
			mq->mq_mess.REP_PROC_NR == proc && 
			mq->mq_mess.REP_REF == ref &&
			(operation == CANCEL_ANY || operation == 
			mq->mq_mess.REP_OPERATION))
		{	/* Found a reply to the canceled request. */
			cancel_reply= mq;
			mq= mq->mq_next;
			continue;
		}
		result= send(mq->mq_mess.m_source, &mq->mq_mess);
		if (result != OK)
			panic("send failed", result);
		tmp_m= mq;
		mq= mq->mq_next;
		mq_free(tmp_m);
	}
	if (cancel_reply)
	{
		result= send(cancel_reply->mq_mess.m_source, 
			&cancel_reply->mq_mess);
		if (result != OK)
			panic("send failed", result);
		mq_free(cancel_reply);
	}
	repl_queue= NULL;
	if (cancel_reply)
		return 1;
	else
		return 0;
}


/*===========================================================================*
 *				serial_init				     *
 *===========================================================================*/
PRIVATE void serial_init()
{
	int i;
	rs232_t *serial;

	repl_queue= NULL;

	ser_tasknr= proc_number(proc_ptr);

	tmrs_initcontext(&ser_cntxt, &exp_timers, ser_tasknr);
	for (i=0, serial= rs_lines; i<NR_RS_LINES; i++, serial++)
	{
		if (serial->rs_mode == RSM_DISABLED)
			continue;

		rs_init(i);
		if (serial->rs_type == RST_UNKNOWN)
			continue;

		serial->rs_flags= RSF_READ_INITIATIVE | RSF_WRITE_INITIATIVE;
		serial->rs_clocal= 0;
		tty_init(&serial->rs_tty, i, 
			serial->rs_tty_readbuf, TTY_RB_SIZ,
			serial->rs_tty_writebuf, TTY_WB_SIZ, ser_tasknr,
			&ser_cntxt);
		serial->rs_tty.t_get= serial_get;
		serial->rs_tty.t_put= serial_put;
		serial->rs_tty.t_reply= serial_reply;
		serial->rs_tty.t_setattr= serial_setattr;
	}
}


/*===========================================================================*
 *				serial_conf				     *
 *===========================================================================*/
PRIVATE void serial_conf()
{
  rs232_t *serial;
  int i, nconf;

  for (i= 0; i<NR_RS_LINES; i++)
	rs_lines[i].rs_mode= RSM_DISABLED;	/* default, don't use this port */

  nconf= sizeof(conf_8250) / sizeof(conf_8250[0]); /* # of configurations */
  if (nconf < NR_RS_LINES)
  {
	printf("rs232: warning more ports (%d) than configurations (%d)\n",
		NR_RS_LINES, nconf);
  }
  else if (nconf > NR_RS_LINES)
  {
	printf("rs232: warning more configurations (%d) than ports (%d)\n",
		nconf, NR_RS_LINES);
	nconf= NR_RS_LINES;
  }

  for (i= 0, serial= &rs_lines[0]; i<nconf; i++, serial++)
	update_conf(serial, &conf_8250[i]);
}


/*===========================================================================*
 *				update_conf				     *
 *===========================================================================*/
PRIVATE void update_conf(serial, rcp)
rs232_t *serial;
rs_conf_t *rcp;
{
	long v;
	static char rc_fmt[] = "x:d";

	/* The default settings may be overridden from the environment. */
	serial->rs_mode= rcp->rc_mode;
	v= rcp->rc_port;
	switch (env_parse(rcp->rc_envvar, rc_fmt, 0, &v, 0x000L, 0x3ffL)) {
	case EP_OFF:
		serial->rs_mode= RSM_DISABLED;
		break;
	case EP_ON:
	case EP_SET:
		serial->rs_mode= RSM_ENABLED;
		break;
	}
	serial->rs_port= v;

	v= rcp->rc_irq;
	(void) env_parse(rcp->rc_envvar, rc_fmt, 1, &v, 0L,
						(long) NR_IRQ_VECTORS - 1);
	serial->rs_irq= v;

	/* Don't let a disabled line be recognized in the interrupt handler. */
	if (serial->rs_mode == RSM_DISABLED)
		serial->rs_irq= IRQ_OFF;
}


/*===========================================================================*
 *				serial_get				     *
 *===========================================================================*/
PRIVATE void serial_get(serial_line)
int serial_line;
{
/* The tty wants some data. */

	int total, size, r;
	int rd_size, rd_read;
	rs232_t *serial;

	assert (serial_line >= 0 || serial_line < NR_RS_LINES);
	serial= &rs_lines[serial_line];

	if (debug && serial->rs_rd_overflow > 256)
	{
		printf("serial[%d]: missed %d characters\n", serial_line,
			serial->rs_rd_overflow);
		serial->rs_rd_overflow= 0;
	}

	lock();
	rd_size= RS_RD_BUF-serial->rs_rd_size;
	unlock();
	rd_read= serial->rs_rd_read;
	total= 0;
	for (;;)
	{
		size= rd_size-total;
		if (size == 0)
		{
			istart(serial);
			serial->rs_flags |= RSF_READ_INITIATIVE;
			tty_put(&serial->rs_tty, NULL, 0);
			break;
		}
		if (RS_RD_BUF-rd_read < size)
			size= RS_RD_BUF-rd_read;
		assert(size > 0);
		r= tty_put(&serial->rs_tty, 
			serial->rs_rd_buf+rd_read, size);
		assert(r<=size);
		total += r;
		rd_read += r;
		if (rd_read == RS_RD_BUF)
			rd_read= 0;
		if (r != size)
		{
			serial->rs_flags &= 
					~RSF_READ_INITIATIVE;
			break;
		}
	}
	lock();
	serial->rs_rd_size += total;
	unlock();
	serial->rs_rd_read= rd_read;
}


/*===========================================================================*
 *				serial_put				     *
 *===========================================================================*/
PRIVATE int serial_put(serial_line, buf, siz)
int serial_line;
char *buf;
int siz;
{
/* We got some data from tty. */
	rs232_t *serial;
	int total;
	int buf_size, size;
	int wr_write;

	assert (serial_line >= 0 || serial_line < NR_RS_LINES);
	serial= &rs_lines[serial_line];

	if (siz == 0)
	{
		/* This is a flush command. */
		return 0;
	}
		
	total= 0;
	wr_write= serial->rs_wr_write;

	/* rs_wr_size is also updated at interrupt level so we have
	 * to use a lock.
	 */
	lock();
	buf_size= RS_WR_BUF-serial->rs_wr_size;
	unlock();
	do
	{
		assert(buf_size >= total);
		if (buf_size == total)
			break;
		size= buf_size-total;
		if (size > siz)
			size= siz;

		if (size > RS_WR_BUF-wr_write)
			size= RS_WR_BUF-wr_write;

		assert(size > 0);

		/* Copy in */
		memcpy(serial->rs_wr_buf+wr_write, buf, size);
		wr_write += size;
		total += size;
		siz -= size;
		buf += size;
		if (wr_write == RS_WR_BUF)
			wr_write= 0;
		assert(wr_write < RS_WR_BUF);

		/* Update wr_size and try to send a character. */
		lock();
		serial->rs_wr_size += size;
		if (txready(serial)) 
			out_int(serial);
		unlock();
	}
	while (siz > 0);
	assert(siz >= 0);

	serial->rs_wr_write= wr_write;

	if (siz != 0)
	{
		/* We have to block output. */
		if (!(serial->rs_flags & RSF_WRITE_INITIATIVE))
		{
			serial->rs_flags |= RSF_WRITE_INITIATIVE;
			/* Let's have another check */
			lock();
			if (txready(serial)) out_int(serial);
			unlock();
		}
	}
	return total;
}


/*===========================================================================*
 *				serial_reply				     *
 *===========================================================================*/
PRIVATE void serial_reply(serial_line, mq)
int serial_line;
mq_t *mq;
{
/* Tty wants to send a reply but the send is block because FS is sending to
 * this task. Enqueue to reply.
 */
	rs232_t *serial;

	assert (serial_line >= 0 || serial_line < NR_RS_LINES);
	serial= &rs_lines[serial_line];

#if DEBUG
	printf("in serial_reply\n");
#endif
	mq->mq_next= NULL;
	if (repl_queue)
		repl_queue_tail->mq_next= mq;
	else
		repl_queue= mq;
	repl_queue_tail= mq;
}


/*==========================================================================*
 *				serial_int			 	    *
 *==========================================================================*/
PRIVATE void serial_int()
{
	int i;
	rs232_t *serial;

	hwint_count= 0;
	for (i=0, serial= rs_lines; i<NR_RS_LINES; i++, serial++)
	{
		if (serial->rs_rd_interrupt)
		{
			serial->rs_rd_interrupt= FALSE;
			serial_get(i);
		}
		if (serial->rs_got_break)
		{
			serial->rs_got_break= FALSE;
			tty_break_int(&serial->rs_tty);
		}
		if (serial->rs_wr_interrupt)
		{
			serial->rs_wr_interrupt= FALSE;
			serial->rs_flags &= ~RSF_WRITE_INITIATIVE;
			tty_get(&serial->rs_tty);
		}
	}
}


/*==========================================================================*
 *				serial_setattr			 	    *
 *==========================================================================*/
PRIVATE void serial_setattr(line, tmios_p)
int line;			/* which rs line */
struct termios *tmios_p;	/* configuration */
{
/* Set various line control parameters for RS232 I/O.
 * If DataBits == 5 and StopBits == 2, 8250 will generate 1.5 stop bits.
 * The 8250 can't handle split speed, but we have propagated both speeds
 * anyway for the benefit of future UART chips.
 */

  int divisor;
  int line_controls, fifo;
  int baud;
  register rs232_t *rs;
  speed_t ospeed;

  rs = rs_addr(line);

  rs->rs_clocal= !!(tmios_p->c_cflag & CLOCAL);

#if (CHIP == INTEL)
  baud= 0;				/* Default is do nothing */
  fifo= 1;				/* Enable Fifo except for
  					 * speed <= 2400 */
  ospeed= cfgetospeed(tmios_p);
  switch (ospeed)
  {
  case B0:
  	printf("serial_setattr: warning modem lines should be dropped\n"); 
  	break;
  case B50:	baud= 50; fifo= 0; break;
  case B75:	baud= 75; fifo= 0; break;
  case B110:	baud= 110; fifo= 0; break;
  case B134:	baud= 134; fifo= 0; break;
  case B150:	baud= 150; fifo= 0; break;
  case B200:	baud= 200; fifo= 0; break;
  case B300:	baud= 300; fifo= 0; break;
  case B600:	baud= 600; fifo= 0; break;
  case B1200:	baud= 1200; fifo= 0; break;
  case B1800:	baud= 1800; fifo= 0; break;
  case B2400:	baud= 2400; fifo= 0; break;
  case B4800:	baud= 4800; break;
  case B9600:	baud= 9600; break;
  case B19200:	baud= 19200; break;
  case B38400:	baud= 38400; break;
  case B57600:	baud= 57600; break;
  case B115200:	baud= 115200; break;
  default:
  	printf(
  	"serial_setattr: warning illegal value for ospeed: %d (ignored)\n",
  		ospeed);
  	break;
  }
  if (baud != 0)
  {
	divisor = (int) (UART_FREQ / baud);	/* 8250 can't hack 2 speeds */
	if (rs->rs_type == RST_16550A)
	{
		if (fifo)
		{
			out_byte(rs->rs_fifo_ctl_reg, FCR_ENABLE |
				FCR_TRIGGER_8);
		}
		else
		{
			out_byte(rs->rs_fifo_ctl_reg, FCR_ENABLE |
				FCR_TRIGGER_1);
		}
	}
  }
  else
	divisor = 0;				/* Error, don't set baud rate */

  line_controls= 0;
  switch(tmios_p->c_cflag & CSIZE)
  {
  case CS5: line_controls |= LCR_CS5; break;
  case CS6: line_controls |= LCR_CS6; break;
  case CS7: line_controls |= LCR_CS7; break;
  case CS8: line_controls |= LCR_CS8; break;
  default:
  	printf("serial_setattr: illegal CSIZE value: 0x%x\n", 
  						tmios_p->c_cflag & CSIZE);
  	if (tmios_p->c_cflag & PARENB)
		line_controls |= LCR_CS7;
	else
		line_controls |= LCR_CS8;
	break;
  }
  if (tmios_p->c_cflag & CSTOPB)
  	line_controls |= LCR_STOPB;
  if (tmios_p->c_cflag & PARENB)
	line_controls |= LCR_PARENB;
  if (!(tmios_p->c_cflag & PARODD))
	line_controls |= LCR_PAREVN;

  /* Lock out interrupts while setting the speed. The receiver register is
   * going to be hidden by the div_low register, but the input interrupt
   * handler relies on reading it to clear the interrupt and avoid looping
   * forever.
   */
  lock();

  /* Select the baud rate divisor registers and change the rate. */
  if (divisor != 0)
  {
	out_byte(rs->line_ctl_port, LCR_ADDRESS_DIVISOR);
	out_byte(rs->div_low_port, divisor);
	out_byte(rs->div_hi_port, divisor >> 8);
  }

  /* Change the line controls and reselect the usual registers. */
  out_byte(rs->line_ctl_port, line_controls);

  rs->rs_flags &= ~RSF_IGNBRK;
  if (tmios_p->c_iflag & IGNBRK)
  	rs->rs_flags |= RSF_IGNBRK;

  rs->rs_flags &= ~RSF_BRKINT;
  if (tmios_p->c_iflag & BRKINT)
  	rs->rs_flags |= RSF_BRKINT;

  rs->rs_flags &= ~RSF_IGNPAR;
  if (tmios_p->c_iflag & IGNPAR)
  	rs->rs_flags |= RSF_IGNPAR;

  rs->rs_flags &= ~RSF_PARMRK;
  if (tmios_p->c_iflag & PARMRK)
  	rs->rs_flags |= RSF_PARMRK;

  rs->rs_flags &= ~RSF_INPCK;
  if (tmios_p->c_iflag & INPCK)
  	rs->rs_flags |= RSF_INPCK;

  rs->rs_flags &= ~RSF_ISTRIP;
  if (tmios_p->c_iflag & ISTRIP)
  	rs->rs_flags |= RSF_ISTRIP;

#else /* MACHINE == ATARI */
#include <warning should be checked>
  line_controls = parity | U_Q16;
  switch (stop_bits) {
	case 1:	line_controls |= U_ST1; break;
	case 2:	line_controls |= U_ST2; break;
  }
  switch (data_bits) {
	case 5:	line_controls |= U_D5; break;
	case 6:	line_controls |= U_D6; break;
	case 7:	line_controls |= U_D7; break;
	case 8:	line_controls |= U_D8; break;
  }
  lock();
  MFP->mf_ucr = line_controls;
  /* Check if baud rate is valid. Some decent baud rates may not work because
   * the timer cannot be programmed exactly enough, e.g. 7200. That is caught
   * by the expressions below which check that the resulting baud rate has a
   * deviation of max. 5%.
   */
  if (((UART_FREQ - (long)  divisor*in_baud >  UART_FREQ / 20) &&
       (UART_FREQ - (long)++divisor*in_baud < -UART_FREQ / 20))   ||
      divisor == 0 || divisor > 256)
	printf ("tty line %d: can't set speed %d\n", minor, in_baud);
  else
	MFP->mf_tddr = divisor;
#endif /* CHIP/MACHINE */

  istart(rs);
  lock();
  if (txready(rs)) 
	out_int(rs);
  unlock();
}


/*==========================================================================*
 *				rs_init					    *
 *==========================================================================*/
PRIVATE void rs_init(serial_line)
int serial_line;			/* which rs line */
{
/* Initialize RS232 for one line. */

  register rs232_t *rs;
  struct termios tmios;
  int id, scratch, res1, res2;

  rs = rs_addr(serial_line) = &rs_lines[serial_line];

  rs->rs_type= RST_UNKNOWN;

  /* Set up input queue. */
  rs->rs_rd_write= 0;
  rs->rs_rd_read= 0;
  rs->rs_rd_size= RS_RD_BUF;
  rs->rs_rd_threshold= RS_RD_BUF/4;
  rs->rs_rd_interrupt= FALSE;
  rs->rs_got_break= FALSE;

  /* Set up output buffer. */
  rs->rs_wr_write= 0;
  rs->rs_wr_read= 0;
  rs->rs_wr_size= 0;
  rs->rs_wr_threshold= RS_WR_BUF/4;
  rs->rs_wr_interrupt= FALSE;

#if (CHIP == INTEL)
  /* Precalculate port numbers for speed. Magic numbers in the code (once). */
  rs->rs_trans_holding_reg = rs->rs_port + 0;
  rs->recv_port = rs->rs_port + 0;
  rs->div_low_port = rs->rs_port + 0;
  rs->div_hi_port = rs->rs_port + 1;
  rs->int_enab_port = rs->rs_port + 1;
  rs->rs_int_id_reg = rs->rs_port + 2;
  rs->rs_fifo_ctl_reg = rs->rs_port + 2;
  rs->line_ctl_port = rs->rs_port + 3;
  rs->modem_ctl_port = rs->rs_port + 4;
  rs->rs_line_status_reg = rs->rs_port + 5;
  rs->rs_modem_status_reg = rs->rs_port + 6;
  rs->rs_scratch_reg = rs->rs_port + 7;

  /* Probe the UART */
  out_byte(rs->rs_fifo_ctl_reg, FCR_ENABLE);
  id= in_byte(rs->rs_int_id_reg);
  if (id == 0xff)
  {
	/* No device */
	return;
  }
  id= id >> 6;
  switch(id)
  {
  case 0:	rs->rs_type= RST_16450; break;		/* or 8250 */
  case 1:	break;					/* unknown */
  case 2:	rs->rs_type= RST_16550;	break;
  case 3:	rs->rs_type= RST_16550A; break;
  }

  if (rs->rs_type == RST_16450)
  {
  	scratch= in_byte(rs->rs_scratch_reg);
  	out_byte(rs->rs_scratch_reg, 0xa5);
  	res1= in_byte(rs->rs_scratch_reg);
  	out_byte(rs->rs_scratch_reg, 0x5a);
  	res2= in_byte(rs->rs_scratch_reg);
  	out_byte(rs->rs_scratch_reg, scratch);
  	if (res1 != 0xa5 || res2 != 0x5a)
  		rs->rs_type= RST_8250;
  }
#endif

  if (debug) {
	printf("tty%02d: port %x irq %d, UART type is ", serial_line,
		rs->rs_port, rs->rs_irq);
	switch(rs->rs_type)
	{
	case RST_8250:	printf("8250\n"); break;
	case RST_16450:	printf("16450\n"); break;
	case RST_16550:	printf("16550 (fifo not enabled)\n"); break;
	case RST_16550A:printf("16550A\n"); break;
	default:	printf("unknown\n");
	}
  }

  /* Set up the hardware to a base state, in particular
   *	o turn off DTR (MC_DTR) to try to stop the external device.
   *	o be careful about the divisor latch.  Some BIOS's leave it enabled
   *	  here and that caused trouble (no interrupts) in version 1.5 by
   *	  hiding the interrupt enable port in the next step, and worse trouble
   *	  (continual interrupts) in an old version by hiding the receiver
   *	  port in the first interrupt.  Call rs_config() early to avoid this.
   *	o disable interrupts at the chip level, to force an edge transition
   *	  on the 8259 line when interrupts are next enabled and active.
   *	  RS232 interrupts are guaranteed to be disabled now by the 8259
   *	  mask, but there used to be trouble if the mask was set without
   *	  handling a previous interrupt.
   */
  istop(rs);			/* sets modem_ctl_port */

  /* Allow this port to be processed during interrupt handling */

  cfsetispeed(&tmios, TSPEED_DEF);
  cfsetospeed(&tmios, TSPEED_DEF);
  tmios.c_cflag= CS8;
  serial_setattr(serial_line, &tmios);
#if (CHIP == INTEL)
  out_byte(rs->int_enab_port, 0);
#endif

  /* Clear any harmful leftover interrupts.  An output interrupt is harmless
   * and will occur when interrupts are enabled anyway.  Set up the output
   * queue using the status from clearing the modem status interrupt.
   */
#if (CHIP == INTEL)
  in_byte(rs->rs_line_status_reg);
  in_byte(rs->recv_port);
#endif
  dummy= devready(rs);				/* reads modem_ctl_port */

#if (CHIP == INTEL)
  lock();
  /* Enable interrupts for both interrupt controller and device. */
  put_irq_handler(rs->rs_irq, rs232handler);

  assert(rs->rs_irq >= 0 && rs->rs_irq < NR_IRQ_VECTORS);
  rs->rs_nextirq= rs_irq[rs->rs_irq];
  rs_irq[rs->rs_irq]= rs;
  enable_irq(rs->rs_irq);
  out_byte(rs->int_enab_port, IE_LINE_STATUS_CHANGE | IE_MODEM_STATUS_CHANGE
				| IE_RECEIVER_READY | IE_TRANSMITTER_READY);
#else /* MACHINE == ATARI */
  /* Initialize the 68901 chip, then enable interrupts. */
  MFP->mf_scr = 0x00;
  MFP->mf_tcdcr |= T_Q004;
  MFP->mf_rsr |= R_ENA;
  MFP->mf_tsr |= T_ENA;
  MFP->mf_aer = (MFP->mf_aer | (IO_SCTS|IO_SDCD)) ^
		 (MFP->mf_gpip & (IO_SCTS|IO_SDCD));
  MFP->mf_ddr = (MFP->mf_ddr & ~ (IO_SCTS|IO_SDCD));
  MFP->mf_iera |= (IA_RRDY|IA_RERR|IA_TRDY|IA_TERR);
  MFP->mf_imra |= (IA_RRDY|IA_RERR|IA_TRDY|IA_TERR);
  MFP->mf_ierb |= (IB_SCTS|IB_SDCD);
  MFP->mf_imrb |= (IB_SCTS|IB_SDCD);
#endif /* CHIP/MACHINE */

  /* Tell external device we are ready. */
  istart(rs);

rs232handler(rs->rs_irq);
}

/* Low level (interrupt) routines. */

#if (CHIP == INTEL)
/*==========================================================================*
 *				rs232handler				    *
 *==========================================================================*/
PRIVATE int rs232handler(irq)
int irq;
{
/* Interrupt handler for IRQ4 and IRQ3.
 */

  int did_something;
  register struct rs232 *rs;
  register int iir;
  int loop_count;

  unlock();	/* enable interrupts */

  if ((++hwint_count & 0xfff) == 0)
  {
  	printf("rs232handler: warning too many interrupts: %d\n",
  		hwint_count);
  }

  /* Because interrupts are edge triggered and multiple device can shared
   * a single IRQ, we must continue processing until we are certain that all
   * devices have negated the interrupt line.
   */

  loop_count= 0;
  do
  {
  	if ((++loop_count & 255) == 0)
  	{
  		printf("rs232handler: warning too long in loop: %d\n",
  			loop_count);
  	}

  	did_something= 0;
	for (rs = rs_irq[irq]; rs; rs= rs->rs_nextirq)
	{
		iir= in_byte(rs->rs_int_id_reg) & 0x7;
		if (iir & IIR_NO_INT)
			continue;			/* Did nothing! */
		switch (iir) {
		case IIR_RECEIVER_READY:
			in_int(rs);
			break;
		case IIR_TRANSMITTER_READY:
			out_int(rs);
			break;
		case IIR_MODEM_STATUS_CHANGE:
			modem_int(rs);
			break;
		case IIR_LINE_STATUS_CHANGE:
			line_int(rs);
			break;
		default:
			panic("strange value for iir", iir);
		}
		did_something |= (iir != 0);
	}
  }
  while (did_something);

  return 1;	/* Reenable RS232 interrupt */
}
#else /* MACHINE == ATARI */
/*==========================================================================*
 *				siaint					    *
 *==========================================================================*/
PUBLIC void siaint(type)
int    type;	       /* interrupt type */
{
/* siaint is the rs232 interrupt procedure for Atari ST's. For ST there are
 * as much as 5 interrupt lines used for rs232. The trap type byte left on the
 * stack by the assembler interrupt handler identifies the interrupt cause.
 */

  register unsigned char  code;
  register struct rs232_s *rs = &rs_lines[0];
  int s = lock();

  switch (type & 0x00FF)
  {
	case 0x00:	       /* receive buffer full */
		in_int(rs);
		break;
	case 0x01:	       /* receive error */
		line_int(rs);
		break;
	case 0x02:	       /* transmit buffer empty */
		out_int(rs);
		break;
	case 0x03:	       /* transmit error */
		code = MFP->mf_tsr;
		if (code & ~(T_ENA | T_UE | T_EMPTY))
		{
		    printf("sia: transmit error: status=%x\r\n", code);
		    /* MFP->mf_udr = lastchar; */ /* retry */
		}
		break;
	case 0x04:		/* modem lines change */
		modem_int(rs);
		break;
  }
  restore(s);
}
#endif

/*==========================================================================*
 *				in_int					    *
 *==========================================================================*/
PRIVATE void in_int(rs)
register struct rs232 *rs;	/* line with input interrupt */
{
/* Read the data which just arrived.
 * Put data in the buffer if room, otherwise discard it.
 */
  int rd_size, rd_write;
  char c;
  int istrip;

  rd_size= rs->rs_rd_size;
  rd_write= rs->rs_rd_write;
  istrip= !!(rs->rs_flags & RSF_ISTRIP);
  for (;;)
  {
  	/* Let's assume a FIFO. */
  	if (!dtready(rs))
  		break;

#if (CHIP == INTEL)
	c = in_byte(rs->recv_port);
#else /* MACHINE == ATARI */
	c = MFP->mf_udr;
#endif /* CHIP/MACHINE */
  	if (rd_size == 0)
  	{
  		rs->rs_rd_overflow++;
  		continue;
	}
	if (istrip)
		c &= 0x7f;

	rs->rs_rd_buf[rd_write] = c;

	rd_size--;
	rd_write++;
	if (rd_write == RS_RD_BUF)
		rd_write= 0;
  }
  rs->rs_rd_size= rd_size;
  rs->rs_rd_write= rd_write;

  if (rd_size <= rs->rs_rd_threshold)
  	istop(rs);

  /* Interrupt if we have a character, the receiver is blocked and
   * we didn't sent an interrupt already. Maybe we should wait a little
   * bit for an other character... */
  if ((rs->rs_flags & RSF_READ_INITIATIVE) && !rs->rs_rd_interrupt)
  {
  	rs->rs_rd_interrupt= TRUE;
  	interrupt(ser_tasknr);
  }
}


/*==========================================================================*
 *				line_int				    *
 *==========================================================================*/
PRIVATE void line_int(rs)
register struct rs232 *rs;	/* line with line status interrupt */
{
/* Check for and record errors. */

  int lstatus, c, mark;
  int rd_size, rd_write;

#if (CHIP == INTEL)
  lstatus = in_byte(rs->rs_line_status_reg);
#else /* MACHINE == ATARI */
  rs->lstatus = MFP->mf_rsr;
  MFP->mf_rsr &= R_ENA;
  rs->pad = MFP->mf_udr;	/* discard char in case of LS_OVERRUN_ERR */
#endif /* CHIP/MACHINE */

  /* Fetch byte, if any */
  if (dtready(rs))
  {
#if (CHIP == INTEL)
	c = in_byte(rs->recv_port);
#else /* MACHINE == ATARI */
	c = MFP->mf_udr;
#endif /* CHIP/MACHINE */
  }
  else
  	c= '\0';
  mark= 0;

  /* Multiple error might code from one bad character. We check the error in 
   * order of usefullness. This means that a break is also framing error but
   * we only process the break, and a framing error might also cause a parity
   * error in which case we only process the framing error.
   */
  if (lstatus & LSR_BREAK_INTERRUPT) 
  {
  	if (rs->rs_flags & RSF_IGNBRK)
  		return;
  	if (rs->rs_flags & RSF_BRKINT)
  	{
  		rs->rs_got_break= 1;
  		interrupt(ser_tasknr);
  		return;
  	}
  	if (rs->rs_flags & RSF_PARMRK)
  		mark= 1;
  	c= 0;
  }
  else if (lstatus & LSR_FRAMING_ERR) 
  {
  	if (rs->rs_flags & RSF_IGNPAR)
  		return;
  	if (rs->rs_flags & RSF_PARMRK)
  		mark= 1;
  	c= 0;
  }
  else if (lstatus & LSR_PARITY_ERR) 
  {
  	if (rs->rs_flags & RSF_INPCK)
  	{
		if (rs->rs_flags & RSF_IGNPAR)
			return;
		if (rs->rs_flags & RSF_PARMRK)
			mark= 1;
		c= 0;
  	}
  }
  else if (lstatus & LSR_OVERRUN_ERR) 
  {
  	/* We do nothing */
  	return;
  }
  else
  {
  	/* Faulty hardware */
  	return;
  }

  /* Add at one or three bytes to the input buffer. */
  rd_size= rs->rs_rd_size;
  rd_write= rs->rs_rd_write;
  if (mark)
  {
  	if (rd_size < 3)
  		return;
	rs->rs_rd_buf[rd_write] = '\377';
	rd_size--;
	rd_write++;
	if (rd_write == RS_RD_BUF)
		rd_write= 0;
	rs->rs_rd_buf[rd_write] = '\0';
	rd_size--;
	rd_write++;
	if (rd_write == RS_RD_BUF)
		rd_write= 0;
  }
  else
  {
  	if (rd_size < 1)
  		return;
  }
  rs->rs_rd_buf[rd_write] = c;
  rd_size--;
  rd_write++;
  if (rd_write == RS_RD_BUF)
	rd_write= 0;

  rs->rs_rd_size= rd_size;
  rs->rs_rd_write= rd_write;

  if (rd_size <= rs->rs_rd_threshold)
  	istop(rs);

  /* Interrupt if we have a character, the receiver is blocked and
   * we didn't sent an interrupt already. Maybe we should wait a little
   * bit for an other character... */
  if ((rs->rs_flags & RSF_READ_INITIATIVE) && !rs->rs_rd_interrupt)
  {
  	rs->rs_rd_interrupt= TRUE;
  	interrupt(ser_tasknr);
  }
}


/*==========================================================================*
 *				modem_int				    *
 *==========================================================================*/
PRIVATE void modem_int(rs)
register struct rs232 *rs;	/* line with modem interrupt */
{
/* Get possibly new device-ready status.
 * If the device just became ready, restart output.
 */
  int msr;
#if (CHIP == INTEL)
  msr= in_byte(rs->rs_modem_status_reg);
#if DEBUG
  if (msr & MSR_DCTS) printf("delta clear to send\n");
  if (msr & MSR_DDSR) printf("delta data set ready\n");
  if (msr & MSR_TERI) printf("trailing edge ring indicator\n");
  if (msr & MSR_DDCD) printf("delta data carrier detect\n");
  if (msr & MSR_CTS) printf("clear to send\n");
  if (msr & MSR_DSR) printf("data set ready\n");
  if (msr & MSR_RI) printf("ring indicator\n");
  if (msr & MSR_DCD) printf("data carrier detect\n");
  printf("\n");
#endif
#endif /* CHIP == INTEL */

#if (MACHINE == ATARI)
  /* Set active edge interrupt so that next change causes a new interrupt */
  MFP->mf_aer = (MFP->mf_aer | (IO_SCTS|IO_SDCD)) ^
		 (MFP->mf_gpip & (IO_SCTS|IO_SDCD));
#endif
  if (devready(rs) && txready(rs))
	out_int(rs);
}


/*==========================================================================*
 *				out_int					    *
 *==========================================================================*/
PRIVATE void out_int(rs)
register struct rs232 *rs;	/* line with output interrupt */
{
/* This routine sends an output character to the uart. It must be protected
 * from interrupts when called from task level. Fifo queueing is assumed so
 * the characters are send in a loop.
 */
  int wr_size, wr_read;
  int first;

  wr_size= rs->rs_wr_size;
  wr_read= rs->rs_wr_read;
  for (first=1;;first=0)
  {
  	if (wr_size == 0)
  		break;				/* Nothing more to do. */

  	if ((!devready(rs) && !rs->rs_clocal) ||
		!txready(rs))
	{
  		break;				/* Device not ready. */
	}

#if (CHIP == INTEL)
	out_byte(rs->rs_trans_holding_reg, rs->rs_wr_buf[wr_read]);
#else /* MACHINE == ATARI */
	MFP->mf_udr = rs->rs_wr_buf[wr_read];
#endif
	wr_read++;
	if (wr_read == RS_WR_BUF)
		wr_read= 0;
	wr_size--;
  }

  rs->rs_wr_size= wr_size;
  rs->rs_wr_read= wr_read;

  /* We have to wakeup the tty is the number of bytes left in the buffer is
   * below the threshold and output from the generic tty driver is suspended.
   */
  if (wr_size <= rs->rs_wr_threshold && rs->rs_flags & RSF_WRITE_INITIATIVE &&
  	!rs->rs_wr_interrupt)
  {
  	rs->rs_wr_interrupt= TRUE;
  	interrupt(ser_tasknr);
  }
}


/*==========================================================================*
 *				ser_nextline				    *
 *==========================================================================*/
PUBLIC int ser_nextline()
{
	do {
		if (++ser_debug_line == NR_RS_LINES) {
			ser_debug_line= -1;
			break;
		}
	} while (rs_lines[ser_debug_line].rs_mode != RSM_ENABLED);
	return ser_debug_line;
}


/*==========================================================================*
 *				ser_putk					    *
 *==========================================================================*/
PUBLIC void ser_putk(c)
int c;	/* line with output interrupt */
{
	struct rs232 *rs;	/* line with output interrupt */
	int i;

	if (ser_debug_line < 0) return;

	if (!ser_initialized) return;

	rs= &rs_lines[ser_debug_line];
	while (!txready(rs))
		for (i= 0; i<100; i++);	/* Do nothing */
	out_byte(rs->rs_trans_holding_reg, c);
	if (c == '\n')
		ser_putk('\r');

}

/*
 * $PchId: rs232.c,v 1.5 1996/01/19 22:27:12 philip Exp $
 */
