/*
rawkbd.c

Created:	Nov 1, 1992 by Philip Homburg
*/

#include "kernel.h"
#include <time.h>	/* For <sys/kbdio.h> */
#include <sys/ioctl.h>
#include <sys/kbdio.h>
#include <minix/keymap.h>
#include <minix/com.h>
#include "assert.h"
INIT_ASSERT
#include "mq.h"
#include "proc.h"

/*
 * Requests:
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_COUNT NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_OPEN    |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_CLOSE   |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_READ    |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_WRITE   |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_IOCTL NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_IOCTL3  |minor dev  | proc nr   |  fd       |  command  | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_OPERATION
 * -------------------------------------------------------------------|
 * | DEV_CANCEL  |minor dev  | proc nr   |  fd       | which operation|
 * |-------------+-----------+-----------+-----------+----------------|
 *
 * Replies:
 *
 *    m_type        REP_PROC_NR   REP_STATUS   REP_REF    REP_OPERATION
 * ----------------------------------------------------------------------|
 * | DEVICE_REPLY |   proc nr   |  status    |  fd     | which operation |
 * |--------------+-------------+------------+---------+-----------------|
 */

#define RAWKBD_BUF_SIZE	256
#define KBDAUX_BUF_SIZE	256

PRIVATE reply_func_t rawkbd_reply;

PRIVATE char rawkbd_buf[RAWKBD_BUF_SIZE];
PRIVATE int rawkbd_buf_size;
PRIVATE char *rawkbd_ptr;
PRIVATE int rawkbd_size;

PRIVATE char kbdaux_buf[KBDAUX_BUF_SIZE];
PRIVATE int kbdaux_buf_size;
PRIVATE char *kbdaux_ptr;
PRIVATE int kbdaux_size;

PRIVATE mq_t *rawkbd_mq;
PRIVATE mq_t *kbdaux_mq;
PRIVATE int rawkbd_initialized;
PRIVATE int rawkbd_open;
PRIVATE int kbdaux_open;
PRIVATE int rawkbd_ttytask= ANY;		/* ANY to prevent accidents */

FORWARD _PROTOTYPE( int do_read, (mq_t *mq)				);
FORWARD _PROTOTYPE( int do_ioctl, (mq_t *mq)				);
FORWARD _PROTOTYPE( int aux_read, (mq_t *mq)				);

/*===========================================================================*
 *				rawkbd_init				     *
 *===========================================================================*/
PUBLIC void rawkbd_init(reply_func, task_nr)
reply_func_t reply_func;
int task_nr;
{

	rawkbd_buf_size= RAWKBD_BUF_SIZE;
	rawkbd_ptr= rawkbd_buf;
	rawkbd_size= 0;

	kbdaux_buf_size= RAWKBD_BUF_SIZE;
	kbdaux_ptr= kbdaux_buf;
	kbdaux_size= 0;

	rawkbd_reply= reply_func;
	rawkbd_ttytask= task_nr;
	rawkbd_initialized= TRUE;
}


/*===========================================================================*
 *				rawkbd_mess				     *
 *===========================================================================*/
PUBLIC void rawkbd_mess(mq)
struct mq *mq;
{
	int result;
	int free_mq;
	
	result= OK;
	free_mq= TRUE;
	
	switch(mq->mq_mess.m_type)
	{
	case DEV_OPEN:
		if (rawkbd_open)
			result= EBUSY;
		else
		{
			rawkbd_open= TRUE;
			rawkbd_enabled= TRUE;	/* This should done by an
						 * ioctl */
		}
		break;
	case DEV_CLOSE:
		assert(rawkbd_open);
		rawkbd_open= FALSE;
		rawkbd_enabled= FALSE;		/* We should also have an ioctl
						 * for this */
		break;
	case DEV_READ:
		result= do_read(mq);
		if (result == SUSPEND)
			free_mq= FALSE;
		break;
	case DEV_WRITE:
		result= EIO;
		break;
	case DEV_IOCTL3:
		result= do_ioctl(mq);
		assert(result != SUSPEND);
		break;
	case DEV_CANCEL:
		assert(rawkbd_mq);
		(*rawkbd_reply)(rawkbd_mq, EINTR, TRUE);
		rawkbd_mq= NULL;
		mq_free(mq);
		return;
	default:
		panic("got unknown request", mq->mq_mess.m_type);
	}
	(*rawkbd_reply)(mq, result, FALSE);
	if (free_mq)
		mq_free(mq);
}


/*===========================================================================*
 *				kbdaux_mess				     *
 *===========================================================================*/
PUBLIC void kbdaux_mess(mq)
struct mq *mq;
{
	int result;
	int free_mq;
	
	result= OK;
	free_mq= TRUE;
	
	switch(mq->mq_mess.m_type)
	{
	case DEV_OPEN:
		if (kbdaux_open)
			result= EBUSY;
		else
		{
			result= kbd_aux(1 /* enable */);
			if (result == OK)
				kbdaux_open= TRUE;
		}
		break;
	case DEV_CLOSE:
		assert(kbdaux_open);
		kbdaux_open= FALSE;
		kbd_aux(0 /* disable */);
		break;
	case DEV_READ:
		result= aux_read(mq);
		if (result == SUSPEND)
			free_mq= FALSE;
		break;
	case DEV_WRITE:
		printf("kbdaux_mess: got a write request: from %d, %d bytes\n",
			mq->mq_mess.NDEV_PROC, 
			mq->mq_mess.NDEV_COUNT);
		result= EIO;
		break;
	case DEV_IOCTL3:
		result= EIO;
		assert(result != SUSPEND);
		break;
	case DEV_CANCEL:
		assert(kbdaux_mq);
		(*rawkbd_reply)(kbdaux_mq, EINTR, TRUE);
		kbdaux_mq= NULL;
		mq_free(mq);
		return;
	default:
		panic("got unknown request", mq->mq_mess.m_type);
	}

	(*rawkbd_reply)(mq, result, FALSE);
	if (free_mq)
		mq_free(mq);
}


/*===========================================================================*
 *				rawkbd_int				     *
 *===========================================================================*/
PUBLIC void rawkbd_int()
{
	mq_t *mq;
	int r;

	rawkbd_int_pending= FALSE;
	if (rawkbd_mq != NULL)
	{
		/* Restart the read */
		mq= rawkbd_mq;
		rawkbd_mq= NULL;

		r= do_read(mq);
		if (r != SUSPEND)
		{
			/* The request is finished */
			(*rawkbd_reply)(mq, r, TRUE);
		}
	}

	if (kbdaux_mq != NULL)
	{
		/* Restart the read */
		mq= kbdaux_mq;
		kbdaux_mq= NULL;

		r= aux_read(mq);

		if (r != SUSPEND)
		{
			/* The request is finished */
			(*rawkbd_reply)(mq, r, TRUE);
		}
	}
}


/*===========================================================================*
 *				do_read					     *
 *===========================================================================*/
PRIVATE int do_read(mq)
mq_t *mq;
{
	int size, offset;
	phys_bytes rawkbd_user_ptr;
	int data_size;
	
	/* We only support one reader */
	if (rawkbd_mq != NULL)
		return EIO;
	if (rawkbd_size == 0)
	{
		/* No data available */
		rawkbd_mq= mq;
		return SUSPEND;
	}

	size= mq->mq_mess.NDEV_COUNT;
	rawkbd_user_ptr= numap(mq->mq_mess.NDEV_PROC, 
				(vir_bytes)mq->mq_mess.NDEV_BUFFER, size);
	if (rawkbd_user_ptr == 0)
		return EFAULT;
	for (offset= 0; offset < size; )
	{
		data_size= rawkbd_size;
		if (rawkbd_ptr-rawkbd_buf+rawkbd_size > rawkbd_buf_size)
			data_size= rawkbd_buf+rawkbd_buf_size - rawkbd_ptr;
		assert(data_size > 0);
		if (data_size > (size-offset))
			data_size= size-offset;
		phys_copy(vir2phys(rawkbd_ptr), rawkbd_user_ptr+offset,
								data_size);
		offset += data_size;
		rawkbd_ptr += data_size;
		if (rawkbd_ptr == rawkbd_buf+rawkbd_buf_size)
			rawkbd_ptr= rawkbd_buf;
		rawkbd_size -= data_size;
		if (rawkbd_size == 0)
			break;
	}
	return offset;
}


/*===========================================================================*
 *				aux_read				     *
 *===========================================================================*/
PRIVATE int aux_read(mq)
mq_t *mq;
{
	int size, offset;
	phys_bytes kbdaux_user_ptr;
	int data_size;
	
	/* We only support one reader */
	if (kbdaux_mq != NULL)
		return EIO;
	if (kbdaux_size == 0)
	{
		/* No data available */
		kbdaux_mq= mq;
		return SUSPEND;
	}

	size= mq->mq_mess.NDEV_COUNT;
	kbdaux_user_ptr= numap(mq->mq_mess.NDEV_PROC, 
				(vir_bytes)mq->mq_mess.NDEV_BUFFER, size);
	if (kbdaux_user_ptr == 0)
		return EFAULT;
	for (offset= 0; offset < size; )
	{
		data_size= kbdaux_size;
		if (kbdaux_ptr-kbdaux_buf+kbdaux_size > kbdaux_buf_size)
			data_size= kbdaux_buf+kbdaux_buf_size - kbdaux_ptr;
		assert(data_size > 0);
		if (data_size > (size-offset))
			data_size= size-offset;
		phys_copy(vir2phys(kbdaux_ptr), kbdaux_user_ptr+offset,
								data_size);
		offset += data_size;
		kbdaux_ptr += data_size;
		if (kbdaux_ptr == kbdaux_buf+kbdaux_buf_size)
			kbdaux_ptr= kbdaux_buf;
		kbdaux_size -= data_size;
		if (kbdaux_size == 0)
			break;
	}
	return offset;
}


/*===========================================================================*
 *				do_ioctl				     *
 *===========================================================================*/
PRIVATE int do_ioctl(mq)
mq_t *mq;
{
	phys_bytes u_phys;
	kio_leds_t kio_leds;
	kio_bell_t kio_bell;
	kio_map_t kio_map;
	unsigned duration;

	switch(mq->mq_mess.NDEV_IOCTL)
	{
	case KIOCSLEDS:
		u_phys= numap(mq->mq_mess.NDEV_PROC, 
			(vir_bytes)mq->mq_mess.NDEV_BUFFER, sizeof(kio_leds));
		if (u_phys == 0)
			return EFAULT;
		phys_copy(u_phys, vir2phys(&kio_leds), sizeof(kio_leds));
		kbd_set_leds(kio_leds.kl_bits);
		return OK;
	case KIOCBELL:
		u_phys= numap(mq->mq_mess.NDEV_PROC, 
			(vir_bytes)mq->mq_mess.NDEV_BUFFER, sizeof(kio_bell));
		if (u_phys == 0)
			return EFAULT;
		phys_copy(u_phys, vir2phys(&kio_bell), sizeof(kio_bell));
#if DEBUG
		printf("kb_pitch= %d, kb_volume= %d, kb_duration= {%d, %d}\n",
			kio_bell.kb_pitch, kio_bell.kb_volume,
			kio_bell.kb_duration.tv_sec,
			kio_bell.kb_duration.tv_usec);
#endif
		/* Calculate the duration in clock ticks. We
		 * interpratate the volume as a percentage of the duration.
		 */
		duration= kio_bell.kb_duration.tv_sec * HZ + 
			kio_bell.kb_duration.tv_usec * HZ / 1000000;
		duration= (duration * kio_bell.kb_volume / 1000) / 1000;
		kbd_bell(kio_bell.kb_pitch, duration);
		return OK;
	case KIOCSMAP:
		u_phys= numap(mq->mq_mess.NDEV_PROC,
			(vir_bytes)mq->mq_mess.NDEV_BUFFER, sizeof(kio_map));
		if (u_phys == 0)
			return EFAULT;
		phys_copy(u_phys, vir2phys(&kio_map), sizeof(kio_map));

		return kbd_loadmap(&kio_map, mq->mq_mess.NDEV_PROC);
	default:
		return ENOTTY;
	}
}


/*===========================================================================*
 *				rawkbd_put				     *
 *===========================================================================*/
PUBLIC void rawkbd_put(buf, size)
char *buf;
unsigned size;
{
	unsigned offset;
	char *optr;
	int data_size;

	if (!rawkbd_initialized)
		return;

	offset= rawkbd_ptr-rawkbd_buf + rawkbd_size;
	if (offset > rawkbd_buf_size)
		offset -= rawkbd_buf_size;
	optr= rawkbd_buf+offset;

	rawkbd_size += size;
	while(size > 0)
	{
		if (optr == rawkbd_buf + rawkbd_buf_size)
			optr= rawkbd_buf;
		data_size= rawkbd_buf+rawkbd_buf_size-optr;
		if (data_size > size)
			data_size= size;
		memcpy(optr, buf, data_size);
		buf += data_size;
		optr += data_size;
		size -= data_size;
	}
	if (rawkbd_size > rawkbd_buf_size)
	{
		rawkbd_size= 0;
		printk("warning keyboard buffer overflow\n");
	}
	rawkbd_int_pending= TRUE;
	interrupt(rawkbd_ttytask);
}


/*===========================================================================*
 *				kbdaux_put				     *
 *===========================================================================*/
PUBLIC void kbdaux_put(buf, size)
char *buf;
unsigned size;
{
	unsigned offset;
	char *optr;
	int data_size;

	if (!rawkbd_initialized)
		return;

	offset= kbdaux_ptr-kbdaux_buf + kbdaux_size;
	if (offset > kbdaux_buf_size)
		offset -= kbdaux_buf_size;
	optr= kbdaux_buf+offset;

	kbdaux_size += size;
	while(size > 0)
	{
		if (optr == kbdaux_buf + kbdaux_buf_size)
			optr= kbdaux_buf;
		data_size= kbdaux_buf+kbdaux_buf_size-optr;
		if (data_size > size)
			data_size= size;
		memcpy(optr, buf, data_size);
		buf += data_size;
		optr += data_size;
		size -= data_size;
	}

	if (kbdaux_size > kbdaux_buf_size)
	{
		kbdaux_size= 0;
		printk("warning: aux keyboard buffer overflow\n");
	}
	rawkbd_int_pending= TRUE;
	interrupt(rawkbd_ttytask);
}

/*
 * $PchId: rawkbd.c,v 1.6 1996/03/12 22:17:52 philip Exp $
 */
