#ifndef PROC_H
#define PROC_H

/* Here is the declaration of the process table.  It contains the process'
 * registers, memory map, accounting, and message send/receive information.
 * Many assembly code routines reference fields in it.  The offsets to these
 * fields are defined in the assembler include file sconst.h.  When changing
 * 'proc', be sure to change sconst.h to match.
 */

struct proc {
  struct stackframe_s p_reg;	/* process' registers saved in stack frame */

#if (CHIP == INTEL)
  reg_t p_ldt_sel;		/* selector in gdt giving ldt base and limit*/
  struct segdesc_s p_ldt[2];	/* local descriptors for code and data */
				/* 2 is LDT_SIZE - avoid include protect.h */
#endif /* (CHIP == INTEL) */

#if (CHIP == M68000)
  reg_t p_splow;		/* lowest observed stack value */
  int p_trap;			/* trap type (only low byte) */
  phys_clicks p_shadow;		/* set if shadowed process image */
  int p_nflips;			/* statistics */
  char p_physio;		/* cannot be (un)shadowed now if set */
  int align;			/* make the struct size a multiple of 4 */
#endif /* (CHIP == M68000) */

  reg_t *p_stguard;		/* stack guard word */

  int p_nr;			/* number of this process (for fast access) */

  int p_int_blocked;		/* nonzero if int msg blocked by busy task */
  int p_int_held;		/* nonzero if int msg held by busy syscall */
  struct proc *p_nextheld;	/* next in chain of held-up int processes */

  int p_flags;			/* P_SLOT_FREE, SENDING, RECEIVING, etc. */
  int p_status;			/* Some statusbits, not reflecting runability */
  struct mem_map p_map[NR_SEGS];/* memory map */
  pid_t p_pid;			/* process id passed in from MM */
  int p_priority;		/* task, server, or user process */

  int p_ticksleft;		/* number of ticks to the next reschedule */
  int p_schedr;			/* schedule round of this process */
  int p_complr;			/* last completed schedule round */
  unsigned long p_schedquant;	/* schedule quantum */
  unsigned long p_cumquant;	/* cumulative quantum */

  clock_t user_time;		/* user time in ticks */
  clock_t sys_time;		/* sys time in ticks */
  clock_t child_utime;		/* cumulative user time of children */
  clock_t child_stime;		/* cumulative sys time of children */
  clock_t p_alarm;		/* time of next alarm in ticks, or 0 */
  watchdog_t p_watchdog;	/* function to call when the alarm goes off */

  struct proc *p_senderq;	/* head of list of procs wishing to send */
  struct proc *p_sendlink;	/* link to next proc wishing to send */
  vir_bytes p_messbuf;		/* pointer to message buffer */
  int p_tofrom;			/* from/to whom does process want to
  				 * send/receive? */

  sigset_t p_pending;		/* bit map for pending signals */
  struct proc *p_signal;	/* next proc to send a sig or a syn_alarm */

  struct proc *p_nextready;	/* pointer to next ready process */
  struct proc *p_synchqueue;	/* synchronization waiting queue */

#if PAGING_VM
  u32_t p_faultaddr;		/* Address on which the page fault occured */
  u32_t p_faulterr;		/* Error generated decribing the kind of 
  				 * fault
  				 */
  struct proc *p_pagefq;	/* Next process in the page fault queue */
  int p_function;		/* SEND, RECEIVE, or SENDREC */
#endif /* PAGING_VM */

  char p_name[32];		/* name of the process */
};

/* Guard word for task stacks. */
#define STACK_GUARD	((reg_t) (sizeof(reg_t) == 2 ? 0xBEEF : 0xDEADBEEF))

/* Bits for p_flags in proc[].  A process is runnable iff p_flags == 0. */
#define P_SLOT_FREE	  0x1	/* set when slot is not in use */
#define NO_MAP		  0x2	/* keeps unmapped forked child from running */
#define SENDING		  0x4	/* set when process blocked trying to send */
#define RECEIVING	  0x8	/* set when process blocked trying to recv */
#define SIG_PENDING	 0x20	/* keeps to-be-signalled proc from running */
#define P_STOP		 0x40	/* set when process is being traced */
#define P_UNREADY	 0x80	/* a process is not to be scheduled */
#define P_PAGEFAULT	0x100	/* the process got a page fault */
#define P_RSTSYS	0x200	/* a system call has to be restarted */

/* Bits for p_status. */
#define P_ST_TRACED	  0x1	/* set when the process has been executed in
				 * trace mode. */
#define P_ST_INRUNQ	  0x2	/* the process is in some run queue */
#define P_ST_UNRDY_1	  0x4	/* the sweeper detected that the process
				 * is not runnable */
#define P_ST_UNRDY_2	  0x8	/* the sweeper detected that the process
				 * was not runnable for some time */
#if PAGING_VM
#define P_ST_VM_INCORE	 0x10	/* all pages of the process are in core and
				 * should remain in core */
#endif /* PAGING_VM */
#define P_ST_FPU_USED	 0x20	/* Does the process have an fpu state */
#define P_ST_FPU_INVAL	 0x40	/* the process has an invalid fpu state */
#define P_ST_FPU_EX	 0x80	/* The process should receive a SIGFPE */
#define P_ST_SIGIRQ	0x100	/* The process wants to receive a signal for
				 * some irqs */
#define P_ST_TRSYS_B	0x200	/* Run the process until it executes a
				 * system call */
#define P_ST_TRSYS_E	0x400	/* Run the process until a system call
				 * is completed */

/* Values for p_priority */
#define PPRI_NONE	0	/* Slot is not in use */
#define PPRI_TASK	1	/* Part of the kernel */
#define PPRI_SERVER	2	/* System process outside the kernel */
#define PPRI_USER	3	/* User process */
#define PPRI_IDLE	4	/* Idle process */

/* Default quantum. */
#define DEF_SCHEDQUANT	0x10000

/* Magic process table addresses. */
#define BEG_PROC_ADDR (&proc[0])
#define END_PROC_ADDR (&proc[NR_TASKS + NR_PROCS])
#define BEG_USER_ADDR (&proc[NR_TASKS + LOW_USER])

#define NIL_PROC          ((struct proc *) 0)
#define isidlehardware(n) ((n) == IDLE || (n) == HARDWARE)
#define isokprocn(n)      ((unsigned) ((n) + NR_TASKS) < NR_PROCS + NR_TASKS)
#define isokindex(n)      ((unsigned)(n) < NR_PROCS + NR_TASKS)
#define isoksrc_dest(n)   (isokprocn(n) || (n) == ANY)
#define issysentn(n)      ((n) == FS_PROC_NR || (n) == MM_PROC_NR)
#define isrxhardware(n)   ((n) == ANY || (n) == HARDWARE)
#define isuserp(p)        ((p)->p_priority == PPRI_USER)
#define isservp(p)        ((p)->p_priority == PPRI_SERVER)
#define istaskp(p)        ((p)->p_priority == PPRI_TASK)
#define isemptyp(p)       ((p)->p_priority == PPRI_NONE)
#define proc_addr(n)      (pproc_addr + NR_TASKS)[(n)]
#define proc_indx2addr(n) (proc_addr(n-NR_TASKS))
#define cproc_addr(n)     (&(proc + NR_TASKS)[(n)])
#define proc_number(p)    ((p)->p_nr)
#define proc_index(p)     (proc_number(p) + NR_TASKS)
#define proc_vir2phys(p, vir) \
  (((phys_bytes)(p)->p_map[SEG_D].mem_phys << CLICK_SHIFT) + (vir_bytes) (vir))
#if (CHIP == M68000)
#define isshadowp(p)      ((p)->p_shadow != 0)
#endif

EXTERN struct proc proc[NR_TASKS + NR_PROCS];	/* process table */
EXTERN struct proc *pproc_addr[NR_TASKS + NR_PROCS];
				/* ptrs to process table slots (fast) */
EXTERN struct proc *bill_ptr;	/* ptr to process to bill for system time */
EXTERN struct proc *sched_ptr;	/* ptr to process to bill for clock ticks */
EXTERN int schedule_round;	/* Virtual clock related to scheduling. When-
				 * ever all ready user processes used their
				 * quantum, a new schedule round begins and
				 * all user processes are given a new quantum.
				 */
EXTERN struct proc *fpu_proc_ptr;	/* pointer to process using the fpu */

/* The following items pertain to the scheduling queues. */
#define TASK_L		0	/* ready tasks are scheduled via queue 0 */
#define SERVER_L	1	/* ready servers are scheduled via queue 1 */
#define USER_L		2	/* ready users are scheduled via queue 2 */
#define NEXT_USER_L	3	/* queue of ready processes for the next round
				 * of scheduling.
				 */
#define NLISTS		4	/* # of scheduling queues */

EXTERN struct proc *rdy_list[NLISTS];	/* pointers to list of ready
					 * processes
					 */
#define TASK_LIST	(rdy_list[TASK_L])
#define SERVER_LIST	(rdy_list[SERVER_L])
#define USER_LIST	(rdy_list[USER_L])
#define NEXT_USER_LIST	(rdy_list[NEXT_USER_L])

#endif /* PROC_H */

/*
 * $PchId: proc.h,v 1.5 1996/01/19 22:32:01 philip Exp $
 */
