/* This file contains essentially all of the process and message handling.
 * It has two main entry points from the outside:
 *
 *   sys_call:   called when a process or task does SEND, RECEIVE or SENDREC
 *   interrupt:	called by interrupt routines to send a message to task
 *
 * It also has several minor entry points:
 *
 *   lock_ready:      put a process on one of the ready queues so it can be run
 *   lock_unready:    remove a process from the ready queues
 *   lock_sched:      a process has run too long; schedule another one
 *   lock_mini_send:  send a message (used by interrupt signals, etc.)
 *   lock_pick_proc:  pick a process to run (used by system initialization)
 *   unhold:          repeat all held-up interrupts
 */

#include "kernel.h"
#include <signal.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "assert.h"
INIT_ASSERT
#include "proc.h"

/* schedule time quantum. */
#define MILLISEC         100	/* how often to call the scheduler (msec) */
#define SCHED_RATE (MILLISEC*HZ/1000)	/* number of ticks per schedule */


PRIVATE unsigned char switching;	/* nonzero to inhibit interrupt() */
PRIVATE int in_check_proc;

PRIVATE unsigned long max_quantum= 1;
PRIVATE unsigned long max_quantum_next= 0;

FORWARD _PROTOTYPE( int mini_send, (struct proc *caller_ptr, int function,
		int dest, vir_bytes m_vir) );
FORWARD _PROTOTYPE( int mini_rec, (struct proc *caller_ptr, int src,
		vir_bytes m_vir) );
FORWARD _PROTOTYPE( void ready, (struct proc *rp) );
FORWARD _PROTOTYPE( void sched, (int in_switch) );
FORWARD _PROTOTYPE( void dequeue, (struct proc *rp) );
FORWARD _PROTOTYPE( void unready, (struct proc *rp) );
FORWARD _PROTOTYPE( void pick_proc, (void) );

#if (CHIP == M68000)
FORWARD _PROTOTYPE( void cp_mess, (int src, struct proc *src_p, message *src_m,
		struct proc *dst_p, message *dst_m) );
#endif
#if DEBUG
FORWARD _PROTOTYPE( void print_queues, (void) );
#else
#define print_queues()
#endif
#if PAGING_VM
FORWARD _PROTOTYPE( void sys_suspend, (struct proc *rp, int function, 
				int src_dest, vir_bytes m_vir)	);
#endif /* PAGING_VM */

/*===========================================================================*
 *				interrupt				     * 
 *===========================================================================*/
PUBLIC void interrupt(task)
int task;			/* number of task to be started */
{
/* An interrupt has occurred.  Schedule the task that handles it. */

  register struct proc *rp;	/* pointer to task's proc entry */
  message *m_ptr;

  assert(task != ANY);
  rp = proc_addr(task);

  /* If this call would compete with other process-switching functions, put
   * it on the 'held' queue to be flushed at the next non-competing restart().
   * The competing conditions are:
   * (1) k_reenter == (typeof k_reenter) -1:
   *     Call from the task level, typically from an output interrupt
   *     routine.  An interrupt handler might reenter interrupt().  Rare,
   *     so not worth special treatment.
   * (2) k_reenter > 0:
   *     Call from a nested interrupt handler.  A previous interrupt handler
   *     might be inside interrupt() or sys_call().
   * (3) switching != 0:
   *     Some process-switching function other than interrupt() is being
   *     called from the task level, typically sched() from CLOCK.  An
   *     interrupt handler might call interrupt and pass the k_reenter test.
   */
  if (k_reenter != 0 || switching) {
	lock();
	if (!rp->p_int_held) {
		rp->p_int_held = TRUE;
		rp->p_nextheld= held_list;
		held_list= rp;
	}
	unlock();
	return;
  }
  assert(!switching);
  switching= TRUE;

  /* If task is not waiting for an interrupt, record the blockage. */
  if ( (rp->p_flags & (RECEIVING | SENDING)) != RECEIVING ||
      !isrxhardware(rp->p_tofrom))
  {
	rp->p_int_blocked = TRUE;
	assert(switching);
	switching= FALSE;
	return;
  }

  /* Destination is waiting for an interrupt.
   * Send it a message with source HARDWARE and type HARD_INT.
   * No more information can be reliably provided since interrupt messages
   * are not queued.
   */
  m_ptr= (message *)rp->p_messbuf;
  m_ptr->m_source = HARDWARE;
  m_ptr->m_type = HARD_INT;
  rp->p_flags &= ~RECEIVING;
  rp->p_int_blocked = FALSE;

  ready(rp);			/* TODO */
  assert(switching);
  switching= FALSE;
}


/*===========================================================================*
 *				sys_call				     * 
 *===========================================================================*/
PUBLIC void sys_call(rp, function, src_dest, m_vir)
register struct proc *rp;	/* Process trying to send or recieve */
int function;			/* SEND, RECEIVE, or BOTH */
int src_dest;			/* source to receive from or dest to send to */
vir_bytes m_vir;		/* virtual address of the message */
{
/* The only system calls that exist in MINIX are sending and receiving
 * messages.  These are done by trapping to the kernel with an INT instruction.
 * The trap is caught and sys_call() is called to send or receive a message
 * (or both).
 */

  int n;
  vir_bytes mem_vir, m_end;
  struct mem_map *map_ptr;

  /* Check for bad system call parameters. */
  if (isuserp(rp))
  {
	/* User processes are only allowed to call sendrec. */
	if (function != SENDREC)
	{
		rp->p_reg.sf_retreg= E_NO_PERM;
		return;
	}
	/* User processes are only allowed to send to FS and MM.  Check for
	 * this.
	 */
	if (!issysentn(src_dest))
	{
		rp->p_reg.sf_retreg= E_BAD_SRC;
		return;
	}
  }
  else
  {
	assert(isoksrc_dest(src_dest)); 
	if ((function & SEND) && (src_dest == HARDWARE || src_dest == IDLE))
		panic("bad dest", src_dest);
  }

  assert(!switching);
  switching= TRUE;

  if (rp->p_status & P_ST_TRSYS_B)
  {
	sys_suspend(rp, function, src_dest, m_vir);
	rp->p_status &= ~P_ST_TRSYS_B;
	assert(switching);
	switching= FALSE;
  	cause_sig(proc_number(rp), SIGTRAP);
  	return;
  }
  
  /* Check the validity of the buffer, this avoids duplication of this
   * code in mini_send and mini_rec.
   */
  m_end= m_vir+sizeof(message)-1;
#ifdef notdef
  /* On systems where the top of the stack segment can be close to
   * the top of the address space we need code to check for a
   * wrap around.
   */
  if (m_end < m_vir)
  {
	rp->p_reg.sf_retreg= E_BAD_ADDR;
	return;
  }
#endif
#ifdef notdef
  /* On non-VM systems there is need to update the stack segment based
   * on the based on the current stackpointer.
   */
#endif
  /* The message either lies in the stack segment or in the data
   * segment. Try the stack segment first.
   */
  map_ptr= &rp->p_map[SEG_S];
  mem_vir= map_ptr->mem_vir;
  if (m_vir >= (mem_vir <<CLICK_SHIFT) &&
	m_end < ((mem_vir+map_ptr->mem_len) << CLICK_SHIFT))
  {
	/* OK, In stack segment, do nothing. */
  }
  else
  {
	map_ptr= &rp->p_map[SEG_D];
	mem_vir= map_ptr->mem_vir;
	if (m_vir >= (mem_vir <<CLICK_SHIFT) &&
		m_end < ((mem_vir+map_ptr->mem_len) << CLICK_SHIFT))
	{
		/* OK, In data segment, do nothing. */
	}
	else
	{
		/* Bad pointer. */
		rp->p_reg.sf_retreg= E_BAD_ADDR;
		return;
	}
  }

  /* The parameters are ok. Do the call. */
  if (function == SEND || function == SENDREC)
  {
	n = mini_send(rp, function, src_dest, m_vir);
	if (n == E_NOPAGE)
	{
		assert(rp->p_flags & P_PAGEFAULT);
		sys_suspend(rp, function, src_dest, m_vir);
	}
	else
	{	/* done, or blocked */
		rp->p_reg.sf_retreg= n;
	}
	assert(switching);
	switching= FALSE;
  }
  else if (function == RECEIVE)
  {
	n= mini_rec(rp, src_dest, m_vir);
	if (n == E_NOPAGE)
	{
		assert(rp->p_flags & P_PAGEFAULT);
		sys_suspend(rp, function, src_dest, m_vir);
	}
	else
		rp->p_reg.sf_retreg= n;
	assert(switching);
	switching= FALSE;
  }
  else
	panic("sys_call: illegal function", function);
}


/*===========================================================================*
 *				mini_send				     * 
 *===========================================================================*/
PRIVATE int mini_send(caller_ptr, function, dest, m_vir)
register struct proc *caller_ptr; /* who is trying to send a message? */
int function;			/* SEND or SENDREC */
int dest;			/* to whom is message being sent? */
vir_bytes m_vir;		/* physical address of the message buffer */
{
	/* Send a message from 'caller_ptr' to 'dest'. If 'dest' is blocked
	 * waiting for this message, copy the message to it and unblock
	 * 'dest'. If 'dest' is not waiting at all, or is waiting for
	 * another source, queue 'caller_ptr'.
	 */

	struct proc *dest_ptr;
	struct proc *prev, *curr;
	phys_bytes src_phys, dest_phys;
	int caller;
	int dest_flags, dest_tofrom;
	u32_t page;

	caller= proc_number(caller_ptr);
	dest_ptr = proc_addr(dest);	/* pointer to destination's proc entry */
	dest_flags= dest_ptr->p_flags;
	assert(!(dest_flags & P_SLOT_FREE));
	dest_tofrom= dest_ptr->p_tofrom;

	/* Check if the send blocks. */
	if (!(dest_flags & (SENDING|RECEIVING)))
	{
		/* Target is not blocked in a system call yet, block. */
	}
	else if (dest_flags & SENDING)
	{
		if (dest_tofrom == caller)
		{
			return ELOCKED;	/* Deadlock */
		}
		/* block */
	}
	else if (dest_tofrom != ANY && dest_tofrom != caller)
	{
		; /* target is not waiting for us. */
	}
	else
	{
		/* We can proceed. */

		src_phys= m_vir+
			(caller_ptr->p_map[SEG_D].mem_phys << CLICK_SHIFT);
		dest_phys= dest_ptr->p_messbuf+
			(dest_ptr->p_map[SEG_D].mem_phys << CLICK_SHIFT);

#if PAGING_VM
		assert(!cpmess_stackptr);
#endif /* PAGING_VM */
		cpmess_unchecked(dest_phys, src_phys, caller);
#if PAGING_VM
		if (cpmess_stackptr == 0)
		{
			assert(switching);
			switching= FALSE;
			vm_suspend(caller_ptr, cpmess_faultaddr);
			assert(!switching);
			switching= TRUE;
			cpmess_faultaddr= 0;
			return E_NOPAGE;
		}
		else
		{
			cpmess_stackptr= 0;
			assert(cpmess_faultaddr == 0);
		}
#endif /* PAGING_VM */

		dest_ptr->p_flags &= ~RECEIVING; /* deblock destination */
		ready(dest_ptr);
		if (dest_ptr->p_status & P_ST_TRSYS_E)
		{
			dest_ptr->p_status &= ~P_ST_TRSYS_E;
			assert(switching);
			switching= FALSE;
			cause_sig(proc_number(dest_ptr), SIGTRAP);
			assert(!switching);
			switching= TRUE;
		}
		if (function == SENDREC)
		{
			/* Block and queue caller. */
			caller_ptr->p_messbuf = m_vir;
			caller_ptr->p_flags |= RECEIVING;
			caller_ptr->p_tofrom= dest;
			unready(caller_ptr);
		}
		return(OK);
	}

	/* Block and queue caller. */
	caller_ptr->p_messbuf = m_vir;
	if (function == SEND)
		caller_ptr->p_flags |= SENDING;
	else
		caller_ptr->p_flags |= SENDING|RECEIVING;
	caller_ptr->p_tofrom= dest;
	unready(caller_ptr);

	/* Process is now blocked.  Put in on the destination's queue. */
	caller_ptr->p_sendlink = NIL_PROC;
	for (prev= NULL, curr= dest_ptr->p_senderq; curr;
		prev= curr, curr= curr->p_sendlink)
	{
		;	/* Do nothing */
	}
	if (prev == NULL)
		dest_ptr->p_senderq= caller_ptr;
	else
		prev->p_sendlink= caller_ptr;
	return OK;
}


/*===========================================================================*
 *				mini_rec				     * 
 *===========================================================================*/
PRIVATE int mini_rec(caller_ptr, src, m_vir)
register struct proc *caller_ptr; /* process trying to get message */
int src;			/* which message source is wanted (or ANY) */
vir_bytes m_vir;		/* pointer to message buffer */
{
/* A process or task wants to get a message.  If one is already queued,
 * acquire it and deblock the sender.  If no message from the desired source
 * is available, block the caller.  No need to check parameters for validity.
 * Users calls are always sendrec(), and mini_send() has checked already.  
 * Calls from the tasks, MM, and FS are trusted.
 */

	register struct proc *sender_ptr;
	register struct proc *prev_ptr;
	phys_bytes src_phys, dst_phys;
#if PAGING_VM
	u32_t page;
#endif /* PAGING_VM */

	/* Check to see if a message from desired source is already available.
	* a receive from ANY can be satisfied with the first entry from 
	* the senderq. Else check if the intended target is sending to us.
	*/
	if (src == ANY)
	{
		if ((sender_ptr= caller_ptr->p_senderq) != NULL)
			caller_ptr->p_senderq= sender_ptr->p_sendlink;
	}
	else
	{
		sender_ptr= proc_addr(src);
		if ((sender_ptr->p_flags & SENDING) &&
			sender_ptr->p_tofrom == proc_number(caller_ptr))
		{
			if (caller_ptr->p_senderq == sender_ptr)
				caller_ptr->p_senderq= sender_ptr->p_sendlink;
			else
			{
				for (prev_ptr= caller_ptr->p_senderq;
					prev_ptr->p_sendlink != sender_ptr;
					prev_ptr = prev_ptr->p_sendlink)
				{
					; /* do nothing */
				}
				prev_ptr->p_sendlink= sender_ptr->p_sendlink;
			}
		}
		else
			sender_ptr= NULL;
	}

	if (sender_ptr == NULL)
	{
		/* Check for blocked interrupt. */
		if (caller_ptr->p_int_blocked && isrxhardware(src))
		{
			message *m_ptr= (message *)m_vir;
			m_ptr->m_source = HARDWARE;
			m_ptr->m_type = HARD_INT;
			caller_ptr->p_int_blocked = FALSE;

			return(OK);
		}

		/* No suitable message is available.  Block the process
		 * trying to receive.
		 */
		caller_ptr->p_tofrom = src;
		caller_ptr->p_messbuf = m_vir;
		caller_ptr->p_flags |= RECEIVING;
		unready(caller_ptr);

		return(OK);
	}

	/* An acceptable message has been found. */
	src_phys= sender_ptr->p_messbuf+
		(sender_ptr->p_map[SEG_D].mem_phys << CLICK_SHIFT);
	dst_phys= m_vir+(caller_ptr->p_map[SEG_D].mem_phys << CLICK_SHIFT);

#if PAGING_VM
	assert(!cpmess_stackptr);
#endif /* PAGING_VM */
	cpmess_unchecked(dst_phys, src_phys, proc_number(sender_ptr));
#if PAGING_VM
	if (cpmess_stackptr == 0)
	{
#if DEBUG
 { printf("pagefault in mini_rec\n"); }
#endif
		assert(switching);
		switching= FALSE;
		vm_suspend(caller_ptr, cpmess_faultaddr);
		assert(!switching);
		switching= TRUE;
		cpmess_faultaddr= 0;

		/* Put the sender back on the list. */
		sender_ptr->p_sendlink= caller_ptr->p_senderq;
		caller_ptr->p_senderq= sender_ptr;

		return E_NOPAGE;
	}
	else
	{
		cpmess_stackptr= 0;
		assert(cpmess_faultaddr == 0);
	}
#endif /* PAGING_VM */
	sender_ptr->p_flags &= ~SENDING;
	ready(sender_ptr);	/* deblock sender */

	return(OK);
}


/*===========================================================================*
 *				pick_proc				     * 
 *===========================================================================*/
PRIVATE void pick_proc()
{
/* Decide who to run now.  A new process is selected by setting 'proc_ptr'.
 * When a fresh user (or idle) process is selected, record it in 'bill_ptr',
 * so the clock task can tell who to bill for system time.
 */

  register struct proc *rp;	/* process to run */

  while ((rp = TASK_LIST) != NIL_PROC)
  {
  	assert(rp->p_status & P_ST_INRUNQ);
  	if (rp->p_flags & P_UNREADY)
  	{
  		TASK_LIST= rp->p_nextready;
  		rp->p_status &= ~P_ST_INRUNQ;
  		continue;
	}
	assert(!rp->p_flags);

	/* We found a runable task. */
	proc_ptr= rp;
	return;
  }
  
  while ((rp = SERVER_LIST) != NIL_PROC)
  {
  	assert(rp->p_status & P_ST_INRUNQ);
  	if (rp->p_flags & P_UNREADY)
  	{
  		SERVER_LIST= rp->p_nextready;
  		rp->p_status &= ~P_ST_INRUNQ;
  		continue;
	}
	assert(!rp->p_flags);

	/* We found a runable server. */
	proc_ptr= rp;
	return;
  }
	
  assert(!in_check_proc);
  sched_ptr= NULL;	/* prevent sched from updating the wrong user process. 
  			 */
  for(;;)
  {
	if (USER_LIST == NULL)
	{
		if (NEXT_USER_LIST == NULL)
			break;
		USER_LIST= NEXT_USER_LIST;
		NEXT_USER_LIST= NULL;
		max_quantum= max_quantum_next;
		max_quantum_next= 0;
		assert(max_quantum);

		schedule_round++;
		if (schedule_round >= MAX_SCHEDULE_ROUND)
		{
			for (rp= BEG_PROC_ADDR; rp < END_PROC_ADDR; rp++)
			{
				rp->p_schedr >>= 1;
				rp->p_complr >>= 1;
			}
			schedule_round= (schedule_round >> 1)+1;
		}
	}
	rp= USER_LIST;
  	assert(rp->p_status & P_ST_INRUNQ);
  	if (rp->p_flags & P_UNREADY)
  	{
  		USER_LIST= rp->p_nextready;
  		rp->p_status &= ~P_ST_INRUNQ;
  		continue;
	}
	assert(!rp->p_flags);

	if (rp->p_ticksleft == 0)	/* User process ran out of quantum. */
	{
		if (rp->p_schedr < schedule_round)
		{
			rp->p_cumquant += rp->p_schedquant;
			rp->p_schedr= schedule_round;
		}
		if (rp->p_cumquant < max_quantum)
		{
			USER_LIST= rp->p_nextready;
			rp->p_nextready= NEXT_USER_LIST;
			NEXT_USER_LIST= rp;
			rp->p_complr= schedule_round;
			if (rp->p_schedquant > max_quantum_next)
			{
				max_quantum_next= rp->p_schedquant;
				if (rp->p_schedquant > max_quantum)
					max_quantum= rp->p_schedquant;
			}
			continue;
		}
		rp->p_ticksleft= SCHED_RATE;
		rp->p_cumquant -= max_quantum;
		if (rp->p_cumquant >= max_quantum)
			rp->p_cumquant= max_quantum-1;
	}
	proc_ptr= rp;
	bill_ptr= rp;
	if (rp->p_nextready != NULL || NEXT_USER_LIST != NULL)
		sched_ptr= rp;
	return;
  }

  /* No one is ready.  Run the idle task.  The idle task might be made an
   * always-ready user task to avoid this special case.
   */
  proc_ptr = cproc_addr(IDLE);
  bill_ptr= proc_ptr;
}


/*===========================================================================*
 *				ready					     * 
 *===========================================================================*/
PRIVATE void ready(rp)
register struct proc *rp;	/* this process is now runnable */
{
/* Add 'rp' to of one of the lists of runnable processes. Three
 * lists are maintained:
 *   TASK_L   - (highest priority) for runnable tasks
 *   SERVER_L - (middle priority) for MM, FS, etc.
 *   USER_L   - (lowest priority) for user processes
 */

  struct proc *pp;
  
  assert(rp != cproc_addr(HARDWARE));
  assert(switching);
  if (rp->p_flags & ~P_UNREADY)
  	return;					/* Process is not ready yet */
  rp->p_flags &= ~P_UNREADY;
  if (rp->p_status & P_ST_INRUNQ)
  	return;			/* Process is still in some run queue */
  rp->p_status= (rp->p_status | P_ST_INRUNQ) & ~(P_ST_UNRDY_1|P_ST_UNRDY_2);

  switch(rp->p_priority)
  {
  case PPRI_TASK:
  	pp= TASK_LIST;
  	if (pp != NULL)
  	{
  		/* Don't care about fairness, we run to completion anyhow */
  		rp->p_nextready= pp->p_nextready;
  		pp->p_nextready= rp;
		return;
	}
	rp->p_nextready= NULL;
	TASK_LIST= rp;
	proc_ptr= rp;
	return;

  case PPRI_SERVER:
  	pp= SERVER_LIST;
  	if (pp != NULL)
  	{
  		rp->p_nextready= pp->p_nextready;
  		pp->p_nextready= rp;
  		return;
	}
	rp->p_nextready= NULL;
	SERVER_LIST= rp;
	
	if (proc_ptr->p_priority < PPRI_SERVER)
		return;
	proc_ptr= rp;
	return;

  case PPRI_USER:
  	compare(rp->p_schedr, <=, schedule_round);
	rp->p_nextready= USER_LIST;
	USER_LIST= rp;
	assert(proc_ptr->p_priority < PPRI_USER);
	return;

  default:
  	panic("unknown priority: ", rp->p_priority);	
  }
}


/*===========================================================================*
 *				unready					     * 
 *===========================================================================*/
PRIVATE void unready(rp)
register struct proc *rp;	/* this process is no longer runnable */
{
/* A process has blocked. */

  /* task stack still ok? */
  if (istaskp(rp) && *rp->p_stguard != STACK_GUARD)
	panic("stack overrun by task", proc_number(rp));

  assert(rp->p_flags & ~P_UNREADY);	/* Process has to be unready for some
  					 * reason.
  					 */
  rp->p_flags |= P_UNREADY;
  if (proc_ptr != rp)
  	return;				/* The process will be dequeued by
  					 * pick_proc if necessary.
  					 */
  assert(rp->p_status & P_ST_INRUNQ);
  assert(switching);
  pick_proc();
}


/*===========================================================================*
 *				sched					     * 
 *===========================================================================*/
PRIVATE void sched(in_switch)
int in_switch;
{
/* We got a clock tick and we have to look at sched_ptr to determine if we
 * should run an other user process.
 */
  static unsigned load_ticks;
  static unsigned long ldav_12_base, ldav_16_base;
  static unsigned ldav_6[64], ldav_12[256], ldav_16[256];
  static int ldav_6_ind, ldav_12_ind, ldav_16_ind;

  int i, ticks;
  unsigned long load;
  struct proc *pp;

  if (in_switch)
  {
  	load= loadav[LDAV_CURR];
  }
  else
  {
  	load= 0;
  	for (i= 0; i<NLISTS; i++)
  	{
		for (pp= rdy_list[i]; pp; pp= pp->p_nextready)
			load += !(pp->p_flags);	/* Only count ready procs */
	}
	loadav[LDAV_CURR]= load;
  }
  loadav[LDAV_TOT] += load;
  load_ticks++;
  loadav[LDAV_6]= loadav[LDAV_6]-ldav_6[ldav_6_ind]+load;
  ldav_6[ldav_6_ind++]= load;
  if (ldav_6_ind == 64) ldav_6_ind= 0;
  if ((load_ticks & 0xF) == 0)
  {
  	load= loadav[LDAV_TOT]-ldav_12_base;
	ldav_12_base= loadav[LDAV_TOT];
	loadav[LDAV_12]= loadav[LDAV_12]-ldav_12[ldav_12_ind]+load;
	ldav_12[ldav_12_ind++]= load;
	if (ldav_12_ind == 256) ldav_12_ind= 0;
  }	
  if ((load_ticks & 0xFF) == 0)
  {
  	load= loadav[LDAV_TOT]-ldav_16_base;
	ldav_16_base= loadav[LDAV_TOT];
	loadav[LDAV_16]= loadav[LDAV_16]-ldav_16[ldav_16_ind]+load;
	ldav_16[ldav_16_ind++]= load;
	if (ldav_16_ind == 256) ldav_16_ind= 0;
  }	

  lock();
  if (sched_ptr == NULL)
  {
	/* Nothing to do, proc_ptr is either a system proc or IDLE,
	 * or we have only one runnable user proc.
	 */
	unlock();
  	return;
  }

  ticks= --sched_ptr->p_ticksleft;
  assert(ticks >= 0);
  if (ticks > 0)
  {
	unlock();
  	return; /* Process can continue. */
  }
  sched_ptr= NULL;
  unlock();
  if (in_switch || k_reenter)
  {
  	/* We are not allowed to change the run queues if switching was true
  	 * when we were called (by lock_sched). We also don't schedule 
  	 * during another interrupt handler. In order to force a call to 
  	 * pick_proc, we ready the CLOCK task by sending it an interrupt.
  	 */
  	interrupt(clck_tasknr);
  	return;
  }
  if (!isuserp(proc_ptr))
  {
  	/* Sched_ptr was set, but a system process is running. This way a
  	 * user process pays for system time.
  	 */
	assert(!in_check_proc);
  	return;
  }
  /* The current user process ran out of its time quantum, call pick_proc.
   */
  pick_proc();
}


/*===========================================================================*
 *				dequeue					     * 
 *===========================================================================*/
PRIVATE void dequeue(rp)
register struct proc *rp;	/* this process is to be removed from the run
				 * queue 
				 */
{
/* A process is going to exit. */

  struct proc *p, *prev_p;
  
  if (bill_ptr == rp)
  	bill_ptr= NULL;
  if (sched_ptr == rp)
  {
assert(!in_check_proc);
  	sched_ptr= NULL;
  }
  if (!(rp->p_status & P_ST_INRUNQ))
  	return;				/* Nothing to do */
  rp->p_status &= ~P_ST_INRUNQ;
  assert(rp->p_priority= PPRI_USER);
  for(prev_p= NULL, p= USER_LIST; p; prev_p= p, p= p->p_nextready)
  {
  	if (p != rp)
  		continue;
	if (prev_p == NULL)
		USER_LIST= rp->p_nextready;
	else
		prev_p->p_nextready= rp->p_nextready;
	return;
  }
  for(prev_p= NULL, p= NEXT_USER_LIST; p; prev_p= p, p= p->p_nextready)
  {
  	if (p != rp)
  		continue;
	if (prev_p == NULL)
		NEXT_USER_LIST= rp->p_nextready;
	else
		prev_p->p_nextready= rp->p_nextready;
	return;
  }
  print_queues();
  panic("unable to dequeue user process", proc_number(rp));
}


/*==========================================================================*
 *				lock_pick_proc				    *
 *==========================================================================*/
PUBLIC void lock_pick_proc()
{
/* Safe gateway to pick_proc() for tasks. */

  assert(!switching);
  switching = TRUE;
  pick_proc();
  assert(switching);
  switching = FALSE;
}


/*==========================================================================*
 *				lock_ready				    *
 *==========================================================================*/
PUBLIC void lock_ready(rp)
struct proc *rp;		/* this process is now runnable */
{
/* Safe gateway to ready() for tasks. */

  assert(!switching);
  switching = TRUE;
  ready(rp);
  assert(switching);
  switching = FALSE;
}


/*==========================================================================*
 *				lock_sched				    *
 *==========================================================================*/
PUBLIC void lock_sched()
{
/* Safe gateway to lock_sched() for tasks. */

  if (switching)
  	sched(TRUE);
  else
  {
	switching = TRUE;
	sched(FALSE);
	assert(switching);
	switching = FALSE;
  }
}


/*==========================================================================*
 *				lock_unready				    *
 *==========================================================================*/
PUBLIC void lock_unready(rp)
struct proc *rp;		/* this process is no longer runnable */
{
/* Safe gateway to ready() for tasks. */

  assert(!switching);
  switching = TRUE;
  unready(rp);
  assert(switching);
  switching = FALSE;
}


/*==========================================================================*
 *				lock_dequeue				    *
 *==========================================================================*/
PUBLIC void lock_dequeue(rp)
struct proc *rp;		/* this process should be no longer in a 
				 * run queue */
{
/* Safe gateway to dequeue() for tasks. */

  assert(!switching);
  switching = TRUE;
  dequeue(rp);
  assert(switching);
  switching = FALSE;
}


/*==========================================================================*
 *				unhold					    *
 *==========================================================================*/
PUBLIC void unhold()
{
/* Flush any held-up interrupts.  k_reenter must be 0.  held_list must not
 * be NIL_PROC.  Interrupts must be disabled.  They will be enabled but will
 * be disabled when this returns.
 */

  register struct proc *rp;	/* current head of held queue */

  if (switching) return;
  rp = held_list;
  do {
	held_list = rp->p_nextheld;
	rp->p_int_held = FALSE;
	unlock();		/* reduce latency; held queue may change! */
	interrupt(proc_number(rp));
	lock();			/* protect the held queue again */
  }
  while ( (rp = held_list) != NIL_PROC);
}


#if (CHIP == M68000)
/*==========================================================================*
 *				cp_mess					    *
 *==========================================================================*/
PRIVATE void cp_mess(src, src_p, src_m, dst_p, dst_m)
int src;			/* sender process */
register struct proc *src_p;	/* source proc entry */
message *src_m;			/* source message */
register struct proc *dst_p;	/* destination proc entry */
message *dst_m;			/* destination buffer */
{
  register vir_clicks clk;

  if (clk = src_p->p_shadow) {
	clk -= src_p->p_map[D].mem_phys;
	src_m = (message *)((char *)src_m + ((phys_bytes)clk << CLICK_SHIFT));
  }
  if (clk = dst_p->p_shadow) {
	clk -= dst_p->p_map[D].mem_phys;
	dst_m = (message *)((char *)dst_m + ((phys_bytes)clk << CLICK_SHIFT));
  }
#ifdef NEEDFSTRUCOPY
  phys_copy(src_m,dst_m,(phys_bytes) sizeof(message));
#else
  *dst_m = *src_m;
#endif
  dst_m->m_source = src;
}
#endif


#if DEBUG
/*==========================================================================*
 *				print_procinfo				    *
 *==========================================================================*/
PUBLIC void print_procinfo()
{
	printf("procinfo of: %d (0x%x) pc= 0x%x\r\n", proc_number(proc_ptr),
		proc_ptr, proc_ptr->p_reg.sf_pc);
}


/*==========================================================================*
 *				print_unready				    *
 *==========================================================================*/
PUBLIC void print_unready()
{
	if (proc_number(proc_ptr) >= 0 || proc_number(proc_ptr) == -3)
		return;
	printf("unready: ");
	print_procinfo();
}


/*==========================================================================*
 *				print_queues				    *
 *==========================================================================*/
PRIVATE void print_queues()
{
	struct proc *p;
	int i;
	
	printf("queues:\n");
	for (i= 0; i<NLISTS; i++)
	{
		if (!rdy_list[i])
			continue;
		switch(i)
		{
		case TASK_L: printf("TASK_L:\t"); break;
		case SERVER_L: printf("SERVER_L:\t"); break;
		case USER_L: printf("USER_L:\t"); break;
		case NEXT_USER_L: printf("NEXT_USER_L:\t"); break;
		default: panic("unknown queue", i);
		}
		for (p= rdy_list[i]; p; p= p->p_nextready)
			printf("%d ", proc_number(p));
		printf("\n");
	}
	printf("\n");
}
			
		
/*==========================================================================*
 *				check_proc_ptr				    *
 *==========================================================================*/
PUBLIC int check_proc_ptr()
{
	if (!switching)
	{
		printf("not switching\n");
		return FALSE;
	}
assert(!in_check_proc);
in_check_proc= TRUE;
	if (sched_ptr)
	{
		if (!( (proc_ptr == sched_ptr && bill_ptr == NULL) ||
			(bill_ptr == sched_ptr && 
			proc_ptr->p_priority < sched_ptr->p_priority)))
		{
			printf("sched_ptr check failed: ");
			printf("proc_ptr= 0x%x (%d), ", proc_ptr, 
				proc_number(proc_ptr));
			printf("bill_ptr= 0x%x ", bill_ptr);
			if (bill_ptr) 
				printf("(%d), ", proc_number(bill_ptr));
			else
				printf(", ");
			printf("sched_ptr= 0x%x (%d)\n", sched_ptr, 
				proc_number(sched_ptr));
			return FALSE;
		}
	}

	assert(in_check_proc);
	in_check_proc= FALSE;

	if (proc_ptr == TASK_LIST)	
	{
		assert(proc_ptr->p_priority == PPRI_TASK);
		return TRUE;
	}
	if (proc_ptr == SERVER_LIST)
	{
		assert(proc_ptr->p_priority == PPRI_SERVER);
		return TRUE;
	}
	if (proc_ptr == USER_LIST)	
	{
		/* When a process signs up as a server, it may still be
		 * in the user run queue.
		 */
		assert(proc_ptr->p_priority < PPRI_USER);
		return TRUE;
	}
	if (proc_ptr == cproc_addr(IDLE))
	{
		assert(proc_ptr->p_priority == PPRI_IDLE);
		return TRUE;
	}
	printf("proc_ptr 0x%x (%d) is not head of queue\n", proc_ptr, 
		proc_number(proc_ptr));
	return FALSE;
}
#endif /* DEBUG */


#if PAGING_VM
/*==========================================================================*
 *				sys_suspend				    *
 *==========================================================================*/
PRIVATE void sys_suspend(rp, function, src_dest, m_vir)
struct proc *rp;
int function;
int src_dest;
vir_bytes m_vir;
{
	rp->p_function= function;
	rp->p_tofrom= src_dest;
	rp->p_messbuf= m_vir;
	rp->p_flags |= P_RSTSYS;
	unready(rp);
}


/*==========================================================================*
 *				sys_restart				    *
 *==========================================================================*/
PUBLIC void sys_restart(rp)
struct proc *rp;
{
	assert(rp->p_flags & P_RSTSYS);
	rp->p_flags &= ~P_RSTSYS;
	sys_call(rp, rp->p_function, rp->p_tofrom, rp->p_messbuf);
	lock_ready(rp);
}
#endif /* PAGING_VM */

/*
 * $PchId: proc.c,v 1.6 1996/01/19 22:33:23 philip Exp $
 */
