/* This file contains a collection of miscellaneous procedures:
 *	mem_init:	initialize memory tables.  Some memory is reported
 *			by the BIOS, some is guesstimated and checked later
 *	misc_vrdwt:	unpack an i/o vector for those block device drivers
 *			which do not do it for themself
 */

#include "kernel.h"
#include <minix/com.h>
#include <stdlib.h>
#include "assert.h"

#if (CHIP == INTEL)

#define EM_BASE     0x100000L	/* base of extended memory on AT's */
#define SHADOW_BASE 0xFA0000L	/* base of RAM shadowing ROM on some AT's */
#define SHADOW_MAX  0x060000L	/* maximum usable shadow memory (16M limit) */

FORWARD _PROTOTYPE (void chunk_init, (void));

/*=========================================================================*
 *				mem_init				   *
 *=========================================================================*/
PUBLIC void mem_init()
{
  chunk_init();

  /* Get the size of ordinary memory from the BIOS. */
  chunk_add(0, k_to_click(low_memsize));

  /* Get the size of extended memory from the BIOS.  This is special
   * except in protected mode, but protected mode is now normal.
   */
  chunk_add(EM_BASE >> CLICK_SHIFT, k_to_click(ext_memsize));

  /* There may be some memory available just under the 16 Mb limit that
   * is used by the system to shadow ROM between 640 kb and 1 Mb.  We
   * try to use this memory if we have less then 4 Mb extended memory.
   */
  if (ext_memsize < 4096 && check_mem(SHADOW_BASE, SHADOW_MAX) == SHADOW_MAX) {
	chunk_add(SHADOW_BASE >> CLICK_SHIFT, SHADOW_MAX >> CLICK_SHIFT);
  }
}
#endif /* (CHIP == INTEL) */


/*=========================================================================*
 *				chunk_init				   *
 *=========================================================================*/
PRIVATE void chunk_init()
{
	int i;

	for (i=0; i<CHUNK_NR; i++)
	{
		chunk_table[i].chk_size= 0;	/* means not allocated */
	}
}


/*=========================================================================*
 *				chunk_add				   *
 *=========================================================================*/
PUBLIC void chunk_add(base, size)
phys_clicks base;
phys_clicks size;
{
	chunk_t chunk, *chk_p;
	int i;

	chunk.chk_base= base;
	chunk.chk_size= size;

	for (i= 0; i<CHUNK_NR; i++)
	{
		chk_p= &chunk_table[i];
		if (!chk_p->chk_size)
		{
			*chk_p= chunk;
			break;
		}
		if (chk_p->chk_base+chk_p->chk_size < base || 
			base+size < chk_p->chk_base)
			continue;
		if (chk_p->chk_base+chk_p->chk_size == base)
		{
			chunk.chk_base= chk_p->chk_base;
			chunk.chk_size += chk_p->chk_size;
			chunk_del(chk_p->chk_base, chk_p->chk_size);
			i= -1;
			continue;
		}
		if (base+size == chk_p->chk_base)
		{
			chunk.chk_size += chk_p->chk_size;
			chunk_del(chk_p->chk_base, chk_p->chk_size);
			i= -1;
			continue;
		}
		panic("Overlapping chunks", NO_NUM);
	}
	if (i == CHUNK_NR)
		panic("chunk_table full", NO_NUM);
}


/*=========================================================================*
 *				chunk_del				   *
 *=========================================================================*/
PUBLIC void chunk_del(base, size)
phys_clicks base;
phys_clicks size;
{
	chunk_t chunk;
	int i,j;

	for (i= 0; i < CHUNK_NR; i++)
	{
		if (base >= chunk_table[i].chk_base && 
			base + size <= chunk_table[i].chk_base +
			chunk_table[i].chk_size)
			break;
	}
	if (i == CHUNK_NR)
	{
		panic("chunk not found", NO_NUM);
	}
	chunk= chunk_table[i];
	
	for (j= i+1; j<CHUNK_NR; j++)
	{	/* find the last chunk */
		if (!chunk_table[j].chk_size)
			break;
	}
	j--;
	if (j != i)
	{
		chunk_table[i]= chunk_table[j];
	}
	chunk_table[j].chk_size= 0;
	if (chunk.chk_base != base)
	{
		chunk_add(chunk.chk_base, base-chunk.chk_base);
		chunk.chk_size -= base - chunk.chk_base;
	}
	if (chunk.chk_size != size)
	{
		chunk_add(base+size, chunk.chk_size-size);
	}
}


/*=========================================================================*
 *				chunk_find				   *
 *=========================================================================*/
PUBLIC int chunk_find(base, size)
phys_clicks *base, *size;
{
	int i;

	for (i=0; i<CHUNK_NR; i++)
	{
		if (chunk_table[i].chk_size >= *size && 
			chunk_table[i].chk_size > 0)
		{
			*base= chunk_table[i].chk_base;
			*size= chunk_table[i].chk_size;
			return 1;
		}
	}
	return 0;
}


/*=========================================================================*
 *				env_parse				   *
 *=========================================================================*/
PUBLIC int env_parse(env, fmt, field, param, min, max)
char *env;		/* environment variable to inspect */
char *fmt;		/* template to parse it with */
int field;		/* field number of value to return */
long *param;		/* address of parameter to get */
long min, max;		/* minimum and maximum values for the parameter */
{
/* Parse an environment variable setting, something like "DPETH0=300:3".
 * Panic if the parsing fails.  Return EP_UNSET if the environment variable
 * is not set, EP_OFF if it is set to "off", EP_ON if set to "on" or a
 * field is left blank, or EP_SET if a field is given (return value through
 * *param).  Commas and colons may be used in the environment and format
 * string, fields in the environment string may be empty, and punctuation
 * may be missing to skip fields.  The format string contains characters
 * 'd', 'o', 'x' and 'c' to indicate that 10, 8, 16, or 0 is used as the
 * last argument to strtol.
 */

  char *val, *end;
  long newpar;
  int i = 0, radix, r;

  if ((val = k_getenv(env)) == NIL_PTR) return(EP_UNSET);
  if (strcmp(val, "off") == 0) return(EP_OFF);
  if (strcmp(val, "on") == 0) return(EP_ON);

  r = EP_ON;
  for (;;) {
	while (*val == ' ') val++;

	if (*val == 0) return(r);	/* the proper exit point */

	if (*fmt == 0) break;		/* too many values */

	if (*val == ',' || *val == ':') {
		/* Time to go to the next field. */
		if (*fmt == ',' || *fmt == ':') i++;
		if (*fmt++ == *val) val++;
	} else {
		/* Environment contains a value, get it. */
		switch (*fmt) {
		case 'd':	radix =   10;	break;
		case 'o':	radix =  010;	break;
		case 'x':	radix = 0x10;	break;
		case 'c':	radix =    0;	break;
		default:	goto badenv;
		}
		newpar = strtol(val, &end, radix);

		if (end == val) break;	/* not a number */
		val = end;

		if (i == field) {
			/* The field requested. */
			if (newpar < min || newpar > max) break;
			*param = newpar;
			r = EP_SET;
		}
	}
  }
badenv:
  printf("Bad environment setting: '%s = %s'\n", env, k_getenv(env));
  panic("", NO_NUM);
  /*NOTREACHED*/
}


#if !NDEBUG
/*=========================================================================*
 *				bad_assertion				   *
 *=========================================================================*/
PUBLIC void bad_assertion(file, line, what)
char *file;
int line;
char *what;
{
  printf("panic at %s(%d): assertion \"%s\" failed\n", file, line, what);
  panic(NULL, NO_NUM);
}


/*=========================================================================*
 *				bad_compare				   *
 *=========================================================================*/
PUBLIC void bad_compare(file, line, lhs, what, rhs)
char *file;
int line;
int lhs;
char *what;
int rhs;
{
  printf("panic at %s(%d): compare (%d) %s (%d) failed\n",
	file, line, lhs, what, rhs);
  panic(NULL, NO_NUM);
}
#endif /* !NDEBUG */

/*
 * $PchId: misc.c,v 1.4 1995/12/22 08:29:39 philip Exp $
 */
