/*
log.c

Created:	Nov 1, 1992 by Philip Homburg
*/

#include "kernel.h"
#include <sys/ioctl.h>
#include <minix/com.h>
#include "assert.h"
INIT_ASSERT
#include "mq.h"
#include "proc.h"


/*
 * Requests:
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_COUNT NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_OPEN    |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_CLOSE   |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_READ    |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_WRITE   |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_IOCTL NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_IOCTL3  |minor dev  | proc nr   |  fd       |  command  | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_OPERATION
 * -------------------------------------------------------------------|
 * | DEV_CANCEL  |minor dev  | proc nr   |  fd       | which operation|
 * |-------------+-----------+-----------+-----------+----------------|
 *
 * Replies:
 *
 *    m_type        REP_PROC_NR   REP_STATUS   REP_REF    REP_OPERATION
 * ----------------------------------------------------------------------|
 * | DEVICE_REPLY |   proc nr   |  status    |  fd     | which operation |
 * |--------------+-------------+------------+---------+-----------------|
 */

#define LOG_BUF_SIZE	4096

PRIVATE reply_func_t log_reply;
PRIVATE char log_buf[LOG_BUF_SIZE];
PRIVATE int log_buf_size;
PRIVATE char *log_ptr;
PRIVATE int log_size;
PRIVATE int log_read_size;
PRIVATE mq_t *log_mq;
PRIVATE char *overflow_ptr;
PRIVATE char *overflow_text;
PRIVATE int overflow_size;
PRIVATE int log_initialized;
PRIVATE int log_ttytask= ANY;			/* ANY to prevent accidents */

struct savelog_header {		/* Header of log saved in high core. */
	u32_t	magic;
	u16_t	log_size;
	u8_t	cksum;
	u8_t	spare1;
};
#define LOG_MAGIC	0x50434821L

FORWARD _PROTOTYPE( int do_read, (mq_t *mq)				);
FORWARD _PROTOTYPE( int do_write, (mq_t *mq)				);

/*===========================================================================*
 *				log_init				     *
 *===========================================================================*/
PUBLIC void log_init()
{
#if (CHIP == INTEL)
	phys_bytes mem_ptr= 0x100000;
	phys_bytes mem_end= 0x100000 + ext_memsize * 1024L;
#endif
	struct savelog_header save_hdr;
	unsigned cksum;
	int i;

	if (log_initialized)
		return;

	log_buf_size= LOG_BUF_SIZE;
	log_ptr= log_buf;
	log_size= 0;
	overflow_ptr= NULL;
	overflow_text= "\n[...] ";
	overflow_size= strlen(overflow_text);
	log_reply= 0;
	log_initialized= TRUE;

	/* Try to find the old log back if rebooting. */
	mem_end -= sizeof(save_hdr) + log_buf_size;
	for (; mem_ptr <= mem_end; mem_ptr += 0x40000)
	{
		phys_copy(mem_ptr, vir2phys(&save_hdr),
						(phys_bytes) sizeof(save_hdr));
		if (save_hdr.magic != LOG_MAGIC
			|| save_hdr.log_size > log_buf_size) continue;

		phys_copy(mem_ptr + sizeof(save_hdr), vir2phys(log_buf),
						(phys_bytes) save_hdr.log_size);

		/* The XOR of header and log must be zero. */
		cksum= 0;
		for (i= 0; i < sizeof(save_hdr); i++)
			cksum ^= ((u8_t *) &save_hdr)[i];
		for (i= 0; i < save_hdr.log_size; i++)
			cksum ^= ((u8_t *) log_buf)[i];
		if (cksum == 0)
		{
			/* Found a good copy! */
			log_size= save_hdr.log_size;
			break;
		}
	}
}


/*===========================================================================*
 *				log_stop				     *
 *===========================================================================*/
PUBLIC void log_stop()
{
	/* Save the log into high memory.  Leave a lot of copies, because
	 * the system seems to use some of the memory at reboot time.
	 */
#if (CHIP == INTEL)
	phys_bytes mem_ptr= 0x100000;
	phys_bytes mem_end= 0x100000 + ext_memsize * 1024L;
#endif
	struct savelog_header save_hdr;
	unsigned cksum;
	int i;

	/* Rotate the log to put byte 0 up front. */
	while (log_ptr > log_buf)
	{
		int tmp;

		tmp= log_buf[0];
		memmove(log_buf, log_buf + 1, LOG_BUF_SIZE);
		log_buf[LOG_BUF_SIZE-1]= tmp;
		log_ptr--;
	}
	save_hdr.magic= LOG_MAGIC;
	save_hdr.log_size= log_size;
	save_hdr.cksum= 0;

	/* Compute the bytewise XOR of header and log so that it is zero. */
	cksum= 0;
	for (i= 0; i < sizeof(save_hdr); i++)
		cksum ^= ((u8_t *) &save_hdr)[i];
	for (i= 0; i < log_size; i++)
		cksum ^= ((u8_t *) log_buf)[i];
	save_hdr.cksum= cksum;

	/* Copy header and log into high memory 256K apart. */
	mem_end -= sizeof(save_hdr) + log_size;
	for (; mem_ptr <= mem_end; mem_ptr += 0x40000)
	{
		phys_copy(vir2phys(&save_hdr), mem_ptr,
						(phys_bytes) sizeof(save_hdr));
		phys_copy(vir2phys(log_buf), mem_ptr + sizeof(save_hdr),
						(phys_bytes) log_size);
	}
}


/*===========================================================================*
 *				log_setreply				     *
 *===========================================================================*/
PUBLIC void log_setreply(reply_func, task_nr)
reply_func_t reply_func;
int task_nr;
{
	log_reply= reply_func;
	log_ttytask= task_nr;
}


/*===========================================================================*
 *				log_mess				     *
 *===========================================================================*/
PUBLIC void log_mess(mq)
struct mq *mq;
{
	int result;
	int can_enqueue;
	
	result= OK;
	can_enqueue= TRUE;
	
	switch(mq->mq_mess.m_type)
	{
	case DEV_OPEN:
	case DEV_CLOSE:
		break;
	case DEV_READ:
		result= do_read(mq);
		if (result == SUSPEND)
			can_enqueue= FALSE;
		break;
	case DEV_WRITE:
		result= do_write(mq);
		break;
	case DEV_IOCTL3:
		result= EIO;
		break;
	case DEV_CANCEL:
		assert(log_mq);
		(*log_reply)(log_mq, EINTR, TRUE);
		log_mq= NULL;
		mq_free(mq);
		return;
	default:
		panic("got unknown request", mq->mq_mess.m_type);
	}
	(*log_reply)(mq, result, can_enqueue);
}


/*===========================================================================*
 *				log_int					     *
 *===========================================================================*/
PUBLIC void log_int()
{
	mq_t *mq;
	int r;

	log_int_pending= FALSE;
	if (log_mq == NULL)
	{
		/* No client */
		return;
	}

	/* Restart the read */
	mq= log_mq;
	log_mq= NULL;

	r= do_read(mq);
	if (r == SUSPEND)
	{
		/* For some reason, no data arrived */
		return;
	}

	/* The request is finished */
	(*log_reply)(mq, r, TRUE);
}


/*===========================================================================*
 *				do_read					     *
 *===========================================================================*/
PRIVATE int do_read(mq)
mq_t *mq;
{
	int size, user_size, offset;
	phys_bytes user_ptr;
	int ov_size, data_size;
	char *ptr;

	/* Discard what has been read previously. */
	log_size -= log_read_size;
	if (log_read_size < log_buf_size - (log_ptr - log_buf))
	{
		log_ptr += log_read_size;
	}
	else
	{
		log_ptr -= log_buf_size - log_read_size;
	}
	ptr= log_ptr;
	size= log_size;
	log_read_size= 0;

	/* We only support one reader */
	if (log_mq != NULL)
		return EIO;
	if (size == 0)
	{
		/* No data available */
		log_mq= mq;
		return SUSPEND;
	}

	user_size= mq->mq_mess.NDEV_COUNT;
	user_ptr= numap(mq->mq_mess.NDEV_PROC, 
				(vir_bytes)mq->mq_mess.NDEV_BUFFER, user_size);
	if (user_ptr == 0)
		return EFAULT;
	for (offset= 0; offset < user_size; )
	{
		if (overflow_ptr)
		{
			/* We got a buffer overflow, overflow_ptr points to
			 * the (remaining) string indicating an overflow.
			 */
			ov_size= overflow_text+overflow_size-overflow_ptr;
			if (ov_size == 0)
			{
				overflow_ptr= NULL;
				continue; /* Let's try some real data */
			}
			if (ov_size > (user_size-offset))
				ov_size= (user_size-offset);
			phys_copy(vir2phys(overflow_ptr), user_ptr+offset,
								ov_size);
			offset += ov_size;
			overflow_ptr += ov_size;
			continue;
		}
		data_size= size;
		if (ptr-log_buf+log_size > log_buf_size)
			data_size= log_buf+log_buf_size - ptr;
		assert(data_size > 0);
		if (data_size > (user_size-offset))
			data_size= user_size-offset;
		phys_copy(vir2phys(ptr), user_ptr+offset, data_size);
		offset += data_size;
		ptr += data_size;
		if (ptr == log_buf+log_buf_size)
			ptr= log_buf;
		log_read_size += data_size;
		size -= data_size;
		if (size == 0)
			break;
	}
	return offset;
}


/*===========================================================================*
 *				do_write				     *
 *===========================================================================*/
PRIVATE int do_write(mq)
mq_t *mq;
{
	return log_print((vir_bytes)mq->mq_mess.NDEV_BUFFER,
		mq->mq_mess.NDEV_COUNT, mq->mq_mess.NDEV_PROC);
}


/*===========================================================================*
 *				log_print				     *
 *===========================================================================*/
PUBLIC int log_print(buf, size, proc_nr)
vir_bytes buf;
vir_bytes size;
int proc_nr;
{
	int i, offset;
	phys_bytes user_ptr;
	char b[256];
	int blck;
	
	user_ptr= numap(proc_nr, buf, size);
	if (user_ptr == 0)
		return EFAULT;

	for (offset= 0; offset<size; offset += blck, user_ptr += blck)
	{
		if (size > sizeof(b))
			blck= sizeof(b);
		else
			blck= size;
		phys_copy(user_ptr, vir2phys(b), (phys_bytes)blck);
		log_put(b, blck, proc_nr);
		if (!isuserp(proc_addr(proc_nr)))
		{
			for (i= 0; i<blck; i++)
				ser_putk(b[i]);
		}
		cons_print(b, blck);
	}
	log_put(NULL, 0, 0);
	cons_print(NULL, 0);
	
	return size;
}


/*===========================================================================*
 *				log_put					     *
 *===========================================================================*/
PUBLIC void log_put(buf, size, proc_nr)
char *buf;
unsigned size;
int proc_nr;
{
	static prev_proc= ANY;
	static vir_clicks prev_tlen;
	static char proc_name[1+6+1+1];
	static int pnam_len= 8;
	static int line_complete= TRUE;
	unsigned len;
	char *bp, *nl;

	while (size > 0) {		/* also ignores flush (size = 0) */
		bp= buf;
		len= size;
		nl= memchr(bp, '\n', len);
		if (nl)
		{
			len= (nl+1) - bp;
		}
		buf+= len;
		size-= len;

		if (nl && len > 1 && bp[len-2] == '\r')
		{
			/* Try to get rid of CRs. Unfortunately this is not
			 * guaranteed to work.
			 */
			bp[len-2]= '\n';
			len--;
		}

		/* Is the same process still writing? */
		if (prev_proc != proc_nr || prev_tlen !=
				proc_addr(proc_nr)->p_map[SEG_T].mem_len)
		{
			if (!line_complete)
			{
				/* Output a newline to finish previous line */
				log_append("\n", 1);
				line_complete= TRUE;
			}
			proc_name[0]= isuserp(proc_addr(proc_nr)) ? '@' : '!';
			strncpy(proc_name+1, proc_nr < 0 ? "kernel"
					: proc_addr(proc_nr)->p_name, 7);
			proc_name[8]= '\0';
			pnam_len= strlen(proc_name);
			proc_name[pnam_len++]= ':';
			if (pnam_len < 9) proc_name[pnam_len++]= '\t';
			prev_proc= proc_nr;
			prev_tlen= proc_addr(proc_nr)->p_map[SEG_T].mem_len;
		}
		if (line_complete && len > 0)
		{
			if (prev_proc == log_ttytask)
				log_append(">", 1);
			else
				log_append(proc_name, pnam_len);
		}
		log_append(bp, len);
		line_complete= (nl != 0);
	}
}


/*===========================================================================*
 *				log_append				     *
 *===========================================================================*/
PUBLIC void log_append(buf, size)
char *buf;
unsigned size;
{
	static in_log_put= FALSE;

	unsigned offset;
	char *optr;
	int data_size;

	if (!log_initialized)
		return;
	if (in_log_put)
		return;
	in_log_put= TRUE;

	offset= log_ptr-log_buf + log_size;
	if (offset > log_buf_size)
		offset -= log_buf_size;
	optr= log_buf+offset;

	log_size += size;
	while(size > 0)
	{
		if (optr == log_buf + log_buf_size)
			optr= log_buf;
		data_size= log_buf+log_buf_size-optr;
		if (data_size > size)
			data_size= size;
		memcpy(optr, buf, data_size);
		buf += data_size;
		optr += data_size;
		size -= data_size;
	}
	if (log_size > log_buf_size)
	{
		if (log_size - log_buf_size > log_read_size)
		{
			log_read_size= 0;
		}
		else
		{
			log_read_size -= log_size - log_buf_size;
		}
		log_size= log_buf_size;
		if (!overflow_ptr)
		{
			overflow_ptr= overflow_text;
		}
		log_ptr= optr;
		if (log_ptr == log_buf + log_buf_size)
			log_ptr= log_buf;
	}
	log_int_pending= TRUE;
	if (log_ttytask != ANY)
		interrupt(log_ttytask);

	assert(in_log_put);
	in_log_put= FALSE;
}


/*===========================================================================*
 *				putk					     *
 *===========================================================================*/
PUBLIC void putk(c)
int c;				/* character to print */
{
  	char chr;

	if (c == 0)
		return;

	ser_putk(c);
	chr= c;
	cons_print(&chr, 1);
	cons_print(NULL, 0);
	log_put(&chr, 1, SYSTASK);
}

/*
 * $PchId: log.c,v 1.7 1996/03/12 22:15:42 philip Exp $
 */
