/*
profile.c

Created:	before Dec 28, 1992 by Philip Homburg
*/

#include "kernel.h"
#include <sys/ioctl.h>
#include <minix/com.h>
#include "assert.h"
INIT_ASSERT
#include "mq.h"
#include "proc.h"
#include "i386/vm386.h"

/*
 * Requests:
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_COUNT NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_OPEN    |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_CLOSE   |minor dev  | proc nr   |  fd       |           |           |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_READ    |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 * | DEV_WRITE   |minor dev  | proc nr   |  fd       |  count    | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_IOCTL NDEV_BUFFER
 * ---------------------------------------------------------------------------
 * | DEV_IOCTL3  |minor dev  | proc nr   |  fd       |  command  | buf ptr   |
 * |-------------+-----------+-----------+-----------+-----------+-----------|
 *
 *    m_type      NDEV_MINOR   NDEV_PROC    NDEV_REF   NDEV_OPERATION
 * -------------------------------------------------------------------|
 * | DEV_CANCEL  |minor dev  | proc nr   |  fd       | which operation|
 * |-------------+-----------+-----------+-----------+----------------|
 *
 * Replies:
 *
 *    m_type        REP_PROC_NR   REP_STATUS   REP_REF    REP_OPERATION
 * ----------------------------------------------------------------------|
 * | DEVICE_REPLY |   proc nr   |  status    |  fd     | which operation |
 * |--------------+-------------+------------+---------+-----------------|
 */

PUBLIC int profiling;

PRIVATE reply_func_t prof_reply;
PRIVATE u32_t *prof_ptr, *prof_end;
PRIVATE u32_t prof_buffer[1024];
PRIVATE phys_bytes prof_user_ptr;
PRIVATE mq_t *prof_mq;
PRIVATE int prof_done;
PRIVATE int profile_client;
PRIVATE int profile_missed;
PRIVATE int prof_ttytask= ANY;			/* ANY to prevent accidents */

FORWARD _PROTOTYPE( int setup_read, (mq_t *mq)				);
FORWARD _PROTOTYPE( int do_ioctl, (mq_t *mq)				);
FORWARD _PROTOTYPE( void profile3, (struct proc *pp, reg_t pc, reg_t bp));
FORWARD _PROTOTYPE( int prof_handler, (int irq)				);

/*===========================================================================*
 *				profile_init				     *
 *===========================================================================*/
PUBLIC void profile_init(reply_func, task_nr)
reply_func_t reply_func;
int task_nr;
{
	prof_reply= reply_func;

	put_irq_handler(REALTIME_IRQ, prof_handler);
	prof_ttytask= task_nr;
}


/*===========================================================================*
 *				profile_mess				     *
 *===========================================================================*/
PUBLIC void profile_mess(mq)
struct mq *mq;
{
	int result;
	int can_enqueue;
	
	result= OK;
	can_enqueue= TRUE;
	
	switch(mq->mq_mess.m_type)
	{
	case DEV_OPEN:
	case DEV_CLOSE:
		break;
	case DEV_READ:
		result= setup_read(mq);
		if (result == SUSPEND)
			can_enqueue= FALSE;
		break;
	case DEV_WRITE:
		result= EIO;
		break;
	case DEV_IOCTL3:
		result= do_ioctl(mq);
		break;
	case DEV_CANCEL:
		assert(prof_mq);
		prof_stop();
		mq_free(mq);
		return;
	default:
		panic("got unknown request", mq->mq_mess.m_type);
	}
	(*prof_reply)(mq, result, can_enqueue);
}


/*===========================================================================*
 *				profile					     *
 *===========================================================================*/
PUBLIC void profile(pp)
struct proc *pp;
{
	reg_t bp, newbp, pc;
	u32_t *ptr;
	int n;
	
	if (pp == cproc_addr(HARDWARE))
	{
		bp= get_bp();
		pc= ((reg_t *)bp)[1];
	}
	else
	{
		bp= pp->p_reg.sf_bp;
		pc= pp->p_reg.sf_pc;
	}
		
	if (profile_client && !prof_done)
	{	
		for(ptr= prof_ptr+1; bp && ptr<prof_end; ptr++)
		{
			*ptr= pc;
			newbp= ((reg_t *)bp)[0];
			assert(newbp == 0 || newbp > bp);
			bp= newbp;
			pc= ((reg_t *)bp)[1];
		}
		if (ptr < prof_end)
		{
			n= ptr-prof_ptr;
			*prof_ptr= (proc_number(pp) << 16) | n;
			prof_ptr= ptr;
			return;
		}
		if (!prof_done)
		{
			prof_done= TRUE;
			interrupt(prof_ttytask);
		}
	}
	printf("missed profile\n");
}


/*===========================================================================*
 *				profile_int				     *
 *===========================================================================*/
PUBLIC void profile_int()
{
	int size;
	
	profile_int_pending= FALSE;
	if (!prof_done)
		return;
	assert(profile_client);
	profile_client= FALSE;
	prof_done= FALSE;
	size= (prof_ptr-prof_buffer)*sizeof(u32_t);
	phys_copy(vir2phys(prof_buffer), prof_user_ptr, size);
	assert(prof_mq);
	assert(prof_mq->mq_allocated);
	(*prof_reply)(prof_mq, size, TRUE);
	prof_mq= NULL;
}


/*===========================================================================*
 *				setup_read				     *
 *===========================================================================*/
PRIVATE int setup_read(mq)
mq_t *mq;
{
	int size;
	
	if (!profiling)
		return EIO;
	if (profile_client)
		return EBUSY;
	size= mq->mq_mess.NDEV_COUNT;
	size &= ~3;					/* Round down to a
							 * multiple of 4 */
	prof_user_ptr= numap(mq->mq_mess.NDEV_PROC, 
				(vir_bytes)mq->mq_mess.NDEV_BUFFER, size);
	if (prof_user_ptr == 0)
		return EFAULT;
	/* Maybe we should lock the buffer into memory here. */
	if (size > sizeof(prof_buffer))
		size= sizeof(prof_buffer);
	prof_ptr= prof_buffer;
	prof_end= prof_buffer+(size >> 2);	/* The profile buffer is an 
						 * array of u32_t */
	assert(!prof_mq);
	prof_mq= mq;
	prof_done= FALSE;
	profile_client= TRUE;
	return SUSPEND;
}


/*===========================================================================*
 *				do_ioctl				     *
 *===========================================================================*/
PRIVATE int do_ioctl(mq)
mq_t *mq;
{
	int regA, regB;
	
	switch(mq->mq_mess.NDEV_IOCTL)
	{
	case PIOCSTART:
		if (profiling)
			return EBUSY;
		profile_missed= 0;
		lock();
		out_byte(0x70, 0xA);
		regA= in_byte(0x71);
		out_byte(0x70, 0xA);
		out_byte(0x71, (regA & ~0xf) | 0x9);
		out_byte(0x70,0xB);
		regB= in_byte(0x71);
		out_byte(0x70, 0xB);
		out_byte(0x71, regB | 0x40);
		unlock();
		enable_irq(REALTIME_IRQ);
		profiling= TRUE;
		return OK;
	case PIOCSTOP:
		prof_stop();
		return OK;
	default:
		return ENOTTY;
	}
}


/*===========================================================================*
 *				profile3					     *
 *===========================================================================*/
PRIVATE void profile3(pp, pc, bp)
struct proc *pp;
reg_t pc;
reg_t bp;
{
	reg_t newbp;
	u32_t *ptr;
	u32_t base;
	int n;
	
	if (k_reenter)
	{
		pp= cproc_addr(HARDWARE);
	}
	else if (!istaskp(pp) && !isservp(pp))
		return;
	else
	{
			
		bp= pp->p_reg.sf_bp;
		pc= pp->p_reg.sf_pc;
	}
		
	if (profile_client && !prof_done)
	{	
		if (isservp(pp))
		{
			base= pp->p_map[SEG_D].mem_phys << CLICK_SHIFT;
			ptr= prof_ptr+1;
			if (ptr < prof_end)
			{
				*ptr= pc;
				ptr++;
			}
			for(; bp && ptr<prof_end; ptr++)
			{
				pc= get_phys_dword(base+bp+4);
				*ptr= pc;
				newbp= get_phys_dword(base+bp);
				assert(newbp == 0 || newbp > bp);
				bp= newbp;
			}
		}
		else
		{
			ptr= prof_ptr+1;
			if (ptr < prof_end)
			{
				*ptr= pc;
				ptr++;
			}
			for(; bp && ptr<prof_end; ptr++)
			{
				pc= ((reg_t *)bp)[1];
				*ptr= pc;
				newbp= ((reg_t *)bp)[0];
				assert(newbp == 0 || newbp > bp);
				bp= newbp;
			}
		}	
		if (ptr < prof_end)
		{
			n= ptr-prof_ptr;
			*prof_ptr= (proc_number(pp) << 16) | n;
			prof_ptr= ptr;
			return;
		}
		if (!prof_done)
		{
			prof_done= TRUE;
			profile_int_pending= TRUE;
			interrupt(prof_ttytask);
		}
	}
	profile_missed++;
}


/*===========================================================================*
 *				prof_handler					     *
 *===========================================================================*/
PRIVATE int prof_handler(irq)
int irq;
{
	assert(irq == REALTIME_IRQ);

	profile3(profile_proc, profile_pc, profile_bp);

	lock();
	enable_irq(REALTIME_IRQ);
	out_byte(0x70, 0x0C);	/* Select the interrupt register */
	(void) in_byte(0x71);	/* Just a read is enough */
	return 0;
}


/*===========================================================================*
 *				prof_stop				     *
 *===========================================================================*/
PUBLIC void prof_stop()
{ 
	int regA, regB;
	
	if (profiling)
	{
		lock();
		out_byte(0x70,0xB);
		regB= in_byte(0x71);
		out_byte(0x70,0xB);
		out_byte(0x71, regB & ~0x40);
		out_byte(0x70, 0xA);
		regA= in_byte(0x71);
		out_byte(0x70, 0xA);
		out_byte(0x71, (regA & ~0xf) | 0x6);
		unlock();
		profiling= FALSE;
		if (profile_client)
		{
			prof_done= TRUE;
			profile_int();
		}
		printf("missed profiles: %d\n", profile_missed);
	}
}

/*
 * $PchId: profile.c,v 1.5 1996/01/19 22:52:29 philip Exp $
 */
