/*
dosdsk.c

Created:	Feb 245, 1994 by Philip Homburg <philip@cs.vu.nl>
*/

#include "kernel.h"
#include "driver.h"
#include "drvlib.h"
#if ENABLE_DOSDSK
#include <stdlib.h>
#include "assert.h"
INIT_ASSERT


#define BOOT_POS		((off_t)0)
#define BOOT_SIZE		512
#define B_NAME_POS		  3
#define B_NAME_SIZE		  8
#define B_BYTESPSEC_POS		 11
#define B_BYTESPSEC_SIZE	  2
#define B_SECSPCLUS_POS		 13
#define B_SECSPCLUS_SIZE	  1
#define B_FATS_POS		 16
#define B_FATS_SIZE		  1
#define B_DIRENTS_POS		 17
#define B_DIRENTS_SIZE		  2
#define B_NRSECS_POS		 19
#define B_NRSECS_SIZE		  2
#define B_SECSPFAT_POS		 22
#define B_SECSPFAT_SIZE		  2
#define B_NRSECS4_POS		 32
#define B_NRSECS4_SIZE		  4
#define B_MAGIC_POS		0x1FE
#define B_MAGIC_SIZE		2
#define B_MAGIC_OK(c1,c2)	(c1 == 0x55 && (c2 & 0xFF) == 0xAA)

struct dirent
{
	char d_name[8];
	char d_ext[3];
	unsigned char d_attr;
	char d_reserved[10];
	unsigned short d_time;
	unsigned short d_date;
	unsigned char d_cluster0;
	unsigned char d_cluster1;
	unsigned char d_size0;
	unsigned char d_size1;
	unsigned char d_size2;
	unsigned char d_size3;
};

#define DA_ISDIR	0x10

#define CMAP_NO	513		/* [0..512] */

typedef struct dd
{
	int dd_init;
	int dd_dead;
	int dd_dev_task;
	int dd_dev_minor;
	off_t dd_rtdir_start;
	off_t dd_fat_start;
	off_t dd_data_start;
	unsigned long dd_img_size;
	int dd_fat_16;
	u16_t dd_cluster_map[CMAP_NO];
	unsigned dd_map_step;
	unsigned dd_step_mask;
	int dd_step_shift;
	size_t dd_cluster_size;
	off_t dd_cache_pos;
	u16_t dd_cache_cluster;
	u16_t dd_open_ct;

	/* for driver.c */
	struct device dd_part[DEV_PER_DRIVE];
} dd_t;

#define DD_NR	2

static struct dd dd_table[DD_NR];

#define NR_DEVICES	(DD_NR * DEV_PER_DRIVE)

#define CACHE_NR	4

static struct
{
	dd_t *c_device;
	off_t c_position;
	char c_block[BLOCK_SIZE];
} cache[CACHE_NR];
static int cache_victim= 0;
static int dosdsk_tasknr= ANY;
static int dd_drive;
static dd_t *dd_curr;
static struct device *dd_dv;

#define isalpha(c)	((unsigned) ((c) - 'a') <= (unsigned) ('z' - 'a'))
#define toupper(c)	((c) - 'a' + 'A')


_PROTOTYPE( static char *dd_name, (void)				);
_PROTOTYPE( static int dd_do_open, (struct driver *dp, message *m_ptr)	);
_PROTOTYPE( static int dd_do_close, (struct driver *dp, message *m_ptr)	);
_PROTOTYPE( static struct device *dd_prepare, (int device)		);
_PROTOTYPE( static int dd_transfer, (int proc_nr, int opcode,
			u64_t position, iovec_t *iov, unsigned nr_req)	);
_PROTOTYPE( static void dd_geometry, (struct partition *entry)		);

_PROTOTYPE( static void init, (dd_t *ddp)				);
_PROTOTYPE( static int init_hd, (dd_t *ddp)				);
_PROTOTYPE( static char *read_bytes, (dd_t *ddp, off_t pos, size_t size));
_PROTOTYPE( static u16_t b2u16, (U8_t c0, U8_t c1)			);
_PROTOTYPE( static u32_t b2u32, (U8_t c0, U8_t c1, U8_t c2, U8_t c3)	);
_PROTOTYPE( static int dir_lookup, (dd_t *ddp, char *name,
		U16_t cluster, size_t size, struct dirent *dirp)	);
_PROTOTYPE( static unsigned next_cluster, (dd_t *ddp, unsigned cluster)	);
_PROTOTYPE( static off_t cluster2block, (dd_t *ddp, unsigned cluster)	);
_PROTOTYPE( static off_t img_cluster2block, (dd_t *ddp, off_t pos)	);
_PROTOTYPE( static int parse_env, (int drive, int *taskp, int *minorp,
							char **namep)	);

static struct driver dd_dtab=
{
	dd_name,
	dd_do_open,
	dd_do_close,
	do_diocntl,
	dd_prepare,
	dd_transfer,
	nop_cleanup,
	dd_geometry,
};

void dosdsk_task()
{
	int i;

	dosdsk_tasknr= proc_number(proc_ptr);

	for (i= 0; i<DD_NR; i++)
		dd_table[i].dd_init= 0;
	driver_task(&dd_dtab);
}

static char *dd_name()
{
	static char name[]= "dosd4";

	name[4]= '0' + dd_drive * DEV_PER_DRIVE;
	return name;
}

static int dd_do_open(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
	if (dd_prepare(m_ptr->DEVICE) == NULL)
		return ENXIO;
	if (!dd_curr->dd_init)
		init(dd_curr);
	if (dd_curr->dd_dead)
		return ENXIO;
	if (dd_curr->dd_open_ct++ == 0)
	{
		partition(&dd_dtab, dd_drive*DEV_PER_DRIVE, P_PRIMARY);
	}
	return OK;
}

static int dd_do_close(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
	if (dd_prepare(m_ptr->DEVICE) == NULL)
		return ENXIO;
	dd_curr->dd_open_ct--;
	return OK;
}

static struct device *dd_prepare(device)
int device;
{
	if (device < NR_DEVICES)
	{
		dd_drive= device / DEV_PER_DRIVE;
		dd_curr= &dd_table[dd_drive];
		dd_dv= &dd_curr->dd_part[device % DEV_PER_DRIVE];
	}
	else
		return NULL;

	if (dd_curr->dd_dead)
		return NULL;
	return dd_dv;
}

/*
dd_transfer
*/
static int dd_transfer(proc_nr, opcode, position, iovec, nr_req)
int proc_nr;
int opcode;
u64_t position;
iovec_t *iovec;
unsigned nr_req;
{
	iovec_t *request, *iovec_end= iovec + nr_req;
	uoff_t first_dos_pos, dos_pos, pos;
	int r;
	unsigned dos_nr_req;
	unsigned nbytes, count;
	unsigned dos_size;
	unsigned long dv_size;
	iovec_t iovec1;
	message m;

	while (nr_req > 0) {
		/* How many bytes to transfer? */
		nbytes= 0;
		for (request= iovec; request < iovec_end; request++)
			nbytes += request->iov_size;

		/* Handle partition translation. */
		dv_size = cv64ul(dd_dv->dv_size);
		pos = cv64ul(position);
		if (pos >= dv_size)
			return OK;
		if (pos + nbytes > dv_size)
			nbytes = dv_size - pos;
		pos += cv64ul(dd_dv->dv_base);
		if (pos >= dd_curr->dd_img_size)
			return EIO;
		if (pos + nbytes > dd_curr->dd_img_size)
			return EIO;

		/* How many bytes can be found in consecutive clusters? */
		count= 0;
		while (count < nbytes) {
			dos_pos= img_cluster2block(dd_curr, pos);
			if (count == 0)
			{
				first_dos_pos= dos_pos;
			}
			else
			{
				if (dos_pos != first_dos_pos + count)
					break;
			}
			dos_size= dd_curr->dd_cluster_size
					- (pos % dd_curr->dd_cluster_size);
			if (count + dos_size > nbytes)
				dos_size = nbytes - count;

			pos += dos_size;
			count += dos_size;
		}

		/* How many I/O requests can be executed on those clusters? */
		dos_nr_req= 0;
		request= iovec;
		while (count > 0)
		{
			if (count < request->iov_size)
				break;
			count -= request->iov_size;
			request++;
			dos_nr_req++;
		}

		if (dos_nr_req > 0)
		{
			/* Do those requests that can be done, then exit. */
			m.m_type= opcode;
			m.DEVICE= dd_curr->dd_dev_minor;
			m.PROC_NR= proc_nr;
			m.POSITION= first_dos_pos;
			m.HIGHPOS= 0;
			m.ADDRESS= (char *)iovec;
			m.COUNT= dos_nr_req;
			r= sendrec(dd_curr->dd_dev_task, &m);
			assert(r == OK);

			/* EOF is checked here, so no half-baked results. */
			return iovec[0].iov_size == 0 ? OK : EIO;
		}

		/* The first request can only be partially executed. */
		iovec1.iov_addr= iovec[0].iov_addr;
		iovec1.iov_size= count;

		m.m_type= opcode;
		m.DEVICE= dd_curr->dd_dev_minor;
		m.PROC_NR= proc_nr;
		m.POSITION= first_dos_pos;
		m.HIGHPOS= 0;
		m.ADDRESS= (char *)&iovec1;
		m.COUNT= 1;
		r= sendrec(dd_curr->dd_dev_task, &m);
		assert(r == OK);

		if (iovec1.iov_size != 0)
			return EIO;

		/* Book the partial I/O and try again with the updated list
		 * unless we happen to be done with the first element.
		 */
		if ((iovec[0].iov_size-= count) == 0)
			break;
		iovec[0].iov_addr+= count;
		position= add64u(position, count);
	}
	return OK;
}

static void dd_geometry(entry)
struct partition *entry;
{
	/* The number of sectors per track is chosen to match the cluster size
	 * to make it easy for people to place partitions on cluster boundaries.
	 */
	entry->cylinders= dd_curr->dd_img_size / dd_curr->dd_cluster_size / 64;
	entry->heads= 64;
	entry->sectors= dd_curr->dd_cluster_size >> SECTOR_SHIFT;
}

/*
init
*/
static void init(ddp)
dd_t *ddp;
{
	char *cp, *name;
	struct dirent dirent;
	int i, r;
	unsigned cluster;
	unsigned long size;
	unsigned off;

	int rtdir_ents;
	unsigned rtdir_size;
	unsigned bytespsec;
	int nr_fats;
	int secspfat;
	unsigned long fatsize;
	int secspclus;
	unsigned long nr_secs;
	int nr_clusters;

	ddp->dd_init= 1;
	ddp->dd_dead= 1;

	if (parse_env(dd_drive, &ddp->dd_dev_task, &ddp->dd_dev_minor,
		&name) != OK)
	{
		return;
	}
	if (init_hd(ddp) != OK)
	{
		printf("dosdsk.init: init_hd failed\n");
		return;
	}
	cp= read_bytes(ddp, BOOT_POS, BOOT_SIZE);
	if (cp == NULL)
		return;
	if (!B_MAGIC_OK(cp[B_MAGIC_POS], cp[B_MAGIC_POS + 1]))
	{
		printf("dosdsk.init: magic AA55 not found\n");
		return;
	}
	rtdir_ents= b2u16(cp[B_DIRENTS_POS], cp[B_DIRENTS_POS + 1]);
	rtdir_size= rtdir_ents * sizeof(struct dirent);
	bytespsec= b2u16(cp[B_BYTESPSEC_POS], cp[B_BYTESPSEC_POS + 1]);
	nr_fats= cp[B_FATS_POS];
	if (nr_fats != 2)
	{
		printf("dosdsk.init: strange number of fats: %d\n", nr_fats);
		return;
	}
	secspfat= b2u16(cp[B_SECSPFAT_POS], cp[B_SECSPFAT_POS + 1]);
	fatsize= secspfat*(unsigned long)bytespsec;
	ddp->dd_fat_start= bytespsec;
	ddp->dd_rtdir_start= ddp->dd_fat_start + nr_fats*fatsize;
	secspclus= cp[B_SECSPCLUS_POS];
	nr_secs= b2u16(cp[B_NRSECS_POS], cp[B_NRSECS_POS+1]);
	if (nr_secs == 0)
		nr_secs= b2u32(cp[B_NRSECS4_POS], cp[B_NRSECS4_POS+1],
			cp[B_NRSECS4_POS+2], cp[B_NRSECS4_POS+3]);
	nr_clusters= nr_secs / (secspclus == 0 ? 1 : secspclus);
	ddp->dd_fat_16= (nr_clusters > 0x1000);
	ddp->dd_cluster_size= bytespsec * secspclus;
	ddp->dd_data_start= ddp->dd_rtdir_start + rtdir_size;

#if DEBUG
	/* DEBUG */
	{
		char name[B_NAME_SIZE+1];
		strncpy(name, cp+B_NAME_POS, B_NAME_SIZE);
		printf("%s: os name is '%s'\n", dd_name(), name);
	}
#endif

	cp= name;
	while (cp[0] == '/') cp++;
	name= strchr(cp, '/');
	if (name != NULL)
	{
		*name= '\0';
		name++;
	}
	r= dir_lookup(ddp, cp, 0, rtdir_size, &dirent);
	if (r != OK)
		return;
	while (name)
	{
		if (!(dirent.d_attr & DA_ISDIR))
		{
			printf("dosdsk.init: directory expected\n");
			return;
		}
		cluster= b2u16(dirent.d_cluster0, dirent.d_cluster1);
		size= b2u32(dirent.d_size0, dirent.d_size1, dirent.d_size2,
			dirent.d_size3);
		cp= name;
		while (cp[0] == '/') cp++;
		name= strchr(cp, '/');
		if (name != NULL)
		{
			*name= '\0';
			name++;
		}
		r= dir_lookup(ddp, cp, cluster, size, &dirent);
		if (r != OK)
			return;
	}
	if (dirent.d_attr & DA_ISDIR)
	{
		printf("dosdsk.init: no directory expected\n");
		return;
	}
	cluster= b2u16(dirent.d_cluster0, dirent.d_cluster1);
	size= b2u32(dirent.d_size0, dirent.d_size1, dirent.d_size2,
		dirent.d_size3);

	ddp->dd_img_size= size;
	ddp->dd_map_step= ddp->dd_cluster_size;
	while (ddp->dd_map_step * (CMAP_NO-1) < ddp->dd_img_size)
		ddp->dd_map_step <<= 1;

	for (i= 0; (1 << i) < ddp->dd_map_step; i++)
		; /* no nothing */
	if ((1 << i) != ddp->dd_map_step)
	{
		printf("dosdsk.init: dd_map_step (= %d) is not a power of 2\n",
			ddp->dd_map_step);
		return;
	}
	ddp->dd_step_shift= i;
	ddp->dd_step_mask= ddp->dd_map_step-1;
	for (i= 0; i<=(ddp->dd_img_size >> ddp->dd_step_shift); i++)
	{
		ddp->dd_cluster_map[i]= cluster;
		for (off= 0; off < ddp->dd_map_step;
			off += ddp->dd_cluster_size)
		{
			cluster= next_cluster(ddp, cluster);
		}
	}
	ddp->dd_cache_pos= 0;
	ddp->dd_cache_cluster= ddp->dd_cluster_map[0];
	ddp->dd_dead= 0;
	ddp->dd_part[0].dv_size= cvul64(ddp->dd_img_size);
}

static int init_hd(ddp)
dd_t *ddp;
{
	message m;
	int r;

	m.m_type= DEV_OPEN;
	m.DEVICE= ddp->dd_dev_minor;
	m.PROC_NR= dosdsk_tasknr;
	m.COUNT= R_BIT|W_BIT;
	r= sendrec(ddp->dd_dev_task, &m);
	assert (r == OK);
	r= m.REP_STATUS;
	if (r != OK)
	{
		printf("dosdsk.init_hd: dev_open failed: %d\n", r);
		return EIO;
	}
	return OK;
}

/*
read_bytes
*/
static char *read_bytes(ddp, pos, size)
dd_t *ddp;
off_t pos;
size_t size;
{
	message m;
	off_t rpos, off;
	int i, r;
	iovec_t iovec1;

	rpos= pos & ~(0x1ffL);
	off= pos-rpos;
	if (off+size > BLOCK_SIZE)
	{
		printf("dosdsk.read_bytes: invalid pos, size pair: %ld, %d\n",
			pos, size);
		return NULL;
	}

	/* search the cache */
	for (i= 0; i<CACHE_NR; i++)
	{
		if (cache[i].c_device == ddp && cache[i].c_position == rpos)
			return cache[i].c_block + off;
	}

	i= cache_victim;
	if (++cache_victim == CACHE_NR)
		cache_victim= 0;

	cache[i].c_device= NULL;	/* in case of failures */
	iovec1.iov_addr= (vir_bytes)cache[i].c_block;
	iovec1.iov_size= BLOCK_SIZE;
	m.m_type= DEV_GATHER;
	m.DEVICE= ddp->dd_dev_minor;
	m.POSITION= rpos;
	m.HIGHPOS= 0;
	m.PROC_NR= dosdsk_tasknr;
	m.ADDRESS= (char *)&iovec1;
	m.COUNT= 1;
	r= sendrec(ddp->dd_dev_task, &m);
	assert(r == OK);
	if (iovec1.iov_size != 0)
	{
		printf("dosdsk.init_hd: dev_read failed: %d\n", m.REP_STATUS);
		return NULL;
	}
	cache[i].c_device= ddp;
	cache[i].c_position= rpos;
	return cache[i].c_block+off;
}

static u16_t b2u16(c0, c1)
u8_t c0;
u8_t c1;
{
	return c0 | (c1 << 8);
}

static u32_t b2u32(c0, c1, c2, c3)
u8_t c0;
u8_t c1;
u8_t c2;
u8_t c3;
{
	return (u32_t)c0 | ((u32_t)c1 << 8)
		| ((u32_t)c2 << 16) | ((u32_t)c3 << 24);
}

/*
dir_lookup
*/
static int dir_lookup(ddp, name, cluster, size, dirp)
dd_t *ddp;
char *name;
u16_t cluster;
size_t size;
struct dirent *dirp;
{
	struct dirent *direntp;
	char base[8], ext[3];
	int i;
	char *cp;
	off_t offset;
	off_t cluster_block;

	cp= name;
	memset(base, ' ', 8);
	for (i= 0; *cp != '\0' && *cp != '.'; i++, cp++)
	{
		if (i >= 8)
			continue;
		if (isalpha(*cp))
			base[i]= toupper(*cp);
		else
			base[i]= *cp;
	}
	if (*cp == '.')
		cp++;
	memset(ext, ' ', 3);
	for (i= 0; *cp != '\0'; i++, cp++)
	{
		if (i >= 3)
			continue;
		if (isalpha(*cp))
			ext[i]= toupper(*cp);
		else
			ext[i]= *cp;
	}

	if (cluster == 0)
	{
		/* Root directory */
		offset= 0;
		while(size >= sizeof(struct dirent))
		{
			cp= read_bytes(ddp, ddp->dd_rtdir_start + offset,
				BLOCK_SIZE);
			if (cp == NULL)
				return EIO;
			for (i= 0; i<BLOCK_SIZE &&
				size >= sizeof(struct dirent);
				i += sizeof(struct dirent))
			{
				direntp= (struct dirent *)&cp[i];
				if (strncmp(direntp->d_name, base, 8) == 0 &&
					strncmp(direntp->d_ext, ext, 3) == 0)
				{
					memcpy(dirp, cp+i,
						sizeof(struct dirent));
					return OK;
				}
				size -= sizeof(struct dirent);
				offset += sizeof(struct dirent);
			}
		}
		printf("dosdsk.dir_lookup: '%s' not found\n", name);
		return ENOENT;
	}
	/* Any sub directory */
	offset= 0;
	cluster_block= cluster2block(ddp, cluster);
	for(;;)
	{
		cp= read_bytes(ddp, cluster_block + offset, BLOCK_SIZE);
		if (cp == NULL)
			return EIO;
		for (i= 0; i<BLOCK_SIZE; i += sizeof(struct dirent))
		{
			direntp= (struct dirent *)&cp[i];
			if (strncmp(direntp->d_name, base, 8) == 0 &&
				strncmp(direntp->d_ext, ext, 3) == 0)
			{
				memcpy(dirp, cp+i,
					sizeof(struct dirent));
				return OK;
			}
			offset += sizeof(struct dirent);
		}
		if (offset >= ddp->dd_cluster_size)
		{
			cluster= next_cluster(ddp, cluster);
			if (cluster == 0)
			{
				printf("dosdsk.dir_lookup: '%s' not found\n",
					name);
				return EIO;
			}
			offset -= ddp->dd_cluster_size;
		}
	}
	/* NOTREACHED */
	return ENOENT;
}

/*
next_cluster
*/
static unsigned next_cluster(ddp, cluster)
dd_t *ddp;
unsigned cluster;
{
	unsigned tmp_cluster;
	unsigned start;
	char *new_cluster;

	if (ddp->dd_fat_16)
	{
		/* printf("dosdsk.new_cluster: 0x%x -> ", cluster); */
		new_cluster= read_bytes(ddp, ddp->dd_fat_start +
			(cluster << 1), 2);
		if (new_cluster == 0)
		{
			printf("EIO\n");
			return 0;
		}
		tmp_cluster= b2u16(new_cluster[0], new_cluster[1]);
		if (tmp_cluster >= 0xfff8)
		{
#if DEBUG
 { printf("(0x%x) ", tmp_cluster); }
#endif
			tmp_cluster= 0;
		}
		/* printf("0x%x\n", tmp_cluster); */
		return tmp_cluster;
	}

	/* 12 bit cluster argh.. */
	start= cluster * 3 / 2;
	new_cluster= read_bytes(ddp, ddp->dd_fat_start + start, 2);
	if (new_cluster == 0)
		return 0;
	tmp_cluster= b2u16(new_cluster[0], new_cluster[1]);

	if ((cluster & 1) == 0)
		tmp_cluster &= 0xfff;
	else
		tmp_cluster= (tmp_cluster >> 4) & 0xfff;
	if (tmp_cluster >= 0xff8)
		tmp_cluster= 0;
	return tmp_cluster;
}

static off_t cluster2block(ddp, cluster)
dd_t *ddp;
unsigned cluster;
{
	return ddp->dd_data_start + (cluster-2)*ddp->dd_cluster_size;
}


static off_t img_cluster2block(ddp, pos)
dd_t *ddp;
off_t pos;
{
	off_t posbase, cachebase;
	int posindx;

	posbase= (pos & ~ddp->dd_step_mask);
	cachebase= (ddp->dd_cache_pos & ~ddp->dd_step_mask);
	if (posbase != cachebase || pos < ddp->dd_cache_pos)
	{
		posindx= posbase >> ddp->dd_step_shift;
		ddp->dd_cache_pos= posbase;
		ddp->dd_cache_cluster= ddp->dd_cluster_map[posindx];
	}
	while (ddp->dd_cache_pos + ddp->dd_cluster_size <= pos)
	{
		ddp->dd_cache_pos += ddp->dd_cluster_size;
		ddp->dd_cache_cluster= next_cluster(ddp, ddp->dd_cache_cluster);
		assert(ddp->dd_cache_cluster != 0);
	}
	assert(pos-ddp->dd_cache_pos < ddp->dd_cluster_size);
	return cluster2block(ddp, ddp->dd_cache_cluster)+
		pos-ddp->dd_cache_pos;
}

static int parse_env(drive, taskp, minorp, namep)
int drive;
int *taskp;
int *minorp;
char **namep;
{
	static char namebuf[256];

	char *env_var;
	char *cp, *op, *check, *taskname;
	int nr;

	env_var= dd_name();
	cp= k_getenv(env_var);
	if (cp == NULL)
	{
		printf("dosdsk: no environment variable '%s'\n", env_var);
		return ESRCH;
	}
	printf("%s: using %s\n", dd_name(), cp);
	op= cp;
	if (cp[0] == 'h' && cp[1] == 'd')
	{
		taskname= WINCHE_NAME;
		cp += 2;
	}
	else if (cp[0] == 's' && cp[1] == 'd')
	{
		taskname= SCSI_NAME;
		cp += 2;
	}
	else if (cp[0] == 'f' && cp[1] == 'd')
	{
		taskname= FLOPPY_NAME;
		cp += 2;
	}
	nr= strtol(cp, &check, 0);
	if (nr < 0 || check == cp || check[0] != ':')
	{
		printf("dosdsk: unable to parse '%s'\n", op);
		return EINVAL;
	}
	cp= check+1;
	if (strlen(cp)+1 > sizeof(namebuf))
	{
		printf("dosdsk: name too long '%s'\n", cp);
		return EINVAL;
	}
	strcpy(namebuf, cp);
	if ((*taskp= findproc(taskname)) == ANY)
	{
		printf("dosdsk: drive for '%s' is not available\n",
			taskname);
		return EINVAL;
	}
	*minorp= nr;
	*namep= namebuf;
	return OK;
}
#endif /* ENABLE_DOSDSK */

/*
 * $PchId: dosdsk.c,v 1.4 1996/01/19 22:59:04 philip Exp $
 */
