/* This file contains the "device dependent" part of a hard disk driver that
 * uses the ROM BIOS.  It makes a call and just waits for the transfer to
 * happen.  It is not interrupt driven and thus will (*) have poor performance.
 * The advantage is that it should work on virtually any PC, XT, 386, PS/2
 * or clone.  The demo disk uses this driver.  It is suggested that all
 * MINIX users try the other drivers, and use this one only as a last resort,
 * if all else fails.
 *
 * (*) The performance is within 10% of the AT driver for reads on any disk
 *     and writes on a 2:1 interleaved disk, it will be DMA_BUF_SIZE bytes
 *     per revolution for a minimum of 60 kb/s for writes to 1:1 disks.
 *
 * The file contains one entry point:
 *
 *	 bios_winchester_task:	main entry when system is brought up
 *
 *
 * Changes:
 *	30 Apr 1992 by Kees J. Bot: device dependent/independent split.
 *	12 Dec 1995 by Kees J. Bot: d-d/i rewrite.
 */

#include "kernel.h"
#include "driver.h"
#include "drvlib.h"
#if ENABLE_BIOS_WINI


/* Error codes */
#define ERR		 (-1)	/* general error */

/* Parameters for the disk drive. */
#define MAX_DRIVES         4	/* this driver supports 4 drives (hd0 - hd19)*/
#define MAX_SECS	 255	/* bios can transfer this many sectors */
#define NR_DEVICES      (MAX_DRIVES * DEV_PER_DRIVE)
#define SUB_PER_DRIVE	(NR_PARTITIONS * NR_PARTITIONS)
#define NR_SUBDEVS	(MAX_DRIVES * SUB_PER_DRIVE)

/* BIOS parameters */
#define BIOS_ASK        0x08	/* opcode for asking BIOS for parameters */
#define BIOS_RESET      0x00	/* opcode for resetting disk BIOS */
#define BIOS_READ       0x02	/* opcode for BIOS read */
#define BIOS_WRITE      0x03	/* opcode for BIOS write */
#define HD_CODE         0x80	/* BIOS code for hard disk drive 0 */

/* Variables. */
PRIVATE struct wini {		/* main drive struct, one entry per drive */
  unsigned cylinders;		/* number of cylinders */
  unsigned heads;		/* number of heads */
  unsigned sectors;		/* number of sectors per track */
  unsigned open_ct;		/* in-use count */
  struct device part[DEV_PER_DRIVE];    /* primary partitions: hd[0-4] */
  struct device subpart[SUB_PER_DRIVE]; /* subpartitions: hd[1-4][a-d] */
} wini[MAX_DRIVES], *w_wn;

PRIVATE int nr_drives = MAX_DRIVES;	/* Number of drives */
PRIVATE int w_drive;			/* selected drive */
PRIVATE struct device *w_dv;		/* device's base and size */

FORWARD _PROTOTYPE( struct device *w_prepare, (int device) );
FORWARD _PROTOTYPE( char *w_name, (void) );
FORWARD _PROTOTYPE( int w_transfer, (int proc_nr, int opcode, u64_t position,
					iovec_t *iov, unsigned nr_req) );
FORWARD _PROTOTYPE( int w_do_open, (struct driver *dp, message *m_ptr) );
FORWARD _PROTOTYPE( int w_do_close, (struct driver *dp, message *m_ptr) );
FORWARD _PROTOTYPE( void w_init, (void) );
FORWARD _PROTOTYPE( void enable_vectors, (void) );
FORWARD _PROTOTYPE( void w_geometry, (struct partition *entry));


/* Entry points to this driver. */
PRIVATE struct driver w_dtab = {
  w_name,	/* current device's name */
  w_do_open,	/* open or mount request, initialize device */
  w_do_close,	/* release device */
  do_diocntl,	/* get or set a partition's geometry */
  w_prepare,	/* prepare for I/O on a given minor device */
  w_transfer,	/* do the I/O */
  nop_cleanup,	/* no cleanup needed */
  w_geometry	/* tell the geometry of the disk */
};


/*===========================================================================*
 *				bios_winchester_task			     *
 *===========================================================================*/
PUBLIC void bios_winchester_task()
{
  driver_task(&w_dtab);
}


/*===========================================================================*
 *				w_prepare				     *
 *===========================================================================*/
PRIVATE struct device *w_prepare(device)
int device;
{
/* Prepare for I/O on a device. */

  if (device < NR_DEVICES) {			/* hd0, hd1, ... */
	w_drive = device / DEV_PER_DRIVE;	/* save drive number */
	w_wn = &wini[w_drive];
	w_dv = &w_wn->part[device % DEV_PER_DRIVE];
  } else
  if ((unsigned) (device -= MINOR_hd1a) < NR_SUBDEVS) {	/* hd1a, hd1b, ... */
	w_drive = device / SUB_PER_DRIVE;
	w_wn = &wini[w_drive];
	w_dv = &w_wn->subpart[device % SUB_PER_DRIVE];
  } else {
	return(NIL_DEV);
  }
  return(w_drive < nr_drives ? w_dv : NIL_DEV);
}


/*===========================================================================*
 *				w_name					     *
 *===========================================================================*/
PRIVATE char *w_name()
{
/* Return a name for the current device. */
  static char name[] = "bios-hd15";
  unsigned device = w_drive * DEV_PER_DRIVE;

  if (device < 10) {
	name[7] = '0' + device;
	name[8] = 0;
  } else {
	name[7] = '0' + device / 10;
	name[8] = '0' + device % 10;
  }
  return name;
}


/*===========================================================================*
 *				w_transfer				     *
 *===========================================================================*/
PRIVATE int w_transfer(proc_nr, opcode, position, iov, nr_req)
int proc_nr;			/* process doing the request */
int opcode;			/* DEV_GATHER or DEV_SCATTER */
u64_t position;			/* offset on device to read or write */
iovec_t *iov;			/* pointer to read or write request vector */
unsigned nr_req;		/* length of request vector */
{
  struct wini *wn = w_wn;
  iovec_t *iop, *iov_end = iov + nr_req;
  int errors;
  unsigned nbytes, count, chunk;
  unsigned long block;
  unsigned cylinder, sector, head;
  unsigned secspcyl = wn->heads * wn->sectors;
  phys_bytes user_base = proc_vir2phys(proc_addr(proc_nr), 0);

  /* Check disk address. */
  if (rem64u(position, SECTOR_SIZE) != 0) return(EINVAL);

  errors = 0;

  while (nr_req > 0) {
	/* How many bytes to transfer? */
	nbytes = 0;
	for (iop = iov; iop < iov_end; iop++) {
		if (nbytes + iop->iov_size > DMA_BUF_SIZE) {
			/* Don't do half a segment if you can avoid it. */
			if (nbytes == 0) nbytes = DMA_BUF_SIZE;
			break;
		}
		nbytes += iop->iov_size;
	}
	if ((nbytes & SECTOR_MASK) != 0) return(EINVAL);

	/* Which block on disk and how close to EOF? */
	if (cmp64(position, w_dv->dv_size) >= 0) return(OK);	/* At EOF */
	if (cmp64(add64u(position, nbytes), w_dv->dv_size) > 0)
		nbytes = diff64(w_dv->dv_size, position);
	block = div64u(add64(w_dv->dv_base, position), SECTOR_SIZE);

	/* Degrade to per-sector mode if there were errors. */
	if (errors > 0) nbytes = SECTOR_SIZE;

	if (opcode == DEV_SCATTER) {
		/* Copy from user space to the DMA buffer. */
		count = 0;
		for (iop = iov; count < nbytes; iop++) {
			chunk = iov->iov_size;
			if (count + chunk > nbytes) chunk = nbytes - count;
			phys_copy(user_base + iop->iov_addr, tmp_phys + count,
							(phys_bytes) chunk);
			count += chunk;
		}
	}

	/* Do the transfer */
	cylinder = block / secspcyl;
	sector = (block % wn->sectors) + 1;
	head = (block % secspcyl) / wn->sectors;

	/* Set up and execute a read or write BIOS call. */
	reg86.b.intno = 0x13;
	reg86.b.ah = opcode == DEV_SCATTER ? BIOS_WRITE : BIOS_READ;
	reg86.b.al = nbytes >> SECTOR_SHIFT;
	reg86.w.bx = tmp_phys % HCLICK_SIZE;
	reg86.w.es = physb_to_hclick(tmp_phys);
	reg86.b.ch = cylinder & 0xFF;
	reg86.b.cl = sector | ((cylinder & 0x300) >> 2);
	reg86.b.dh = head;
	reg86.b.dl = HD_CODE + w_drive;
	level0(int86);
	if (reg86.w.f & 0x0001) {
		/* An error occurred, try again sector by sector unless */
		if (++errors == 2) return(EIO);
		continue;
	}

	if (opcode == DEV_GATHER) {
		/* Copy from the DMA buffer to user space. */
		count = 0;
		for (iop = iov; count < nbytes; iop++) {
			chunk = iov->iov_size;
			if (count + chunk > nbytes) chunk = nbytes - count;
			phys_copy(tmp_phys + count, user_base + iop->iov_addr,
							(phys_bytes) chunk);
			count += chunk;
		}
	}

	/* Book the bytes successfully transferred. */
	position = add64u(position, nbytes);
	for (;;) {
		if (nbytes < iov->iov_size) {
			/* Not done with this one yet. */
			iov->iov_addr += nbytes;
			iov->iov_size -= nbytes;
			break;
		}
		nbytes -= iov->iov_size;
		iov->iov_addr += iov->iov_size;
		iov->iov_size = 0;
		if (nbytes == 0) {
			/* The rest is optional, so we return to give FS a
			 * chance to think it over.
			 */
			return(OK);
		}
		iov++;
		nr_req--;
	}
  }
  return(OK);
}


/*============================================================================*
 *				w_do_open				      *
 *============================================================================*/
PRIVATE int w_do_open(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
/* Device open: Initialize the controller and read the partition table. */

  static int init_done = FALSE;

  if (!init_done) { w_init(); init_done = TRUE; }

  if (w_prepare(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);

  if (w_wn->open_ct++ == 0) {
	/* Partition the disk. */
	partition(&w_dtab, w_drive * DEV_PER_DRIVE, P_PRIMARY);
  }
  return(OK);
}


/*============================================================================*
 *				w_do_close				      *
 *============================================================================*/
PRIVATE int w_do_close(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
/* Device close: Release a device. */

  if (w_prepare(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);
  w_wn->open_ct--;
  return(OK);
}


/*===========================================================================*
 *				w_init					     *
 *===========================================================================*/
PRIVATE void w_init()
{
/* This routine is called at startup to initialize the drive parameters. */

  int drive;
  struct wini *wn;

  /* Enable real mode BIOS vectors. */
  enable_vectors();

  /* Set the geometry of the drives */
  for (drive = 0; drive < nr_drives; drive++) {
	(void) w_prepare(drive * DEV_PER_DRIVE);
	wn = w_wn;
	reg86.b.intno = 0x13;
	reg86.b.ah = BIOS_ASK;
	reg86.b.dl = HD_CODE + drive;
	level0(int86);
	nr_drives = !(reg86.w.f & 0x0001) ? reg86.b.dl : drive;
	if (nr_drives > MAX_DRIVES) nr_drives = MAX_DRIVES;
	if (drive >= nr_drives) break;

	wn->heads = reg86.b.dh + 1;
	wn->sectors = reg86.b.cl & 0x3F;
	wn->cylinders = (reg86.b.ch | ((reg86.b.cl & 0xC0) << 2)) + 1;

	wn->part[0].dv_size = mul64u((unsigned long) wn->cylinders
				* wn->heads * wn->sectors, SECTOR_SIZE);

	printf("%s: %d cylinders, %d heads, %d sectors per track\n",
		w_name(), wn->cylinders, wn->heads, wn->sectors);
  }
}


/*===========================================================================*
 *				enable_vectors				     *
 *===========================================================================*/
PRIVATE void enable_vectors()
{
/* Protected mode Minix has reprogrammed the interrupt controllers to
 * use different vectors from the BIOS.  This means that the BIOS vectors
 * must be copied to the Minix locations for use in real mode.  The int86()
 * function enables all interrupts that Minix doesn't use, and masks all
 * interrupts that Minix does use when it makes the "INT" call.  Alas
 * more than one clock tick may occur while the disk is active, so we need
 * a special real mode clock interrupt handle to gather lost ticks.
 */

  int irq;
  static u8_t clock_handler[] = {
	0x50, 0xB0, 0x20, 0xE6, 0x20, 0xEB, 0x06, 0xE4,
	0x61, 0x0C, 0x80, 0xE6, 0x61, 0x58, 0x53, 0x1E,
	0xE8, 0x00, 0x00, 0x5B, 0x2E, 0xC5, 0x5F, 0x0A,
	0xFF, 0x07, 0x1F, 0x5B, 0xCF,
	0x00, 0x00, 0x00, 0x00,
  };
  u16_t vector[2];

  if (!protected_mode) return;

  vm_map_zp(TRUE);
  for (irq = 0; irq < NR_IRQ_VECTORS; irq++) {
	phys_copy(BIOS_VECTOR(irq)*4L, VECTOR(irq)*4L, 4L);
  }

  /* Install a clock interrupt handler to collect clock ticks when the
   * BIOS disk driver is active.  The handler is a simple bit of 8086 code
   * that increments "lost_ticks".
   */

  /* Let the clock interrupt point to the handler. */
  vector[0] = vir2phys(clock_handler) % HCLICK_SIZE;
  vector[1] = vir2phys(clock_handler) / HCLICK_SIZE;
  phys_copy(vir2phys(vector), VECTOR(CLOCK_IRQ)*4L, 4L);

  /* Store the address of lost_ticks in the handler. */
  vector[0] = vir2phys(&lost_ticks) % HCLICK_SIZE;
  vector[1] = vir2phys(&lost_ticks) / HCLICK_SIZE;
  memcpy(clock_handler + 0x1D, vector, 4);

  if (ps_mca) clock_handler[6]= 0;	/* (PS/2 port B clock ack) */
  vm_map_zp(FALSE);
}


/*============================================================================*
 *				w_geometry				      *
 *============================================================================*/
PRIVATE void w_geometry(entry)
struct partition *entry;
{
  entry->cylinders = w_wn->cylinders;
  entry->heads = w_wn->heads;
  entry->sectors = w_wn->sectors;
}
#endif /* ENABLE_BIOS_WINI */
