/*	ATAPI CD-ROM driver code			Author: Kees J. Bot
 *								3 Mar 1996
 * This routines in this file allow access to a CD-ROM that
 * implements the ATA Packet Interface.  This file is conceptionally a
 * part of at_wini.c, but is #included to keep at_wini.c clean.
 */

#define ATAPI_PKTCMD	0xA0	/* ATAPI packet command */
#define ATAPI_IDENTIFY	0xA1	/* ATAPI identify device */

#define ATAPI		0x08	/* wn->state: device is an ATAPI device */
#define READY		0x10	/* wn->state: device is ready */

/* Reroute the entry points. */
PRIVATE struct driver atapi_dtab = {
  w_name,		/* current device's name */
  atapi_do_open,	/* open or mount request, initialize device */
  w_do_close,		/* release device */
  atapi_ioctl,		/* get or set a partition's geometry, eject CD */
  w_prepare,		/* prepare for I/O on a given minor device */
  atapi_transfer,	/* do the I/O */
  w_cleanup,		/* start a timer to do a spin down */
  w_geometry,		/* tell the geometry of the device */
};
#define w_dtab	atapi_dtab	/* Let at_wini use the ATAPI entry points. */

FORWARD _PROTOTYPE( int atapi_do_open, (struct driver *dp, message *m_ptr) );
FORWARD _PROTOTYPE( int atapi_identify, (void)				);


/*============================================================================*
 *				atapi_do_open				      *
 *============================================================================*/
PRIVATE int atapi_do_open(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
/* Device open: Initialize the controller and read the partition table. */

  struct wini *wn;
  int r;

  if (w_prepare(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);
  wn = w_wn;

  if (wn->state == 0) {
	/* Try to identify the device as an ATAPI device. */
	if (atapi_identify() != OK) {
		printf("%s: probe failed\n", w_name());
		if (wn->state & DEAF) w_reset();
		wn->state = 0;
		return(ENXIO);
	}
  }
  if (!(wn->state & ATAPI)) {
	/* There is something there, but it is not ATAPI.  Probably a disk. */
	return(w_do_open(dp, m_ptr));
  }

  if ((r = atapi_probe()) != OK) return(r);

  if (m_ptr->COUNT & W_BIT) return(EACCES);

  if (wn->open_ct++ == 0) {
	/* Partition the device. */
	partition(&atapi_dtab, w_drive * DEV_PER_DRIVE, P_PRIMARY);
  }
  return(OK);
}


/*===========================================================================*
 *				atapi_identify				     *
 *===========================================================================*/
PRIVATE int atapi_identify()
{
  /* Check if a device is an ATAPI device by sending it an ATAPI identify
   * command.
   */

  struct wini *wn = w_wn;
  struct command cmd;
  char id_string[40];
  int i, r;
  unsigned multiple;
  unsigned long size;
#define id_byte(n)	(&tmp_buf[2 * (n)])
#define id_word(n)	(((u16_t) id_byte(n)[0] <<  0) \
			|((u16_t) id_byte(n)[1] <<  8))
#define id_longword(n)	(((u32_t) id_byte(n)[0] <<  0) \
			|((u32_t) id_byte(n)[1] <<  8) \
			|((u32_t) id_byte(n)[2] << 16) \
			|((u32_t) id_byte(n)[3] << 24))

  /* Check if the one of the registers exists. */
  r = in_byte(wn->base + REG_CYL_LO);
  out_byte(wn->base + REG_CYL_LO, ~r);
  if (in_byte(wn->base + REG_CYL_LO) == r) return(ERR);

  /* Looks OK; register IRQ and try an ATAPI identify command. */
  put_irq_handler(wn->irq, w_handler);
  enable_irq(wn->irq);

  cmd.ldh     = wn->ldhpref;
  cmd.command = ATAPI_IDENTIFY;
  if (com_simple(&cmd) != OK) return(OK);	/* OK, but not ATAPI. */

  /* This is an ATAPI device. */
  wn->state |= ATAPI;

  /* Device information. */
  port_read(wn->base + REG_DATA, tmp_phys, SECTOR_SIZE);

  /* Like ATA, the strings are byte swapped. */
  for (i = 0; i < 40; i++) id_string[i] = id_byte(27)[i^1];

#if 0
  wn->part[0].dv_size = XXX;
  wn->lheads = 64;
  wn->lsectors = 32;
  wn->lcylinders = div64u(wn->part[0].dv_size, SECTOR_SIZE) / (64 * 32);
#endif

  printf("%s: ATAPI %.40s\n", w_name(), id_string);
}


/*===========================================================================*
 *				atapi_probe				     *
 *===========================================================================*/
PRIVATE int atapi_probe()
{
/* See if the device is ready. */
  struct wini *wn = w_wn;

  if (!(wn->state & INITIALIZED)) {
	/* First contact with a new device, what type is it? */

	if ((r = atapi_inquiry()) != OK) return(r);
  }

  /* See if the unit is ready for I/O. */
  while ((key = atapi_simple(SCSI_UNITRDY, 0)) != SENSE_NO_SENSE) {
	/* Not ready, why? */

	wn->state &= ~READY;

	switch (key) {
	case SENSE_UNIT_ATT:
		/* A media change or something, try again. */
		break;
	case SENSE_NOT_READY:
		/* Look at the additional sense data to see why it isn't
		 * ready.
		 */
		sense = &ccb_sense(rq);
		switch ((sense->add_code << 8) | sense->add_qual) {
		case 0x0401:
			/* "It is becoming ready."  Fine, we wait. */
			micro_delay(1000000);
			break;
		case 0x0402:
			/* "Initialization command required."  So we tell it
			 * to spin up.
			 */
			if (scsi_simple(SCSI_STRTSTP, 1) != SENSE_NO_SENSE)
				return(EIO);
			break;
		case 0x0403:
			/* "Manual intervention required." */
		case 0x3A00:
			/* "No media present." */
			printf("%s: no media loaded\n", s_name());
			return(EIO);
		default:
			/* For some reason it is not usable. */
			printf("%s: not ready\n", s_name());
			return(EIO);
		}
		break;
	default:
		/* The device is in some odd state.  */
		if (key != SENSE_NOT_READY) {
			printf("%s: hardware error\n", s_name());
			return(EIO);
		}
	}
  }

  if (!(sp->state & S_PRESENT)) {
	/* Do the inquiry again, the message may have changed. */
	if (scsi_inquiry() != OK) return(EIO);

	/* Tell what kind of device it is we have found. */

	printf("%s: %-7s %.48s\n",
		s_name(),
		inqdata.devtype > SCSI_DEVMAX ? "UNKNOWN"
				: scsi_devstr[inqdata.devtype],
		inqdata.vendor /* + product + revision + extra */);

	if (debug && aha_model != AHA1540) {
		/* The ISA bus limits throughput to 11/(11+4) * 5MB/s = 3.7MB/s
		 * for default settings of bus on, bus off and DMA speed.
		 * Nevertheless it is nice to know if sync SCSI is used:
		 */
		cmd[0] = AHACOM_GETSETUP;
		cmd[1] = 16;
		aha_command(2, cmd, 16, setup);

		if (setup[8 + sp->targ] & 0x80) {
			printf("%s: synchronous at %s MB/s\n", s_name(),
				speed[(setup[8 + sp->targ] >> 4) & 0x07]);
		}
	}

	if (inqdata.devtype == SCSI_DEVDISK
					&& scsi_ansiver(inqdata.stdver) >= 2) {
		/* A SCSI-2 device should be able to spin down. */
		sp->state |= S_CANSLEEP;
	}
  }

  if (!(sp->state & S_READY)) {
	/* Get the geometry, limits, etc. */

	switch (sp->devtype) {
	case SCSI_DEVDISK:
	case SCSI_DEVWORM:
	case SCSI_DEVCDROM:
		if (scsi_ndisk() != OK) return(EIO);
		break;
	case SCSI_DEVTAPE:
		if (scsi_ntape() != OK) return(EIO);
		break;
	default:
		printf("%s: unsupported\n", s_name());
		return(EIO);
	}
  }
  return(OK);
}


/*===========================================================================*
 *				scsi_sense				     *
 *===========================================================================*/
PRIVATE int scsi_sense()
{
  int key;
  sense_t *sense = (sense_t *) tmp_buf;

  /* Do a request sense to find out if a target exists or to check out
   * a unit attention condition.
   */
  key = scsi_simple(SCSI_REQSENSE, sizeof(sense_t));

  if (rq->ccb.hastat == HST_TIMEOUT) return(ENXIO);	/* nothing there */
  if (rq->ccb.hastat != 0) return(EIO);		/* something very bad */

  /* There is something out there for sure. */
  if (key == SENSE_UNIT_ATT || sense_key(sense->key) == SENSE_UNIT_ATT) {
	/* Device is in a "look at me" state, probably changed media. */
	s_sp->state &= ~S_READY;
  }
  return(OK);
}


/*===========================================================================*
 *				scsi_inquiry				     *
 *===========================================================================*/
PRIVATE int scsi_inquiry()
{
  /* Prefill with nulls. */
  memset(tmp_buf, '\0', sizeof(inquiry_t));

  /* Do a SCSI inquiry. */
  if (scsi_simple(SCSI_INQUIRY, sizeof(inquiry_t)) != SENSE_NO_SENSE)
	return(EIO);
  inqdata = * (inquiry_t *) tmp_buf;

  if (inqdata.len == 0) {
	/* The device doesn't return meaningful text fields. */
	strcpy(inqdata.vendor, "(unknown)");
  }

  /* The top three bits of devtype must be zero for the lun to exist. */
  if ((inqdata.devtype & 0xE0) != 0) return(ENXIO);

  return(OK);
}


/*===========================================================================*
 *				scsi_ndisk				     *
 *===========================================================================*/
PRIVATE int scsi_ndisk()
{
/* Gather disk data, capacity and block size. */

  struct scsi *sp = s_sp;
  unsigned long capacity = -1, block_size = SECTOR_SIZE;
  byte *buf = tmp_buf;

  /* Minor device type must be for a disk. */
  if (s_type != TYPE_SD) return(EIO);

  if (sp->devtype == SCSI_DEVCDROM) {
	/* Read-only by definition. */
	sp->state |= S_RDONLY;
  } else {
	/* SCSI modesense to find out if the disk is write protected. */
	if (scsi_simple(SCSI_MDSENSE, 255) != SENSE_NO_SENSE) return(EIO);

	/* Write protected? */
	sp->state &= ~S_RDONLY;
	if (buf[2] & 0x80) sp->state |= S_RDONLY;

	/* Don't write a worm disk, not wise at the moment. */
	if (sp->devtype == SCSI_DEVWORM) sp->state |= S_RDONLY;
  }

  /* Get drive capacity and block size. */
  group1();
  rq->ccb.opcode = CCB_INIT;
  ccb_cmd1(rq).scsi_op = SCSI_CAPACITY;

  if (scsi_command(tmp_phys, 8) == SENSE_NO_SENSE) {
	capacity = b2h32(buf + 0) + 1;
	block_size = b2h32(buf + 4);
	printf("%s: capacity %lu x %lu bytes\n",
					s_name(), capacity, block_size);
  } else {
	printf("%s: unknown capacity\n", s_name());
  }

  /* We do not believe block sizes over 4 kb. */
  if (block_size > 4096) {
	printf("%s: can't handle %lu byte blocks\n", s_name(), block_size);
	return(EIO);
  }

  sp->block_size = block_size;
  sp->part[0].dv_size = mul64u(capacity, block_size);

#if _WORD_SIZE > 2
  /* Keep it within reach of a group 0 command. */
  sp->count_max = 0x100 * block_size;
#else
  sp->count_max = block_size > UINT_MAX/0x100 ? UINT_MAX : 0x100 * block_size;
#endif

  /* Finally we recognize its existence. */
  sp->state |= S_PRESENT|S_READY;

  return(OK);
}
