/*
vm386.c

Virtual memory implementation for the Intel 386 and higher.

Created:	July 23, 1992 by Philip Homburg <philip@cs.vu.nl>
*/

#include "kernel.h"	/* Must be first */
#include <stddef.h>
#include <minix/com.h>
#include <minix/minlib.h>
#include <minix/queryparam.h>
#include <signal.h>
#include "assert.h"
INIT_ASSERT
#include "proc.h"
#include "proto.h"
#include "vm386.h"

/* We have 12 bits for page types. 0x1 .. 0x7ff are used for the tasks, servers
 * and user processes: the index in the process table plus 1. This allows
 * for more than 2000 processes. The values 0x800 .. 0xfff are used for pages
 * that are used by the paging system or by the hardware. */

#define PT_SPECIAL_PAGES	0x800
#define PT_PAGE_BASE		0x800
#define PT_SHADOW_BASE		0x801
#define PT_PAGE_TABLE		0x802
#define PT_SHADOW_TABLE		0x803
#define PT_ID_TABLE		0x804
#define PT_HARDWARE		0x805
#define PT_FREEPAGE		0x806
#define PT_NOSWAP		0x807
#define PT_BOUNCE_PAGE		0x808

#define HI_VIRT_MEM	0x80000000U		/* 2G */

#define is_diskpage(e)	((e) & VM_SWAPPAGE)
#define is_dmapage(e)	(no_isa_dma || (e) < VM_ISA_LIMIT)
#define proc2vmtype(p)	(proc_index(p) + 1)
#define vmtype2proc(t)	(proc_indx2addr(t-1))

PRIVATE u32_t page_base, shadow_base;	/* root of the page table and the
					 * shadow page table
					 */
PRIVATE u32_t paging_base;		/* page faults below paging_base
					 * should not occur
					 */
PRIVATE u32_t hi_coremem;		/* all core is below hi_coremem */
PRIVATE u32_t coremem_table_base;	/* address of the table that keeps
					 * track of the allocation of core
					 * memory
					 */
PRIVATE u32_t coremem_lock_base;	/* lock table for core memory */
PRIVATE unsigned coremem_pages;		/* number of entries in the
					 * coremem_table
					 */
PRIVATE u32_t free_dmacore_head;	/* head of the list of free core
					 * pages that can be used for DMA
					 */
PRIVATE u32_t freecore_head;		/* head of the list of the remaining
					 * free core pages
					 */
PRIVATE int locked_pages,		/* allocation in numbers */
	floating_pages, free_dmacore_mem, free_coremem, total_dmacore_mem,
	total_coremem, free_swapmem, total_swapmem, transfer_pages;
PRIVATE unsigned long pagein_cnt, pageout_cnt, bounce_pagein_cnt,
	bounce_pageout_cnt;		/* page in/page out statistics */
PRIVATE u32_t intl_pages[2],
	taskl_pages[2];			/* preallocated pages for the page
					 * fault handler.
					 */
PRIVATE int no_isa_dma= 0;		/* At the moment there is no support
					 * for changing no_isa_dma after
					 * vm_init() is called.
					 */
PRIVATE u32_t bounce_page;		/* page to be used to page in or
					 * page out pages that are not dma
					 * pages
					 */


#define MIN_UNLOCKED_PAGES	(0x10000/VM_PAGESIZE)

typedef struct swapdesc
{
	int sd_proc;
	int sd_minor;
	u32_t sd_offset;
	i32_t sd_priority;
	unsigned sd_flags;
	u32_t sd_map;
	u32_t sd_phys_space;
	u32_t sd_freehead;
	struct swapdesc *sd_nextfree;
} swapdesc_t;

#define SDP_MIN		(-0x7FFFFFFF)
#define SDF_INUSE	0x1
#define SDF_INFREEQ	0x2

#define NR_SWAPSEGS	64
PRIVATE swapdesc_t swaptable[NR_SWAPSEGS], *swap_freehead, *swap_freetail;

PRIVATE struct proc *pagefq_head, *page_ptr;
PRIVATE int pager_running;

#define MIN_PAGES2SWAP	2
#define MAX_PAGES2SWAP	16

#define SLEEPING_PRIO	((unsigned)-1)

PRIVATE unsigned curr_pageno;
PRIVATE int sweeper_round;
PRIVATE int sweeper_running;
PRIVATE int pager_tasknr= ANY;

PRIVATE volatile u32_t sink;	/* Hope this is enough to force
				 * the assignment
				 */

#define SWEEPER_MAX	60	/* We continue running the sweeper 60 times
				 * after the last time pages were swapped to
				 * disk */
#define SWEEPER_FREQ	2	/* The sweeper runs 2 times per second. */


/* Allow these variables to be queried by an outside program. */
PRIVATE struct export_param_list vm_ex_param_list[] = {
	QP_VARIABLE(locked_pages),
	QP_VARIABLE(floating_pages),
	QP_VARIABLE(free_dmacore_mem),
	QP_VARIABLE(total_dmacore_mem),
	QP_VARIABLE(free_coremem),
	QP_VARIABLE(total_coremem),
	QP_VARIABLE(free_swapmem),
	QP_VARIABLE(total_swapmem),
	QP_VARIABLE(vm_not_alloc),
	QP_VARIABLE(transfer_pages),
	QP_VARIABLE(pagein_cnt),
	QP_VARIABLE(bounce_pagein_cnt),
	QP_VARIABLE(pageout_cnt),
	QP_VARIABLE(bounce_pageout_cnt),
	QP_END()
};
PRIVATE struct export_params vm_ex_params = { vm_ex_param_list };


FORWARD _PROTOTYPE( void coremem_init, (u32_t *hi_coremem_p)		);
FORWARD _PROTOTYPE( void phys_check, (void)				);
FORWARD _PROTOTYPE( void vm_nomem, (void)				);
FORWARD _PROTOTYPE( void mem_free, (u32_t mem_ptr, int *account_p)	);
FORWARD _PROTOTYPE( u32_t dmamem_alloc, (int *account_p)		);
FORWARD _PROTOTYPE( u32_t dmamem_alloc_forced, (int *account_p)		);
FORWARD _PROTOTYPE( u32_t coremem_alloc, (int *account_p)		);
FORWARD _PROTOTYPE( void set_page_info, (u32_t page_address, int type,
						u32_t virtual_address)	);
FORWARD _PROTOTYPE( u32_t get_page_info, (u32_t page_address)		);
FORWARD _PROTOTYPE( int lock_page, (u32_t page_address, int *account_p)	);
FORWARD _PROTOTYPE( void unlock_page, (u32_t page_address,
							int *account_p)	);
FORWARD _PROTOTYPE( int lock_count, (u32_t page_address)		);
FORWARD _PROTOTYPE( void map_table, (u32_t vir_addr, u32_t table,
					u32_t shadow, int region_type)	);
FORWARD _PROTOTYPE( void map_id_page, (u32_t real_vir_addr)		);
FORWARD _PROTOTYPE( void map_page, (u32_t addr, u32_t virtual,
							u32_t flags)	);
FORWARD _PROTOTYPE( void mk_space, (struct proc *pp, u32_t base,
							u32_t size)	);
FORWARD _PROTOTYPE( void mk_table, (u32_t base, u32_t size)		);
FORWARD _PROTOTYPE( void mk_image, (u32_t image_base, u32_t base,
							u32_t size)	);
FORWARD _PROTOTYPE( void mk_clear, (u32_t base, u32_t size)		);
FORWARD _PROTOTYPE( void page_dequeue, (u32_t real_addr, u32_t vm_addr,
		int vm_type, int type, u32_t next, int *account_p)	);
FORWARD _PROTOTYPE( void suspend, (struct proc *pp, u32_t addr,
							u32_t err)	);
FORWARD _PROTOTYPE( void adjust_user_stack, (struct proc *pp)		);
FORWARD _PROTOTYPE( void tell_pages, (char *label, u32_t n)		);
#if DEBUG
FORWARD _PROTOTYPE( void dump_map, (struct mem_map *mp)			);
#endif
FORWARD _PROTOTYPE( struct swapdesc *get_swapdesc, (int pnr,
			int minor, u32_t offset, i32_t priority)	);
FORWARD _PROTOTYPE( u32_t swapmem_alloc, (int *account_p)		);
FORWARD _PROTOTYPE( void do_pagefault, (void)				);
FORWARD _PROTOTYPE( void pages2swap, (struct proc *pp)			);
FORWARD _PROTOTYPE( u32_t move2dma, (u32_t page)			);
FORWARD _PROTOTYPE( int check_page_prio, (u32_t qhead,
						unsigned priority)	);
FORWARD _PROTOTYPE( void remap_pages, (u32_t prev, u32_t next)		);
FORWARD _PROTOTYPE( unsigned proc_prio, (struct proc *pp)		);
FORWARD _PROTOTYPE( void transfer_page, (u32_t src_page, u32_t
							dst_page)	);
FORWARD _PROTOTYPE( void req_swapout, (int procn)			);
FORWARD _PROTOTYPE( int check_list, (u32_t head)			);
FORWARD _PROTOTYPE( int check_free_list, (void)				);
FORWARD _PROTOTYPE( int check_disk_pages, (void)			);
FORWARD _PROTOTYPE( void sweeper, (void)				);
FORWARD _PROTOTYPE( void sweeper_int, (void)				);

/*
 * Initialization routines.
 */

/*==========================================================================*
 *				vm_init					    *
 *==========================================================================*/
PUBLIC void vm_init()
{
	u32_t tab_addr, page_table, shadow_table, page_addr,
		virt_base;

	/* Check if acessing physical bytes works. */
	phys_check();

	/* Create tables for the main memory in the system. And report some
	 * statistics.
	 */
	coremem_init(&hi_coremem);

	/* allocate a page directory */
	page_base= coremem_alloc(&locked_pages);

	/* allocate a shadow page directory */
	shadow_base= coremem_alloc(&locked_pages);
	assert (page_base != 0 && shadow_base != 0);

	/* Mark de pages to be owned by the kernel. */
	set_page_info(page_base, PT_PAGE_BASE, 0);
	set_page_info(shadow_base, PT_SHADOW_BASE, 0);

	/* Clear the pages, this means not allocated for anything. */
	phys_clr_page(page_base);
	phys_clr_page(shadow_base);

	/* Allocate some page tables for the indentity mapping. */
	for (tab_addr= 0; tab_addr<hi_coremem; tab_addr += VM_DIRSIZE)
	{
		/* Allocate a page table and a shadow page table */
		page_table= coremem_alloc(&locked_pages);
		shadow_table= coremem_alloc(&locked_pages);
		assert(page_table != 0 && shadow_table != 0);

		/* Mark de pages to be owned by the kernel. */
		set_page_info(page_base, PT_PAGE_TABLE, tab_addr);
		set_page_info(shadow_base, PT_SHADOW_TABLE, tab_addr);

		/* A clear page table entry means a non exitent virtual address
		 * and a clear shadow page table entry means the end of the
		 * list of virual pages that translate to the same physical
		 * address
		 */
		phys_clr_page(page_table);
		phys_clr_page(shadow_table);

		/* Enter the page table in the page directory */
		map_table(tab_addr, page_table, shadow_table, PT_ID_TABLE);
	}

	/* Now we map in all memory. */
	for (page_addr= 0; page_addr<hi_coremem; page_addr += VM_PAGESIZE)
	{
		map_id_page(page_addr);
	}

	vm_enable(page_base);	/* This is what it's all about */

	/* Calculate where to start the virtual memory available to MM and
	 * the ram disk.
	 */
	virt_base= (hi_coremem + VM_DIRSIZE-1) & ~VM_DIRMASK;

	/* record the start of virtual memory for consistency checks. */
	paging_base= virt_base;

	/* Make the virtual memory available as 'normal' memory. */
	chunk_add (virt_base >> CLICK_SHIFT,
			(HI_VIRT_MEM-virt_base) >> CLICK_SHIFT);

	/* Allocate a bounce page (if not all pages are dma pages) */
	if (free_coremem != 0)
	{
		bounce_page= dmamem_alloc(&locked_pages);
		assert(bounce_page != 0);
		set_page_info(bounce_page, PT_BOUNCE_PAGE, 0);
	}
	else
		bounce_page= 0;

	/* Record the number of pages available for user process. Kernel
	 * pages that are allocated after this point should be subtracted
	 * from the not reserved pages.
	 */
	vm_not_alloc= free_dmacore_mem + free_coremem;
	vm_lowest_free= vm_not_alloc;

	/* These variables may be queried. */
	qp_export(&vm_ex_params);
}


/*==========================================================================*
 *				coremem_init				    *
 *==========================================================================*/
PRIVATE void coremem_init(hi_coremem_p)
u32_t *hi_coremem_p;
{
	phys_clicks hi_mem_clicks, coremem_table_clicks,  coremem_lock_clicks,
		chunk_base_clicks, chunk_size_clicks;
	u32_t hi_mem, chunk_base, chunk_size, mem_ptr;
	unsigned coremem_table_size, coremem_lock_size;
	int i, r;

	hi_mem_clicks= 0;

	/* Calculate the amount of memory (address space) available (in clicks).
	 */
	for (i=0; i<CHUNK_NR; i++)
	{
		if (!chunk_table[i].chk_size)
			break;
		if (chunk_table[i].chk_base + chunk_table[i].chk_size >
								hi_mem_clicks)
		{
			hi_mem_clicks= chunk_table[i].chk_base +
							chunk_table[i].chk_size;
		}
	}

	hi_mem= hi_mem_clicks << CLICK_SHIFT;	/* Address space in bytes */

	/* We are now going to allocate at table that keeps track of
	 * allocated memory. We need sizeof(u32_t) bytes for each page.
	 */
	coremem_pages= (hi_mem + VM_PAGESIZE-1) >> VM_PAGESHIFT;
	coremem_table_size= coremem_pages * sizeof(u32_t);
	coremem_lock_size= coremem_pages;
	coremem_table_clicks= (coremem_table_size + CLICK_SIZE-1) >>
								CLICK_SHIFT;
	coremem_lock_clicks= (coremem_lock_size + CLICK_SIZE-1) >>
								CLICK_SHIFT;

	/* Find a place for the table */
	chunk_size_clicks= coremem_table_clicks+coremem_lock_clicks;
	r= chunk_find(&chunk_base_clicks, &chunk_size_clicks);
	if (!r)
	{
#if DEBUG
 { printW(); }
#endif
		panic("Unable to find a place for the memory allocation table",
									NO_NUM);
	}
	chunk_del(chunk_base_clicks, coremem_table_clicks+coremem_lock_clicks);

	coremem_table_base= chunk_base_clicks << CLICK_SHIFT;
	coremem_lock_base= coremem_table_base + (coremem_table_clicks <<
								CLICK_SHIFT);
	for (i= 0; i<coremem_pages; i++)
	{
		set_page_info(i << VM_PAGESHIFT, PT_HARDWARE, 0);
	}
	for (i= 0; i<coremem_lock_size; i += VM_PAGESIZE)
		phys_clr_page(coremem_lock_base+i);

	/* Mark all available memory as free */
	free_coremem= 0;
	free_dmacore_mem= 0;
	freecore_head= 0;
	free_dmacore_head= 0;
	total_coremem= coremem_pages;
	while (chunk_table[0].chk_size)
	{
		chunk_base_clicks= chunk_table[0].chk_base;
		chunk_size_clicks= chunk_table[0].chk_size;
		chunk_base= chunk_base_clicks << CLICK_SHIFT;
		chunk_size= chunk_size_clicks << CLICK_SHIFT;

		for (mem_ptr= chunk_base; mem_ptr<chunk_base+chunk_size;
			mem_ptr += VM_PAGESIZE)
		{
			mem_free(mem_ptr, &total_coremem);
		}
		chunk_del(chunk_base_clicks, chunk_size_clicks);
	}
	assert(total_coremem + free_dmacore_mem + free_coremem ==
		coremem_pages);

	locked_pages= 0;
	floating_pages= 0;
	total_dmacore_mem= free_dmacore_mem;
	total_coremem= free_coremem;
	*hi_coremem_p= coremem_pages << VM_PAGESHIFT;
}


/*==========================================================================*
 *				phys_check				    *
 *==========================================================================*/
PRIVATE void phys_check()
{
/* Let's check get_phys_byte, put_phys_byte, get_phys_dword, and
 * put_phys_dword. These function can be coded as macros iff the kernel
 * data segment is mapped linearly.
 */
	u8_t byte;
	u32_t dword;
	phys_bytes byte_a, dword_a;

	byte_a= vir2phys(&byte);
	dword_a= vir2phys(&dword);

	byte= 0xA5;
	if (get_phys_byte(byte_a) != 0xA5)
		panic("phys_check: get_phys_byte failed", NO_NUM);
	put_phys_byte(byte_a, 0x5A);
	if (byte != 0x5A)
		panic("phys_check: put_phys_byte failed", NO_NUM);
	dword= 0xA5B4C3D2;
	if (get_phys_dword(dword_a) != 0xA5B4C3D2)
		panic("phys_check: get_phys_dword failed", NO_NUM);
	put_phys_dword(dword_a, 0x5A4B3C2D);
	if (dword != 0x5A4B3C2D)
		panic("phys_check: put_phys_dword failed", NO_NUM);
}


/*==========================================================================*
 *				vm_alloc_space				    *
 *==========================================================================*/
PUBLIC u32_t vm_alloc_space(pp, size)
struct proc *pp;
u32_t size;
{
	u32_t base;
	phys_clicks clicks, chunk_base_clicks, chunk_size_clicks;
	unsigned space;
	int r;

	size= (size + VM_DIRSIZE-1) & ~VM_DIRMASK;

	/* Reserve enough memory for our pagetables. */
	space= 2*(size >> VM_DIRSHIFT);
	if (vm_not_alloc < space) vm_nomem();
	vm_not_alloc -= space;

	clicks= size >> CLICK_SHIFT;
	chunk_size_clicks= clicks;
	r= chunk_find(&chunk_base_clicks, &chunk_size_clicks);
	assert(r);
	chunk_del(chunk_base_clicks, clicks);
	base= chunk_base_clicks << CLICK_SHIFT;
	assert(!(base & VM_DIRMASK));

	mk_space(pp, base, size);
	mk_table(base, size);
	return base;
}


/*==========================================================================*
 *				vm_nomem				    *
 *==========================================================================*/
PRIVATE void vm_nomem()
{
	panic("Not enough memory", NO_NUM);
}


/*==========================================================================*
 *				vm_map_image				    *
 *==========================================================================*/
PUBLIC void vm_map_image(pp, vir_base, img_base, size)
struct proc *pp;
u32_t vir_base;
u32_t img_base;
u32_t size;
{
	int r;

	mk_image(img_base, vir_base, size);
	if (pp->p_status & P_ST_VM_INCORE)
	{
		r= vm_lock(vir_base, size, TRUE /* force lock */);
		assert(r == OK);
	}
}


/*==========================================================================*
 *				vm_make_clear				    *
 *==========================================================================*/
PUBLIC void vm_make_clear(pp, base, size)
struct proc *pp;
u32_t base;
u32_t size;
{
	u32_t space;
	int r;

	size= (size + VM_PAGESIZE-1) & ~VM_PAGEMASK;
	space= (size >> VM_PAGESHIFT);
	if (vm_not_alloc < space) vm_nomem();
	vm_not_alloc -= space;

	mk_clear(base, size);
	if (pp->p_status & P_ST_VM_INCORE)
	{
		r= vm_lock(base, size, TRUE /* force lock */);
		assert(r == OK);
	}
}


/*==========================================================================*
 *				vm_map_zp				    *
 *==========================================================================*/
PUBLIC void vm_map_zp(map_page)
int map_page;
{
	u32_t table, pageent;
	unsigned tableno, pageno;
#if DEBUG
	u32_t shadow;
#endif

	/* Paging enabled? */
	if (page_base == 0) return;

	tableno= 0;
	pageno= 0;

	table= get_phys_dword(page_base + 4*tableno);
	assert(table & VM_PRESENT);
	table &= VM_ADDRMASK;
	assert(table);
#if DEBUG
	/* assert that the shadow entry is 0 */
	shadow= get_phys_dword(shadow_base + 4*tableno);
	assert((shadow & VM_PAGEMASK) == PT_ID_TABLE);
	shadow &= VM_ADDRMASK;
	assert(get_phys_dword(shadow + 4*pageno) == 0);
#endif
	if (map_page)
	{
		/* Map page zero */
		pageent= VM_PRESENT | VM_WRITE | VM_USER;
	}
	else
	{
		/* Unmap page zero */
		pageent= 0;
	}
	put_phys_dword(table + 4*pageno, pageent);
	if (k_reenter != -1)
		vm_reload();
	else
		level0(vm_reload);
}


/*==========================================================================*
 *				vm_lock_hardware			    *
 *==========================================================================*/
PUBLIC void vm_lock_hardware(phys_base, phys_size)
phys_bytes phys_base;
phys_bytes phys_size;
{
	u32_t base, top;
	int r;

	base= phys_base;
	top= phys_base+phys_size;
	assert(!(base & VM_PAGEMASK));
	assert(!(top & VM_PAGEMASK));
	for (; base < top; base += VM_PAGESIZE)
	{
		r= lock_page(base, &locked_pages);
		assert(r == OK);
	}
}


/*==========================================================================*
 *				vm_map_hardware				    *
 *==========================================================================*/
PUBLIC void vm_map_hardware(hw_addr, phys_base, phys_size)
phys_bytes hw_addr;
phys_bytes phys_base;
phys_bytes phys_size;
{
	u32_t base, top, hw_base;
	unsigned tableno, pageno, type;
	u32_t table, shadow, pageent, hwent;
	volatile u32_t dummy;

	base= phys_base;
	top= phys_base+phys_size;
	hw_base= hw_addr;
	assert(!(base & VM_PAGEMASK));
	assert(!(top & VM_PAGEMASK));
	assert(!(hw_base & VM_PAGEMASK));

	if (vm_not_alloc < (phys_size >> VM_PAGESHIFT)) vm_nomem();
	vm_not_alloc -= phys_size >> VM_PAGESHIFT;

	for (tableno= base >> VM_DIRSHIFT; base < top; tableno++)
	{
		table= get_phys_dword(page_base + 4*tableno);
		shadow= get_phys_dword(shadow_base + 4*tableno);
		if (table == 0)
		{
			/* Cause a pagefault to allocate a table. */
			dummy= get_phys_dword(base);
			/* and unmap the page */
			vm_unmap_pages(base, VM_PAGESIZE);
			table= get_phys_dword(page_base + 4*tableno);
			assert(table != 0);
		}
		table &= VM_ADDRMASK;
		type= shadow & VM_PAGEMASK;
		shadow &= VM_ADDRMASK;
		assert(table != 0);
		assert(shadow != 0);

		for (pageno= (base & VM_DIRMASK) >> VM_PAGESHIFT;
			pageno < VM_PAGESIZE/4 && base < top;
			base += VM_PAGESIZE, hw_base += VM_PAGESIZE, pageno++)
		{
			assert(get_phys_dword(table + 4*pageno) == 0);
			pageent= hw_base | VM_PRESENT | VM_WRITE | VM_USER |
				VM_HARDWARE;
			put_phys_dword(table+4*pageno, pageent);
			hwent= get_page_info(hw_base);
			put_phys_dword(shadow+4*pageno, hwent);
			set_page_info(hw_base, type, base);
		}
	}
}


/*==========================================================================*
 *				mk_space				    *
 *==========================================================================*/
PRIVATE void mk_space(pp, base, size)
struct proc *pp;
u32_t base;
u32_t size;
{
	u32_t table_base, table_end, addr, shadow;
	unsigned tableno;
	int type;

	type= proc2vmtype(pp);
	assert(proc + type-1 == pp);

	table_base= base & ~VM_DIRMASK;
	table_end= (base+size + VM_DIRSIZE-1) & ~VM_DIRMASK;

	for (addr= table_base, tableno= addr >> VM_DIRSHIFT;
				addr < table_end; addr += VM_DIRSIZE, tableno++)
	{
		shadow= get_phys_dword(shadow_base + 4*tableno);

		if (shadow == 0)
		{
			shadow= type;
			put_phys_dword(shadow_base + 4*tableno, shadow);
		}
		else
			assert((shadow & VM_PAGEMASK) == type);
	}
}


/*==========================================================================*
 *				mk_table				    *
 *==========================================================================*/
PRIVATE void mk_table(base, size)
u32_t base;
u32_t size;
{
	u32_t table_base, table_end, addr, type, table, shadow;
	unsigned tableno;

#if DEBUG & 256
 { printW(); printf("in mk_table(0x%x, 0x%x)\n", base, size); }
#endif

	table_base= base & ~VM_DIRMASK;
	table_end= (base+size + VM_DIRSIZE-1) & ~VM_DIRMASK;

	for (addr= table_base, tableno= addr >> VM_DIRSHIFT;
				addr < table_end; addr += VM_DIRSIZE, tableno++)
	{
		assert(!get_phys_dword(page_base + 4*tableno));
		type= get_phys_dword(shadow_base + 4*tableno);
#if DEBUG & 256
 { printW(); printf("shadow table 0x%x: *(0x%x)= 0x%x\n", tableno,
	shadow_base + 4*tableno, type); }
#endif
		assert(!(type & VM_ADDRMASK));
		assert(type & VM_PAGEMASK);

		while (free_dmacore_mem + free_coremem < 2)
			req_swapout(proc_number(vmtype2proc(type)));
		table= coremem_alloc(&locked_pages);
		shadow= coremem_alloc(&locked_pages);

		assert(table != 0 && shadow != 0);

		phys_clr_page(table);
		phys_clr_page(shadow);

		set_page_info(table, PT_PAGE_TABLE, addr);
		set_page_info(shadow, PT_SHADOW_TABLE, addr);

		map_table(addr, table, shadow, type);
	}
}


/*==========================================================================*
 *				mk_image				    *
 *==========================================================================*/
PRIVATE void mk_image(image_base, base, size)
u32_t image_base;
u32_t base;
u32_t size;
{
	u32_t coreinfo;

	while(size)
	{
		coreinfo= get_page_info(image_base);
		assert(coreinfo == PT_HARDWARE);
		set_page_info(image_base, 0, 0);
		if (is_dmapage(image_base))
			total_dmacore_mem++;
		else
			total_coremem++;
		floating_pages++;
		map_page(image_base, base,  VM_PRESENT | VM_WRITE | VM_USER);
		assert((get_page_info(image_base) & VM_ADDRMASK) == base);
		assert(size >= VM_PAGESIZE);
		image_base += VM_PAGESIZE;
		base += VM_PAGESIZE;
		size -= VM_PAGESIZE;
	}
}


/*==========================================================================*
 *				mk_clear				    *
 *==========================================================================*/
PRIVATE void mk_clear(base, size)
u32_t base;
u32_t size;
{
	u32_t clear_page;

	while(size)
	{
		clear_page= coremem_alloc(&floating_pages);
		assert(clear_page != 0);

		phys_clr_page(clear_page);
		map_page(clear_page, base,  VM_PRESENT | VM_WRITE | VM_USER);
		assert((get_page_info(clear_page) & VM_ADDRMASK) == base);
		assert(size >= VM_PAGESIZE);
		base += VM_PAGESIZE;
		size -= VM_PAGESIZE;
	}
}


/*
 * Operations on core memory allocation.
 */

/*==========================================================================*
 *				dmamem_alloc				    *
 *==========================================================================*/
PRIVATE u32_t dmamem_alloc(account_p)
int *account_p;
{
	u32_t page, next;

	page= free_dmacore_head;
	if (page == 0)
	{
		assert(free_dmacore_mem == 0);
		return page;
	}
	assert ((page & VM_PAGEMASK) == PT_FREEPAGE);
	page &= VM_ADDRMASK;
	next= get_page_info(page);
	free_dmacore_head= next;
	assert(free_dmacore_mem > 0);
	free_dmacore_mem--;
	assert(is_dmapage(page));

	(*account_p)++;
	assert(page < hi_coremem);
	return page;
}


/*==========================================================================*
 *				coremem_alloc				    *
 *==========================================================================*/
PRIVATE u32_t coremem_alloc(account_p)
int *account_p;
{
	u32_t page, next;

	if (free_coremem != 0)
	{
		page= freecore_head;
		assert(page != 0);
		assert ((page & VM_PAGEMASK) == PT_FREEPAGE);
		page &= VM_ADDRMASK;
		next= get_page_info(page);
		freecore_head= next;
		assert(free_coremem > 0);
		free_coremem--;
		assert (!is_dmapage(page));
	}
	else
	{
		assert(freecore_head == 0);
		page= free_dmacore_head;
		if (page == 0)
		{
			assert(free_dmacore_mem == 0);
			return page;
		}
		assert ((page & VM_PAGEMASK) == PT_FREEPAGE);
		page &= VM_ADDRMASK;
		next= get_page_info(page);
		free_dmacore_head= next;
		assert(free_dmacore_mem > 0);
		free_dmacore_mem--;
		assert(is_dmapage(page));
	}
	(*account_p)++;
	assert(page < hi_coremem);
	return page;
}


/*==========================================================================*
 *				dmamem_alloc_forced			    *
 *==========================================================================*/
PRIVATE u32_t dmamem_alloc_forced(account_p)
int *account_p;
{
	static u32_t curr_page= 0;
	u32_t page, cntpage, dmapage;
	int locks, r;
	u32_t info;

	page= dmamem_alloc(account_p);
	if (page != 0)
		return page;			/* We were lucky */
	page= coremem_alloc(account_p);
	assert(page != 0);

	dmapage= 0;

	assert ((coremem_pages << VM_PAGESHIFT) >= VM_ISA_LIMIT);

	/* First loop. Look at dma pages. Clear the referenced bit if
	 * necessary.
	 */
	for (cntpage= 0; cntpage <  VM_ISA_LIMIT; cntpage += VM_PAGESIZE,
		curr_page += VM_PAGESIZE)
	{
		if (curr_page >= VM_ISA_LIMIT)
		{
			assert(curr_page == VM_ISA_LIMIT);
			curr_page= 0;
		}
		info= get_page_info(curr_page);
		if ((info & VM_PAGEMASK) >= PT_SPECIAL_PAGES ||
			(info & VM_PAGEMASK) == 0)
		{
			/* This page is not interesting. */
			continue;
		}
		locks= lock_count(curr_page);
		if (locks != 0)
		{
			/* This page is locked. */
			continue;
		}
		r= check_page_prio(info, 0);
		if (r == EBUSY)
			continue;
		if (r == EAGAIN)
			continue;

		assert(r == OK);

		/* OK, we found a decent page. Allocate a swap page
		 * and remap the page table entries. */
		dmapage= curr_page;
		break;
	}

	/* Second loop. Look at all pages (referenced bits are ignored) */

	if (dmapage == 0)
	{
		for (cntpage= 0; cntpage <  VM_ISA_LIMIT;
			cntpage += VM_PAGESIZE, curr_page += VM_PAGESIZE)
		{
			if (curr_page >= VM_ISA_LIMIT)
			{
				assert(curr_page == VM_ISA_LIMIT);
				curr_page= 0;
			}
			info= get_page_info(curr_page);
			if ((info & VM_PAGEMASK) >= PT_SPECIAL_PAGES ||
				(info & VM_PAGEMASK) == 0)
			{
				/* This page is not interesting. */
				continue;
			}
			locks= lock_count(curr_page);
			if (locks != 0)
			{
				/* This page is locked. */
				continue;
			}
			r= check_page_prio(info, 0);
			if (r == EBUSY)
				continue;

			dmapage= curr_page;
			break;
		}
	}

	assert(dmapage != 0);
	phys_copy(dmapage, page, VM_PAGESIZE);
	remap_pages(dmapage, page);
	level0(vm_reload);
	return dmapage;
}


/*
 * general page operations
 */

/*==========================================================================*
 *				set_page_info				    *
 *==========================================================================*/
PRIVATE void set_page_info(page_address, type, virtual_address)
u32_t page_address;
int type;
u32_t virtual_address;
{
	unsigned pageno, swapdesc;
	swapdesc_t *sd;

	assert(!(page_address & VM_PAGEMASK));
	assert(!(virtual_address & VM_PAGEMASK));
	assert(!(type & VM_ADDRMASK));
	virtual_address |= type;
	if (!is_diskpage(page_address))
	{
		/* core memory page */
		pageno= page_address >> VM_PAGESHIFT;
		assert(pageno < coremem_pages);
		put_phys_dword(coremem_table_base + 4*pageno, virtual_address);
		return;
	}
#if DEBUG & 256
 { printW(); printf("set_page_info(0x%x, 0x%x, 0x%x)\n", page_address, type,
	virtual_address); }
#endif
	swapdesc= (page_address & ~VM_SWAPPAGE) >> VM_DIRSHIFT;
	assert(swapdesc < NR_SWAPSEGS);
	sd= &swaptable[swapdesc];
	assert((sd->sd_flags & SDF_INUSE) && sd->sd_map != 0);
	pageno= (page_address & VM_DIRMASK) >> VM_PAGESHIFT;
	put_phys_dword(sd->sd_map + 4*pageno, virtual_address);
}


/*==========================================================================*
 *				get_page_info				    *
 *==========================================================================*/
PRIVATE u32_t get_page_info(page_address)
u32_t page_address;
{
	unsigned pageno, swapdesc;
	swapdesc_t *sd;

#if DEBUG & 256
 { printW(); printf("in get_page_info(0x%x)\n", page_address); }
#endif
	assert(!(page_address & VM_PAGEMASK));
	if (!is_diskpage(page_address))
	{
		pageno= page_address >> VM_PAGESHIFT;
		assert(pageno < coremem_pages);
		return get_phys_dword(coremem_table_base + 4*pageno);
	}
	swapdesc= (page_address & ~VM_SWAPPAGE) >> VM_DIRSHIFT;
	assert(swapdesc < NR_SWAPSEGS);
	sd= &swaptable[swapdesc];
	assert((sd->sd_flags & SDF_INUSE) && sd->sd_map != 0);
	pageno= (page_address & VM_DIRMASK) >> VM_PAGESHIFT;
	return get_phys_dword(sd->sd_map + 4*pageno);
}


/*==========================================================================*
 *				lock_page				    *
 *==========================================================================*/
PRIVATE int lock_page(page_address, account_p)
u32_t page_address;
int *account_p;
{
	unsigned pageno;
	u8_t locks;

	assert(!(page_address & VM_PAGEMASK));
	assert (!is_diskpage(page_address));
	pageno= page_address >> VM_PAGESHIFT;
	assert(pageno < coremem_pages);
	locks= get_phys_byte(coremem_lock_base + pageno)+1;
	if (locks == 0)
	{
		/* Overflow */
		return ENOLCK;
	}

	if (locks == 1)
	{
		if (locked_pages + MIN_UNLOCKED_PAGES >= total_dmacore_mem)
		{
			return ENOMEM;
		}
		assert(*account_p > 0);
		(*account_p)--;
		locked_pages++;
	}
	put_phys_byte(coremem_lock_base + pageno, locks);
	return OK;
}


/*==========================================================================*
 *				unlock_page				    *
 *==========================================================================*/
PRIVATE void unlock_page(page_address, account_p)
u32_t page_address;
int *account_p;
{
	unsigned pageno;
	u8_t locks;

	assert(!(page_address & VM_PAGEMASK));
	assert (!is_diskpage(page_address));
	pageno= page_address >> VM_PAGESHIFT;
	assert(pageno < coremem_pages);
	locks= get_phys_byte(coremem_lock_base + pageno);
	assert(locks > 0);
	if (locks == 1)
	{
		assert(locked_pages > 0);
		locked_pages--;
		(*account_p)++;
	}
	put_phys_byte(coremem_lock_base + pageno, locks-1);
}


/*==========================================================================*
 *				lock_count				    *
 *==========================================================================*/
PRIVATE int lock_count(page_address)
u32_t page_address;
{
	unsigned pageno;
	u8_t locks;

	assert(!(page_address & VM_PAGEMASK));
	assert (!is_diskpage(page_address));
	pageno= page_address >> VM_PAGESHIFT;
	assert(pageno < coremem_pages);
	locks= get_phys_byte(coremem_lock_base + pageno);
	return locks;
}


/*==========================================================================*
 *				mem_free				    *
 *==========================================================================*/
PRIVATE void mem_free(mem_ptr, account_p)
u32_t mem_ptr;
int *account_p;
{
	unsigned swapdesc;
	swapdesc_t *sd;

	assert(!(mem_ptr & VM_PAGEMASK));
	assert(*account_p > 0);
	(*account_p)--;

	if (!is_diskpage(mem_ptr))
	{
		assert(lock_count(mem_ptr) == 0);
		if (is_dmapage(mem_ptr))
		{
			free_dmacore_mem++;
			set_page_info(mem_ptr, free_dmacore_head & VM_PAGEMASK,
				free_dmacore_head & VM_ADDRMASK);
			free_dmacore_head= mem_ptr | PT_FREEPAGE;
		}
		else
		{
			free_coremem++;
			set_page_info(mem_ptr, freecore_head & VM_PAGEMASK,
				freecore_head & VM_ADDRMASK);
			freecore_head= mem_ptr | PT_FREEPAGE;
		}
		return;
	}
	free_swapmem++;
	swapdesc= (mem_ptr & ~VM_SWAPPAGE) >> VM_DIRSHIFT;
	assert(swapdesc < NR_SWAPSEGS);
	sd= &swaptable[swapdesc];
	assert((sd->sd_flags & SDF_INUSE) && sd->sd_map != 0);
	set_page_info(mem_ptr, sd->sd_freehead & VM_PAGEMASK,
		sd->sd_freehead & VM_ADDRMASK);
	sd->sd_freehead= mem_ptr | PT_FREEPAGE;
	if (sd->sd_flags & SDF_INFREEQ)
	{
		return;
	}
	sd->sd_flags |= SDF_INFREEQ;
	if (swap_freehead == NULL ||
				sd->sd_priority >= swap_freehead->sd_priority)
	{
		sd->sd_nextfree= swap_freehead;
		if (!swap_freehead)
			swap_freetail= sd;
		swap_freehead= sd;
	}
	else
	{
		sd->sd_nextfree= NULL;
		swap_freetail->sd_nextfree= sd;
		swap_freetail= sd;
	}
}


/*==========================================================================*
 *				page_dequeue				    *
 *==========================================================================*/
PRIVATE void page_dequeue(real_addr, vm_addr, vm_type, type, next, account_p)
u32_t real_addr;
u32_t vm_addr;
int vm_type;
int type;
u32_t next;
int *account_p;
{
	u32_t pageent, nextent, shadow;
	unsigned tableno, pageno;

#if DEBUG & 256
 { printW(); printf("page_dequeue(0x%x, 0x%x, 0x%x, 0x%x, 0x%x)\n",
	real_addr, vm_addr, type, next, account_p); }
#endif
	assert(!(real_addr & VM_PAGEMASK));
	assert(!(vm_addr & VM_PAGEMASK));
	assert(!(type & VM_ADDRMASK));

	vm_addr |= type;

	pageent= get_page_info(real_addr);
	if (pageent == vm_addr)
	{
		/* Simple case, remove from head of the list. */
		if (next == 0)
		{
			/* Free page */
			mem_free(real_addr, account_p);
		}
		else
		{
			/* Set head of the list to next. */
#if DEBUG & 256
 { printW(); printf("calling set_page_info\n"); }
#endif
			set_page_info(real_addr, next & VM_PAGEMASK,
							next & VM_ADDRMASK);
#if DEBUG & 256
 { printW(); printf("set_page_info done\n"); }
#endif
		}
		return;
	}
	/* We have to find the virtual address that contains the pointer
	 * to the virtual address we want to remove.
	 */
	while(pageent)
	{
		tableno= pageent >> VM_DIRSHIFT;
		pageno= (pageent & VM_DIRMASK) >> VM_PAGESHIFT;

		assert(get_phys_dword(page_base + 4*tableno) & VM_ADDRMASK);
		shadow= get_phys_dword(shadow_base + 4*tableno);

		assert((shadow & VM_PAGEMASK) == (pageent & VM_PAGEMASK));
		shadow &= VM_ADDRMASK;
		assert(shadow);

		nextent= get_phys_dword(shadow + 4*pageno);
		if (nextent != vm_addr)
		{
			pageent= nextent;
			continue;
		}

		put_phys_dword(shadow + 4*pageno, next);
		return;
	}
#if DEBUG
 { printW(); }
#endif
	panic("unable to dequeue page", NO_NUM);
}


/*==========================================================================*
 *				remap_pages				    *
 *==========================================================================*/
PRIVATE void remap_pages(prev, next)
u32_t prev;
u32_t next;
{
	u32_t curr, table, shadow, pageent, shadowent, oldent, newent;
	unsigned tableno, pageno;

	assert(!(prev & VM_PAGEMASK));
	assert(!(next & VM_PAGEMASK));

	curr= get_page_info(prev);

assert(check_list(curr));

	set_page_info(next, curr & VM_PAGEMASK, curr & VM_ADDRMASK);
	newent= 0;
	while(curr)
	{
		tableno= curr >> VM_DIRSHIFT;
		table= get_phys_dword(page_base + 4*tableno);
		shadow= get_phys_dword(shadow_base + 4*tableno);
		assert((shadow & VM_PAGEMASK) == (curr & VM_PAGEMASK));

		table &= VM_ADDRMASK;
		assert(table);
		shadow &= VM_ADDRMASK;

		pageno= (curr & VM_DIRMASK) >> VM_PAGESHIFT;
		pageent= get_phys_dword(table + 4*pageno);
		shadowent= get_phys_dword(shadow + 4*pageno);

		assert((pageent & VM_ADDRMASK) == prev);
		if (newent == 0)
		{
			oldent= pageent & (VM_ADDRMASK | VM_USER |
				VM_WRITE | VM_PRESENT);
			if (next & VM_SWAPPAGE)
			{
				/* Page out, so not present */
				newent= (pageent & VM_PAGEMASK) | next;
				newent &= ~VM_PRESENT;
			}
			else
			{
				/* Page in, present iff not VM_WRITE */
				newent= (pageent & VM_PAGEMASK) | next;
				if (!(newent & VM_WRITE))
					newent |= VM_PRESENT;
			}
		}
		assert((pageent & (VM_ADDRMASK | VM_USER | VM_WRITE |
			VM_PRESENT)) == oldent);
		put_phys_dword(table + 4*pageno, newent);
		curr= shadowent;
	}
assert(check_list(get_page_info(next)));
}

/*
 * operations on the page table and page directories
 */

/*==========================================================================*
 *				map_table				    *
 *==========================================================================*/
PRIVATE void map_table(vir_addr, table, shadow, region_type)
u32_t vir_addr;
u32_t table;
u32_t shadow;
int region_type;
{
	unsigned tableno;

	assert(!(vir_addr & VM_DIRMASK));
	assert(!(table & VM_PAGEMASK));
	assert(!(shadow & VM_PAGEMASK));
	assert(!(region_type & VM_ADDRMASK));

	tableno= vir_addr >> VM_DIRSHIFT;
	shadow |= region_type;
	table |= VM_PRESENT | VM_WRITE | VM_USER;
	put_phys_dword(page_base + 4*tableno, table);
	put_phys_dword(shadow_base + 4*tableno, shadow);
}


/*==========================================================================*
 *				map_id_page				    *
 *==========================================================================*/
PRIVATE void map_id_page(real_vir_addr)
u32_t real_vir_addr;
{
	u32_t table;
	unsigned tableno, pageno;
#if DEBUG
	u32_t shadow;
#endif

	assert(!(real_vir_addr & VM_PAGEMASK));
	tableno= real_vir_addr >> VM_DIRSHIFT;

	table= get_phys_dword(page_base + 4*tableno);
	assert(table & VM_PRESENT);
	table &= VM_ADDRMASK;
	assert(table);
	pageno= (real_vir_addr & VM_DIRMASK) >> VM_PAGESHIFT;
	real_vir_addr |= VM_PRESENT | VM_WRITE | VM_USER;
	if (processor >= 486 && real_vir_addr >= 0xA0000 &&
		real_vir_addr < 0x100000)
	{
		real_vir_addr |= VM_PCD;
	}
	put_phys_dword(table + 4*pageno, real_vir_addr);
#if DEBUG
	/* Let's assert that the shadow entry is 0 */
	shadow= get_phys_dword(shadow_base + 4*tableno);
	assert((shadow & VM_PAGEMASK) == PT_ID_TABLE);
	shadow &= VM_ADDRMASK;
	assert(get_phys_dword(shadow + 4*pageno) == 0);
#endif
}


/*==========================================================================*
 *				map_page				    *
 *==========================================================================*/
PRIVATE void map_page(addr, virtual, flags)
u32_t addr;
u32_t virtual;
u32_t flags;
{
	u32_t table, shadow, pageent;
	unsigned tableno, pageno;
	int type;

	assert(!(virtual & VM_PAGEMASK));
	tableno= virtual >> VM_DIRSHIFT;
	table= get_phys_dword(page_base + 4*tableno);
	shadow= get_phys_dword(shadow_base + 4*tableno);

	table &= VM_ADDRMASK;
	assert(table != 0);
	type = shadow & VM_PAGEMASK;
	assert(isokindex(type-1));
	shadow &= VM_ADDRMASK;
	assert(shadow != 0);

	pageno= (virtual & VM_DIRMASK) >> VM_PAGESHIFT;
	assert(get_phys_dword(table + 4*pageno) == 0);
	assert(get_phys_dword(shadow + 4*pageno) == 0);

	assert(!(addr & VM_PAGEMASK));
	assert(!(flags & ~(VM_PRESENT|VM_WRITE|VM_USER)));

	pageent= get_page_info(addr);
	switch(pageent & VM_PAGEMASK)
	{
	case 0:
	case PT_FREEPAGE:
		break;
	default:
		put_phys_dword(shadow + 4*pageno, pageent);
		break;
	}
#if DEBUG & 256
 { printW(); printf("calling set_page_info\n"); }
#endif
	set_page_info(addr, type, virtual);
#if DEBUG & 256
 { printW(); printf("set_page_info done\n"); }
#endif
	put_phys_dword(table + 4*pageno, addr | flags);
#if DEBUG  & 256
 { printW(); printf("pagent: 0x%x: *(0x%x) := 0x%x\n", virtual, table+4*pageno,
	addr|flags); }
#endif
}


/*
 * public interface routines for process management.
 */

/*==========================================================================*
 *				vm_unmap_pages				    *
 *==========================================================================*/
PUBLIC void vm_unmap_pages(phys_base, phys_size)
phys_bytes phys_base;
phys_bytes phys_size;
{
	u32_t base, size, table, shadow, page, shadow_page,
		alloc_info;
	unsigned tableno, pageno;
	int type;

#if DEBUG & 256
 { printW(); printf("vm_unmap_pages(0x%x, 0x%x)\n", phys_base, phys_size); }
#endif

	base= phys_base & VM_ADDRMASK;
	size= ((phys_base + phys_size + VM_PAGESIZE-1) & VM_ADDRMASK) - base;
	assert(base == phys_base);
	assert(size == phys_size);

	while (size > 0)
	{
		tableno= base >> VM_DIRSHIFT;
		table= get_phys_dword(page_base + 4*tableno);
		shadow= get_phys_dword(shadow_base + 4*tableno);

		table &= VM_ADDRMASK;
		type= shadow & VM_PAGEMASK;
		shadow &= VM_ADDRMASK;
		if (table == 0)
		{
			if (size <= VM_DIRSIZE)
			{
				size= 0;
				break;
			}
			base += VM_DIRSIZE;
			size -= VM_DIRSIZE;
			vm_not_alloc += VM_DIRSIZE/VM_PAGESIZE;
			continue;
		}
		assert(table != 0 && shadow != 0);
		for (pageno= (base & VM_DIRMASK) >> VM_PAGESHIFT;
			pageno < VM_PAGESIZE/4 && size > 0;
			pageno++, base += VM_PAGESIZE, size -= VM_PAGESIZE)
		{
			vm_not_alloc++;
			page= get_phys_dword(table + 4*pageno);
			if (!page)
			{
				assert(get_phys_dword(shadow + 4*pageno) == 0);
				continue;
			}
			shadow_page= get_phys_dword(shadow + 4*pageno);
			page_dequeue(page & VM_ADDRMASK, base, page & VM_PAGEMASK,
					 type, shadow_page, &floating_pages);
			put_phys_dword(table + 4*pageno, 0);	/* Clear */
			if (!(page & VM_WRITE))
			{
				/* we might have to compensate for read/only
				 * pages
				 */
				alloc_info= get_page_info(page & VM_ADDRMASK);
				switch (alloc_info & VM_PAGEMASK)
				{
				case PT_FREEPAGE:
				case 0:
					break;
				default:
					vm_not_alloc--;
				}
			}
		}
	}
	assert(size == 0);
	level0(vm_reload);
}


/*==========================================================================*
 *				vm_unmap_space				    *
 *==========================================================================*/
PUBLIC void vm_unmap_space(phys_base, phys_size)
phys_bytes phys_base;
phys_bytes phys_size;
{
	u32_t base, size, table, shadow;
	unsigned tableno;
	int type;

#if DEBUG & 256
 { printW(); printf("vm_unmap_space(0x%x, 0x%x)\n", phys_base, phys_size); }
#endif

	base= phys_base & VM_ADDRMASK;
	size= ((phys_base + phys_size) & VM_ADDRMASK) - base;
	assert(base == phys_base);
	assert(size == phys_size);

	assert(!(base & VM_DIRMASK));
	size= (size + VM_DIRSIZE-1) & ~VM_DIRMASK;

	tableno= base >> VM_DIRSHIFT;
	for(;size > 0; size -= VM_DIRSIZE, tableno++)
	{
		table= get_phys_dword(page_base + 4*tableno);
		shadow= get_phys_dword(shadow_base + 4*tableno);

		vm_not_alloc += 2;		/* We paid for this space */
		if (!table)
		{
			assert(shadow);		/* We always have a type */
			assert(!(shadow & VM_ADDRMASK)); /* but no page */
			put_phys_dword(shadow_base + 4*tableno, 0);
			continue;
		}
		table &= VM_ADDRMASK;
		type= shadow & VM_PAGEMASK;
		shadow &= VM_ADDRMASK;
		assert(table != 0 && shadow != 0);

		put_phys_dword(page_base + 4*tableno, 0);
		put_phys_dword(shadow_base + 4*tableno, 0);

		mem_free(table, &locked_pages);
		mem_free(shadow, &locked_pages);
	}
	assert(size == 0);
assert(vm_check_stats());
}


/*==========================================================================*
 *				vm_remap_space				    *
 *==========================================================================*/
PUBLIC void vm_remap_space(phys_base, phys_size, srcproc, dstproc)
phys_bytes phys_base;
phys_bytes phys_size;
int srcproc;
int dstproc;
{
	u32_t base, size, table, shadow;
	unsigned tableno;
	int srctype, dsttype;

#if DEBUG & 256
 { printW(); printf("vm_remap_space(0x%x, 0x%x, %d, %d)\n",
	phys_base, phys_size, srcproc, dstproc); }
#endif

	base= phys_base & VM_ADDRMASK;
	size= ((phys_base + phys_size) & VM_ADDRMASK) - base;
	assert(base == phys_base);
	assert(size == phys_size);

	assert(!(base & VM_DIRMASK));
	size= (size + VM_DIRSIZE-1) & ~VM_DIRMASK;

	srctype= proc2vmtype(proc_addr(srcproc));
	dsttype= proc2vmtype(proc_addr(dstproc));

	tableno= base >> VM_DIRSHIFT;
	for(;size > 0; size -= VM_DIRSIZE, tableno++)
	{
		table= get_phys_dword(page_base + 4*tableno);
		shadow= get_phys_dword(shadow_base + 4*tableno);
		assert((shadow & VM_PAGEMASK) == srctype);
		if (!table)
		{
			assert(shadow);		/* We always have a type */
			assert(!(shadow & VM_ADDRMASK)); /* but no page */
		}
		shadow= (shadow & VM_ADDRMASK) | dsttype;
		put_phys_dword(shadow_base + 4*tableno, shadow);

	}
	assert(size == 0);
}


/*==========================================================================*
 *				vm_remap_pages				    *
 *==========================================================================*/
PUBLIC void vm_remap_pages(phys_base, phys_size, srcproc, dstproc)
phys_bytes phys_base;
phys_bytes phys_size;
int srcproc;
int dstproc;
{
	u32_t base, size, table, shadow, page, shadow_page;
	u32_t real_addr, pageent, nextent;
	unsigned tableno, pageno, tmp_tableno, tmp_pageno;
	int srctype, dsttype;

#if DEBUG & 256
 { printW(); printf("vm_remap_pages(0x%x, 0x%x, %d, %d)\n",
	phys_base, phys_size, srcproc, dstproc); }
#endif

	base= phys_base & VM_ADDRMASK;
	size= ((phys_base + phys_size + VM_PAGESIZE-1) & VM_ADDRMASK) - base;
	assert(base == phys_base);
	assert(size == phys_size);

	srctype= proc2vmtype(proc_addr(srcproc));
	dsttype= proc2vmtype(proc_addr(dstproc));

	while (size > 0)
	{
		tableno= base >> VM_DIRSHIFT;
		table= get_phys_dword(page_base + 4*tableno);
		shadow= get_phys_dword(shadow_base + 4*tableno);

		table &= VM_ADDRMASK;
		shadow &= VM_ADDRMASK;
		if (table == 0)
		{
			if (size <= VM_DIRSIZE)
			{
				size= 0;
				break;
			}
			base += VM_DIRSIZE;
			size -= VM_DIRSIZE;
			continue;
		}
		assert(table != 0 && shadow != 0);
		for (pageno= (base & VM_DIRMASK) >> VM_PAGESHIFT;
			pageno < VM_PAGESIZE/4 && size > 0;
			pageno++, base += VM_PAGESIZE, size -= VM_PAGESIZE)
		{
			page= get_phys_dword(table + 4*pageno);
			if (!page)
			{
				assert(get_phys_dword(shadow + 4*pageno) == 0);
				continue;
			}
			shadow_page= get_phys_dword(shadow + 4*pageno);

			real_addr= page & VM_ADDRMASK;
			pageent= get_page_info(real_addr);
			if (pageent == (base|srctype))
			{
				set_page_info(real_addr, dsttype, base);
				continue;
			}
			/* We have to find the virtual address that contains
			 * the pointer to the virtual address we want to
			 * remap.
			 */
			while(pageent)
			{
				tmp_tableno= pageent >> VM_DIRSHIFT;
				tmp_pageno= (pageent & VM_DIRMASK) >>
					VM_PAGESHIFT;

				assert(get_phys_dword(page_base +
					4*tmp_tableno) & VM_ADDRMASK);
				shadow= get_phys_dword(shadow_base +
					4*tmp_tableno);

				assert((shadow & VM_PAGEMASK) ==
					(pageent & VM_PAGEMASK));
				shadow &= VM_ADDRMASK;
				assert(shadow);

				nextent= get_phys_dword(shadow +
					4*tmp_pageno);
				if (nextent != (base|srctype))
				{
					pageent= nextent;
					continue;
				}
				put_phys_dword(shadow + 4*tmp_pageno,
					base|dsttype);
				break;
			}
			assert(pageent);
		}
	}
	assert(size == 0);
}


/*==========================================================================*
 *				vm_allocmap				    *
 *==========================================================================*/
PUBLIC void vm_allocmap(pp, base_clicks, size_clicks)
struct proc *pp;
phys_clicks base_clicks;
phys_clicks size_clicks;
{
	u32_t base, size;
	unsigned tablepages;

#if DEBUG & 256
 { printW(); printf("vm_allocmap(&proc[%d], 0x%x, 0x%x)\n", proc_number(pp),
	base_clicks, size_clicks); }
#endif
	base= base_clicks << CLICK_SHIFT;
	size= size_clicks << CLICK_SHIFT;
	assert(!(base & VM_DIRMASK));
	mk_space(pp, base, size);

	tablepages= (size + VM_DIRSIZE-1) >> VM_DIRSHIFT;
	vm_not_alloc -= 2*tablepages;
}


/*==========================================================================*
 *				vm_fork					    *
 *==========================================================================*/
PUBLIC void vm_fork(source_clicks, dest_clicks, size_clicks, ro)
phys_clicks source_clicks;
phys_clicks dest_clicks;
phys_clicks size_clicks;
int ro;
{
	u32_t source, dest, size, srctab, src_table, src_shadow,
		dst_table, dst_shadow, srcpage, next;
	unsigned src_tableno, dst_tableno, pageno;
	int src_type, dst_type;

#if DEBUG & 256
 { printW(); printf("in vm_fork(0x%x, 0x%x, 0x%x, %d)\n", source_clicks,
	dest_clicks, size_clicks, ro); }
#endif
	source= source_clicks << CLICK_SHIFT;
	dest= dest_clicks << CLICK_SHIFT;
	size= size_clicks << CLICK_SHIFT;

	assert((source & VM_DIRMASK) == (dest & VM_DIRMASK));
	while (size)
	{
		srctab= source & ~VM_DIRMASK;
		src_tableno= source >> VM_DIRSHIFT;
		dst_tableno= dest >> VM_DIRSHIFT;

		assert(src_tableno != 0 && dst_tableno != 0 &&
			src_tableno != dst_tableno);

		src_table= get_phys_dword(page_base + 4*src_tableno);
		src_shadow= get_phys_dword(shadow_base + 4*src_tableno);

		src_table &= VM_ADDRMASK;
		src_type= src_shadow & VM_PAGEMASK;
		src_shadow &= VM_ADDRMASK;
		if (!src_table)
		{
			assert(!src_shadow);
			if (size <= VM_DIRSIZE)
			{
assert(vm_check_stats());
				return;
			}
			size -= VM_DIRSIZE;
			source += VM_DIRSIZE;
			dest += VM_DIRSIZE;
			continue;
		}
		assert(src_shadow);

		dst_table= get_phys_dword(page_base + 4*dst_tableno);
		if (dst_table == 0)
		{
			mk_table(dest, 1);
			dst_table= get_phys_dword(page_base + 4*dst_tableno);
		}
		dst_shadow= get_phys_dword(shadow_base + 4*dst_tableno);
#if DEBUG  & 256
 { printW(); printf("dst_table 0x%x: *(0x%x)= 0x%x\n", dst_tableno,
	page_base+4*dst_tableno, dst_table); }
 { printW(); printf("dst_shadow 0x%x: *(0x%x)= 0x%x\n", dst_tableno,
	shadow_base+4*dst_tableno, dst_shadow); }
#endif

		dst_table &= VM_ADDRMASK;
		dst_type= dst_shadow & VM_PAGEMASK;
		dst_shadow &= VM_ADDRMASK;
		assert(dst_table);
		assert(dst_shadow);

		pageno= (source & VM_DIRMASK) >> VM_PAGESHIFT;
		for(; size && (source & ~VM_DIRMASK) == srctab;
				size -= VM_PAGESIZE, source += VM_PAGESIZE,
						dest += VM_PAGESIZE, pageno++)
		{
			assert(get_phys_dword(dst_table + 4*pageno) == 0);
			assert(get_phys_dword(dst_shadow + 4*pageno) == 0);

			srcpage= get_phys_dword(src_table + 4*pageno);
			next= get_phys_dword(src_shadow + 4*pageno);
			switch(srcpage & (VM_PRESENT|VM_WRITE|VM_SWAPPAGE))
			{
			case 0:
				/* Not present, read only, core memory.
				 * This should be a zero page (srcpage == 0)
				 * which is allowed iff !ro
				 */
				if (!ro)
					break;
#if DEBUG
 { printW(); }
#endif
				panic("zero page in text segment", source);
			case VM_PRESENT:
				/* Present, read only, core memory.
				 * This is allowed iff ro.
				 */
				if (!ro)
				{
#if DEBUG
 { printW(); }
#endif
					panic("read only page in rw fork",
									source);
				}
				map_page(srcpage & VM_ADDRMASK, dest,
							VM_USER|VM_PRESENT);
				break;
			case VM_WRITE:
				/* Not present, read/write, core memory.
				 * This is allowed iff !ro
				 */
				if (ro)
				{
#if DEBUG
 { printW(); }
#endif
					panic("copy on write page in ro fork",
									source);
				}
				map_page(srcpage & VM_ADDRMASK, dest,
							VM_USER|VM_WRITE);
				break;
			case VM_WRITE|VM_PRESENT:
				/* Present, read/write, core memory.
				 * if ro, this page should become read only,
				 * and if !ro, this page should become not
				 * present.
				 * In both cases, this page should be the only
				 * one in the list.
				 */
				assert(next == 0);
				assert(get_page_info(srcpage & VM_ADDRMASK) ==
					source | src_type);
				if (ro)
					srcpage &= ~VM_WRITE;
				else
					srcpage &= ~VM_PRESENT;
				put_phys_dword(src_table + 4*pageno, srcpage);
				map_page(srcpage & VM_ADDRMASK, dest,
						srcpage & (VM_USER|VM_WRITE|
								VM_PRESENT));
				break;
			case VM_SWAPPAGE:
				/* Not present, read only, swapped out.
				 * This is allowed iff ro.
				 */
				if (!ro)
				{
#if DEBUG
 { printW(); }
#endif
					panic("read only page in rw fork",
									source);
				}
				map_page(srcpage & VM_ADDRMASK, dest, VM_USER);
				break;
			case VM_SWAPPAGE|VM_WRITE:
				/* Not present, read/write, swapped out.
				 * if ro, this should be the only page in
				 * the list and it should be changed to
				 * read only. It can mapped as is if !ro.
				 */
				if (ro)
				{
					assert(next == 0);
					assert(get_page_info(srcpage &
						VM_ADDRMASK) == source |
								src_type);
					srcpage &= ~VM_WRITE;
					put_phys_dword(src_table + 4*pageno,
								srcpage);
				}
				map_page(srcpage & VM_ADDRMASK, dest,
						srcpage & (VM_USER|VM_WRITE));
				break;
			/* The following cases are impossible.
			 * case VM_SWAPPAGE|VM_WRITE|VM_PRESENT:
			 * case VM_SWAPPAGE|VM_PRESENT:
			 */
			default:
#if DEBUG
 { printW(); }
#endif
				panic("invalid mode in page table entry",
									source);
			}
			if (ro)
			{
				/* We paid to much */
				vm_not_alloc++;
			}
		}
	}
}


/*==========================================================================*
 *				vm_lock					    *
 *==========================================================================*/
PUBLIC int vm_lock(phys_addr, phys_count, force_lock)
phys_bytes phys_addr;
phys_bytes phys_count;
int force_lock;
{
	u32_t base, top, orig_base;
	u32_t table, page;
	volatile u32_t dummy;
	unsigned tableno, pageno;
	int r;

	base= phys_addr & VM_ADDRMASK;
	top= (phys_addr + phys_count + VM_PAGESIZE-1) & VM_ADDRMASK;

	orig_base= base;
	while(base < top)
	{
		tableno= (base >> VM_DIRSHIFT);
		table= get_phys_dword(page_base + 4*tableno);
		if (!(table & VM_PRESENT))
		{
			if (!force_lock)
			{
				vm_unlock(orig_base, base-orig_base);
				return E_NOPAGE;
			}
			dummy= get_phys_dword(base);
			table= get_phys_dword(page_base + 4*tableno);
			assert (table & VM_PRESENT);
		}
		table &= VM_ADDRMASK;
		for (pageno= (base & VM_DIRMASK) >> VM_PAGESHIFT;
			base<top && pageno < VM_PAGESIZE/4;
			pageno++, base += VM_PAGESIZE)
		{
			page= get_phys_dword(table + 4*pageno);
			if (!(page & VM_PRESENT))
			{
				if (!force_lock)
				{
					vm_unlock(orig_base, base-orig_base);
					return E_NOPAGE;
				}
				dummy= get_phys_dword(base);
				page= get_phys_dword(table + 4*pageno);
				assert (page & VM_PRESENT);
			}
			if (!is_dmapage(page))
			{
				if (!force_lock)
				{
					vm_unlock(orig_base, base-orig_base);
					return E_NOPAGE;
				}
				page= move2dma(page);
			}
			r= lock_page(page & VM_ADDRMASK, &floating_pages);
			if (r != OK)
			{
				vm_unlock(orig_base, base-orig_base);
				return r;
			}
		}
	}
	return OK;
}


/*==========================================================================*
 *				vm_unlock					    *
 *==========================================================================*/
PUBLIC void vm_unlock(phys_addr, phys_count)
phys_bytes phys_addr;
phys_bytes phys_count;
{
	u32_t base, top, orig_base;
	u32_t table, page;
	unsigned tableno, pageno;

	base= phys_addr & VM_ADDRMASK;
	top= (phys_addr + phys_count + VM_PAGESIZE-1) & VM_ADDRMASK;

	orig_base= base;
	while(base < top)
	{
		tableno= (base >> VM_DIRSHIFT);
		table= get_phys_dword(page_base + 4*tableno);
		assert (table & VM_PRESENT);
		table &= VM_ADDRMASK;
		for (pageno= (base & VM_DIRMASK) >> VM_PAGESHIFT;
			base<top && pageno < VM_PAGESIZE/4;
			pageno++, base += VM_PAGESIZE)
		{
			page= get_phys_dword(table + 4*pageno);
			assert (page & VM_PRESENT);
			unlock_page(page & VM_ADDRMASK, &floating_pages);
		}
	}
}


/*==========================================================================*
 *				vm_ismapped				    *
 *==========================================================================*/
PUBLIC int vm_ismapped(addr)
u32_t addr;
{
	unsigned tableno, pageno;
	u32_t pageent, table;

	tableno= addr >> VM_DIRSHIFT;
	table= get_phys_dword(page_base + 4*tableno);
	if (table == 0)
		return 0;
	table &= VM_ADDRMASK;
	assert(table != 0);
	pageno= (addr & VM_DIRMASK) >> VM_PAGESHIFT;
	pageent= get_phys_dword(table + 4*pageno);
	if (pageent == 0)
		return 0;
	return 1;
}


/*
 * the page fault handler
 */

/*==========================================================================*
 *				vm_page_fault				    *
 *==========================================================================*/
PUBLIC void vm_page_fault(err, addr, fault_proc)
u32_t err;
u32_t addr;
struct proc *fault_proc;
{
	struct proc *pp, *app;
	int int_level, i, r, type;
	u32_t *pages_p, table, shadow, page, next, newpage;
	u32_t *sp;
	unsigned tableno, pageno;
	phys_clicks addr_click;

	if (cpmess_stackptr)
	{
		sp= (u32_t *)cpmess_stackptr;

		/* Reset cpmess_stackptr to prevent recursion in case of
		 * a bug.
		 */
		cpmess_stackptr= 0;

		/* Save the fault address */
		cpmess_faultaddr= addr;

		/* Normally, we expect k_reenter to be equal to 1. This
		 * happens when a system call causes the fault.
		 * k_reenter is equal to 0 when the fault is caused by
		 * the restart of a system call.
		 * Unfortunately, the location of the pc differs in those
		 * two situations.
		 */
		if (k_reenter == 1)
		{
#if 0
			printf("vm_page_fault: during cpmess\n");
			printf("sp= 0x%x, addr= 0x%x, f= 0x%x, e= 0x%x\n",
				sp, addr, (unsigned)cpmess_unchecked,
				(unsigned)cpmess_end);
			printf("sp[-1]= 0x%x\n", sp[-1]);
			printf("sp[-2]= 0x%x\n", sp[-2]);
			printf("sp[-3]= 0x%x\n", sp[-3]);
			printf("sp[-4]= 0x%x\n", sp[-4]);
			printf("sp[-5]= 0x%x\n", sp[-5]);
#endif

			sp[-3]= (u32_t)cpmess_end;
		}
		else if (k_reenter == 0)
		{
			/* The program counter is stored in the process
			 * table.
			 */
			pp= proc_ptr;

#if 0
			printf("vm_page_fault: during cpmess\n");
			printf("pp= &[%d], addr= 0x%x, f= 0x%x, e= 0x%x\n",
				proc_number(pp), addr,
				(unsigned)cpmess_unchecked,
				(unsigned)cpmess_end);
			printf("p_reg.sf_pc= 0x%x\n", pp->p_reg.sf_pc);
#endif

			pp->p_reg.sf_pc= (u32_t)cpmess_end;
		}
		else
		{
			panic("vm_page_fault: strange value for k_reenter",
				k_reenter);
		}
		return;
	}

	if (!fault_proc)
	{
		int_level= TRUE;
		pp= proc_ptr;
		unlock();			/* If we unlock earlier,
						 * proc_ptr might be changed
						 * by a clock_tick...
						 */
		compare(k_reenter, ==, 0);
						/* Interrupt handlers don't
						 * cause page faults */

#if DEBUG
		assert(pp != proc_addr(pager_tasknr) ||
		(printW(), printf(
			"process %d got a page fault at 0x%x due to a %s.\n",
		proc_number(pp), addr,
		(err & 1) ? "protection violation" : "not present page"),
		printf(
		"The process was running in %s mode and issuing a %s.\n",
		(err & 4) ? "user" : "supervisor",
		(err & 2) ? "write" : "read"),
		stacktrace(pp), 0));
		assert(vm_check_stats() ? 1 :
			(printf("task: %d\n", proc_number(pp)), 0));
#endif
	}
	else
	{
		int_level= FALSE;
		pp= fault_proc;
	}

	assert(vm_check_stats());

#if DEBUG
	assert (addr >= paging_base ||
		(printW(),
		printf("process %d got a page fault at 0x%x due to a %s.\n",
			proc_number(pp), addr,
			(err & 1) ? "protection violation" :
			"not present page"),
		printf(
		"The process was running in %s mode and issuing a %s.\n",
		(err & 4) ? "user" : "supervisor",
		(err & 2) ? "write" : "read"),
		printf("pc= 0x%x\n", pp->p_reg.sf_pc), 0));
#endif

	/* Fetch two empty pages. */
	if (int_level)
	{
		for (i= 0; i<2; i++)
		{
			if (intl_pages[i] != 0)
				continue;	/* We already have a page */
			intl_pages[i]= coremem_alloc(&locked_pages);
			if (intl_pages[i] == 0)
			{
				suspend(pp, addr, err);
				return;
			}
		}
		pages_p= intl_pages;
	}
	else
	{
		for (i= 0; i<2; i++)
		{
			if (taskl_pages[i] != 0)
				continue;	/* We already have a page */
			taskl_pages[i]= coremem_alloc(&locked_pages);
			if (taskl_pages[i] == 0)
			{
				pages2swap(fault_proc);
				sweeper_round= 0;	/* Restart sweeper */
				taskl_pages[i]= coremem_alloc(&locked_pages);
				assert(taskl_pages[i] != 0);
				if (page_ptr == NULL)
				{
					/* Our process died when we were
					 * aiting for free pages, return
					 * (and continue with the next
					 * one).
					 */
					return;
				}
			}
		}
		pages_p= taskl_pages;
	}

	/* Look at the page table */
	tableno= addr >> VM_DIRSHIFT;
	table= get_phys_dword(page_base + 4*tableno);
	shadow= get_phys_dword(shadow_base + 4*tableno);

	table &= VM_ADDRMASK;
	type= shadow & VM_PAGEMASK;
	shadow &= VM_ADDRMASK;

	if (!isokindex(type-1))
	{
		printf(
		"vm386: process %d got a page fault at 0x%x due to a %s.\n",
			proc_number(pp), addr,
			(err & 1) ? "protection violation" :
			"not present page");
		printf(
		"The process was running in %s mode and issuing a %s.\n",
			(err & 4) ? "user" : "supervisor",
			(err & 2) ? "write" : "read");
		stacktrace(pp);
		panic("vm386: invalid type:", type);
	}

	app= &proc[type-1];

	app->p_status &= ~(P_ST_UNRDY_1|P_ST_UNRDY_2);

	if (table == 0)
	{
		/* No page table yet */
		assert(shadow == 0);

		table= pages_p[0];
		shadow= pages_p[1];
		pages_p[0]= 0;
		pages_p[1]= 0;
		assert(table != 0 && shadow != 0);
		phys_clr_page(table);
		phys_clr_page(shadow);

		map_table(addr & ~VM_DIRMASK, table, shadow, type);


		/* We simply restart this fault, and expect another
		 * trap on the same location for the missing page.
		 * It is possible to also allocate the page here, but
		 * this situation doesn't occur very often.
		 */

		return;
	}

	assert(table != 0 && shadow != 0);

	pageno= (addr & VM_DIRMASK) >> VM_PAGESHIFT;
	page= get_phys_dword(table + 4*pageno);

	if (page & VM_PRESENT)
	{
		/* We got a page fault, but no reason for the fault
		 * since the page in present and protection is done
		 * by the segmentation scheme.
		 * However, two process may get queued for the same
		 * page that is swapped out.
		 */
		if (!int_level)
		{
			return;
		}
#if DEBUG
 { printW(); }
#endif
		panic("page fault on a present page", NO_NUM);
	}

	/* check zero pages first. */
	if (page == 0)
	{
		addr_click= addr >> CLICK_SHIFT;
		if (pp != app)
		{
			/* If copying, addresses are checked by umap, so
			 * there is no reason to grow the stack, and the
			 * addr should be within the processes maps.
			 */
#if !NDEBUG
			if (!((addr_click >= app->p_map[SEG_T].mem_phys+
				app->p_map[SEG_T].mem_vir && addr_click <
				app->p_map[SEG_T].mem_phys+
				app->p_map[SEG_T].mem_vir+
				app->p_map[SEG_T].mem_len) ||
				(addr_click >= app->p_map[SEG_D].mem_phys+
				app->p_map[SEG_D].mem_vir && addr_click <
				app->p_map[SEG_D].mem_phys+
				app->p_map[SEG_D].mem_vir+
				app->p_map[SEG_D].mem_len) ||
				(addr_click >= app->p_map[SEG_S].mem_phys+
				app->p_map[SEG_S].mem_vir && addr_click <
				app->p_map[SEG_S].mem_phys+
				app->p_map[SEG_S].mem_vir+
				app->p_map[SEG_S].mem_len)))
			{
				printf(
			"Process %d got a page fault at 0x%x due to a %s.\n",
					proc_number(pp), addr,
					(err & 1) ? "protection violation" :
						"not present page");
				printf("int_level= %d\n", int_level);
				stacktrace(pp);
				printf(
		"The process was running in %s mode and issuing a %s.\n",
					(err & 4) ? "user" : "supervisor",
					(err & 2) ? "write" : "read");
				printf("app= %d\n", proc_number(app));
				stacktrace(app);
				assert(0);
			}
#endif
		}
		else
		{
			/* We can't have a zero entry below the text
			 * segment.
			 */
			if (addr_click < app->p_map[SEG_T].mem_phys)
			{
#if DEBUG
 { printW(); }
#endif
				panic("page fault outside address space",
								NO_NUM);
			}

			/* A zero entry can be in the first page of the
			 * text segment. We don't check if common I&D.
			 */
			if (app->p_map[SEG_T].mem_len &&
				addr_click >= app->p_map[SEG_T].mem_phys &&
				addr_click < app->p_map[SEG_T].mem_phys+
				app->p_map[SEG_T].mem_vir)
			{
#if !DEBUG
				if (!isuserp(pp)) {
#endif
 { printW(); printf("process %d got a page fault at 0x%x due to a %s.\n",
	proc_number(pp), addr, (err & 1) ? "protection violation" :
	"not present page");
	printf("The process was running in %s mode and issuing a %s.\n",
	(err & 4) ? "user" : "supervisor", (err & 2) ? "write" : "read");
	stacktrace(pp); }
#if !DEBUG
				}
#endif
				cause_sig(proc_number(app), SIGSEGV);
				return;
			}

			/* We can't have a page fault below the data segment
			 * other than the two mentioned above.
			 */
			if (addr_click < app->p_map[SEG_D].mem_phys)
			{
#if DEBUG
 { printW(); printf("addr_click= 0x%x, mem_phys= 0x%x, k_reenter= %d\n",
	addr_click, app->p_map[SEG_D].mem_phys, k_reenter); }
#endif
				panic("page fault outside address space",
								NO_NUM);
			}

			/* A zero entry can be in the first page of the
			 * data segment.
			 */
			if (addr_click >= app->p_map[SEG_D].mem_phys &&
				addr_click <
					app->p_map[SEG_D].mem_phys+
					app->p_map[SEG_D].mem_vir)
			{
#if !DEBUG
				if (!isuserp(app)) {
#endif
 { printW(); printf(
	"process %d has a page fault in the zero data page, offset 0x%x\n",
	proc_number(app), addr-(app->p_map[SEG_D].mem_phys << CLICK_SHIFT)); }
 { printW(); printf("process %d got a page fault at 0x%x due to a %s.\n",
	proc_number(pp), addr, (err & 1) ? "protection violation" :
	"not present page");
	printf("The process was running in %s mode and issuing a %s.\n",
	(err & 4) ? "user" : "supervisor", (err & 2) ? "write" : "read");
	stacktrace(pp); }
#if !DEBUG
				}
#endif
				cause_sig(proc_number(app), SIGSEGV);
				return;
			}

			/* One other reason for a panic is an addr above
			 * the stack segment.
			 */
			if (addr_click >= app->p_map[SEG_S].mem_phys+
				app->p_map[SEG_S].mem_vir+
				app->p_map[SEG_S].mem_len)
			{
#if DEBUG
 { printW(); }
#endif
				panic("page fault outside address space",
								NO_NUM);
			}

			/* Now the address is in a part of the user address
			 * space where we might put a new page. We allocate
			 * a new page if the address is in the data segment, or
			 * if the address is in the stack segment.
			 */

			/* If the address is between the data segment and the
			 * stack segment we might need to adjust to stack
			 * segment.
			 */
			if (addr_click >= app->p_map[SEG_D].mem_phys+
				app->p_map[SEG_D].mem_vir+
				app->p_map[SEG_D].mem_len &&
				addr_click < app->p_map[SEG_S].mem_phys+
				app->p_map[SEG_S].mem_vir)
			{
				adjust_user_stack(app);
				if (addr_click >= app->p_map[SEG_D].mem_phys+
					app->p_map[SEG_D].mem_vir+
					app->p_map[SEG_D].mem_len &&
					addr_click < app->p_map[SEG_S].mem_phys+
					app->p_map[SEG_S].mem_vir)
				{
					/* Either the address is realy bad or
					 * we are out of memory and can't grow
					 * the stack.
					 */
#if !DEBUG
					if (!isuserp(app)) {
#endif
 { printW(); printf(
	"process %d has a bad pointer in the data segment, offset 0x%x\n",
	proc_number(app), addr-(app->p_map[SEG_D].mem_phys << CLICK_SHIFT));
	printf("The process was running in %s mode and issuing a %s.\n",
	(err & 4) ? "user" : "supervisor", (err & 2) ? "write" : "read");
	stacktrace(pp); }
#if !DEBUG
					}
#endif
					cause_sig(proc_number(app), SIGSEGV);
					return;
				}
			}
		}

		if ((app->p_status & P_ST_VM_INCORE) &&
			int_level && !is_dmapage(pages_p[0]))
		{
			suspend(pp, addr, err);
			return;
		}

		/* Now we can map the new page. */
		page= pages_p[0];
		pages_p[0]= 0;

		assert(locked_pages > 0);
		locked_pages--;
		floating_pages++;

		assert(page);
		phys_clr_page(page);
		map_page(page, addr & VM_ADDRMASK,
						VM_PRESENT|VM_WRITE|VM_USER);

		if (app->p_status & P_ST_VM_INCORE)
		{
			assert (is_dmapage(page) || !int_level);
			if (!is_dmapage(page))
				page= move2dma(page);

			/* Lock the page */
			r= lock_page(page, &floating_pages);
			assert (r == OK);
		}
		return;
	}

	/* Now we have an existing entry, so it must be within the process'
	 * address space.
	 */
	addr_click= addr >> CLICK_SHIFT;
	assert((addr_click >= app->p_map[SEG_T].mem_phys+
					app->p_map[SEG_T].mem_vir &&
		addr_click < app->p_map[SEG_T].mem_phys+
					app->p_map[SEG_T].mem_vir+
					app->p_map[SEG_T].mem_len) ||
		(addr_click >= app->p_map[SEG_D].mem_phys+
					app->p_map[SEG_D].mem_vir &&
		addr_click < app->p_map[SEG_D].mem_phys+
					app->p_map[SEG_D].mem_vir+
					app->p_map[SEG_D].mem_len) ||
		(addr_click >= app->p_map[SEG_S].mem_phys+
					app->p_map[SEG_S].mem_vir &&
		addr_click <  app->p_map[SEG_S].mem_phys+
					app->p_map[SEG_S].mem_vir+
					app->p_map[SEG_S].mem_len));

	/* We have two options left, a page that is swapped out, or a copy
	 * on access page. Since a page that is swapped in may be marked
	 * copy on access, we do the swapping part first.
	 */
	if (page & VM_SWAPPAGE)
	{
		if (int_level)
		{
			suspend(pp, addr, err);
			return;
		}

		/* Transfer the page to memory */
		page &= VM_ADDRMASK;
		newpage= pages_p[0];
		if (!is_dmapage(newpage) && free_dmacore_mem)
		{
			newpage= dmamem_alloc(&locked_pages);
			assert(newpage != 0);
		}
		else
		{
			/* Allocate the page */
			pages_p[0]= 0;
		}

#if DEBUG & 256
		assert(check_disk_pages());
#endif

		transfer_page(page & VM_ADDRMASK, newpage);
		if (page_ptr == NULL)
		{
			/* Our process died while we fetched the page.
			 * We can savely discard the page. */
			mem_free(newpage, &locked_pages);
			return;
		}
		remap_pages(page, newpage);
		mem_free(page, &locked_pages);
		return;
	}

	/* Now we must have a copy on access page. */
	compare((page & (VM_SWAPPAGE|VM_USER|VM_WRITE|VM_PRESENT)), ==,
		(VM_USER|VM_WRITE));

	/* If we are the only user of the page, there is no need to copy. */
	next= get_phys_dword(shadow + 4*pageno);
	if (next == 0 && get_page_info(page & VM_ADDRMASK) ==
						((addr & VM_ADDRMASK) | type))
	{
		page |= VM_PRESENT;
		put_phys_dword(table + 4*pageno, page);
		return;
	}

	/* We dequeue our selves. And clear out entry in the page table */
	assert(!istaskp(app));
	page_dequeue(page & VM_ADDRMASK, addr & VM_ADDRMASK,
			page & VM_PAGEMASK, type, next, &floating_pages);
	put_phys_dword(table + 4*pageno, 0);
	put_phys_dword(shadow + 4*pageno, 0);

	newpage= pages_p[0];
	pages_p[0]= 0;
	assert(locked_pages > 0);
	locked_pages--;
	floating_pages++;
	assert(newpage);
	phys_copy(page & VM_ADDRMASK, newpage, VM_PAGESIZE);
	map_page(newpage, addr & VM_ADDRMASK, VM_PRESENT|VM_WRITE|VM_USER);
}


/*==========================================================================*
 *				adjust_user_stack			    *
 *==========================================================================*/
PRIVATE void adjust_user_stack(pp)
struct proc *pp;
{
	u32_t sp;
	phys_clicks sp_click, new_clicks;
	struct mem_map *mp;

	sp= pp->p_reg.sf_sp;
	sp_click= sp >> CLICK_SHIFT;

	mp= pp->p_map;
	if (sp_click >= mp[SEG_S].mem_vir+mp[SEG_S].mem_len ||
		sp_click < mp[SEG_D].mem_vir+mp[SEG_D].mem_len)
	{
#if DEBUG
 { printW(); printf("process %d: strange value in sp: 0x%x\n", proc_number(pp),
	sp); dump_map(mp); }
#endif /* DEBUG */
		return;
	}
	sp_click--;	/* Allocate one extra click because the current
			 * intruction might try to push sp over the edge.
			 */
	if (sp_click < mp[SEG_D].mem_vir+mp[SEG_D].mem_len+1)
	{
#if DEBUG
 { printW(); printf("process %d: stack full, sp= 0x%x\n", proc_number(pp),
	sp); }
#endif /* DEBUG */
		cause_sig(proc_number(pp), SIGSEGV);
		return;
	}
	if (sp_click >= mp[SEG_S].mem_vir)
	{
		/* No adjustment needed. */
		return;
	}
	new_clicks= mp[SEG_S].mem_vir - sp_click;
	if (vm_not_alloc >= new_clicks)
	{
		vm_not_alloc -= new_clicks;
		mp[SEG_S].mem_vir -= new_clicks;
		mp[SEG_S].mem_len += new_clicks;
		return;
	}

	/* Now have a problem, we want to grow the stack but we are out of
	 * memory. We kill a user process, but a system process may continue
	 * until a kernel panic.
	 */
	if (isuserp(pp))
	{
#if DEBUG
 { printW(); printf("process %d: out of memory, sp= 0x%x\n", proc_number(pp),
	sp); }
#endif /* DEBUG */
		cause_sig(proc_number(pp), SIGSEGV);
		return;
	}
#if DEBUG
 { printW(); printf("system process %d: out of memory, sp= 0x%x, (continued)\n",
	proc_number(pp),  sp); }
#endif /* DEBUG */
	vm_not_alloc -= new_clicks;
	mp[SEG_S].mem_vir -= new_clicks;
	mp[SEG_S].mem_len += new_clicks;
	return;
}


/*==========================================================================*
 *				suspend					    *
 *==========================================================================*/
PRIVATE void suspend(pp, addr, err)
struct proc *pp;
u32_t addr;
u32_t err;
{
	assert(!cpmess_stackptr);
	assert(pp != proc_addr(pager_tasknr) && pp != cproc_addr(HARDWARE));

	pp->p_faultaddr= addr;
	pp->p_faulterr= err;
	pp->p_pagefq= pagefq_head;
	pagefq_head= pp;
	if (!pager_running)
		interrupt(pager_tasknr);
	pp->p_flags |= P_PAGEFAULT;
	lock_unready(pp);
}


/*==========================================================================*
 *				do_pagefault				    *
 *==========================================================================*/
PRIVATE void do_pagefault()
{
	struct proc *currp, *prevp, *prev_ptr;

	if (pagefq_head->p_pagefq == NULL)
	{
		page_ptr= pagefq_head;
		pagefq_head= NULL;
	}
	else
	{
		page_ptr= NULL;
		for (prevp= NULL, currp= pagefq_head; currp;
			prevp= currp, currp= currp->p_pagefq)
		{
			if (page_ptr == NULL)
			{
				page_ptr= currp;
				prev_ptr= prevp;
				continue;
			}
			if (currp->p_complr > page_ptr->p_complr ||
				currp > page_ptr)
			{
				continue;
			}
			page_ptr= currp;
			prev_ptr= prevp;
		}
		if (prev_ptr == NULL)
			pagefq_head= page_ptr->p_pagefq;
		else
			prev_ptr->p_pagefq= page_ptr->p_pagefq;
		/* No we have the process with highest priority (== least
		 * used a quotum). Note that system process will have to
		 * highest priority because they never use a quotum
		 * (p_complr == 0) and their index in the process table
		 * is lower than that of the user processes.
		 */
	}
	assert(page_ptr->p_flags & P_PAGEFAULT);
	vm_page_fault(page_ptr->p_faulterr, page_ptr->p_faultaddr, page_ptr);
	if (page_ptr)
	{
		page_ptr->p_flags &= ~P_PAGEFAULT;
		if (page_ptr->p_flags & P_RSTSYS)
			sys_restart(page_ptr);
		lock_ready(page_ptr);
	}
}

/*
 * public swapping related routines.
 */

/*==========================================================================*
 *				vm_swapon				    *
 *==========================================================================*/
PUBLIC int vm_swapon(pnr, minor, offset, size, priority)
int pnr;
int minor;
u32_t offset;
u32_t size;
i32_t priority;
{
	int dummy_account;
	u32_t top, phys_space, vaddr, info;
	swapdesc_t *sd;

assert(vm_check_stats());
	/* Adjust offset and size to page boundaries. */
	top= offset + size;
	offset= (offset + VM_PAGESIZE-1) & VM_ADDRMASK;
	top &= VM_ADDRMASK;

	dummy_account= size >> VM_PAGESHIFT;

	while (offset < top)
	{
		sd= get_swapdesc(pnr, minor, offset & ~VM_DIRMASK, priority);
		if (!sd)
			return ENOMEM;
		phys_space= sd->sd_phys_space;
		vaddr= phys_space | (offset & VM_DIRMASK);
		while(offset < top)
		{
			info= get_page_info(vaddr);
			if (info != PT_NOSWAP)
			{
				/* Someone tries to mark some disk space as
				 * swap space for the second time...
				 */
assert(vm_check_stats());
				return EBUSY;
			}
			mem_free(vaddr, &dummy_account);
			total_swapmem++;
			vm_not_alloc++;
			vm_lowest_free++;
			offset += VM_PAGESIZE;
			vaddr += VM_PAGESIZE;
			if (!(offset & VM_DIRMASK))
				break;	/* New segment */
		}
	}
assert(vm_check_stats());
assert(check_free_list());
	return OK;
}


/*==========================================================================*
 *				vm_swapoff				    *
 *==========================================================================*/
PUBLIC int vm_swapoff(segm)
int segm;
{
	swapdesc_t *sd, *prev, *sd_p, *best, *best_prev;
	u32_t map, o, page;
	unsigned pages, free_pages;
	unsigned type;

	if (segm < 0 || segm >= NR_SWAPSEGS)
		return EINVAL;

	assert(check_disk_pages());

	sd= &swaptable[segm];
	if (!(sd->sd_flags & SDF_INUSE))
		return EINVAL;

	/* Count the number of pages in this segment to see if we
	 * have enough unresevered memory available.
	 */
	map= sd->sd_map;
	pages= 0;
	for (o= 0; o<VM_PAGESIZE; o += 4)
	{
		page= get_phys_dword(map+o);
		if ((page & VM_PAGEMASK) != PT_NOSWAP)
			pages++;
	}
#if DEBUG
 { printW(); printf("segm %d has %d pages\n", segm, pages); }
#endif
	if (vm_not_alloc < pages)
		return ENOMEM;
	vm_not_alloc -= pages;

	/* Give the segment a very low priority */
	sd->sd_priority= SDP_MIN;

	/* If the segment is the head of the queue we have look for a new
	 * head. */
	if (sd->sd_flags & SDF_INFREEQ)
	{
		for (prev= 0, sd_p= swap_freehead; sd_p;
					prev= sd_p, sd_p= sd_p->sd_nextfree)
		{
			if (sd_p == sd)
				break;
		}
		assert(sd_p);
		if (prev == 0)
		{
			best= NULL;
			best_prev= NULL;
			for (prev= 0, sd_p= swap_freehead; sd_p;
					prev= sd_p, sd_p= sd_p->sd_nextfree)
			{
				if (best == NULL)
				{
					best= sd_p;
					best_prev= prev;
					continue;
				}
				if (sd_p->sd_priority <= best->sd_priority)
					continue;
				best= sd_p;
				best_prev= prev;
			}
			if (best && best_prev)
			{
				best_prev->sd_nextfree= best->sd_nextfree;
				best->sd_nextfree= swap_freehead;
				swap_freehead= best;
				if (best_prev->sd_nextfree == NULL)
					swap_freetail= best_prev;
			}
		}
	}

	/* Now we are going to free the segment by marking the free pages
	 * as hardware and forcing the used pages into memory.
	 */
	free_pages= 0;
	for (o= 0; o<VM_PAGESIZE; o += 4)
	{
		page= get_phys_dword(map+o);
		assert (page == get_page_info(sd->sd_phys_space +
			((o >> 2) << VM_PAGESHIFT)));

		type= page & VM_PAGEMASK;
		if (type == PT_NOSWAP)
			continue;
		if (type != PT_FREEPAGE && type != 0)
		{
			page &= VM_ADDRMASK;

			assert(type < PT_SPECIAL_PAGES);

			/* Force the page into memory */
			sink= get_phys_dword(page);

			page= get_phys_dword(map+o);

			type= page & VM_PAGEMASK;
			assert(type == PT_FREEPAGE || type == 0);
		}
		while(sd->sd_freehead)
		{
			page= sd->sd_freehead;
			assert((page & VM_PAGEMASK) == PT_FREEPAGE);
			page &= VM_ADDRMASK;
			sd->sd_freehead= get_page_info(page);
			assert(free_swapmem > 0);
			free_swapmem--;
			free_pages++;
			total_swapmem--;
			set_page_info(page, PT_NOSWAP, (u32_t)0);
		}
	}
	compare(free_pages, ==, pages);

	/* Now we can remove the segment from the queue */
	if (sd->sd_flags & SDF_INFREEQ)
	{
		for (prev= 0, sd_p= swap_freehead; sd_p;
					prev= sd_p, sd_p= sd_p->sd_nextfree)
		{
			if (sd_p == sd)
				break;
		}
		assert(sd_p);
		if (prev == 0)
			swap_freehead= sd_p->sd_nextfree;
		else
		{
			prev->sd_nextfree= sd_p->sd_nextfree;
			if (prev->sd_nextfree == 0)
				swap_freetail= prev;
		}
		sd->sd_flags &= ~SDF_INFREEQ;
	}

	assert(check_disk_pages());

	mem_free(sd->sd_map, &locked_pages);
	sd->sd_map= 0;
	vm_not_alloc++;
	sd->sd_flags= 0;
	return 0;
}


/*==========================================================================*
 *				vm_swapinfo				    *
 *==========================================================================*/
PUBLIC int vm_swapinfo(pnr, minor, index_p)
int pnr;
int minor;
i32_t *index_p;
{
	int i;
	swapdesc_t *sd;

	for (i= 0, sd= swaptable; i<NR_SWAPSEGS; i++, sd++)
	{
		if (!(sd->sd_flags & SDF_INUSE))
			continue;
		if (sd->sd_proc != pnr || sd->sd_minor != minor)
			continue;
		*index_p= i;
		return OK;
	}
	return ESRCH;
}


/*==========================================================================*
 *				pager_task				    *
 *==========================================================================*/
PUBLIC void pager_task()
{
	message m;
	int r;

	pager_tasknr= proc_number(proc_ptr);
	clck_tasknr= findproc(CLOCK_NAME);

	pager_running= TRUE;

	for(;;)
	{
		pager_running= FALSE;
		r= receive(ANY, &m);
		assert(r == OK);
		pager_running= TRUE;
		if (m.m_type == PAGER_SWAPOUT)
		{
			assert(m.m_source == SYSTASK);
			assert(isokprocn(m.m1_i1));
			pages2swap(proc_addr(m.m1_i1));
			m.m1_i1= OK;
			r= send(m.m_source, &m);
			assert(r == OK);
			continue;
		}
		if (0) assert(m.m_source == HARDWARE && m.m_type == HARD_INT);
		while(pagefq_head)
		{
			do_pagefault();
			if (sweeper_round == 0 && !sweeper_running)
				sweeper();
		}
		if (!sweeper_running && sweeper_round < SWEEPER_MAX)
		{
			sweeper_round++;
			sweeper();
		}

		if (m.m_type == PAGER_SYNC)
		{
			/* Force handling of current pagefaults. All
			 * pagefaults that were present when the request was
			 * sent should be handled by now.
			 */
			assert(m.m_source == SYSTASK);
			m.m1_i1= OK;
			r= send(m.m_source, &m);
			assert(r == OK);
		}
	}
}


/*==========================================================================*
 *				vm_suspend				    *
 *==========================================================================*/
PUBLIC void vm_suspend(pp, addr)
struct proc *pp;
u32_t addr;
{
	suspend(pp, addr, 0);
}


/*==========================================================================*
 *				vm_phys2bus				    *
 *==========================================================================*/
int vm_phys2bus(pp, tab, tabsize)
struct proc *pp;
u32_t *tab;
size_t tabsize;
{
	size_t i;
	u32_t phys;
	unsigned tableno, pageno, type;
	u32_t table, shadow, page, real;


	for (i= 0; i < tabsize; i++, tab++)
	{
		phys= *tab;

		tableno= phys >> VM_DIRSHIFT;
		table= get_phys_dword(page_base + 4*tableno);
		shadow= get_phys_dword(shadow_base + 4*tableno);
		if (table == 0)
			return EINVAL;	/* Empty space */
		assert(shadow != 0);
		type= shadow & VM_PAGEMASK;
		if (pp != &proc[type-1])
			return EINVAL;	/* Wrong process */

		assert(table & VM_PRESENT); /* Tables are either 0 or present */
		pageno= (phys & VM_DIRMASK) >> VM_PAGESHIFT;
		page= get_phys_dword(table + 4*pageno);
		if (!(page & VM_PRESENT))
			return EINVAL;	/* Page is not present */
		if (lock_count(page & VM_ADDRMASK))
			return EINVAL;	/* Page is not locked */
		real= (page & VM_ADDRMASK) | (phys & VM_PAGEMASK);
		*tab= real;
	}
	return OK;
}


/*==========================================================================*
 *				vm_1phys2bus				    *
 *==========================================================================*/
u32_t vm_1phys2bus(phys)
u32_t phys;
{
	unsigned tableno, pageno, type;
	u32_t table, shadow, page, real;

	tableno= phys >> VM_DIRSHIFT;
	table= get_phys_dword(page_base + 4*tableno);
	shadow= get_phys_dword(shadow_base + 4*tableno);
	assert(table != 0);
	assert(shadow != 0);
	type= shadow & VM_PAGEMASK;

	assert(table & VM_PRESENT); /* Tables are either 0 or present */
	table &= VM_ADDRMASK;
	pageno= (phys & VM_DIRMASK) >> VM_PAGESHIFT;
	page= get_phys_dword(table + 4*pageno);
	assert(page & VM_PRESENT);

	real= (page & VM_ADDRMASK) | (phys & VM_PAGEMASK);
#if DEBUG
 { if (!is_dmapage(real)) { printW(); printf("0x%x is not a dma page\n",
 real); } }
#endif

	return real;
}
/*==========================================================================*
 *				vm_check_addr				    *
 *==========================================================================*/
PUBLIC u32_t vm_check_addr(addr, size)
phys_bytes addr;
phys_bytes size;
{
	u32_t base, top, table, pageent;
	unsigned tableno, pageno;

	base= (addr & VM_ADDRMASK);
	top= (addr + size + VM_PAGESIZE-1) & VM_ADDRMASK;

	while (base < top)
	{
		tableno= base >> VM_DIRSHIFT;
		table= get_phys_dword(page_base + 4*tableno);
		table &= VM_ADDRMASK;
		if (!table)
		{
			return base;
		}
		pageno= (base & VM_DIRMASK) >> VM_PAGESHIFT;
		for(; base < top && pageno < VM_PAGESIZE/4;
			base += VM_PAGESIZE, pageno++)
		{
			pageent= get_phys_dword(table + 4*pageno);
			if (!(pageent & VM_PRESENT))
			{
				return base;
			}
		}
	}
	return 0;
}


/*
 * swap space management routines
 */

/*==========================================================================*
 *				get_swapdesc				    *
 *==========================================================================*/
PRIVATE struct swapdesc *get_swapdesc(pnr, minor, offset, priority)
int pnr;
int minor;
u32_t offset;
i32_t priority;
{
	swapdesc_t *sd;
	int i;
	u32_t map, phys_space;

	/* Let's look for an existing descriptor first. */
	for (i= 0, sd= swaptable; i< NR_SWAPSEGS; i++, sd++)
	{
		if (!(sd->sd_flags & SDF_INUSE))
			continue;
		if (sd->sd_proc == pnr && sd->sd_minor == minor &&
							sd->sd_offset == offset)
		{
			return sd;
		}
	}

	/* We didn't find an existing one, so let's create one. First we
	 * have to find an empty slot.
	 */
	for (i= 0, sd= swaptable; i< NR_SWAPSEGS; i++, sd++)
	{
		if (!(sd->sd_flags & SDF_INUSE))
			break;
	}

	if (i == NR_SWAPSEGS)
		return NULL;

	/* If the slot doesn't have a map already, we have to allocate
	 * a map.
	 */

	if (sd->sd_map == 0)
	{
		map= coremem_alloc(&locked_pages);
		if (map == 0)
		{
			/* Bad luck no free pages. */
			return NULL;
		}

		/* This page is clearly not available to user processes */
		vm_not_alloc--;

		sd->sd_map= map;
	}
	assert(!(offset & VM_DIRMASK));

	if (priority <= SDP_MIN)
		priority= SDP_MIN+1;
	sd->sd_proc= pnr;
	sd->sd_minor= minor;
	sd->sd_offset= offset;
	sd->sd_priority= priority;
	sd->sd_flags |= SDF_INUSE;
	sd->sd_flags &= ~SDF_INFREEQ;
	sd->sd_phys_space= VM_SWAPPAGE | ((sd-swaptable) << VM_DIRSHIFT);
	sd->sd_freehead= 0;

	/* mark the disk pages as not available */
	phys_space= sd->sd_phys_space;
	for (offset= 0; offset < VM_DIRSIZE; offset += VM_PAGESIZE)
	{
		set_page_info(phys_space | offset, PT_NOSWAP, 0);
	}

	return sd;
}


/*==========================================================================*
 *				pages2swap				    *
 *==========================================================================*/
PRIVATE void pages2swap(pp)
struct proc *pp;
{
	u32_t corepages[MAX_PAGES2SWAP], swappages[MAX_PAGES2SWAP];
	u32_t curr_page, info, swap_page;
	int no_pages2swap;
	int use2nd, i, r;
	int priority, locks;

	/* Only the pager task is allowed to page out pages. */
	assert(proc_ptr == proc_addr(pager_tasknr));

	assert(check_disk_pages());

	no_pages2swap= 0;
	curr_page= curr_pageno << VM_PAGESHIFT;
	use2nd= FALSE;
	priority= proc_prio(pp);

	/* First loop. Look at pages with a lower priority than ours. Clear
	 * the referenced bit if necessary.
	 */
	for (i= 0; i<coremem_pages && no_pages2swap < MAX_PAGES2SWAP;
		i++, curr_pageno++, curr_page += VM_PAGESIZE)
	{
		if (curr_pageno == coremem_pages)
		{
			curr_pageno= 0;
			curr_page= 0;
		}
		info= get_page_info(curr_page);
		if ((info & VM_PAGEMASK) >= PT_SPECIAL_PAGES ||
			(info & VM_PAGEMASK) == 0)
		{
			/* This page is not interresting. */
			continue;
		}
		locks= lock_count(curr_page);
		if (locks != 0)
		{
			/* This page is locked. */
			continue;
		}
		r= check_page_prio(info, priority);
		if (r == EBUSY)
			continue;
		if (r == EAGAIN)
		{
			use2nd= TRUE;
			continue;
		}
		assert(r == OK);
		/* OK, we found a decent page. Allocate a swap page
		 * and remap the page table entries. */
		swap_page= swapmem_alloc(&transfer_pages);

		if (swap_page == 0)
		{
			/* Out of swap pages. We don't care if we already
			 * found one page to free.
			 */
			if (no_pages2swap != 0)
				break;
			panic("unable to allocate a swap page", NO_NUM);
		}
		remap_pages(curr_page, swap_page);
		corepages[no_pages2swap]= curr_page;
		swappages[no_pages2swap]= swap_page;
		no_pages2swap++;
		set_page_info(curr_page, 0, 0);
	}

	/* Second loop. Look at pages with a lower priority than ours. The
	 * reference bit should be cleared in the first round unless the
	 * page is shared...
	 */
	if (use2nd)
	{
		for (i= 0; i<coremem_pages && no_pages2swap < MAX_PAGES2SWAP;
			i++, curr_pageno++, curr_page += VM_PAGESIZE)
		{
			if (curr_pageno == coremem_pages)
			{
				curr_pageno= 0;
				curr_page= 0;
			}
			info= get_page_info(curr_page);
			if ((info & VM_PAGEMASK) >= PT_SPECIAL_PAGES ||
				(info & VM_PAGEMASK) == 0)
			{
				/* This page is not interresting. */
				continue;
			}
			locks= lock_count(curr_page);
			if (locks != 0)
			{
				/* This page is locked. */
				continue;
			}
			r= check_page_prio(info, priority);
			if (r != OK)
				continue;
			/* OK, we found a decent page. Let's allocate a swap page
			 * and remap the page table entries. */
			swap_page= swapmem_alloc(&transfer_pages);

			if (swap_page == 0)
			{
				/* Out of swap pages. We don't care if we already
				 * found one page to free.
				 */
				if (no_pages2swap != 0)
					break;
				panic("unable to allocate a swap page", NO_NUM);
			}
			remap_pages(curr_page, swap_page);
			corepages[no_pages2swap]= curr_page;
			swappages[no_pages2swap]= swap_page;
			no_pages2swap++;
			set_page_info(curr_page, 0, 0);
		}
	}

	/* Third loop. Look at all pages, that are owned by user processes,
	 * clear the referenced bit if necessary. We don't care about
	 * shared pages.
	 */
	/* Fourth loop. Now all pages are available to us. */
	/* We take the third and fourth loop together. */

	for (i= 2*coremem_pages;
		i > 0 && no_pages2swap < MIN_PAGES2SWAP;
		i--, curr_pageno++, curr_page += VM_PAGESIZE)
	{
		if (curr_pageno == coremem_pages)
		{
			curr_pageno= 0;
			curr_page= 0;
		}
		info= get_page_info(curr_page);
		if ((info & VM_PAGEMASK) >= PT_SPECIAL_PAGES ||
			(info & VM_PAGEMASK) == 0)
		{
			/* This page is not interresting. */
			continue;
		}
		locks= lock_count(curr_page);
		if (locks != 0)
		{
			/* This page is locked. */
			continue;
		}
		r= check_page_prio(info, 0);
		if (r != OK)
			continue;
		/* OK, we found a decent page. Let's allocate a swap page
		 * and remap the page table entries. */
#if DEBUG & 256
 { printW(); printf("\n"); }
#endif
		swap_page= swapmem_alloc(&transfer_pages);

		if (swap_page == 0)
		{
			/* Out of swap pages. We don't care if we already
			 * found one page to free.
			 */
			if (no_pages2swap != 0)
				break;
			panic("unable to allocate a swap page", NO_NUM);
		}
		remap_pages(curr_page, swap_page);
		corepages[no_pages2swap]= curr_page;
		swappages[no_pages2swap]= swap_page;
		no_pages2swap++;
		set_page_info(curr_page, 0, 0);
	}
	assert(i != coremem_pages);
	level0(vm_reload);
	for (i= 0; i<no_pages2swap; i++)
	{
		transfer_page(corepages[i], swappages[i]);
		mem_free(corepages[i], &transfer_pages);
	}
}


/*==========================================================================*
 *				move2dma				    *
 *==========================================================================*/
PRIVATE u32_t move2dma(page)
u32_t page;
{
	u32_t newpage;

	page &= VM_ADDRMASK;

	/* Make sure we have at least one free page. */
	if (free_dmacore_mem + free_coremem == 0)
	{
		if (proc_ptr == proc_addr(pager_tasknr))
			pages2swap(proc_ptr);
		else
			req_swapout(SYSTASK);
		assert(free_dmacore_mem + free_coremem != 0);
	}
	newpage= dmamem_alloc_forced(&floating_pages);
	assert(newpage != 0);
	phys_copy(page, newpage, VM_PAGESIZE);
	remap_pages(page, newpage);
	level0(vm_reload);
	mem_free(page, &floating_pages);
	return newpage;
}


/*==========================================================================*
 *				swapmem_alloc				    *
 *==========================================================================*/
PRIVATE u32_t swapmem_alloc(account_p)
int *account_p;
{
	swapdesc_t *sd, *prev, *curr, *best, *best_prev;
	u32_t page;
	i32_t prio;

	while(swap_freehead)
	{
		sd= swap_freehead;
		page= sd->sd_freehead;
		if (page)
		{
			assert((page & VM_PAGEMASK) == PT_FREEPAGE);
			page &= VM_ADDRMASK;
			sd->sd_freehead= get_page_info(page);
			assert(free_swapmem > 0);
			free_swapmem--;
			(*account_p)++;
			return page;
		}
		prio= sd->sd_priority;
		swap_freehead= sd->sd_nextfree;
		sd->sd_flags &= ~SDF_INFREEQ;
		if (swap_freehead == NULL ||
			swap_freehead->sd_priority == prio)
		{
			continue;
		}
		/* Move the descriptor with highest priority upfront */
		assert(swap_freehead->sd_priority < prio);
		prio= swap_freehead->sd_priority;
		best= NULL;
		best_prev= NULL;
		for (prev= NULL, curr= swap_freehead; curr; prev= curr,
			curr= curr->sd_nextfree)
		{
			if (curr->sd_priority <= prio)
				continue;
			prio= curr->sd_priority;
			best= curr;
			best_prev= prev;
		}
		if (best)
		{
			assert(best_prev);
			best_prev->sd_nextfree= best->sd_nextfree;
			if (best_prev->sd_nextfree == NULL)
				swap_freetail= best_prev;
			best->sd_nextfree= swap_freehead;
			swap_freehead= best;
		}
	}
	/* No free pages left. */
	assert(free_swapmem == 0);
	return 0;
}


/*==========================================================================*
 *				check_page_prio				    *
 *==========================================================================*/
PRIVATE int check_page_prio(qhead, priority)
u32_t qhead;
unsigned priority;
{
	u32_t curr, table, shadow, pageent;
	unsigned type, currpri, tableno, pageno;
	struct proc *pp;

assert(check_list(qhead));

	curr= qhead;
	while(curr)
	{
		type= curr & VM_PAGEMASK;
		assert(isokindex(type-1));
		pp= &proc[type-1];
		if (pp->p_status & P_ST_VM_INCORE)
		{
 { printW(); printf("locked process %d, page 0x%x\n", proc_number(pp), curr); }
			return EBUSY;	/* For now */
		}
		assert(proc_number(pp) >= 0);

		currpri= proc_prio(pp);
		if (currpri <= priority)
			return EBUSY;

		tableno= curr >> VM_DIRSHIFT;
		table= get_phys_dword(page_base + 4*tableno);
		shadow= get_phys_dword(shadow_base + 4*tableno);
		assert((shadow & VM_PAGEMASK) == (curr & VM_PAGEMASK));

		table &= VM_ADDRMASK;
		shadow &= VM_ADDRMASK;
		assert(table && shadow);

		pageno= (curr & VM_DIRMASK) >> VM_PAGESHIFT;
		pageent= get_phys_dword(table +  4*pageno);
		if ((pageent & VM_ACCESSED) && currpri != SLEEPING_PRIO)
		{
			pageent &= ~VM_ACCESSED;
			put_phys_dword(table + 4*pageno, pageent);
			return EAGAIN;
		}
		curr= get_phys_dword(shadow + 4*pageno);
	}
	return OK;
}


/*==========================================================================*
 *				proc_prio				    *
 *==========================================================================*/
PRIVATE unsigned proc_prio(pp)
struct proc *pp;
{
	if (pp->p_status & P_ST_UNRDY_2)
		return SLEEPING_PRIO;
	return (pp->p_complr << 16) | (proc_number(pp) + 1);
}


/*==========================================================================*
 *				transfer_page					    *
 *==========================================================================*/
PRIVATE void transfer_page(src_page, dst_page)
u32_t src_page;
u32_t dst_page;
{
	u32_t disk_page, core_page, b_page;
	message m;
	int op, r;
	unsigned swapdesc;
	swapdesc_t *sd;
	iovec_t iovec1;

	assert(!(src_page & VM_PAGEMASK));
	assert(!(dst_page & VM_PAGEMASK));

	if (src_page & VM_SWAPPAGE)
	{
		assert(!(dst_page & VM_SWAPPAGE));
		disk_page= src_page;
		core_page= dst_page;
		if (is_dmapage(core_page))
		{
			b_page= core_page;
			pagein_cnt++;
		}
		else
		{
			b_page= bounce_page;
			bounce_pagein_cnt++;
		}
		op= DEV_GATHER;
	}
	else
	{
		disk_page= dst_page;
		core_page= src_page;
		if (is_dmapage(core_page))
		{
			b_page= core_page;
			pageout_cnt++;
		}
		else
		{
			b_page= bounce_page;
			phys_copy(core_page, bounce_page, VM_PAGESIZE);
			bounce_pageout_cnt++;
		}
		op= DEV_SCATTER;
	}
	swapdesc= (disk_page & ~VM_SWAPPAGE) >> VM_DIRSHIFT;
	assert(swapdesc < NR_SWAPSEGS);
	sd= &swaptable[swapdesc];
	assert((sd->sd_flags & SDF_INUSE) && sd->sd_map != 0);

	iovec1.iov_addr= b_page;
	iovec1.iov_size= VM_PAGESIZE;
	m.m_type= op;
	m.DEVICE= sd->sd_minor;
	m.PROC_NR= HARDWARE;	/* Used to indicate absolute memory. */
	m.POSITION= sd->sd_offset + (disk_page & VM_DIRMASK);
	m.HIGHPOS= 0;
	m.ADDRESS= (char *)&iovec1;
	m.COUNT= 1;

	r= sendrec(sd->sd_proc, &m);
	compare(r, ==, OK);
	compare(iovec1.iov_size, ==, 0);

	if (op == DEV_GATHER && b_page != core_page)
		phys_copy(b_page, core_page, VM_PAGESIZE);
}


/*==========================================================================*
 *				req_swapout				    *
 *==========================================================================*/
PRIVATE void req_swapout(procn)
int procn;
{
	message m;
	int r;

	assert(proc_ptr == cproc_addr(SYSTASK));

	m.m_type= PAGER_SWAPOUT;
	m.m1_i1= procn;
	r= sendrec(pager_tasknr, &m);
	assert(r == OK && m.m1_i1 == OK);
}


/*==========================================================================*
 *				sweeper					    *
 *==========================================================================*/
PRIVATE void sweeper()
{
	message m;
	struct proc *pp;
	int r;

	assert(!sweeper_running);
	sweeper_running= TRUE;

	/* Let's go over the process table. */
	for (pp= BEG_PROC_ADDR; pp < END_PROC_ADDR; pp++)
	{
		if (pp->p_status & P_ST_UNRDY_1)
			pp->p_status |= P_ST_UNRDY_2;
		else if (pp->p_flags && !(pp->p_flags & P_PAGEFAULT))
			pp->p_status |= P_ST_UNRDY_1;
	}

	/* Let's set the watchdog timer. */
	m.m_type = SET_ALARM;
	m.CLOCK_PROC_NR = proc_number(proc_ptr);
	m.DELTA_TICKS = (long) (HZ/SWEEPER_FREQ);
	m.FUNC_TO_CALL = (sighandler_t) sweeper_int;
	r= sendrec(clck_tasknr, &m);
	assert(r == OK);
}


/*==========================================================================*
 *				sweeper_int				    *
 *==========================================================================*/
PRIVATE void sweeper_int()
{
	assert(sweeper_running);
	sweeper_running= FALSE;
	interrupt(pager_tasknr);
}


/*
 * debug routines
 */

/*==========================================================================*
 *				tell_pages				    *
 *==========================================================================*/
PRIVATE void tell_pages(label, n)
char *label;
u32_t n;
{
	char *K= itoa(click_to_round_k(n));
	int len= strlen(K);

	if (len <= 5) {
		printf("%s%5u%*s(%sK)", label, n, 6 - len, "", K);
	} else {
		printf("%s%*s(%sK)", label, 11 - len, "", K);
	}
}


/*==========================================================================*
 *				vm_dump					    *
 *==========================================================================*/
PUBLIC void vm_dump()
{
	static int recursive= 0;
	int pseudo_allocated;

	if (recursive)
		return;
	recursive++;

	pseudo_allocated= (intl_pages[0] != 0) + (intl_pages[1] != 0) +
		(taskl_pages[0] != 0) + (taskl_pages[1] != 0);

	printf("\nVM dump:");
	tell_pages("\nlocked pages:        ", locked_pages);
	tell_pages("   floating pages:    ", floating_pages);
	tell_pages("\nfree dma pages:      ", free_dmacore_mem);
	tell_pages("   total dma pages:   ", total_dmacore_mem);
	tell_pages("\nfree core pages:     ", free_coremem);
	tell_pages("   total core pages:  ", total_coremem);
	tell_pages("\nfree swap pages:     ", free_swapmem);
	tell_pages("   total swap pages:  ", total_swapmem);
	tell_pages("\nnot reserved pages:  ", vm_not_alloc);
	tell_pages("   total virt pages:  ",
			total_dmacore_mem + total_coremem + total_swapmem);
	tell_pages("\nlowest non-res pages:", vm_lowest_free);
	printf("   [transfer: %d, pseudo: %d]", transfer_pages, pseudo_allocated);
	tell_pages("\n# page in:           ", pagein_cnt);
	tell_pages("   # bounced page in: ", bounce_pagein_cnt);
	tell_pages("\n# page out:          ", pageout_cnt);
	tell_pages("   # bounced page out:", bounce_pageout_cnt);
	printf("\n");
	vm_lowest_free= vm_not_alloc;
	check_free_list();
	recursive--;
}


/*==========================================================================*
 *				vm_check_stats				    *
 *==========================================================================*/
PUBLIC int vm_check_stats()
{
	int pseudo_allocated;

	if (locked_pages + floating_pages + free_dmacore_mem + free_coremem +
		free_swapmem + transfer_pages != total_dmacore_mem +
		total_coremem + total_swapmem)
	{
		vm_dump();
		printf("total pages check failed\n");
		return FALSE;
	}
	pseudo_allocated= (intl_pages[0] != 0) + (intl_pages[1] != 0) +
		(taskl_pages[0] != 0) + (taskl_pages[1] != 0);
	if (vm_not_alloc > free_dmacore_mem + free_coremem + free_swapmem +
		pseudo_allocated + transfer_pages)
	{
		vm_dump();
		printf("reserved memory check failed\n");
		return FALSE;
	}
	return TRUE;
}


/*==========================================================================*
 *				vm_hard_check_stats			    *
 *==========================================================================*/
PUBLIC int vm_hard_check_stats()
{
	if (!vm_check_stats())
		return FALSE;
	if (vm_not_alloc != free_dmacore_mem + free_coremem + free_swapmem)
	{
		vm_dump();
		printf("hard reserved memory check failed\n");
		return FALSE;
	}
	return TRUE;
}

/*==========================================================================*
 *				vm_check_unmapped			    *
 *==========================================================================*/
PUBLIC void vm_check_unmapped(base_clicks, size_clicks)
phys_clicks base_clicks;
phys_clicks size_clicks;
{
	u32_t base, size, table, shadow;
	unsigned tableno;

#if DEBUG & 256
 { printW(); printf("vm_check_unmapped(0x%x, 0x%x)\n", base_clicks,
	size_clicks); }
#endif
	base= base_clicks << VM_PAGESHIFT;
	size= size_clicks << VM_PAGESHIFT;
assert(!(base & VM_DIRMASK));	/* aligned on a 4M boundary */

	size= (size + VM_DIRSIZE-1) & ~VM_DIRMASK;
	for (tableno= base >> VM_DIRSHIFT; size; size -= VM_DIRSIZE, tableno++)
	{
		table= get_phys_dword(page_base + 4*tableno);
		shadow= get_phys_dword(shadow_base + 4*tableno);
#if DEBUG  & 256
 { printW(); printf("shadow 0x%x: *(0x%x)= 0x%x\n", tableno,
	shadow_base+4*tableno, shadow); }
 { printW(); printf("table 0x%x: *(0x%x)= 0x%x\n", tableno,
	page_base+4*tableno, table); }
#endif
		assert(table == 0 && shadow == 0);
	}
}


#if DEBUG
/*==========================================================================*
 *				dump_map				    *
 *==========================================================================*/
PRIVATE void dump_map(mp)
struct mem_map *mp;
{
	printf("%7x %7x %7x, %7x %7x %7x, %7x %7x %7x\n",
		mp[SEG_T].mem_phys, mp[SEG_T].mem_vir, mp[SEG_T].mem_len,
		mp[SEG_D].mem_phys, mp[SEG_D].mem_vir, mp[SEG_D].mem_len,
		mp[SEG_S].mem_phys, mp[SEG_S].mem_vir, mp[SEG_S].mem_len);
}
#endif


/*==========================================================================*
 *				check_list				    *
 *==========================================================================*/
PRIVATE int check_list(head)
u32_t head;
{
	int i, type;
	u32_t vaddr, shadow, table, pageent, physpage;
	unsigned tableno, pageno;

#if DEBUG & 256
 { printW(); printf("check_list(0x%x)\n", head); stacktrace(0); }
#endif
	if (head == 0)
	{
		printf("end of list\n");
		return FALSE;
	}
	for(i= 0; head; i++)
	{
		type= head & VM_PAGEMASK;

		vaddr= head & VM_ADDRMASK;
		tableno= vaddr >> VM_DIRSHIFT;
		shadow= get_phys_dword(shadow_base + 4*tableno);
		if ((shadow & VM_PAGEMASK) != type)
		{
			printf("no %d: type check failed: %d != %d\n",
				i, type, shadow & VM_PAGEMASK);
			return FALSE;
		}
		shadow &= VM_ADDRMASK;
		if (shadow == 0)
		{
			printf("no %d: no shadow\n", i);
			return FALSE;
		}
		table= get_phys_dword(page_base + 4*tableno) & VM_ADDRMASK;
		if (table == 0)
		{
			printf("no %d: no table\n", i);
			return FALSE;
		}
		pageno= (head & VM_DIRMASK) >> VM_PAGESHIFT;
		pageent= get_phys_dword(table + 4*pageno);
		pageent &= (VM_ADDRMASK | VM_USER | VM_WRITE | VM_PRESENT);
		if (i == 0)
			physpage= pageent;
		else if (pageent != physpage)
		{
			printf("no %d: pageent check failed: 0x%x != 0x%x\n", i,
				pageent, physpage);
			return FALSE;
		}
		head= get_phys_dword(shadow + 4*pageno);
	}
#if DEBUG & 256
 { printW(); printf("check_list(...)= TRUE\n"); }
#endif
	return TRUE;
}


/*==========================================================================*
 *				check_free_list				    *
 *==========================================================================*/
PRIVATE int check_free_list()
{
	u32_t page;
	unsigned pages;
	int type;
	swapdesc_t *sd, *prev;

	pages= 0;
	page= free_dmacore_head;
	while(page)
	{
		pages++;
		type= page & VM_PAGEMASK;
		if (type != PT_FREEPAGE)
		{
			printf("page in freelist has wrong type: 0x%x\n", type);
			return FALSE;
		}
		page= get_page_info(page & VM_ADDRMASK);
	}
	if (pages != free_dmacore_mem)
	{
		printf("free_dmacore_mem (%d) != #freelist (%d)\n",
		free_dmacore_mem, pages);
		return FALSE;
	}
	pages= 0;
	page= freecore_head;
	while(page)
	{
		pages++;
		type= page & VM_PAGEMASK;
		if (type != PT_FREEPAGE)
		{
			printf("page in freelist has wrong type: 0x%x\n", type);
			return FALSE;
		}
		page= get_page_info(page & VM_ADDRMASK);
	}
	if (pages != free_coremem)
	{
		printf("free_coremem (%d) != #freelist (%d)\n",
		free_coremem, pages);
		return FALSE;
	}
	pages= 0;
	for (prev= 0, sd= swap_freehead; sd; prev= sd, sd= sd->sd_nextfree)
	{
		page= sd->sd_freehead;
		while(page)
		{
			pages++;
			type= page & VM_PAGEMASK;
			if (type != PT_FREEPAGE)
			{
				printf("page in sd %d has wrong type: 0x%x\n",
					sd-swaptable, type);
				return FALSE;
			}
			page= get_page_info(page & VM_ADDRMASK);
		}
	}
	if (swap_freehead != NULL && swap_freetail != prev)
	{
		printf("swap_freetail (%d) != end of list (%d)\n",
			swap_freetail-swaptable, prev-swaptable);
		return FALSE;
	}
	if (pages != free_swapmem)
	{
		printf("free_swapmem (%d) != #free swap list (%d)\n",
			free_swapmem, pages);
		return FALSE;
	}
	return TRUE;
}


/*==========================================================================*
 *				check_disk_pages			    *
 *==========================================================================*/
PRIVATE int check_disk_pages()
{
	int i, j, type;
	u32_t o, info, table_no, page_no, table, shadow, page;
	swapdesc_t *swap;

	for (i= 0, swap= swaptable; i<NR_SWAPSEGS; i++, swap++)
	{
		if (!(swap->sd_flags & SDF_INUSE))
			continue;
		for (j= 0, o= swap->sd_phys_space; j < VM_PAGESIZE; j += 4,
			o += VM_PAGESIZE)
		{
			info= get_page_info(o);
			assert(get_phys_dword(swap->sd_map + j) == info);
			type= info & VM_PAGEMASK;
			if (type == 0 || type == PT_FREEPAGE ||
				type == PT_NOSWAP)
			{
				continue;
			}
			assert(type < PT_SPECIAL_PAGES);

			table_no= info >> VM_DIRSHIFT;
			table= get_phys_dword(page_base + 4*table_no);
			shadow= get_phys_dword(shadow_base + 4*table_no);

			assert ((shadow & VM_PAGEMASK) == type);

			table &= VM_ADDRMASK;
			page_no= (info & VM_DIRMASK) >> VM_PAGESHIFT;
			page= get_phys_dword(table + 4*page_no);
			if ((page & VM_ADDRMASK) != o)
			{
				printW();
				printf(
			"i= %d, j= %d, o= 0x%x, info= 0x%x, page= 0x%x\n",
					i, j, o, info, page);
				return 0;
			}
		}
	}
	return 1;
}

/*
 * $PchId: vm386.c,v 1.8 1996/05/07 08:14:17 philip Exp $
 */
