/* This file contains the C startup code for Minix on Intel processors.
 * It cooperates with mpx.x to set up a good environment for main().
 *
 * This code runs in real mode for a 16 bit kernel and may have to switch
 * to protected mode for a 286.
 *
 * For a 32 bit kernel this already runs in protected mode, but the selectors
 * are still those given by the BIOS with interrupts disabled, so the
 * descriptors need to be reloaded and interrupt descriptors made.
 */

#include "kernel.h"
#include "protect.h"
#include <stdlib.h>

FORWARD _PROTOTYPE( int k_atoi, (char *s) );


/*==========================================================================*
 *				cstart					    *
 *==========================================================================*/
PUBLIC void cstart(cs, ds, mcs, mds, parmoff, parmsize)
U16_t cs, ds;			/* Kernel code and data segment */
U16_t mcs, mds;			/* Monitor code and data segment */
U16_t parmoff, parmsize;	/* boot parameters offset and length */
{
/* Perform system initializations prior to calling main(). */

  register char *envp;
  phys_bytes mcode_base, mdata_base;
  unsigned mon_start;

  /* Record where the kernel and the monitor are. */
  kernel_code_base = seg2phys(cs);
  kernel_data_base = seg2phys(ds);
  mcode_base = seg2phys(mcs);
  mdata_base = seg2phys(mds);

  /* Initialize protected mode descriptors. */
  prot_init();

  /* Copy the boot parameters to kernel memory. */
  if (parmsize > sizeof k_environ - 2) parmsize = sizeof k_environ - 2;
  phys_copy(mdata_base + parmoff, vir2phys(k_environ), (phys_bytes) parmsize);

  /* Type of VDU: */
  envp = k_getenv("video");
  if (envp != NIL_PTR) {
  	if (strcmp(envp, "ega") == 0) ega = TRUE;
  	if (strcmp(envp, "vga") == 0) vga = ega = TRUE;
  }

  /* Memory sizes: */
  low_memsize = k_atoi(k_getenv("memsize"));
  ext_memsize = k_atoi(k_getenv("emssize"));

  /* Processor? */
  processor = k_atoi(k_getenv("processor"));	/* 386, 486, 586, ... */

  /* AT or MCA bus? */
  envp = k_getenv("bus");
  if (envp != NIL_PTR && strcmp(envp, "mca") == 0) ps_mca = TRUE;

  /* Is there a monitor to return to?  If so then keep it safe. */
  mon_start = mcode_base / 1024;
  if (mon_return && low_memsize > mon_start) low_memsize = mon_start;

  /* Return to assembler code to reload selectors and call main(). */
}


/*==========================================================================*
 *				k_atoi					    *
 *==========================================================================*/
PRIVATE int k_atoi(s)
register char *s;
{
/* Convert string to integer. */

  return strtol(s, (char **) NULL, 10);
}


/*==========================================================================*
 *				k_getenv				    *
 *==========================================================================*/
PUBLIC char *k_getenv(name)
char *name;
{
/* Get environment value - kernel version of getenv to avoid setting up the
 * usual environment array.
 */

  register char *cp;
  register char *envp;
  size_t l, lname;

  lname= strlen(name);

  for (envp = k_environ; (l= strlen(envp)) != 0; envp += l+1) {
  	cp= strchr(envp, '=');
  	if (cp == NULL) continue;
  	if (lname != (cp-envp) || strncmp(name, envp, lname) != 0)
  		continue;
  	return cp+1;
  }
  return(NIL_PTR);
}

/*
 * $PchId: start.c,v 1.4 1995/12/22 09:40:09 philip Exp $
 */
