/*
i387.c

This file implementes support for the Intel nummerical coprocessors 8087,
80287, 80387 and the floating point unit built into the 486DX and 487
processors.

FPU means that the FPU should be initialized, and that the state of the FPU
should be saved and restored at a context switch. The easiest way to save
and restore the context is to save floating point registers at the same
place where the integer registers are saved. This is not very efficient,
forinstance, the kernel tasks and servers don't use the floating registers,
so it is useless to save and restore them when switching to an interrupt 
handler, or when switching to a kernel task or server. Furthermore, most
user processes use floating point only occaisonally, or not at all.

An alternative approach would be to switch the contents of the FPU on demand.
This means that one process gets exclusive access to the FPU and when another
process asks for the FPU the state of the first process will be saved, the 
state of the requesting process will be restored and the requesting process 
will be given exclusive access to the FPU. This approach is efficient if
on the avarage only one process uses the FPU. This scheme is supported by the
286 (and above) hardware and is the one implemented.

If more than one process uses the FPU, it might be more efficient to tell 
kernel (or write code to let kernel figure out) which processes use the FPU
very often and require a context switch of the FPU together with the normal
registers. This is not implemented.

Implementation details.

For each process three bits are maintained; whether the process uses the FPU
(P_ST_FPU_USED), whether is the state reflects and exception (P_ST_FPU_INVAL)
and if the process should get the signal SIGFPE, (P_ST_FPU_EX).
The P_ST_FPU_USED flag is used to give an excutable a clean FPU after an exec,
but avoids the overhead of actually creating a clean FPU state. The P_ST_FPU_EX
flag is used indicate that the process should get a SIGFPU by the exception
handler which caught the FPU interrupt or exception because the exception 
handler might be unable to send the signal from the handler.

The variable fpu_proc_ptr keeps track of which process uses the FPU. The msw
field in the stackframe allows a task to use the FPU or not. The EM bit is
used indicate the absence of an FPU (the msw can be read by the user process), 
and the TS bit will be used to indicate the the process is not the exclusive
owner of the FPU.

Created: 	Dec 13, 1992 by Philip Homburg
*/

#include "kernel.h"
#include "assert.h"
INIT_ASSERT
#include "proc.h"
#include "proto.h"
#include "protect.h"

PRIVATE int fpu_type;
PUBLIC int fpu_usremu= 0;

/*===========================================================================*
 *				fpu_init				     *
 *===========================================================================*/
void fpu_init()
{
  reg_t msw;
  char *fpu_str;

  msw= get_msw() & ~(CR0_MP|CR0_EM|CR0_TS|CR0_NE);
  set_msw(msw);

  fpu_str= k_getenv("fpu");
  if (fpu_str != NULL)
  {
  	if (strcmp(fpu_str, "off") == 0)
  	{
  		fpu_type= 0;
  	}
  	else if (strcmp(fpu_str, "on") == 0)
  	{
  		fpu_type= 387;
  	}
  	else
  	{
  		fpu_type= 0;
  		printf("i387: invalid value in variable 'fpu': '%s'\n",
  			fpu_str);
  	}
  	if (debug)
  	{
  		printf("i387: fpu configured o%s\n", fpu_type==0 ? "ff" : "n");
	}
  }
  else
  {
	fpu_type = fpu_probe() ? 387 : 0;
	if (debug)
	{
		printf("i387: %s\n", fpu_type==0 ? "no fpu" : "fpu detected");
		if (fpu_type != 0 && !fpu_fdiv_test())
			printf("i387: buggy FDIV\n");
	}
  }

  if (fpu_type == 0)
  {
  	/* No fpu */
  	msw |= CR0_EM;
  }
  else
  {
  	msw |= CR0_MP|CR0_TS;
  }
  if (processor == 486)
  {
  	/* A 486 is a problem since some motherboard don't seem to handle
  	 * reporting fpu exception via IRQ 13 very well. This seems to be
  	 * no problem in 386 motherboards. Fortunately, the 486 can handle
  	 * fpu exceptions internally by setting the CR0_NE bit in the 
  	 * processor status word.
  	 */
  	msw |= CR0_NE;
  }

  set_msw(msw);
  put_irq_handler(FPU_EX_IRQ, fpu_exception);
  enable_irq(FPU_EX_IRQ);
}


/*===========================================================================*
 *				fpu_switch				     *
 *===========================================================================*/
void fpu_switch()
{
  struct proc *pp;
  /* We are called with interrupts off. Save the proc_ptr because
   * a clocktick might cause a taskswitch.
   */
  pp= proc_ptr;
  unlock();

  /* Do we really have an fpu */
  if (fpu_type == 0 || fpu_usremu)
  {
  	/* No */
  	cause_sig(proc_number(pp), fpu_usremu ? SIGFPEMU : SIGFPE);
  	return;
  }
  /* Now save the old context. Note that fpu_save will sync with the
   * fpu to make sure errors are signaled, check if there is a current
   * user of the fpu (there is no user initially and after an error), and
   * clear the TS flag, before the context is saved.
   */
  fpu_save();

  /* If the process never used the fpu, we create an intial state. */
  if (!(pp->p_status & P_ST_FPU_USED))
  {
#if DEBUG & 0
 { printW(); printf("creating new context\n"); }
#endif
  	memset(&pp->p_reg.sf_fp_cw, '\0',
  		offsetof(struct stackframe_s, sf_msw) -
  		offsetof(struct stackframe_s, sf_fp_cw));
	fpu_proc_ptr= pp;
	fpu_reload();
  	fpu_empty();
  	pp->p_status &= ~(P_ST_FPU_INVAL|P_ST_FPU_EX);
  	pp->p_status |= P_ST_FPU_USED;
  	pp->p_reg.sf_msw &= ~MSW_TS;
  	return;
  }

  /* If the process got an exception, we send a SIGFPE */
  if (pp->p_status & P_ST_FPU_EX)
  {
#if DEBUG & 0
 { printW(); printf("got an exception\n"); }
#endif
  	pp->p_status &= ~P_ST_FPU_EX;
  	pp->p_status |= P_ST_FPU_INVAL;
  	cause_sig(proc_number(pp), SIGFPE);
  	return;
  }

  /* Finally we get to the reload */
  pp->p_status &= ~P_ST_FPU_INVAL;	/* if the context is really unusable
  					 * we will get an exception.
  					 */
  fpu_proc_ptr= pp;
  pp->p_reg.sf_msw &= ~MSW_TS;
  fpu_reload();
}

/*===========================================================================*
 *				fpu_exception				     *
 *===========================================================================*/
PUBLIC int fpu_exception(irq)
int irq;
{
	out_byte(0xF0, 0);	/* Acknowledge the FPU. */

	assert(fpu_proc_ptr != NULL);
	fpu_store();
  	fpu_proc_ptr->p_status |= P_ST_FPU_EX;
	fpu_proc_ptr->p_reg.sf_msw |= MSW_TS;
	fpu_proc_ptr= NULL;

	return 1;	/* Reenable FPU interrupts */
}

/*
 * $PchId: i387.c,v 1.6 1996/01/19 23:03:32 philip Exp $
 */
