/* This file contains a simple exception handler.  Exceptions in user
 * processes are converted to signals.  Exceptions in the kernel, MM and
 * FS cause a panic.
 */

#include "kernel.h"
#include <signal.h>
#include <minix/com.h>
#include "proc.h"
#include "protect.h"


/*==========================================================================*
 *				exception				    *
 *==========================================================================*/
PUBLIC void exception(ex_number, trap_errno, old_eip, old_cs, old_eflags)
unsigned ex_number;
u32_t trap_errno;
u32_t old_eip;
U16_t old_cs;
u32_t old_eflags;
{
/* An exception or unexpected interrupt has occurred. */

  struct ex_s {
	char *msg;
	int signum;
	int minprocessor;
  };
  static struct ex_s ex_data[] = {
	{ "Divide error", SIGFPE, 86 },
	{ "Debug exception", SIGTRAP, 86 },
	{ "Nonmaskable interrupt", SIGSEGV, 86 },
	{ "Breakpoint", SIGTRAP, 86 },
	{ "Overflow", SIGFPE, 86 },
	{ "Bounds check", SIGFPE, 186 },
	{ "Invalid opcode", SIGILL, 186 },
	{ "Coprocessor not available", SIGFPE, 186 },
	{ "Double fault", SIGSEGV, 286 },
	{ "Copressor segment overrun", SIGSEGV, 286 },
	{ "Invalid TSS", SIGSEGV, 286 },
	{ "Segment not present", SIGSEGV, 286 },
	{ "Stack exception", SIGSEGV, 286 },	/* STACK_FAULT already used */
	{ "General protection", SIGSEGV, 286 },
	{ "Page fault", SIGSEGV, 386 },		/* not close */
	{ NIL_PTR, SIGILL, 0 },			/* probably software trap */
	{ "Coprocessor error", SIGFPE, 386 },
	{ "Alignment fault", SIGSEGV, 486 },
  };
  register struct ex_s *ep;
  struct proc *saved_proc;

  saved_proc= proc_ptr;	/* Save proc_ptr, because it may be changed by debug 
  			 * statements.
  			 */

#if DEBUG
 { printW(); printf("got exception in proc %d: ex_number= %d, trap_errno= 0x%x, eip= 0x%x, cs= 0x%x, eflags= 0x%x\n",
 	proc_number(saved_proc), ex_number, trap_errno, old_eip, old_cs,
 	old_eflags); }
#endif
  ep = &ex_data[ex_number];

  if (ex_number == 2)		/* spurious NMI on some machines */
  {
	printf("got spurious NMI\n");
	return;
  }

  if (ex_number == 17) {	/* Alignment fault. */
	phys_bytes phys_pc;
	u8_t opcode;

	/* No more alignment "warnings." */
	saved_proc->p_reg.sf_msw &= ~CR0_AM;

	/* Determine the first opcode byte.  If it is a coprocessor excape
	 * (0xD8 - 0xDF) then don't complain, because the compilers align
	 * floats at word boundaries, which is not good enough.
	 */
	phys_pc = umap(saved_proc, SEG_T, (vir_bytes) saved_proc->p_reg.sf_pc,
								1L);
	if (phys_pc == 0) return;
	phys_copy(phys_pc, vir2phys(&opcode), 1L);

	if ((opcode & 0xF8) != 0xD8) {
		printf(
		"%s, process number %d, alignment fault at pc = 0x%lx\r\n",
			saved_proc->p_name, proc_number(saved_proc), 
			(unsigned long) saved_proc->p_reg.sf_pc);
		stacktrace(saved_proc);
	}
	/* This was far too forgiving. */
	return;
  }

  if (k_reenter == 0 && isuserp(saved_proc)) 
  {
#if DEBUG
	printf("%s, ", ep->msg);
	printf("process number %d, pc = 0x%04x:0x%08lx\r\n",
			proc_number(saved_proc), 
			(unsigned) saved_proc->p_reg.sf_cs,
			(unsigned long) saved_proc->p_reg.sf_pc);
#endif
	unlock();		/* this is protected like sys_call() */
	cause_sig(proc_number(saved_proc), ep->signum);
	return;
  }

  /* This is not supposed to happen. */
  if (ep->msg == NIL_PTR || processor < ep->minprocessor)
	printf("\r\nIntel-reserved exception %d\r\n", ex_number);
  else
	printf("\r\n%s\r\n", ep->msg);
  printf("process number %d, pc = 0x%04x:0x%08x\r\n",
	proc_number(saved_proc),
	(unsigned) saved_proc->p_reg.sf_cs,
	(unsigned) saved_proc->p_reg.sf_pc);
  printf("ax=%x, bx=%x, cx=%x, dx=%x, si=%x, di=%x, bp=%x, sp=%x\n",
	(unsigned) saved_proc->p_reg.sf_ax, (unsigned) saved_proc->p_reg.sf_bx,
	(unsigned) saved_proc->p_reg.sf_cx, (unsigned) saved_proc->p_reg.sf_dx,
	(unsigned) saved_proc->p_reg.sf_si, (unsigned) saved_proc->p_reg.sf_di,
	(unsigned) saved_proc->p_reg.sf_bp, (unsigned) saved_proc->p_reg.sf_sp);
  stacktrace(saved_proc);
  panic("exception in system code", NO_NUM);
}
	
/*
 * $PchId: exception.c,v 1.4 1995/12/22 09:49:05 philip Exp $
 */
