/* This file contains some dumping routines for debugging. */

#include "kernel.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include "proc.h"

FORWARD _PROTOTYPE(void prname, (int i));
FORWARD _PROTOTYPE(void time2str, (unsigned long tim, char *str));

#define mem_units vir_clicks
#define mem_units_to_round_k(n) click_to_round_k(n)
#define phys_mem_units phys_clicks
#define vir_mem_units vir_clicks

PRIVATE char user_time_str[10];
PRIVATE char sys_time_str[10];


/*===========================================================================*
 *				DEBUG routines here			     *
 *===========================================================================*/
#if (CHIP == INTEL)
PUBLIC void p_dmp()
{
/* Proc table dump */

  register struct proc *rp;
  static struct proc *oldrp = BEG_PROC_ADDR;
  int n = 0;
  vir_mem_units size;
  int unit;

  printf("\n-pid- --pc-- ---sp-- flag --user- --sys-- -size -recv- command\n");

  for (rp = oldrp; rp < END_PROC_ADDR; rp++) {
	if (rp->p_flags & P_SLOT_FREE) continue;
	if (++n > 20) break;
	size = rp->p_map[SEG_T].mem_len + rp->p_map[SEG_D].mem_len +
						rp->p_map[SEG_S].mem_len;
	size = mem_units_to_round_k(size);
	unit = 'K';
	if (size >= 10000) {
		size = (size + 512) / 1024;
		unit = 'M';
	}
	time2str(rp->user_time + rp->child_utime, user_time_str);
	time2str(rp->sys_time + rp->child_stime, sys_time_str);
	printf("%5d %6lx %7lx %4x %7s %7s %4lu%c ",
		proc_number(rp) < 0 ? proc_number(rp) : rp->p_pid,
		(unsigned long) rp->p_reg.sf_pc,
		(unsigned long) rp->p_reg.sf_sp,
		rp->p_flags,
		user_time_str,
		sys_time_str,
		size, unit);
	if (!(rp->p_flags & (RECEIVING | SENDING)))
		printf("      ");
	else {
		if (rp->p_flags & SENDING)
		{
			if (rp->p_flags & RECEIVING)
				printf("S/R: ");
			else
				printf("S: ");
		}
		prname(rp->p_tofrom);
	}

	if (proc_number(rp) < LOW_USER) {
		printf(" ");
		prname(proc_number(rp));
	} else {
		printf(" %.25s", rp->p_name);
	}
	printf("\n");
  }
  if (rp == END_PROC_ADDR) rp = BEG_PROC_ADDR; else printf("--more--\r");
  oldrp = rp;
}


PUBLIC void map_dmp()
{
  vir_mem_units size;
  register struct proc *rp;
  static struct proc *oldrp = cproc_addr(0);
  int n = 0;
  int unit;

  printf("\nPROC   -------TEXT------- -------DATA------- --------STACK------- -SIZE-\n");
  for (rp = oldrp; rp < END_PROC_ADDR; rp++) {
	if (rp->p_flags & P_SLOT_FREE) continue;
	if (++n > 20) break;
#if SEGMENTED_MEMORY
	size=  (rp->p_map[SEG_T].mem_phys != rp->p_map[SEG_D].mem_phys ?
		rp->p_map[SEG_T].mem_vir + rp->p_map[SEG_T].mem_len : 0) +
		rp->p_map[SEG_S].mem_vir+rp->p_map[SEG_S].mem_len;
#else /* !SEGMENTED_MEMORY */
	size = rp->p_map[SEG_T].mem_len
		+ (vir_bytes) (rp->p_map[SEG_S].mem_phys
					- rp->p_map[SEG_D].mem_phys)
		+ rp->p_map[SEG_S].mem_len;
#endif /* SEGMENTED_MEMORY */
	size = mem_units_to_round_k(size);
	unit = 'K';
	if (size >= 100000) {
		size = (size + 512) / 1024;
		unit = 'M';
	}
	prname(proc_number(rp));
	printf(" %4x %6x %6x %4x %6x %6x %6x %6x %6x %5lu%c\n",
		rp->p_map[SEG_T].mem_vir, rp->p_map[SEG_T].mem_phys,
						rp->p_map[SEG_T].mem_len,
		rp->p_map[SEG_D].mem_vir, rp->p_map[SEG_D].mem_phys,
						rp->p_map[SEG_D].mem_len,
		rp->p_map[SEG_S].mem_vir, rp->p_map[SEG_S].mem_phys,
						rp->p_map[SEG_S].mem_len,
		size, unit);
  }
  if (rp == END_PROC_ADDR) rp = cproc_addr(0); else printf("--more--\r");
  oldrp = rp;
}


PUBLIC void sched_dmp()
{
  register struct proc *rp;
  static struct proc *oldrp = BEG_PROC_ADDR;
  int n = 0;

  printf("\nPROC   PRIO TICK SRND CPLR P B S STATE\n");
  for (rp = oldrp; rp < END_PROC_ADDR; rp++) {
	if (rp->p_flags & P_SLOT_FREE) continue;
	if (++n > 20) break;
	prname(proc_number(rp));
	printf(" %4x %4d %4x %4x %s %s %s %5x\n", rp->p_priority,
		rp->p_ticksleft, rp->p_schedr, rp->p_complr,
		rp == proc_ptr ? "p" : " ", rp == bill_ptr ? "b" : " ",
		rp == sched_ptr ? "s" : " ",
		rp->p_status);
  }
  if (rp == END_PROC_ADDR) rp = BEG_PROC_ADDR; else printf("--more--\r");
  oldrp = rp;
}

#endif				/* (CHIP == INTEL) */


PRIVATE void prname(i)
int i;
{
  struct proc *pp;
  char buf[7];
  size_t l;
  int j;

  if (i == ANY)
	printf("ANY   ");
  else
  {
	pp= proc_addr(i);
	switch(pp->p_priority)
	{
	case PPRI_TASK:
	case PPRI_SERVER:
	case PPRI_IDLE:
		l= strlen(pp->p_name);
		for (j= 0; j<sizeof(buf)-1; j++)
		{
			if (j<l)
				buf[j]= pp->p_name[j];
			else
				buf[j]= ' ';
		}
		buf[j]= '\0';
		printf("%s", buf);
		break;
	default:
		printf("%4d  ", i);
		break;
	}
  }
}


PRIVATE void time2str(tim, str)
unsigned long tim;
char *str;
{
	int centisec, decisec, sec, sec10, minu, minu10, hour, hour10,
		day, day10;

	if (tim < HZ*60)	/* Less than one minute */
	{
		tim= (tim * 100)/HZ;
		centisec= tim % 10;
		tim /= 10;
		decisec= tim % 10;
		tim /= 10;
		sec= tim %10;
		tim /= 10;
		str[0]= tim ? tim+'0' : ' ';
		str[1]= sec+'0';
		str[2]= '.';
		str[3]= decisec+'0';
		str[4]= centisec+ '0';
		str[5]= '\0';
		return;
	}
	if (tim < HZ*60*60)	/* Less than one hour */
	{
		tim /= HZ;
		sec= tim % 10;
		tim /= 10;
		sec10= tim % 6;
		tim /= 6;
		minu= tim % 10;
		tim /= 10;
		str[0]= tim ? tim+'0' : ' ';
		str[1]= minu+'0';
		str[2]= 'm';
		str[3]= sec10+'0';
		str[4]= sec+ '0';
		str[5]= '\0';
		return;
	}
	if (tim < HZ*60*60*24)	/* Less than one day */
	{
		tim /= (60*HZ);
		minu= tim % 10;
		tim /= 10;
		minu10= tim % 6;
		tim /= 6;
		hour= tim % 10;
		tim /= 10;
		str[0]= tim ? tim+'0' : ' ';
		str[1]= hour+'0';
		str[2]= 'h';
		str[3]= minu10+'0';
		str[4]= minu+ '0';
		str[5]= '\0';
		return;
	}
	/* else no more than 1000 days */
	{
		tim /= (60*60*HZ);
		hour= tim % 24;
		tim /= 24;
		hour10= hour / 10;
		hour %= 10;
		day= tim % 10;
		tim /= 10;
		day10= tim % 10;
		tim /= 10;
		str[0]= tim ? tim+'0' : ' ';
		str[1]= day10+'0';
		str[2]= day+'0';
		str[3]= 'd';
		str[5]= hour10+'0';
		str[5]= hour+ '0';
		str[6]= '\0';
		return;
	}
}


#if (CHIP == INTEL)
/*===========================================================================*
 *				stacktrace				     *
 *===========================================================================*/
PUBLIC void stacktrace(pp)
struct proc *pp;
{
  /* Show a stack trace of a process. */
  struct { reg_t bp, pc; } frame;
  phys_bytes phys_bp;
  char *name;
  int i;

  if (pp == NIL_PROC) {
	pp = cproc_addr(SYSTASK);
	frame.bp = get_bp();
	frame.pc = -1;
	name = "kernel";
  } else {
	frame.bp = pp->p_reg.sf_bp;
	frame.pc = pp->p_reg.sf_pc;
	name = pp->p_name;
  }
  printf("%s stacktrace:", name);
  for (i = 0; i < 10; i++) {
	if (frame.pc != -1) printf(" 0x%lx", (unsigned long) frame.pc);
	if (frame.bp == 0) break;
	phys_bp = umap(pp, SEG_D, (vir_bytes) frame.bp, sizeof(frame));
	if (phys_bp == 0 || !vm_ismapped(phys_bp)
			|| !vm_ismapped(phys_bp + sizeof(frame)-1)) break;
	phys_copy(phys_bp, vir2phys(&frame), (phys_bytes) sizeof(frame));
  }
  printf("\n");
}
#endif /* (CHIP == INTEL) */


/*===========================================================================*
 *				loadav_dmp				     *
 *===========================================================================*/
void loadav_dmp()
{
	unsigned long ldav[LDAV_NR];

	/* avoid race conditions */
	lock();
	memcpy(ldav, loadav, sizeof(ldav));
	unlock();

	printf("%ld", ldav[LDAV_CURR]);
	printf(" %ld", ldav[LDAV_6] >> 6);
	printf(".%02ld", ((ldav[LDAV_6] & ((1 << 6)-1))*100) >> 6);
	printf(" %ld", ldav[LDAV_12] >> 12);
	printf(".%02ld", ((ldav[LDAV_12] & ((1 << 12)-1))*100) >> 12);
	printf(" %ld", ldav[LDAV_16] >> 16);
	printf(".%02ld", ((ldav[LDAV_16] & ((1 << 16)-1))*100) >> 16);
	printf(" %ld\n", ldav[LDAV_TOT]);
	printf("(%ld %ld %ld)\n", ldav[LDAV_6], ldav[LDAV_12], ldav[LDAV_16]);
}

/*
 * $PchId: dmp.c,v 1.4 1996/01/19 23:06:02 philip Exp $
 */
