/* This file contains the code and data for the clock task.  The clock task
 * accepts six message types:
 *
 *   HARD_INT:    a clock interrupt has occurred
 *   GET_TIME:    a process wants the real time in seconds
 *   SET_TIME:    a process wants to set the real time in seconds
 *   SET_ALARM:   a process wants to be alerted after a specified interval
 *   GET_UPTIME:  get the time since boot in ticks
 *   SET_SYN_AL:  set the sync alarm
 *   
 *
 * The input message is format m6.  The parameters are as follows:
 *
 *     m_type    CLOCK_PROC   FUNC    NEW_TIME
 * ---------------------------------------------
 * | HARD_INT   |          |         |         |
 * |------------+----------+---------+---------|
 * | GET_TIME   |          |         |         |
 * |------------+----------+---------+---------|
 * | SET_TIME   |          |         | newtime |
 * |------------+----------+---------+---------|
 * | SET_ALARM  | proc_nr  |f to call|  delta  |
 * |------------+----------+---------+---------|
 * | GET_UPTIME |          |         |         |
 * |------------+----------+---------+---------|
 * | SET_SYN_AL |          |         |  delta  |
 * ---------------------------------------------
 *
 * When an alarm goes off, if the caller is a user process, a SIGALRM signal
 * is sent to it.  If it is a task, a function specified by the caller will
 * be invoked.  This function may, for example, send a message, but only if
 * it is certain that the task will be blocked when the timer goes off.
 */

#include "kernel.h"
#include <signal.h>
#include <time.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "assert.h"
INIT_ASSERT
#include "proc.h"

#include "timer.h"

/* Clock parameters. */
#if (CHIP == INTEL)
#include "ibm/timerreg.h"

#define LATCH_COUNT	(TIMER_SEL0 | TIMER_LATCH)
#define TIMER_MODE	(TIMER_SEL0 | TIMER_RATEGEN | TIMER_16BIT)

/* Initial counter value and tick cycle fraction. */
#define DEF_TICK_CYCLES	(TIMER_FREQ / HZ)	/* initial value for counter */
#define DEF_TICK_FRAQ	(TIMER_FREQ % HZ * 1024 * 1024 / HZ * 4096)

#define CLOCK_ACK_BIT	0x80	/* PS/2 clock interrupt acknowledge bit */

PRIVATE unsigned tick_cycles= DEF_TICK_CYCLES+1;
PRIVATE unsigned long tick_cycles_frac= DEF_TICK_FRAQ;
PRIVATE int tick_low;
PRIVATE unsigned long tick_frac;
#endif

#if (CHIP == M68000)
#define TIMER_FREQ  2457600L	/* timer 3 input clock frequency */
#endif

/* Clock task variables. */
PRIVATE time_t boot_time;	/* time in seconds of system boot */
PRIVATE clock_t next_alarm;	/* probable time of next alarm */
PRIVATE clock_t pending_ticks;	/* ticks seen by low level only */
PRIVATE clock_t realtime;	/* real time clock */
PRIVATE message mc;		/* message buffer for both input and output */

PRIVATE int syn_al_alive= TRUE; /* don't wake syn_alrm_task before inited*/
PRIVATE int watchdog_proc;	/* contains proc_nr at call of *watch_dog[]*/
PRIVATE struct proc *syn_alarm_head= NULL;
				/* head of a list of processes waiting for
				 * a syn alarm
				 */
PRIVATE int synal_tasknr= ANY;

FORWARD _PROTOTYPE( void common_setalarm, (int proc_nr,
		long delta_ticks, watchdog_t fuction) );
FORWARD _PROTOTYPE( void do_clocktick, (void) );
FORWARD _PROTOTYPE( void do_get_time, (void) );
FORWARD _PROTOTYPE( void do_getuptime, (void) );
FORWARD _PROTOTYPE( void do_set_time, (message *m_ptr) );
FORWARD _PROTOTYPE( void do_setalarm, (message *m_ptr) );
FORWARD _PROTOTYPE( int do_sysutime, (message *m_ptr) );
FORWARD _PROTOTYPE( void init_clock, (void) );
FORWARD _PROTOTYPE( void cause_alarm, (void) );
FORWARD _PROTOTYPE( void do_setsyn_alrm, (message *m_ptr) );
FORWARD _PROTOTYPE( int clock_handler, (int irq) );
FORWARD _PROTOTYPE( long getusecs, (clock_t *ticks) );

/*===========================================================================*
 *				clock_task				     *
 *===========================================================================*/
PUBLIC void clock_task()
{
/* Main program of clock task.  It determines which of the 4 possible
 * calls this is by looking at 'mc.m_type'.   Then it dispatches.
 */
 
  int opcode;
  int reply;

  clck_tasknr= proc_number(proc_ptr);

  init_clock();			/* initialize clock task */

  /* Main loop of the clock task.  Get work, process it, sometimes reply. */
  while (TRUE) {
     receive(ANY, &mc);		/* go get a message */
     opcode = mc.m_type;	/* extract the function code */

     lock();
     realtime += pending_ticks;	/* transfer ticks from low level handler */
     pending_ticks = 0;		/* so we don't have to worry about them */
     unlock();

     reply= OK;
     switch (opcode) {
	case HARD_INT:   do_clocktick();	break;
	case GET_TIME:	 do_get_time();		break;
	case SET_TIME:	 do_set_time(&mc);	break;
	case SET_ALARM:	 do_setalarm(&mc);	break;
	case GET_UPTIME: do_getuptime();	break;
	case SET_SYNC_AL:do_setsyn_alrm(&mc);	break;
	case SYS_UTIME:  reply= do_sysutime(&mc);	break;
	default: panic("clock task got bad message", mc.m_type);
     }

    /* Send reply, except for clock tick. */
    mc.m_type = reply;
    if (opcode != HARD_INT) send(mc.m_source, &mc);
  }
}


/*===========================================================================*
 *				do_clocktick				     *
 *===========================================================================*/
PRIVATE void do_clocktick()
{
/* This routine called on clock ticks when a lot of work needs to be done. */

  register struct proc *rp;
  register int proc_nr;

  if (tac_exp_time && tac_exp_time <= realtime)
  	tmra_exptimers();

  if (next_alarm <= realtime) {
	/* An alarm may have gone off, but proc may have exited, so check. */
	next_alarm = LONG_MAX;	/* start computing next alarm */
	for (rp = BEG_PROC_ADDR; rp < END_PROC_ADDR; rp++) {
		if (rp->p_alarm != 0) {
			/* See if this alarm time has been reached. */
			if (rp->p_alarm <= realtime) {
				/* A timer has gone off.  If it is a user proc,
				 * send it a signal.  If it is a task, call the
				 * function previously specified by the task.
				 */
				proc_nr = proc_number(rp);
				if (rp->p_watchdog) {
					watchdog_proc= proc_nr;
					(*rp->p_watchdog)();
				}
				else
				{
					cause_sig(proc_nr, SIGALRM);
				}
				rp->p_alarm = 0;
			}

			/* Work on determining which alarm is next. */
			if (rp->p_alarm != 0 && rp->p_alarm < next_alarm)
				next_alarm = rp->p_alarm;
		}
	}
  }

/* Scheduling is dome at interrupt level by the new scheduler. This avoids
 * tricks with a prev_ptr.
 */
#if (CHIP == M68000)
  if (rdy_head[SHADOW_Q]) unshadow(rdy_head[SHADOW_Q]);
#endif
}


/*===========================================================================*
 *				do_get_time				     *
 *===========================================================================*/
PRIVATE void do_get_time()
{
/* Get and return the current clock time in seconds. */

  mc.NEW_TIME = boot_time + realtime/HZ;	/* current real time */
}


/*===========================================================================*
 *				do_set_time				     *
 *===========================================================================*/
PRIVATE void do_set_time(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* Set the real time clock.  Only the superuser can use this call. */

  boot_time = m_ptr->NEW_TIME - realtime/HZ;
}


/*===========================================================================*
 *				do_setalarm				     *
 *===========================================================================*/
PRIVATE void do_setalarm(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* A process wants an alarm signal or a task wants a given watch_dog function
 * called after a specified interval.
 */

  register struct proc *rp;
  int proc_nr;			/* which process wants the alarm */
  long delta_ticks;		/* in how many clock ticks does he want it? */
  watchdog_t function;		/* function to call (tasks only) */

  /* Extract the parameters from the message. */
  proc_nr = m_ptr->CLOCK_PROC_NR;	/* process to interrupt later */
  delta_ticks = m_ptr->DELTA_TICKS;	/* how many ticks to wait */
  function = (watchdog_t) m_ptr->FUNC_TO_CALL;
  					/* function to call (tasks only) */
  rp = proc_addr(proc_nr);
  mc.SECONDS_LEFT = (rp->p_alarm == 0 ? 0 : (rp->p_alarm - realtime)/HZ );
  if (!istaskp(rp)) function= 0;	/* user processes get signaled */
  assert(!isservp(rp));			/* Servers need synchronous alarms */
  common_setalarm(proc_nr, delta_ticks, function);
}


/*===========================================================================*
 *				do_setsyn_alrm				     *
 *===========================================================================*/
PRIVATE void do_setsyn_alrm(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* A process wants an synchronous alarm. 
 */

  register struct proc *rp;
  int proc_nr;			/* which process wants the alarm */
  long delta_ticks;		/* in how many clock ticks does he want it? */

  /* Extract the parameters from the message. */
  proc_nr = m_ptr->CLOCK_PROC_NR;	/* process to interrupt later */
  delta_ticks = m_ptr->DELTA_TICKS;	/* how many ticks to wait */
  rp = proc_addr(proc_nr);
  mc.SECONDS_LEFT = (rp->p_alarm == 0 ? 0 : (rp->p_alarm - realtime)/HZ );
  common_setalarm(proc_nr, delta_ticks, cause_alarm);
}


/*===========================================================================*
 *				do_getuptime				     *
 *===========================================================================*/
PRIVATE void do_getuptime()
{
/* Get and return the current clock uptime in ticks. */

  mc.NEW_TIME = realtime;	/* current uptime */
}


/*===========================================================================*
 *				do_sysutime				     *
 *===========================================================================*/
PRIVATE int do_sysutime(m_ptr)
message *m_ptr;			/* pointer to request message */
{
/* A process requests time/timer operations with micro second resolution. */

	int proc, op;
	struct timeval tv;
	char *tvp;
	clock_t reply_ticks, req_ticks;
	time_t reply_offset;
	long usecs;
	phys_bytes u_phys, k_phys;
	struct proc *pp;

	proc= m_ptr->m1_i1;
	op= m_ptr->m1_i2;
	tvp= m_ptr->m1_p1;
	pp= proc_addr(proc);

	k_phys= vir2phys(&tv);
	u_phys= numap(proc, (vir_bytes)tvp, sizeof(tv));
	if (u_phys == 0)
		return EFAULT;

	usecs= 0;
	switch(op)
	{
	case UTIME_TIMEOFDAY:
		reply_ticks= realtime;
		reply_offset= boot_time;
		usecs= getusecs(&reply_ticks);
		break;
	case UTIME_BOOTTIME:
		reply_ticks= 0;
		reply_offset= boot_time;
		break;
	case UTIME_UPTIME:
		reply_ticks= realtime;
		reply_offset= 0;
		usecs= getusecs(&reply_ticks);
		break;
	case UTIME_GETALARM:
		reply_ticks= pp->p_alarm;
		if (reply_ticks != 0)
			reply_offset= boot_time;
		else
			reply_offset= 0;
		break;
	case UTIME_SETALARM:
		reply_ticks= pp->p_alarm;
		if (reply_ticks != 0)
			reply_offset= boot_time;
		else
			reply_offset= 0;
		phys_copy(u_phys, k_phys, sizeof(tv));
		if (tv.tv_sec == 0 && tv.tv_usec == 0)
			req_ticks= 0;
		else if (tv.tv_sec <= boot_time)
			req_ticks= 1;
		else
		{
			req_ticks= (tv.tv_sec-boot_time)*HZ;
			if (req_ticks/HZ != (tv.tv_sec-boot_time))
			{
				/* Overflow. */
				req_ticks= LONG_MAX;
			}
			else
			{
				if (tv.tv_usec != 0)
					req_ticks += ((tv.tv_usec-1) * HZ) / 
							1000000 + 1;
			}
		}
		pp->p_alarm= req_ticks;
		if (req_ticks != 0 && 
				(next_alarm == 0 || req_ticks < next_alarm))
		{
			next_alarm= req_ticks;
		}
		break;
	case UTIME_DEFRATE:
		tv.tv_sec= DEF_TICK_CYCLES;
		tv.tv_usec= 0;
		phys_copy(k_phys, u_phys, sizeof(tv));
		return OK;
	case UTIME_GETRATE:
		tv.tv_sec= tick_cycles-1;
		tv.tv_usec= tick_cycles_frac;
		phys_copy(k_phys, u_phys, sizeof(tv));
		return OK;
	case UTIME_SETRATE:
		phys_copy(u_phys, k_phys, sizeof(tv));
		if (tv.tv_sec < DEF_TICK_CYCLES/2 ||
			tv.tv_sec > DEF_TICK_CYCLES*2)
		{
			return EINVAL;
		}
		lock();
		tick_cycles= tv.tv_sec+1;
		tick_cycles_frac= tv.tv_usec;
		tick_low= -1;
		unlock();
		return OK;
	default:
		printf(
		"do_sysutime: process %d requested illegal operation %d\n",
			proc, op);
		return EINVAL;
	}

	/* Convert the number of ticks we are to reply to a 
	 * struct timeval.
	 */
	tv.tv_sec= reply_ticks / HZ + reply_offset;
	tv.tv_usec= (reply_ticks % HZ) * 1000000 / HZ + usecs;
	if (tv.tv_usec >= 1000000)
	{
		tv.tv_usec -= 1000000;
		tv.tv_sec++;
	}
	phys_copy(k_phys, u_phys, sizeof(tv));
	return OK;
}


/*===========================================================================*
 *				common_setalarm				     *
 *===========================================================================*/
PRIVATE void common_setalarm(proc_nr, delta_ticks, function)
int proc_nr;			/* which process wants the alarm */
long delta_ticks;		/* in how many clock ticks does he want it? */
watchdog_t function;		/* function to call (0 if cause_sig is
				 * to be called */
{
/* Record an alarm request and check to see it is the next alarm needed.  */

  register struct proc *rp;

  rp = proc_addr(proc_nr);
  rp->p_alarm = (delta_ticks == 0 ? 0 : realtime + delta_ticks);
  rp->p_watchdog= function;

  /* Which alarm is next? */
  next_alarm = LONG_MAX;
  for (rp = BEG_PROC_ADDR; rp < END_PROC_ADDR; rp++)
	if(rp->p_alarm != 0 && rp->p_alarm < next_alarm)next_alarm=rp->p_alarm;

}


/*===========================================================================*
 *				get_uptime				     *
 *===========================================================================*/
PUBLIC clock_t get_uptime()
{
/* Get and return the current clock uptime in ticks.  This function is
 * designed to be called from other tasks, so it has to be careful about
 * pending_ticks.
 */

  clock_t uptime;

  lock();
  uptime = realtime + pending_ticks;
  unlock();
  return(uptime);
}


/*===========================================================================*
 *				syn_alrm_task				     *
 *===========================================================================*/
PUBLIC void syn_alrm_task()
{
/* Main program of syn_alrm task.  It sends CLOCK_INT messages to process that
 * requested a syn_alrm.
 */
 
  struct proc *pp, *plist;
  message mess;
  int work_done;	/* ready to sleep ? */

  synal_tasknr= proc_number(proc_ptr);

  syn_al_alive= TRUE;

  while (TRUE) {
	work_done= TRUE;
	plist= syn_alarm_head;
	syn_alarm_head= NULL;
	while (plist)
	{
		pp= plist;
		plist= pp->p_signal;
		pp->p_signal= NULL;

		mess.m_type= CLOCK_INT;
		send (proc_number(pp), &mess); 
		work_done= FALSE;
	}
	if (sig_proclist != NULL)
	{
		mess.m_type= KSIG;
		send(MM_PROC_NR, &mess);
		work_done= FALSE;
	}
	if (work_done) {
		syn_al_alive= FALSE;
		receive (ANY, &mess);
		syn_al_alive= TRUE;
	}
  }
}


/*===========================================================================*
 *				cause_alarm				     *
 *===========================================================================*/
PRIVATE void cause_alarm()
{
/* Routine called if a timer goes off and the process requested a synchronous
 * alarm. The process number is in the global variable watchdog_proc (HACK).
 */
  struct proc *pp;

  pp= proc_addr(watchdog_proc);
  assert(pp->p_signal == NULL);
  pp->p_signal= syn_alarm_head;
  syn_alarm_head= pp;
  if (!syn_al_alive) interrupt(synal_tasknr);
}
#if (CHIP == INTEL)


/*===========================================================================*
 *				init_clock				     *
 *===========================================================================*/
PRIVATE void init_clock()
{
/* Initialize channel 0 of the 8253A timer to e.g. 60 Hz. */

  out_byte(TIMER_CTRL, TIMER_MODE);	/* set timer to run continuously */
  out_byte(TIMER0, tick_cycles);	/* load timer low byte */
  out_byte(TIMER0, tick_cycles >> 8);	/* load timer high byte */
  put_irq_handler(CLOCK_IRQ, clock_handler);	/* set the interrupt handler */
  enable_irq(CLOCK_IRQ);		/* ready for clock interrupts */
}


/*===========================================================================*
 *				clock_stop				     *
 *===========================================================================*/
PUBLIC void clock_stop()
{
/* Reset the clock to the BIOS rate. */

  out_byte(TIMER_CTRL, 0x36);
  out_byte(TIMER0, 0);
  out_byte(TIMER0, 0);
}


/*==========================================================================*
 *				micro_init				    *
 *==========================================================================*/
PUBLIC void micro_init(tvp)
struct timeval *tvp;
{
  /* Prepare for calls to micro_elapsed(). */
  tvp->tv_sec = -1;
}


/*==========================================================================*
 *				micro_elapsed				    *
 *==========================================================================*/
PUBLIC unsigned long micro_elapsed(tvp)
struct timeval *tvp;
{
  /* Return the number of microseconds that have elapsed since tvp was
   * initalized by micro_init().
   *
   * IMPORTANT: this function has to be called at least once per
   * clocktick.
   *
   * Micro_elapsed() is used by micro_delay() to busy wait until some
   * number of microseconds have elapsed.  Micro_elapsed() can also be
   * used to poll a device for some time.
   */
  unsigned counter;

  /* Read the counter of channel 0 of the 8253A timer. */
  lock();
  out_byte(TIMER_CTRL, LATCH_COUNT);
  counter= in_byte(TIMER0);
  counter |= (in_byte(TIMER0) << 8);
  unlock();

  if (tvp->tv_sec == -1) {
	/* This is the first call, use 'counter' as the start. */
	tvp->tv_usec= 0;
	tvp->tv_sec= counter;
  }

  /* Add difference between previous and new count unless the counter has
   * increased (restarted its cycle).  In that case add 1, which should be
   * correct when polling rapidly.
   */
  tvp->tv_usec += counter < tvp->tv_sec ? (tvp->tv_sec - counter) : 1;
  tvp->tv_sec = counter;

  /* Return the number of microseconds counted.  Note that the counter
   * cycles through HZ times per second: HZ * tick_cycles == 1000000 us.
   * So one timer count = 1000000 / tick_cycles / HZ microseconds.
   */
  if (tvp->tv_usec < ULONG_MAX / 1000000) {
	/* Precise for about 3600 us. */
	return tvp->tv_usec * 1000000 / tick_cycles / HZ;
  } else {
	/* Longer periods need not be so precise. */
	return tvp->tv_usec * (1000000 / tick_cycles) / HZ;
  }
}


/*==========================================================================*
 *				micro_delay				    *
 *==========================================================================*/
PUBLIC void micro_delay(micros)
unsigned long micros;
{
/* Delay some microseconds. */
  struct timeval tv;

  micro_init(&tv);
  while (micro_elapsed(&tv) < micros)
	/* do nothing */;
}


/*==========================================================================*
 *				getusecs				    *
 *==========================================================================*/
PRIVATE long getusecs(ticks)
clock_t *ticks;
{
	/* Return the current number of pending ticks and the
	 * number of microseconds in the current tick.
	 *
	 * Algorithm: read the number of pending ticks followed by value
	 * of the timer and again the number of ticks. If the number of
	 * ticks differs, repeat the process.
	 */

	unsigned low_byte, v, tck0, tck1;
	long usecs;
	int i;

	lock();
	tck0= pending_ticks;
	unlock();

	for (i= 0; i<100; i++)
	{
		lock();
		out_byte(TIMER_CTRL, LATCH_COUNT);
		low_byte = in_byte(TIMER0);
		v= (in_byte(TIMER0) << 8) + low_byte;
		unlock();
		if (v > tick_cycles)
			continue;

		/* Give the clock interrupt handler a chance to update
		 * pending_ticks
		 */
		lock();
		tck1= pending_ticks;
		unlock();
		if (tck0 != tck1)
		{
			tck0= tck1;
			continue;
		}

		/* Convert a timer value to micro seconds using fixed point
		 * arithmatics. We are to calculate the following:
		 * 1000000 / HZ * (tick_cycles - v) / tick_cycles;
		 * tick_cycles is between 19882/2 and 19982*2 so we can savely
		 * calculate 100000 * (tick_cycles - v) / tick_cycles;
		 * we divide that result by HZ and multiply by 10.
		 */
		 usecs= 100000 * (tick_cycles - v) / tick_cycles / HZ;
		 usecs *= 10;

		 (*ticks) += tck1;
		 return usecs;
	}
	panic("getusecs: unable to get a consistent readout of the timer",
		NO_NUM);
	/*NOTREACHED*/
}
#endif

#if (CHIP == M68000)
#include "staddr.h"
#include "stmfp.h"

/*===========================================================================*
 *				init_clock				     *
 *===========================================================================*/
PRIVATE void init_clock()
{
/* Initialize the timer C in the MFP 68901.
 * Reducing to HZ is not possible by hardware.  The resulting interrupt
 * rate is further reduced by software with a factor of 4.
 * Note that the expression below works for both HZ=50 and HZ=60.
 */
  do {
	MFP->mf_tcdr = TIMER_FREQ/(64*4*HZ);
  } while ((MFP->mf_tcdr & 0xFF) != TIMER_FREQ/(64*4*HZ));
  MFP->mf_tcdcr |= (T_Q064<<4);
}
#endif

/*===========================================================================*
 *				clock_handler				     *
 *===========================================================================*/
PRIVATE int clock_handler(irq)
int irq;
{
/* Switch context to do_clocktick if an alarm has gone off.
 * Also switch there to reschedule if the reschedule will do something.
 * This happens when
 *	(1) quantum has expired
 *	(2) current process received full quantum (as clock sampled it!)
 *	(3) something else is ready to run.
 * Also call TTY and PRINTER and let them do whatever is necessary.
 *
 * Many global global and static variables are accessed here.  The safety
 * of this must be justified.  Most of them are not changed here:
 *	k_reenter:
 *		This safely tells if the clock interrupt is nested.
 *	proc_ptr, bill_ptr:
 *		These are used for accounting.  It does not matter if proc.c
 *		is changing them, provided they are always valid pointers,
 *		since at worst the previous process would be billed.
 *	next_alarm, realtime, sched_ticks, bill_ptr, prev_ptr,
 *	rdy_head[USER_Q]:
 *		These are tested to decide whether to call interrupt().  It
 *		does not matter if the test is sometimes (rarely) backwards
 *		due to a race, since this will only delay the high-level
 *		processing by one tick, or call the high level unnecessarily.
 * The variables which are changed require more care:
 *	rp->user_time, rp->sys_time:
 *		These are protected by explicit locks in system.c.  They are
 *		not properly protected in dmp.c (the increment here is not
 *		atomic) but that hardly matters.
 *	pending_ticks:
 *		This is protected by explicit locks in clock.c.  Don't
 *		update realtime directly, since there are too many
 *		references to it to guard conveniently.
 *	lost_ticks:
 *		Clock ticks counted outside the clock task.
 *	sched_ticks, prev_ptr:
 *		Updating these competes with similar code in do_clocktick().
 *		No lock is necessary, because if bad things happen here
 *		(like sched_ticks going negative), the code in do_clocktick()
 *		will restore the variables to reasonable values, and an
 *		occasional missed or extra sched() is harmless.
 *
 * Are these complications worth the trouble?  Well, they make the system 15%
 * faster on a 5MHz 8088, and make task debugging much easier since there are
 * no task switches on an inactive system.
 */

  register struct proc *rp;
  register struct proc *bp;
  register unsigned ticks;
  unsigned long new_frac;
  int low;
  unsigned cycles;

  assert(irq == CLOCK_IRQ);

  if (ps_mca) {
	/* Acknowledge the PS/2 clock interrupt. */
	out_byte(PORT_B, in_byte(PORT_B) | CLOCK_ACK_BIT);
  }

  /* Set counter */
  new_frac= (tick_frac + tick_cycles_frac) & 0xffffffff;
  low= (new_frac >= tick_frac);
  tick_frac= new_frac;
  if (low != tick_low)
  {
	tick_low= low;
	cycles= tick_cycles-low;
	/* out_byte(TIMER_CTRL, TIMER_MODE); */
	out_byte(TIMER0, cycles);
	in_byte(TIMER0);
	out_byte(TIMER0, cycles >> 8);
	in_byte(TIMER0);
  }
#ifdef SHOW_EVENT
  if (debug)
	show_event(irq, 'C');
#endif

  /* Update user and system accounting times.
   * First charge the current process for user time.
   * If the current process is not the billable process (usually because it
   * is a task), charge the billable process for system time as well.
   * Thus the unbillable tasks' user time is the billable users' system time.
   */
  rp= proc_ptr;
  bp= bill_ptr;
  if (k_reenter != 0)
  {
  	if (!bp)
  		bp= rp;
  	rp = cproc_addr(HARDWARE);
  }
  ticks = lost_ticks + 1;
  lost_ticks = 0;
  rp->user_time += ticks;
  if (bp && bp != rp)
  	bp->sys_time += ticks;

  pending_ticks += ticks;

#if PROFILING && SYNC_PROFILING
  if (profiling)
	profile(rp);
#endif /* PROFILING && SYNC_PROFILING */

  if (next_alarm <= realtime + pending_ticks) {
  	interrupt(clck_tasknr);
  } else
  if (tac_exp_time && tac_exp_time <= realtime + pending_ticks) {
  	interrupt(clck_tasknr);
  }
  lock_sched();	

#ifdef SHOW_EVENT
  if (debug)
	show_event(irq, ' ');
#endif

  return 1;	/* Reenable clock interrupt */
}

/*
 * $PchId: clock.c,v 1.4 1996/01/19 23:07:29 philip Exp $
 */
