/*	this file contains the interface of the network software with rest of
	minix. Furthermore it contains the main loop of the network task.

Copyright 1995 Philip Homburg

The valid messages and their parameters are:

from FS:
 __________________________________________________________________
|		|           |         |       |          |         |
| m_type	|   DEVICE  | PROC_NR |	COUNT |	POSITION | ADDRESS |
|_______________|___________|_________|_______|__________|_________|
|		|           |         |       |          |         |
| NW_OPEN 	| minor dev | proc nr | mode  |          |         |
|_______________|___________|_________|_______|__________|_________|
|		|           |         |       |          |         |
| NW_CLOSE 	| minor dev | proc nr |       |          |         |
|_______________|___________|_________|_______|__________|_________|
|		|           |         |       |          |         |
| NW_IOCTL	| minor dev | proc nr |       |	NWIO..	 | address |
|_______________|___________|_________|_______|__________|_________|
|		|           |         |       |          |         |
| NW_READ	| minor dev | proc nr |	count |          | address |
|_______________|___________|_________|_______|__________|_________|
|		|           |         |       |          |         |
| NW_WRITE	| minor dev | proc nr |	count |          | address |
|_______________|___________|_________|_______|__________|_________|
|		|           |         |       |          |         |
| NW_CANCEL	| minor dev | proc nr |       |          |         |
|_______________|___________|_________|_______|__________|_________|

from DL_ETH:
 _______________________________________________________________________
|		|           |         |          |            |         |
| m_type	|  DL_PORT  | DL_PROC |	DL_COUNT |  DL_STAT   | DL_TIME |
|_______________|___________|_________|__________|____________|_________|
|		|           |         |          |            |         |
| DL_TASK_INT 	| minor dev | proc nr | rd_count |  0  | stat |  time   |
|_______________|___________|_________|__________|____________|_________|
|		|           |         |          |            |         |
| DL_TASK_REPLY	| minor dev | proc nr | rd_count | err | stat |  time   |         |
|_______________|___________|_________|__________|____________|_________|
*/

#include "inet.h"

#define _MINIX_SOURCE 1

#include <unistd.h>
#include <sys/stat.h>
#include <sys/svrctl.h>

#include "mq.h"
#include "qp.h"
#include "proto.h"
#include "generic/type.h"

#include "generic/arp.h"
#include "generic/assert.h"
#include "generic/buf.h"
#include "generic/clock.h"
#include "generic/eth.h"
#include "generic/event.h"
#include "generic/ip.h"
#include "generic/psip.h"
#include "generic/sr.h"
#include "generic/tcp.h"
#include "generic/udp.h"

INIT_PANIC();

int this_proc;		/* Process number of this server. */

static int dpeth_tasknr= ANY;
static int synal_tasknr= ANY;

/* Killing Solaris */
int killer_inet= 0;

#ifdef BUF_CONSISTENCY_CHECK
extern int inet_buf_debug;
#endif

_PROTOTYPE( void main, (void) );

FORWARD _PROTOTYPE( void nw_init, (void) );

PUBLIC void main()
{
	mq_t *mq;
	int r;
	int source;
	struct stat stb;
	struct fssignon device;
	struct systaskinfo info;

	/* What device group am I to manage? */
	if (stat("/dev/eth0", &stb) < 0) {
		write(2, "inet: can't access device /dev/eth0\n", 36);
		_exit(1);
	}

	/* Sign on as a server at all offices in the proper order. */
	if (svrctl(MMSIGNON, (void *) NULL) == -1) {
		write(2, "inet: server signon failed\n", 27);
		_exit(1);
	}
	if (svrctl(SYSSIGNON, (void *) &info) == -1) pause();

	/* Our new identity as a server. */
	this_proc = info.proc_nr;

	/* Register the device group. */
	device.dev= stb.st_rdev;
	device.style= STYLE_CLONE;
	if (svrctl(FSSIGNON, (void *) &device) == -1) {
		printf("error %d on registering ethernet devices\n", errno);
		pause();
	}

	DBLOCK(1, printf("%s\n", version));

#ifdef BUF_CONSISTENCY_CHECK
	inet_buf_debug= (getenv("inetbufdebug") && 
		(strcmp(getenv("inetbufdebug"), "on") == 0));
	inet_buf_debug= 100;
	if (inet_buf_debug)
	{
		ip_warning(( "buffer consistency check enabled" ));
	}
#endif

	if (getenv("killerinet"))
	{
		ip_warning(( "killer inet active" ));
		killer_inet= 1;
	}

	r= sys_findproc(DP8390_NAME, &dpeth_tasknr, 0);
	if (r != OK)
		ip_panic(( "unable to find dp8390 task: %d\n", r ));
	r= sys_findproc(SYN_AL_NAME, &synal_tasknr, 0);
	if (r != OK)
		ip_panic(( "unable to find synchronous alarm task: %d\n", r ));

	nw_init();
	while (TRUE)
	{
#ifdef BUF_CONSISTENCY_CHECK
		if (inet_buf_debug)
		{
			static int buf_debug_count= 0;

			if (buf_debug_count++ > inet_buf_debug)
			{
				buf_debug_count= 0;
				if (!bf_consistency_check())
					break;
			}
		}
#endif
		if (ev_head)
		{
			ev_process();
			continue;
		}
		if (clck_call_expire)
		{
			clck_expire_timers();
			continue;
		}
		mq= mq_get();
		if (!mq)
			ip_panic(("out of messages"));

		r= receive (ANY, &mq->mq_mess);
		if (r<0)
		{
			ip_panic(("unable to receive: %d", r));
		}
		reset_time();
		source= mq->mq_mess.m_source;
		if (source == FS_PROC_NR)
		{
			sr_rec(mq);
		}
		else if (source == dpeth_tasknr)
		{
compare(mq->mq_mess.m_type, ==, DL_TASK_REPLY);
			eth_rec(&mq->mq_mess);
			mq_free(mq);
		}
		else if (source == synal_tasknr)
		{
			clck_tick (&mq->mq_mess);
			mq_free(mq);
		}
		else
		{
			ip_panic(("message from unknown source: %d",
				mq->mq_mess.m_source));
		}
	}
	ip_panic(("task is not allowed to terminate"));
}

PRIVATE void nw_init()
{
	mq_init();
	qp_init();
	bf_init();
	clck_init();
	sr_init();
	eth_init();
	arp_init();
	psip_init();
	ip_init();
	tcp_init();
	udp_init();
}

void abort()
{
	static message m;

	sys_abort(RBT_PANIC);
	while (receive(HARDWARE, &m) == OK)
		;
	_exit(1);
}

/*
 * $PchId: inet.c,v 1.10 1995/11/23 11:26:28 philip Exp $
 */
