/* This file takes care of those system calls that deal with time.
 *
 * The entry points into this file are
 *   do_utime:	perform the UTIME system call
 *   do_time:	perform the TIME system call
 *   do_stime:	perform the STIME system call
 *   do_tims:	perform the TIMES system call
 */

#include "fs.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "param.h"
#include "assert.h"
INIT_ASSERT

PRIVATE message clock_mess;

/*===========================================================================*
 *				do_utime				     *
 *===========================================================================*/
PUBLIC int do_utime()
{
/* Perform the utime(name, timep) system call. */

  register struct inode *rip;
  register int r;
  int fs_err= EGENERIC;

  assert(fs_call == UTIME);

  if (utime_namelen == 0) {
	/* Fix standard Minix weirdness. */
	utime_namelen = utime_flag;
	utime_flag = 1;
  }

  /* Temporarily open the file. */
  if ((fs_err= fetch_name(utime_name, utime_namelen, M1)) != OK) return(fs_err);
  if ( (rip = eat_path(user_path, &fs_err)) == NIL_INODE) return(fs_err);

  /* Only the owner of a file or the super_user can change its time. */
  r = OK;
  if (rip->i_uid != fp->fp_effuid && !super_user) r = EPERM;
  if (read_only(rip) != OK) r = EROFS;	/* not even su can touch if R/O */
  if (r == OK) {
	if (utime_flag) {
		rip->i_atime = clock_time();
		rip->i_mtime = rip->i_atime;
	} else {
		rip->i_atime = utime_actime;
		rip->i_mtime = utime_modtime;
	}
	rip->i_update = CTIME;	/* discard any stale ATIME and MTIME flags */
	rip->i_dirt = DIRTY;
  }

  put_inode(rip);
  return(r);
}


/*===========================================================================*
 *				do_time					     *
 *===========================================================================*/
PUBLIC int do_time()

{
/* Perform the time(tp) system call. */

  assert(fs_call == TIME);

  reply_l1 = clock_time();	/* return time in seconds */
  return(OK);
}


/*===========================================================================*
 *				do_stime				     *
 *===========================================================================*/
PUBLIC int do_stime()
{
/* Perform the stime(tp) system call. */

  register int k;

  assert(fs_call == STIME);

  if (!super_user) return(EPERM);
  clock_mess.m_type = SET_TIME;
  clock_mess.NEW_TIME = (long) stime_tp;
  if ( (k = sendrec(clck_tasknr, &clock_mess)) != OK)
  	panic("do_stime error", k);
  return (OK);
}


/*===========================================================================*
 *				do_tims					     *
 *===========================================================================*/
PUBLIC int do_tims()
{
/* Perform the times(buffer) system call. */

  clock_t t[5];

  assert(fs_call == TIMES);

  sys_times(who, t);
  reply_t1 = t[0];
  reply_t2 = t[1];
  reply_t3 = t[2];
  reply_t4 = t[3];
  reply_t5 = t[4];
  return(OK);
}


/*===========================================================================*
 *				do_sysutime				     *
 *===========================================================================*/
PUBLIC int do_sysutime()
{
	assert(fs_call == SYSUTIME);

	return clock_utime(who, sysutime_op, sysutime_tvp);
}

/*
 * $PchId: time.c,v 1.4 1995/11/28 07:29:58 philip Exp $
 */
