/* Super block table.  The root file system and every mounted file system
 * has an entry here.  The entry holds information about the sizes of the bit
 * maps and inodes.  The s_ninodes field gives the number of inodes available
 * for files and directories, including the root directory.  Inode 0 is 
 * on the disk, but not used.  Thus s_ninodes = 4 means that 5 bits will be
 * used in the bit map, bit 0, which is always 1 and not used, and bits 1-4
 * for files and directories.  The disk layout is:
 *
 *      Item        # blocks
 *    boot block      1
 *    super block     1
 *    inode map     s_imap_blocks
 *    zone map      s_zmap_blocks
 *    inodes        (s_ninodes + 'inodes per block' - 1)/'inodes per block'
 *    unused        whatever is needed to fill out the current zone
 *    data zones    (s_zones - s_firstdatazone) << s_log_zone_size
 *
 * A super_block slot is free if s_dev == NO_DEV. 
 */


EXTERN struct super_block {
  ino_t s_ninodes;		/* # usable inodes on the minor device */
  zone1_t  s_nzones;		/* total device size, including bit maps etc */
  short s_imap_blocks;		/* # of blocks used by inode bit map */
  short s_zmap_blocks;		/* # of blocks used by zone bit map */
  zone1_t s_firstdatazone;	/* number of first data zone */
  char s_log_zone_size;		/* log2 of blocks/zone */
  char s_flags;			/* flex and clean flags */
  uoff_t s_max_size;		/* maximum file size on this device */
  short s_magic;		/* magic number to recognize super-blocks */
  char s_fsck_magic[2];		/* magic number to enable the clean flag */
  zone_t s_zones;		/* number of zones (replaces s_nzones in V2) */

  /* The following items are only used when the super_block is in memory. */
  struct buf *s_imap[I_MAP_SLOTS]; /* pointers to the in-core inode bit map */
  struct buf *s_zmap[Z_MAP_SLOTS]; /* pointers to the in-core zone bit map */
  struct buf *s_superblock;	/* ptr to the superblock iff r/w and clean */
  unsigned s_inodes_per_block;	/* precalculated from magic number */
  dev_t s_dev;			/* whose super block is this? */
  int s_rd_only;		/* set to 1 iff file sys mounted read only */
  char s_nosuid;		/* set to 1 iff set-uid not allowed */
  char s_grpid;			/* set to 1 iff group ids must be inherited */
  char s_nf;			/* set to 1 iff next-fit should be used */
  unsigned s_sync_ct;		/* mark clean after N sync's */
  int s_native;			/* set to 1 iff not byte swapped file system */
  int s_version;		/* file system version, zero means bad magic */
  int s_ndzones;		/* # direct zones in an inode */
  int s_nindirs;		/* # indirect zones per indirect block */
  unsigned s_link_max;		/* max # of links per file */
  bit_t s_isearch;		/* inodes below this bit number are in use */
  bit_t s_zsearch;		/* all zones below this bit number are in use*/
} super_block[NR_SUPERS];

/* Flag bits in s_flags: */
#define S_FLEX	0x01		/* if the file system has flex dirs */
#define S_CLEAN	0x02		/* if the file system was properly unmounted */

#define N_SYNCS_CLEAN	4	/* N syncs with no write: mark clean */

/* Macro for efficient clean flag updates. */
#define super_clean(sp, clean) \
	(((!(clean) && ((sp)->s_flags & S_CLEAN)) || \
	  ((clean) && !((sp)->s_flags & S_CLEAN))) ? \
	 (void)do_super_clean((sp), (clean)) : (void)0)

#define NIL_SUPER (struct super_block *) 0

struct inode;

EXTERN struct mount {
  char m_special[64];		/* name of mounted device */
  char m_mountp[64];		/* mount point */
  struct inode *m_iroot;	/* associated inodes (null when free) */
  struct inode *m_imount;
} mount_table[NR_MOUNTS];

/*
 * $PchId: super.h,v 1.5 1995/11/28 07:31:56 philip Exp $
 */
