/* This file manages the super block table and the related data structures,
 * namely, the bit maps that keep track of which zones and which inodes are
 * allocated and which are free.  When a new inode or zone is needed, the
 * appropriate bit map is searched for a free entry.
 *
 * The entry points into this file are
 *   load_bit_maps:   get the bit maps for the root or a newly mounted device
 *   unload_bit_maps: write the bit maps back to disk after an UMOUNT
 *   alloc_bit:       somebody wants to allocate a zone or inode; find one
 *   free_bit:        indicate that a zone or inode is available for allocation
 *   search_super:    search the 'superblock' table for a device
 *   get_super:       get the 'superblock' table for a device (can't fail)
 *   read_super:      read a superblock
 *   do_super_clean:  set or unset the superblock clean flag
 */

#include "fs.h"
#include <string.h>
#include "assert.h"
INIT_ASSERT
#include "buf.h"
#include "inode.h"
#include "super.h"

#define BITCHUNK_BITS	(usizeof(bitchunk_t) * CHAR_BIT)
#define BITS_PER_BLOCK	(BITMAP_CHUNKS * BITCHUNK_BITS)

/*===========================================================================*
 *				load_bit_maps				     *
 *===========================================================================*/
PUBLIC int load_bit_maps(dev)
dev_t dev;			/* which device? */
{
/* Load the bit map for some device into the cache and set up superblock. */

  register int i;
  register struct super_block *sp;
  struct buf *bp;
  unsigned word;

  sp = get_super(dev);		/* get the superblock pointer */
  if (bufs_in_use + sp->s_imap_blocks + sp->s_zmap_blocks >= nr_bufs - 3)
	return(EGENERIC);		/* insufficient buffers left for bit maps */
  if (sp->s_imap_blocks > I_MAP_SLOTS || sp->s_zmap_blocks > Z_MAP_SLOTS)
	return(EGENERIC);

  /* Clear the inode map pointers to indicate that no map blocks loaded yet. */
  for (i = 0; i < sp->s_imap_blocks; i++)
	sp->s_imap[i] = NULL;

  /* Clear the zone map. */
  for (i = 0; i < sp->s_zmap_blocks; i++)
	sp->s_zmap[i] = NULL;

  /* Inodes 0 and 1 are never allocated.  Mark them as busy. */
  bp= get_block(dev, SUPER_BLOCK + 1 , BF_NORMAL);
  word= conv2(sp->s_native, bp->b_bitmap[0]);
  if ((word & 3) != 3)					/* inodes 0, 1 busy */
  	panic("inconsistent filesystem", NO_NUM);
  put_block(bp);

  bp= get_block(sp->s_dev, SUPER_BLOCK + 1 + sp->s_imap_blocks, BF_NORMAL);
  word= conv2(sp->s_native, bp->b_bitmap[0]);
  if ((word & 1) != 1)
  	panic("inconsistent filesystem", NO_NUM);
  put_block(bp);
  return(OK);
}



/*===========================================================================*
 *				unload_bit_maps				     *
 *===========================================================================*/
PUBLIC int unload_bit_maps(dev)
dev_t dev;			/* which device is being unmounted? */
{
/* Unload the bit maps so a device can be unmounted. */

  /* Nothing to do */

  return(OK);
}


/*===========================================================================*
 *				alloc_bit				     *
 *===========================================================================*/
PUBLIC bit_t alloc_bit(sp, map, origin)
struct super_block *sp;
int map;			/* MP_INODE or MP_ZONE */
bit_t origin;			/* number of bit to start searching at */
{
/* Allocate a bit from a bit map and return its bit number. */

  unsigned i, o, w, w_off, b, count;
  unsigned first_map_block, bit_blocks;
  struct buf **map_ptr;
  bit_t map_bits;
  bit_t a;
  register bitchunk_t *wptr, *wlim;
  bitchunk_t k;
  unsigned block_count;
  struct buf *bp;

  if (map == MP_ZONE)
  {
	map_bits = (bit_t) (sp->s_zones - (sp->s_firstdatazone - 1) );
	bit_blocks = (unsigned) sp->s_zmap_blocks; /* # blocks in the bit map */
	first_map_block= SUPER_BLOCK + 1 + sp->s_imap_blocks;
	map_ptr= sp->s_zmap;
  }
  else
  {
  	map_bits = (bit_t)sp->s_ninodes+1;
	bit_blocks = sp->s_imap_blocks;
	first_map_block= SUPER_BLOCK + 1;
	map_ptr= sp->s_imap;
  }

  if (sp->s_rd_only) panic("can't allocate bit on read-only filesys.", NO_NUM);
  
  /* Figure out where to start the bit search (depends on 'origin'). */
  assert(origin > 0 && origin < map_bits);

  /* Truncation of the next expression from a bit_t to an int is safe because
   * it it is inconceivable that the number of blocks in the bit map > 32K.
   */  
  b = (unsigned) (origin / BITS_PER_BLOCK);	/* relevant bit map block. */

  /* Truncation of the next expression from a bit_t to an int is safe because
   * its value is smaller than BITS_PER_BLOCK and easily fits in an int.
   */  
  o = (unsigned) (origin % BITS_PER_BLOCK);
  w = o / BITCHUNK_BITS;
  block_count = (w == 0 ? bit_blocks : bit_blocks + 1);

  /* The outer while loop iterates on the blocks of the map.  The inner
   * while loop iterates on the words of a block.  The for loop iterates
   * on the bits of a word.
   */
  while (block_count--) {
	/* If need be, loop on all the blocks in the bit map. */
	bp = map_ptr[b];
	bp= buf_vrfy_ptr(bp, sp->s_dev, b+first_map_block);
	map_ptr[b]= bp;
	wptr = &bp->b_bitmap[w];
	wlim = &bp->b_bitmap[BITMAP_CHUNKS];
	count = 0;
	while (count < BITMAP_CHUNKS) {
		/* Loop on all the words of one of the bit map blocks. */
		if (*wptr != 0xFFFF) {
			/* This word contains a free bit.  Allocate it. */
			k = conv2(sp->s_native, (int) *wptr);
			for (i = 0; i < BITCHUNK_BITS; i++)
				if (((k >> i) & 1) == 0) {
					w_off = (int)(wptr - &bp->b_bitmap[0]);
					w_off = w_off * BITCHUNK_BITS;
					a = i + w_off
					    + ((bit_t) b * BITS_PER_BLOCK);
					/* If 'a' beyond map check other blks*/
					if (a >= map_bits) {
						wptr = wlim - 1;
						break;
					}
					k |= 1 << i;
					*wptr = conv2(sp->s_native, (int) k);
					bp->b_dirt |= GRIMY;
					put_block(bp);
					return(a);
				}
		}
		if (++wptr == wlim) wptr = &bp->b_bitmap[0];	/* wrap */
		count++;
	}
	if (++b == bit_blocks) b = 0;	/* we have wrapped around */
	w = 0;
	put_block(bp);
  }
  return(NO_BIT);		/* no bit could be allocated */
}


/*===========================================================================*
 *				free_bit				     *
 *===========================================================================*/
PUBLIC void free_bit(sp, map, bit_returned)
struct super_block *sp;
int map;			/* MP_INODE or MP_ZONE */
bit_t bit_returned;		/* number of bit to insert into the map */
{
/* Return a zone or inode by turning off its bitmap bit. */

  unsigned b, r, w, bit;
  unsigned first_map_block;
  struct buf **map_ptr;
  struct buf *bp;
  bitchunk_t k;

  if (map == MP_ZONE)
  {
	first_map_block= SUPER_BLOCK + 1 + sp->s_imap_blocks;
	map_ptr= sp->s_zmap;
  }
  else
  {
	first_map_block= SUPER_BLOCK + 1;
	map_ptr= sp->s_imap;
  }

  if (sp->s_rd_only) panic("can't free bit on read-only file system", NO_NUM);

  /* The truncations in the next two assignments are valid by the same
   * reasoning as in alloc_bit.
   */
  b = (unsigned) (bit_returned / BITS_PER_BLOCK);   /* which block it is in */
  r = (unsigned) (bit_returned % BITS_PER_BLOCK);
  w = r / BITCHUNK_BITS;		/* 'w' tells which word it is in */
  bit = r % BITCHUNK_BITS;
  bp = map_ptr[b];
  bp= buf_vrfy_ptr(bp, sp->s_dev, b+first_map_block);
  map_ptr[b]= bp;

  k = conv2(sp->s_native, (int) bp->b_bitmap[w]);
  if (((k >> bit) & 1) == 0) {
	printf("Cannot free bit %ld\n", bit_returned);
	panic("freeing unused block or inode--check file sys", NO_NUM);
  }
  k &= ~(1 << bit);	/* turn the bit off */
  bp->b_bitmap[w] = conv2(sp->s_native, (int) k);
  bp->b_dirt |= GRIMY;
  put_block(bp);
}


/*===========================================================================*
 *				search_super				     *
 *===========================================================================*/
PUBLIC struct super_block *search_super(dev)
dev_t dev;			/* device number whose super_block is sought */
{
/* Search the superblock table for this device.  It need not be there. */

  register struct super_block *sp;

  for (sp = &super_block[0]; sp < &super_block[NR_SUPERS]; sp++)
	if (sp->s_dev == dev) return(sp);

  return(NIL_SUPER);
}


/*===========================================================================*
 *				get_super				     *
 *===========================================================================*/
PUBLIC struct super_block *get_super(dev)
dev_t dev;			/* device number whose super_block is sought */
{
/* Search the superblock table for this device.  It is supposed to be there. */

  register struct super_block *sp;

  for (sp = &super_block[0]; sp < &super_block[NR_SUPERS]; sp++)
	if (sp->s_dev == dev) return(sp);

  /* Search failed.  Something wrong. */
  printf("fs: can't find superblock for device 0x%x", dev);
  panic(NULL, NO_NUM);
  /*NOTREACHED*/
}


/*===========================================================================*
 *				read_super				     *
 *===========================================================================*/
PUBLIC void read_super(sp)
register struct super_block *sp; /* pointer to a superblock */
{
/* Read a superblock. */

  register struct buf *bp;
  dev_t dev;
  int magic;
  int version = 0, native = 0;	/* unknown file systems are version zero */

  dev = sp->s_dev;		/* save device (will be overwritten by copy) */
  bp = get_block(sp->s_dev, SUPER_BLOCK, BF_NORMAL);
  memcpy( (char *) sp, bp->b_data, (size_t) SUPER_SIZE);
  sp->s_dev = dev;		/* restore device number */
  magic = sp->s_magic;		/* determines file system type */

  /* Get file system version and type. */
  if (magic == SUPER_MAGIC || magic == conv2(BYTE_SWAP, SUPER_MAGIC)) {
	version = V1;
	native  = (magic == SUPER_MAGIC);
  } else if (magic == SUPER_V2 || magic == conv2(BYTE_SWAP, SUPER_V2)) {
	version = V2;
	native  = (magic == SUPER_V2);
  }

  /* If the super block has the wrong byte order, swap the fields; the magic
   * number doesn't need conversion. */
  sp->s_ninodes =       conv2(native, (int) sp->s_ninodes);
  sp->s_nzones =        conv2(native, (int) sp->s_nzones);
  sp->s_imap_blocks =   conv2(native, (int) sp->s_imap_blocks);
  sp->s_zmap_blocks =   conv2(native, (int) sp->s_zmap_blocks);
  sp->s_firstdatazone = conv2(native, (int) sp->s_firstdatazone);
  sp->s_max_size =      conv4(native, sp->s_max_size);
  sp->s_zones =         conv4(native, sp->s_zones);

  /* In V1, the device size was kept in a short, s_nzones, which limited
   * devices to 32K zones.  For V2, it was decided to keep the size as a
   * long.  However, just changing s_nzones to a long would not work, since
   * then the position of s_magic in the super block would not be the same
   * in V1 and V2 file systems, and there would be no way to tell whether
   * a newly mounted file system was V1 or V2.  The solution was to introduce
   * a new variable, s_zones, and copy the size there.
   *
   * Calculate some other numbers that depend on the version here too, to
   * hide some of the differences.
   */
  if (version == V1) {
	sp->s_zones = sp->s_nzones;	/* only V1 needs this copy */
	sp->s_inodes_per_block = V1_INODES_PER_BLOCK;
	sp->s_ndzones = V1_NR_DZONES;
	sp->s_nindirs = V1_INDIRECTS;
	sp->s_link_max = V1_LINK_MAX;
  } else {
	sp->s_inodes_per_block = V2_INODES_PER_BLOCK;
	sp->s_ndzones = V2_NR_DZONES;
	sp->s_nindirs = V2_INDIRECTS;
	sp->s_link_max = V2_LINK_MAX;
  }
  sp->s_isearch = 1;		/* inode searches initially start at 1 */
  sp->s_zsearch = 1;		/* zone searches initially start at 1 */
  sp->s_version = version;
  sp->s_native  = native;

  put_block(bp);
}


/*===========================================================================*
 *				do_super_clean				     *
 *===========================================================================*/
PUBLIC void do_super_clean(sp, clean)
register struct super_block *sp; /* pointer to a superblock */
int clean;			 /* TRUE or FALSE */
{
/* Set or unset the file system clean flag.  (Note: called by the macro
 * super_clean() that tests if the flag is not already as required.)
 */

  struct buf *bp;
  struct super_block *dsp;	/* superblock on disk */
  int enabled;

#if DEBUG & 256
 { where(); printf("do_super_clean(sp[%d], %d)\n", sp-super_block, clean); }
#endif

  if (sp->s_rd_only)
  {
  	/* Don't touch the filesystem, the call is (?) caused by access to the
  	 * underlying block device.
  	 */
  	return;
  }

  /* Read the on disk superblock. */
  bp = get_block(sp->s_dev, SUPER_BLOCK, BF_NORMAL);
  dsp = (struct super_block *) &bp->b_data;

  /* Is the clean flag enabled? */
  enabled= (sp->s_fsck_magic[0] == FSCK_MAGIC0
	 && sp->s_fsck_magic[1] == FSCK_MAGIC1
	&& dsp->s_fsck_magic[0] == FSCK_MAGIC0
	&& dsp->s_fsck_magic[1] == FSCK_MAGIC1);

  /* Register new clean flag. */
  if (clean) {
	sp->s_flags |= S_CLEAN;
	if (enabled) dsp->s_flags |= S_CLEAN;

	/* Keep the superblock in core so it may be easily marked
	 * not-clean next time without needing to read.
	 */
	assert(sp->s_superblock == NULL);
	sp->s_superblock = get_block(sp->s_dev, SUPER_BLOCK, BF_NORMAL);
  } else {
	sp->s_flags &= ~S_CLEAN;
	if (enabled) dsp->s_flags &= ~S_CLEAN;

	assert(sp->s_superblock != NULL);
	assert(sp->s_superblock->b_blocknr == SUPER_BLOCK);
	put_block(sp->s_superblock);
	sp->s_superblock = NULL;
  }

  /* Write the block immediately. The superblock is marked not-clean
   * just before other data is written to the filesystem, and the
   * superblock is marked clean if nothing has be written to the
   * filesystem for some number of syncs.
   */
  if (enabled) { bp->b_list = NULL; rw_scattered(bp, WRITING); }

  put_block(bp);
}


/*
 * $PchId: super.c,v 1.7 1996/01/19 23:10:36 philip Exp $
 */
