/* This file deals with protection in the file system.  It contains the code
 * for four system calls that relate to protection.
 *
 * The entry points into this file are
 *   do_chmod:	perform the CHMOD system call
 *   do_chown:	perform the CHOWN system call
 *   do_umask:	perform the UMASK system call
 *   do_access:	perform the ACCESS system call
 *   forbidden:	check to see if a given access is allowed on a given inode
 */

#include "fs.h"
#include <unistd.h>
#include "buf.h"
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "param.h"
#include "super.h"
#include "assert.h"
INIT_ASSERT
#if VMDEXT_HIDDENDIRS
#include <sys/stat.h>
#endif /* VMDEXT_HIDDENDIRS */

/*===========================================================================*
 *				do_chmod				     *
 *===========================================================================*/
PUBLIC int do_chmod()
{
/* Perform the chmod(name, mode) system call. */

  register struct inode *rip;
  register int r;
  int fs_err= EGENERIC;

  assert(fs_call == CHMOD);

  /* Temporarily open the file. */
  if ((fs_err= fetch_name(chmod_name, chmod_namelen, M3)) != OK) return(fs_err);
  if ( (rip = eat_path(user_path, &fs_err)) == NIL_INODE) return(fs_err);

  /* Only the owner or the super_user may change the mode of a file.
   * No one may change the mode of a file on a read-only file system.
   */
  if (rip->i_uid != fp->fp_effuid && !super_user)
	r = EPERM;
  else
	r = read_only(rip);

  /* If error, return inode. */
  if (r != OK)	{
	put_inode(rip);
	return(r);
  }

  /* Now make the change. Clear setgid bit if file is not in caller's grp */
  rip->i_mode = (rip->i_mode & ~ALL_MODES) | (chmod_mode & ALL_MODES);
  if ((rip->i_mode & I_TYPE) == I_REGULAR && !super_user
  		&& rip->i_gid != fp->fp_effgid && !group_member(rip->i_gid)) {
	rip->i_mode &= ~I_SET_GID_BIT;
  }
  rip->i_update |= CTIME;
  rip->i_dirt = DIRTY;

  put_inode(rip);
  return(OK);
}


/*===========================================================================*
 *				do_chown				     *
 *===========================================================================*/
PUBLIC int do_chown()
{
/* Perform the chown(name, owner, group) system call. */

  register struct inode *rip;
  register int r;
  int fs_err= EGENERIC;

  assert(fs_call == CHOWN);

  /* Temporarily open the file. */
  if ((fs_err= fetch_name(chown_name, chown_namelen, M1)) != OK) return(fs_err);
  if ( (rip = eat_path(user_path, &fs_err)) == NIL_INODE) return(fs_err);

  /* Not permitted to change the owner of a file on a read-only file sys. */
  r = read_only(rip);
  if (r == OK) {
	/* FS is R/W.  Whether call is allowed depends on ownership, etc. */
	if (super_user) {
		/* The super user can do anything. */
		rip->i_uid = chown_owner;	
		rip->i_gid = chown_group;
		rip->i_update |= CTIME;
		rip->i_dirt = DIRTY;
	} else {
		/* Regular users can only change groups of their own files. */
		if (rip->i_uid != fp->fp_effuid) r = EPERM;
		if (rip->i_uid != chown_owner) r = EPERM; /* no giving away */
		if (fp->fp_effgid != chown_group &&
			!group_member(chown_group)) {
			r = EPERM;
		}
		if (r == OK) {
			rip->i_gid = chown_group;
			rip->i_mode &= ~I_SET_GID_BIT;
			rip->i_update |= CTIME;
			rip->i_dirt = DIRTY;
		}
	}
  }

  put_inode(rip);
  return(r);
}


/*===========================================================================*
 *				do_umask				     *
 *===========================================================================*/
PUBLIC int do_umask()
{
/* Perform the umask(co_mode) system call. */
  register mode_t r;

  assert(fs_call == UMASK);

  r = ~fp->fp_umask;		/* set 'r' to complement of old mask */
  fp->fp_umask = ~(umask_cmpl_mode & RWX_MODES);
  return(r);			/* return complement of old mask */
}


/*===========================================================================*
 *				do_access				     *
 *===========================================================================*/
PUBLIC int do_access()
{
/* Perform the access(name, mode) system call. */

  struct inode *rip;
  register int r;
  int fs_err= EGENERIC;

  assert(fs_call == ACCESS);

  /* First check to see if the mode is correct. */
  if ( (access_mode & ~(R_OK | W_OK | X_OK)) != 0 && access_mode != F_OK)
  {
	return(EINVAL);
  }

  /* Temporarily open the file whose access is to be checked. */
  if ((fs_err= fetch_name(access_name, access_namelen, M3)) != OK) 
  {
  	return(fs_err);
  }
  if ( (rip = eat_path(user_path, &fs_err)) == NIL_INODE) 
  {
  	return(fs_err);
  }

  /* Now check the permissions. */
  r = forbidden(rip, (mode_t) access_mode);
  put_inode(rip);
  return(r);
}


/*===========================================================================*
 *				forbidden				     *
 *===========================================================================*/
PUBLIC int forbidden(rip, access_desired)
register struct inode *rip;	/* pointer to inode to be checked */
mode_t access_desired;	/* RWX bits */
{
/* Given a pointer to an inode, 'rip', and the access desired, determine
 * if the access is allowed, and if not why not.  The routine looks up the
 * caller's uid in the 'fproc' table.  If access is allowed, OK is returned
 * if it is forbidden, EACCES is returned.
#if VMDEXT_HIDDENDIRS
 * A special check must be made for a hidden directory, as its protection
 * is frozen to rwxr-xr-x.
#endif
 */

  register mode_t bits, perm_bits;
  int r, shift, test_uid, test_gid;

  /* If /dev/hd2 is mounted on /usr, then to lookup ".." in hd2:/ one needs to
   * lookup ".." in /usr instead.  But the permissions of hd2:/ need to be
   * checked to see if the lookup is allowed.
   */
  while (rip->i_mounted_by != NIL_INODE) rip = rip->i_mounted_by;

  /* Isolate the relevant rwx bits from the mode. */
#if VMDEXT_HIDDENDIRS
  bits = S_ISHIDDEN(rip->i_mode) ? I_DIRECTORY | S_HIDDEN | 0755 : rip->i_mode;
#else /* !VMDEXT_HIDDENDIRS */
  bits = rip->i_mode;
#endif /* !VMDEXT_HIDDENDIRS */
  test_uid = (fs_call == ACCESS ? fp->fp_realuid : fp->fp_effuid);
  test_gid = (fs_call == ACCESS ? fp->fp_realgid : fp->fp_effgid);
  if (test_uid == SU_UID) {
	/* Grant read and write permission.  Grant search permission for
	 * directories.  Grant execute permission (for non-directories) if
	 * and only if one of the 'X' bits is set.
	 */
	if ( (bits & I_TYPE) == I_DIRECTORY ||
	     bits & ((X_BIT << 6) | (X_BIT << 3) | X_BIT))
		perm_bits = R_BIT | W_BIT | X_BIT;
	else	
		perm_bits = R_BIT | W_BIT;
  } else
  if ((bits & (bits >> 3) & (bits >> 6) & access_desired) == access_desired) {
	/* Access allowed in all cases, no need to figure it out precisely. */
	perm_bits = access_desired;
  } else {
	if (test_uid == rip->i_uid) shift = 6;		/* owner */
	else if (test_gid == rip->i_gid
		|| group_member(rip->i_gid)) shift = 3;	/* group */
	else shift = 0;					/* other */
	perm_bits = (bits >> shift) & (R_BIT | W_BIT | X_BIT);
  }

  /* If access desired is not a subset of what is allowed, it is refused. */
  r = OK;
  if ((perm_bits & access_desired) != access_desired) r = EACCES;

  /* Check to see if someone is trying to write a file or directory on a file
   * system that is mounted read-only.
   */
  if (r == OK && (access_desired & W_BIT)) {
	bits &= I_TYPE;
  	if (bits == I_DIRECTORY || bits == I_REGULAR) r = read_only(rip);
  }

  return(r);
}


/*===========================================================================*
 *				read_only				     *
 *===========================================================================*/
PUBLIC int read_only(ip)
struct inode *ip;		/* ptr to inode whose file sys is to be cked */
{
/* Check to see if the file system on which the inode 'ip' resides is mounted
 * read only.  If so, return EROFS, else return OK.
 */

  register struct super_block *sp;

  sp = ip->i_sp;
  return(sp->s_rd_only ? EROFS : OK);
}


/*===========================================================================*
 *				group_member				     *
 *===========================================================================*/
PUBLIC int group_member(gid)
Gid_t gid;			/* group to check */
{
/* True if the group to test is part of the supplementary group list. */

  int i;

  for (i = 0; i < fp->fp_ngroups; i++)
	if (fp->fp_groups[i] == gid) return(TRUE);
  return(FALSE);
}

/*
 * $PchId: protect.c,v 1.4 1995/11/28 07:40:10 philip Exp $
 */
