#if VMDEXT_SYMLINK
/* This file contains the procedures that look up path names in the directory
 * system and determine the inode number that goes with a given path name.
 *
 *  The entry points into this file are
 *   parse_path: the 'main' routine of the path-to-inode conversion mechanism
 *   eat_path:	 historical entry point to parse_path with transparent links
 *   last_dir:	 find the final directory on a given path
 *   advance:	 parse one component of a path name
 *   search_dir: search a directory for a string and return its inode number
 */
#else /* !VMDEXT_SYMLINK */
/* This file contains the procedures that look up path names in the directory
 * system and determine the inode number that goes with a given path name.
 *
 *  The entry points into this file are
 *   eat_path:	 the 'main' routine of the path-to-inode conversion mechanism
 *   last_dir:	 find the final directory on a given path
 *   advance:	 parse one component of a path name
 *   search_dir: search a directory for a string and return its inode number
 */
#endif /* VMDEXT_SYMLINK */

#include "fs.h"
#if VMDEXT_SYMLINK || VMDEXT_HIDDENDIRS
#include <sys/stat.h>
#endif /* VMDEXT_SYMLINK || VMDEXT_HIDDENDIRS */
#include <string.h>
#include <minix/callnr.h>
#include "assert.h"
INIT_ASSERT
#include "buf.h"
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "name.h"
#include "super.h"

PUBLIC char dot1[2] = ".";	/* used for search_dir to bypass the access */
PUBLIC char dot2[3] = "..";	/* permissions for . and ..		    */

FORWARD _PROTOTYPE( char *get_name, (char *old_name, char string [NAME_MAX],
								int *err_p) );
#if VMDEXT_SYMLINK
FORWARD _PROTOTYPE( struct inode *ltraverse, (struct inode *rip,
	char *path, char *suffix, struct inode *ldip, int *err_p)	);
#endif /* VMDEXT_SYMLINK */
#if VMDEXT_FLEXDIR
FORWARD _PROTOTYPE( int dots, (char string[NAME_MAX])			);
FORWARD _PROTOTYPE( int nslots, (char string[NAME_MAX])			);
FORWARD _PROTOTYPE( void setname, (char *name, char string[NAME_MAX], 
							char *high)	);
#endif /* VMDEXT_FLEXDIR */

#if VMDEXT_HIDDENDIRS
#define T_SHALLOW	0x01	/* one component transparent */
#define T_DEEP		0x02	/* full path transparent by use of ".@" */
PRIVATE int transparent;	/* if hidden directories must be visible */
#endif /* VMDEXT_HIDDENDIRS */


#if !VMDEXT_SYMLINK
/*===========================================================================*
 *				eat_path				     *
 *===========================================================================*/
PUBLIC struct inode *eat_path(path, err_p)
char *path;			/* the path name to be parsed */
int *err_p;			/* pointer to error varaible */
{
/* Parse the path 'path' and put its inode in the inode table. If not possible,
 * return NIL_INODE as function value and an error code in '*err_p'.
 */

  register struct inode *rip;
  struct inode *ldip;
  char string[NAME_MAX];	/* hold 1 path component name here */

  /* First open the path down to the final directory. */
  if ( (ldip = last_dir(path, string)) == NIL_INODE)
	return(NIL_INODE);	/* we couldn't open final directory */

  /* The path consisting only of "/" is a special case, check for it. */
  if (string[0] == '\0') return(ldip);

  /* Get final component of the path. */
  rip = advance(&ldip, string);
  put_inode(ldip);
  return(rip);
}
#endif /* !VMDEXT_SYMLINK */


#if VMDEXT_SYMLINK
/*===========================================================================*
 *				parse_path				     *
 *===========================================================================*/
PUBLIC struct inode *parse_path(path, string, action, err_p)
char *path;			/* the path name to be parsed */
char string[NAME_MAX];		/* the final component is returned here */
int action;			/* action on last part of path */
int *err_p;			/* pointer to error variable */
#else /* !VMDEXT_SYMLINK */
/*===========================================================================*
 *				last_dir				     *
 *===========================================================================*/
PUBLIC struct inode *last_dir(path, string, err_p)
char *path;			/* the path name to be parsed */
char string[NAME_MAX];		/* the final component is returned here */
int *err_p;			/* pointer to error variable */
#endif /* VMDEXT_SYMLINK */
{
#if VMDEXT_SYMLINK
/* This is the actual code for last_dir and eat_path. Return the inode of
 * the last directory and the name of object within that directory, or the
 * inode of the last object (an empty name will be returned). Names are
 * returned in string. If string is null the name is discarded. The action
 * code determines how "last" is defined. If an error occurs, NIL_INODE
 * will be returned with an error code in *err_p.
 */
#else /* !VMDEXT_SYMLINK */
/* Given a path, 'path', located in the fs address space, parse it as
 * far as the last directory, fetch the inode for the last directory into
 * the inode table, and return a pointer to the inode.  In
 * addition, return the final component of the path in 'string'.
 * If the last directory can't be opened, return NIL_INODE and
 * the reason for failure in '*err_p'.
 */
#endif /* VMDEXT_SYMLINK */

  register struct inode *rip;
  register char *new_name;
  struct inode *dir_ip;
#if VMDEXT_SYMLINK
  int symloop;
  char lstring[NAME_MAX];
  char *slash;
#endif /* VMDEXT_SYMLINK */
  int fs_err= EGENERIC;

#if VMDEXT_HIDDENDIRS
  /* Hidden directories are hidden unless transparent is set. */
  transparent = 0;
#endif /* VMDEXT_HIDDENDIRS */

  /* Is the path absolute or relative?  Initialize 'rip' accordingly. */
  rip = (*path == '/' ? fp->fp_rootdir : fp->fp_workdir);
  assert(rip || (where(), printf(
		  "fp= proc %d, path= '%s', rootdir= 0x%x, workdir= 0x%x\n",
  	fp-fproc, path, fp->fp_rootdir, fp->fp_workdir), 0));

  /* Switch to the device mounted on this directory, if any. */
  while (rip->i_mounted_by != NIL_INODE) rip = rip->i_mounted_by;

  /* If dir has been removed or path is empty, return ENOENT. */
  if (rip->i_nlinks == 0 || *path == '\0') {
	*err_p = ENOENT;
	return(NIL_INODE);
  }

  dup_inode(rip);		/* inode will be returned with put_inode */

#if VMDEXT_SYMLINK
  symloop = 0;			/* symbolic link traversal count */
  if (string == (char *) 0) string = lstring;
#endif /* VMDEXT_SYMLINK */

  /* Scan the path component by component. */
  while (TRUE) {
	/* Extract one component. */
	if ( (new_name = get_name(path, string, &fs_err)) == (char*) 0) {
		put_inode(rip);	/* bad path in user space */
		*err_p= fs_err;
		return(NIL_INODE);
	}

#if VMDEXT_SYMLINK
	/* Skip separators, but remember if a path ended with one. */
	if (*(slash = new_name) == '/') while (*++new_name == '/') {}

	/* Terminate on penultimate name in path. */
	if (*new_name == '\0' && (action & PATH_PENULTIMATE) != 0)
#else /* !VMDEXT_SYMLINK */
	if (*new_name == '\0')
#endif /* VMDEXT_SYMLINK */
		if ( (rip->i_mode & I_TYPE) == I_DIRECTORY)
			return(rip);	/* normal exit */
		else {
			/* last file of path prefix is not a directory */
			put_inode(rip);
			*err_p = ENOTDIR;			
			return(NIL_INODE);
		}

	/* There is more path.  Keep parsing. */
	dir_ip = rip;
	rip = advance(&dir_ip, string, &fs_err);

#if VMDEXT_SYMLINK
	if (rip == NIL_INODE) {
		if (*new_name == '\0' && (action & PATH_NONSYMBOLIC) != 0)
		{
			return(dir_ip);
		}
		else {
			put_inode(dir_ip);
			*err_p= fs_err;
			return(NIL_INODE);
		}
	}	

	/* The call to advance() succeeded.  Fetch next component. */
	if (S_ISLNK(rip->i_mode)) {
		if (*slash != '\0' || (action & PATH_OPAQUE) == 0) {
			new_name = slash;
			rip = ltraverse(rip, path, new_name, dir_ip, &fs_err);
			put_inode(dir_ip);
			if (++symloop > SYMLOOP) {
				put_inode(rip);
				fs_err= ELOOP;
				rip = NIL_INODE;
			}
			if (rip == NIL_INODE) 
			{
				*err_p = fs_err;
				return(NIL_INODE);
			}
			continue;
		}
	} else if (*new_name != '\0') {
		put_inode(dir_ip);
		path = new_name;
		continue;
	}
	
	/* Either last name reached or symbolic link is opaque */
	if ((action & PATH_NONSYMBOLIC) != 0) {
		put_inode(rip);
		return(dir_ip);
	} else {
		put_inode(dir_ip);
		return(rip);
	}
#else /* !VMDEXT_SYMLINK */
	put_inode(dir_ip);	/* dir_ip either obsolete or irrelevant */
	if (rip == NIL_INODE)
	{
		*err_p= fs_err;
		return(NIL_INODE);
	}

	/* The call to advance() succeeded.  Fetch next component. */
	path = new_name;
#endif /* VMDEXT_SYMLINK */
  }
}


/*===========================================================================*
 *				get_name				     *
 *===========================================================================*/
PRIVATE char *get_name(old_name, string, err_p)
char *old_name;			/* path name to parse */
char string[NAME_MAX];		/* component extracted from 'old_name' */
int *err_p;			/* pointer to error variable */
{
/* Given a pointer to a path name in fs space, 'old_name', copy the next
 * component to 'string' and pad with zeros.  A pointer to that part of
 * the name as yet unparsed is returned.  Roughly speaking,
 * 'get_name' = 'old_name' - 'string'.
 *
 * This routine follows the standard convention that /usr/ast, /usr//ast,
 * //usr///ast and /usr/ast/ are all equivalent.
 */

  register int c;
  register char *np, *rnp;

  np = string;			/* 'np' points to current position */
  rnp = old_name;		/* 'rnp' points to unparsed string */
  while ( (c = *rnp) == '/') rnp++;	/* skip leading slashes */

  /* Copy the unparsed path, 'old_name', to the array, 'string'. */
  while ( rnp < &old_name[PATH_MAX]  &&  c != '/'   &&  c != '\0') {
	if (np < &string[NAME_MAX]) *np++ = c;
	c = *++rnp;		/* advance to next character */
#if VMDEXT_HIDDENDIRS
	if (c == '@' && rnp < &old_name[PATH_MAX-1]
				&& (rnp[1] == '/' || rnp[1] == '\0')) {
		if (string[0] == '.' && np == &string[1]) {
			/* ".@" signals full path transparency */
			transparent |= T_DEEP;
		} else {
			/* shallow transparency */
			transparent |= T_SHALLOW;
		}
		c = *++rnp;
	}
#endif /* VMDEXT_HIDDENDIRS */
  }

#if !VMDEXT_SYMLINK
  /* To make /usr/ast/ equivalent to /usr/ast, skip trailing slashes. */
  while (c == '/' && rnp < &old_name[PATH_MAX]) c = *++rnp;
#endif

  if (np < &string[NAME_MAX]) *np = '\0';	/* Terminate string */

  if (rnp >= &old_name[PATH_MAX]) {
	*err_p = ENAMETOOLONG;
	return((char *) 0);
  }
  return(rnp);
}


/*===========================================================================*
 *				dots					     *
 *===========================================================================*/
PRIVATE int dots(string)
char string[NAME_MAX];
{
/* Return 1, 2, 0 if string is ".", "..", or something else.  */

  if (string[0] != '.') return 0;
  if (string[1] ==  0 ) return 1;
  if (string[1] != '.') return 0;
  if (string[2] ==  0 ) return 2;
  return 0;
}


/*===========================================================================*
 *				advance					     *
 *===========================================================================*/
PUBLIC struct inode *advance(pdirp, string, err_p)
struct inode **pdirp;		/* inode for directory to be searched */
char string[NAME_MAX];		/* component name to look for */
int *err_p;			/* pointer to error variable */
{
/* Given a directory and a component of a path, look up the component in
 * the directory, find the inode, open it, and return a pointer to its inode
 * slot.  If it can't be done, return NIL_INODE.
 */

  register struct inode *rip, *dirp = *pdirp;
  struct name *np;
  int hash, r;
  ino_t numb;
  int fs_err= EGENERIC;
#if VMDEXT_HIDDENDIRS
  int transp;

  /* Transparency, shallow or deep? */
  transp = transparent;
  transparent &= ~T_SHALLOW;
#endif /* !VMDEXT_HIDDENDIRS */

  /* If 'string' is empty, yield same inode straight away. */
  if (string[0] == '\0')
  {
  	dup_inode(dirp);
  	return dirp;
  }

  /* Check for NIL_INODE. */
  if (dirp == NIL_INODE)
  {
  	*err_p= EINVAL;
where();
  	return(NIL_INODE);
  }

  /* Look for this name in the name cache. */
  hash= hash_name(dirp->i_dev, dirp->i_num, string);
  np= lookup_name(hash, dirp->i_dev, dirp->i_num, string);

  if (np != NULL)
  {
  	/* Found an entry. Are we allowed to access this directory? */
	r = forbidden(dirp, X_BIT); /* check access permissions */
	if (r != OK)
	{
		*err_p= r;
		return NIL_INODE;
	}
	if (np->n_tdev == NO_DEV)
	{
		/* Non-existant entry. */
		*err_p= ENOENT;
		return NIL_INODE;
	}

	/* Lookup the inode. */
	rip= np->n_ip;
	rip= inode_vrfy_ptr(rip, np->n_tdev, np->n_tino);
	if (rip == NIL_INODE)
	{
		*err_p= EMFILE;
		return NIL_INODE;
	}
	np->n_ip= rip;
  }
  else
  {
  	/* No entry in the name cache, do the normal lookup. */

	/* If 'string' is not present in the directory, signal error. */
	if ( (r = search_dir(dirp, string, &numb, LOOK_UP)) != OK) {
		if (r == ENOENT)
		{
			enter_name(hash, dirp->i_dev, dirp->i_num, string,
				NO_DEV, 0, NIL_INODE);
		}
		*err_p = r;
		return(NIL_INODE);
	}

	/* The component has been found in the directory.  Get inode. */
	if ( (rip = get_inode(dirp->i_dev, numb, &fs_err)) == NIL_INODE)
	{
		*err_p= fs_err;
		return(NIL_INODE);
	}

	/* Enter the result in the name cache. */
	enter_name(hash, dirp->i_dev, dirp->i_num, string, dirp->i_dev,
		numb, rip);
  }

  /* If the inode is loopback mounted onto something then it is not supposed
   * to be visible by its old path.  Replace by a symlink to the mount point.
   */
  if (rip->i_mounted_on != NIL_INODE && dots(string) == 0) {
	struct mount *mp;
	struct buf *bp;

	for (mp = mount_table; mp < &mount_table[NR_SUPERS]; mp++) {
		if (mp->m_iroot == rip) break;
	}
	assert(mp != &mount_table[NR_SUPERS]);
	put_inode(rip);
	rip = alloc_inode(pipe_fs, I_SYMBOLIC_LINK | RWX_MODES, &fs_err);
	if (rip == NIL_INODE) {
		*err_p = fs_err;
		return(NIL_INODE);
	}
	if ((bp = new_block(rip, (uoff_t) 0, FALSE, &fs_err)) == NIL_BUF) {
		*err_p = fs_err;
		put_inode(rip);
		return(NIL_INODE);
	}
	strcpy(bp->b_data, mp->m_mountp);
	rip->i_size = strlen(bp->b_data);
	put_block(bp);
	return(rip);
  }

  /* Following ".." won't get you past your (chroot-ed) root directory,
   * unless string == dot2.
   */
  if (dirp == fp->fp_rootdir && string != dot2 && dots(string) == 2) {
  	put_inode(rip);
  	dup_inode(dirp);
  	return(dirp);
  }

  /* Assume that /dev/hd2 is mounted on /usr, so that any reference to /usr
   * is translated to the root of /dev/hd2 (hd2:/).  Following ".." in a
   * mounted on directory, i.e. hd2:/.. must then be replaced by /usr/.. to
   * get to /.  (That's why you can only mount directories on directories.)
   */
  if (dirp->i_mounted_on != NIL_INODE && dots(string) == 2) {
	/* Release the directory.  Replace by the directory mounted on.  */
	put_inode(rip);
	put_inode(dirp);
	*pdirp = dirp = dirp->i_mounted_on;
	dup_inode(dirp);
	rip = advance(pdirp, string, &fs_err);

	if (rip == NIL_INODE)
	{	
		*err_p= fs_err;
where();
		return(NIL_INODE);
	}
  }

  /* See if the inode is mounted on.  If so, switch to the file mounted on
   * it.  This translates /usr to hd2:/
   */
  while (rip->i_mounted_by != NIL_INODE) {
	put_inode(rip);
	rip = rip->i_mounted_by;
	dup_inode(rip);
  }

#if VMDEXT_HIDDENDIRS
  /* A hidden directory must be replaced by the entry it hides, unless it
   * is referenced as "." or "..".
   */
  if (S_ISHIDDEN(rip->i_mode) && !transp) {
  	r = rip->i_mode & S_HTYPE;
  	if (r >= _UTS_MAX) {
  		put_inode(rip);
		*err_p = EACCES;
where();
  		return(NIL_INODE);
	}
	switch (dots(string)) {
	case 0:
		/* This directory hides one of the names in uts_tbl[] */
		put_inode(dirp);
		*pdirp = rip;
		strncpy(string, uts_tbl[r], NAME_MAX);
where();
		return advance(pdirp, string, err_p);
	case 2:
		/* ".." Should reference the parent of a hidden dir */
		put_inode(dirp);
		*pdirp = rip;
where();
		return advance(pdirp, string, err_p);
	}
  }
#endif /* VMDEXT_HIDDENDIRS */
  return(rip);		/* return pointer to inode's component */
}


/*===========================================================================*
 *				nslots					     *
 *===========================================================================*/
PRIVATE int nslots(string)
char string[NAME_MAX];
{
/* Computes number of directory slots needed for the given string. */
  char *s = string;
  int n = NAME_MAX;

  while (--n >= 0 && *s != 0) s++;

  return 1 + _EXTENT(s - string);
}


/*===========================================================================*
 *				setname					     *
 *===========================================================================*/
PRIVATE void setname(name, string, high)
char *name;
char string[NAME_MAX];
char *high;
{
/* Set name of directory entry name to string, extend with zeros until high
 * is reached.
 */
  char *s = string;
  int n = NAME_MAX;

  while (--n >= 0 && name < high && *s != 0) *name++ = *s++;

  while (name < high) *name++ = 0;
}


/*===========================================================================*
 *				search_dir				     *
 *===========================================================================*/
PUBLIC int search_dir(ldir_ptr, string, numb, flag)
register struct inode *ldir_ptr;	/* ptr to inode for dir to search */
char string[NAME_MAX];		/* component to search for */
ino_t *numb;			/* pointer to inode number */
int flag;			/* LOOK_UP, ENTER, DELETE or IS_EMPTY */
{
/* This function searches the directory whose inode is pointed to by 'ldip':
 * if (flag == ENTER)  enter 'string' in the directory with inode # '*numb';
 * if (flag == DELETE) delete 'string' from the directory;
 * if (flag == LOOK_UP) search for 'string' and return inode # in 'numb';
 * if (flag == IS_EMPTY) return OK if only . and .. in dir else ENOTEMPTY;
 *
 *    if 'string' is dot1 or dot2, no access permissions are checked.
 */
#if VMDEXT_FLEXDIR
/*
 * The function is changed to handle both V7 directories and flex
 * directories.  It now depends on new_block returning zeroed blocks.
 */
#endif /* VMDEXT_FLEXDIR */

  register struct buf *bp;
  int r, nfree, t, match;
  mode_t bits;
  uoff_t pos, new_size;
  block_t b;
  struct super_block *sp;
  int fs_err= EGENERIC;
  int extended = 0;

  /* If 'ldir_ptr' is not a pointer to a dir inode, error. */
  if ( (ldir_ptr->i_mode & I_TYPE) != I_DIRECTORY) return(ENOTDIR);

  r = OK;

  if (flag != IS_EMPTY) {
	bits = (flag == LOOK_UP ? X_BIT : W_BIT | X_BIT);

	if (string == dot1 || string == dot2) {
		if (flag != LOOK_UP) r = read_only(ldir_ptr);
				     /* only a writable device is required. */
        }
	else r = forbidden(ldir_ptr, bits); /* check access permissions */
  }
  if (r != OK) return(r);

  if (flag == ENTER || flag == DELETE)
  {
	  /* Flush possibly cached name. */
	  flush_name(ldir_ptr->i_dev, ldir_ptr->i_num, string);
  }

  new_size = 0;
  nfree = 0;
  match = 0;

  /* Is this a good old Version 7 directory? */
  if (!(get_super(ldir_ptr->i_dev)->s_flags & S_FLEX)) {{
	register struct _v7_direct *dp;
  
  /* Step through the directory one block at a time. */
  for (pos = 0; pos < ldir_ptr->i_size; pos += BLOCK_SIZE) {
	b = read_map(ldir_ptr, pos);	/* get block number */

	/* Since directories don't have holes, 'b' cannot be NO_BLOCK. */
	bp = get_block(ldir_ptr->i_dev, b, BF_NORMAL);	/* get a dir block */

	/* Search a directory block. */
	for (dp = &bp->b_dir[0]; dp < &bp->b_dir[NR_DIR_ENTRIES]; dp++) {
		new_size += DIR_ENTRY_SIZE;

		/* Match occurs if string found. */
		if (flag != ENTER && dp->d_ino != 0) {
			if (flag == IS_EMPTY) {
				/* If this test succeeds, dir is not empty. */
				if (dots(dp->d_name) == 0) match = 1;
			} else {
				if (strncmp(dp->d_name, string,
						sizeof(dp->d_name)) == 0)
					match = 1;
			}
		}

		if (match) {
			/* LOOK_UP or DELETE found what it wanted. */
			r = OK;
			if (flag == IS_EMPTY) r = ENOTEMPTY;
			else if (flag == DELETE) {
				/* Save d_ino for recovery. */
				((ino_t *) &dp[1])[-1] = dp->d_ino;
				dp->d_ino = 0;	/* erase entry */
				bp->b_dirt = DIRTY;
				ldir_ptr->i_update |= CTIME | MTIME;
				ldir_ptr->i_dirt = DIRTY;
			} else {
				sp = ldir_ptr->i_sp;	/* 'flag' is LOOK_UP */
				*numb = conv2(sp->s_native, (int) dp->d_ino);
			}
			put_block(bp);
			return(r);
		}


		/* Check for free slot for the benefit of ENTER. */
		if (flag == ENTER && dp->d_ino == 0) {
			nfree = 1;	/* we found a free slot */
			break;
		}
	}

	/* The whole block has been searched or ENTER has a free slot. */
	if (nfree > 0) break;	/* nfree > 0 if ENTER can be performed now */
	put_block(bp);		/* otherwise, continue searching dir */
  }

  /* The whole directory has now been searched. */
  if (flag != ENTER) return(flag == IS_EMPTY ? OK : ENOENT);

  /* This call is for ENTER.  If no free slot has been found so far, try to
   * extend directory.
   */
  if (nfree == 0) { /* directory is full and no room left in last block */
	if ( (bp = new_block(ldir_ptr, new_size, TRUE, &fs_err)) == NIL_BUF)
		return(fs_err);
	dp = &bp->b_dir[0];
	new_size += DIR_ENTRY_SIZE;
	extended = 1;
  }

  /* 'bp' now points to a directory block with space. 'dp' points to slot. */
  strncpy(dp->d_name, string, sizeof(dp->d_name));
  sp = ldir_ptr->i_sp; 
  dp->d_ino = conv2(sp->s_native, (int) *numb);

  }} else {{
	/* Flexible directory. */
  register struct _fl_direct *dp;
  int need;	/* ENTER needs this many slots */

  /* Step through the directory one block at a time. */
  need = flag == ENTER ? nslots(string) : -1;

  for (pos = 0; pos < ldir_ptr->i_size; pos += BLOCK_SIZE) {
	b = read_map(ldir_ptr, pos);	/* get block number */

	/* Since directories don't have holes, 'b' cannot be NO_BLOCK. */
	bp = get_block(ldir_ptr->i_dev, b, BF_NORMAL);	/* get a dir block */

	nfree = 0;

	/* Search a directory block. */
	for (dp = &bp->b_fl_dir[0]; dp < &bp->b_fl_dir[FL_NR_DIR_ENTRIES];
		dp += 1 + dp->d_extent
	) {
		new_size += (1 + dp->d_extent) * FL_DIR_ENTRY_SIZE;

		/* Match occurs if string found. */
		if (flag != ENTER && dp->d_ino != 0) {
			if (flag == IS_EMPTY) {
				/* If this test succeeds, dir is not empty. */
				if (dots(dp->d_name) == 0) match = 1;
			} else {
				if (strncmp(dp->d_name, string, NAME_MAX) == 0)
					match = 1;
			}
		}

		if (match) {
			/* LOOK_UP or DELETE found what it wanted. */
			r= OK;
			if (flag == IS_EMPTY) r = ENOTEMPTY;
			else if (flag == DELETE) {
				t = 1 + dp->d_extent;  /* erase t entries */
				/* But first save d_ino for recovery. */
				((ino_t *) &dp[t])[-1] = dp->d_ino;
				do {
					dp->d_ino = 0;
					dp->d_extent = 0;
					dp++;
				} while (--t > 0);
				bp->b_dirt = DIRTY;
				ldir_ptr->i_update = CTIME | MTIME;
				ldir_ptr->i_dirt = DIRTY;
			} else {
				sp = ldir_ptr->i_sp;	/* 'flag' is LOOK_UP */
				*numb = conv2(sp->s_native, (int) dp->d_ino);
			}
			put_block(bp);
			return(r);
		}

		/* Check for free slot for the benefit of ENTER. */
		if (dp->d_ino == 0) {
			/* we found a free slot */
			if (++nfree == need) break;
		} else
			nfree = 0;
	}

	/* The whole block has been searched or ENTER has a free slot. */
	if (nfree == need) break;	/* Enough free slots for ENTER */
	put_block(bp);			/* otherwise, continue searching dir */
  }

  /* The whole directory has now been searched. */
  if (flag != ENTER) return(flag == IS_EMPTY ? OK : ENOENT);

  /* This call is for ENTER.  If no free slot has been found so far, try to
   * extend directory.
   */
  if (nfree != need) { /* directory is full and no room left in last block */
	if ( (bp = new_block(ldir_ptr, new_size, TRUE, &fs_err)) == NIL_BUF)
		return(fs_err);
	dp = &bp->b_fl_dir[0];
	new_size += need * FL_DIR_ENTRY_SIZE;
	extended = 1;
  } else {
	/* There is free space, step back to the start of it */
	dp -= nfree-1;
  }

  /* 'bp' now points to a directory block with space. 'dp' points to slot. */
  setname(dp->d_name, string, (char *) &dp[need]);
  sp = ldir_ptr->i_sp; 
  dp->d_ino = conv2(sp->s_native, (int) *numb);
  dp->d_extent = need - 1;
  }}

  bp->b_dirt = DIRTY;
  put_block(bp);
  ldir_ptr->i_update |= CTIME | MTIME;	/* mark mtime for update later */
  ldir_ptr->i_dirt = DIRTY;
  if (new_size > ldir_ptr->i_size) {
	ldir_ptr->i_size = new_size;
	/* Send the change to disk if the directory is extended. */
	if (extended) rw_inode(ldir_ptr, WRITING);
  }
  return(OK);
}


#if VMDEXT_SYMLINK
/*===========================================================================*
 *				eat_path				     *
 *===========================================================================*/
PUBLIC struct inode *eat_path(path, err_p)
char *path;			/* the path name to be parsed */
int *err_p;			/* pointer to error variable */
{
/* Parse the path 'path' and put its inode in the inode table. If not possible,
 * return NIL_INODE as function value and an error code in '*err_p'.
 */
  struct inode *rip;

  rip= parse_path(path, (char *) 0, EAT_PATH, err_p);
  return rip;
}
#endif /* VMDEXT_SYMLINK */


#if VMDEXT_SYMLINK
/*===========================================================================*
 *				last_dir				     *
 *===========================================================================*/
PUBLIC struct inode *last_dir(path, string, err_p)
char *path;			/* the path name to be parsed */
char string[NAME_MAX];		/* the final component is returned here */
int *err_p;			/* pointer to error variable */
{
/* Given a path, 'path', located in the fs address space, parse it as
 * far as the last directory, fetch the inode for the last directory into
 * the inode table, and return a pointer to the inode.  In
 * addition, return the final component of the path in 'string'.
 * If the last directory can't be opened, return NIL_INODE and
 * the reason for failure in '*err_p'.
 */

  return parse_path(path, string, LAST_DIR, err_p);
}
#endif /* VMDEXT_SYMLINK */


#if VMDEXT_SYMLINK
/*===========================================================================*
 *				ltraverse				     *
 *===========================================================================*/
PRIVATE struct inode *ltraverse(rip, path, suffix, ldip, err_p)
register struct inode *rip;	/* symbolic link */
char *path;			/* path containing link */
char *suffix;			/* suffix following link within path */
register struct inode *ldip;	/* directory containing link */
int *err_p;			/* pointer to error variable */
{
/* Traverse a symbolic link. Copy the link text from the inode and insert
 * the text into the path. Return the inode of base directory and the
 * ammended path. The symbolic link inode is always freed. The inode
 * returned is already duplicated. NIL_INODE is returned on error.
 */

  block_t b;			/* block containing link text */
  struct inode *bip;		/* inode of base directory */
  struct buf *bp;		/* buffer containing link text */
  size_t sl;			/* length of link */
  size_t tl;			/* length of suffix */
  char *sp;			/* start of link text */

  bip = NIL_INODE;
  bp  = NIL_BUF;

  if ((b = read_map(rip, (uoff_t) 0)) != NO_BLOCK) {
	bp = get_block(rip->i_dev, b, BF_NORMAL);
	sl = rip->i_size;
	sp = bp->b_data;

	/* Insert symbolic text into path name. */
	tl = strlen(suffix);
	if (sl > 0 && sl + tl <= PATH_MAX-1) {
		memmove(path+sl, suffix, tl);
		memmove(path, sp, sl);
		path[sl+tl] = 0;
		dup_inode(bip = path[0] == '/' ? fp->fp_rootdir : ldip);
	}
  }

  put_block(bp);
  put_inode(rip);
  if (bip == NIL_INODE) 
	*err_p = ENOENT;
  return (bip);
}
#endif /* VMDEXT_SYMLINK */

/*
 * $PchId: path.c,v 1.6 1995/11/28 07:42:55 philip Exp $
 */
