/* This file contains the procedures for creating, opening, closing, and
 * seeking on files.
 *
 * The entry points into this file are
 *   do_creat:	perform the CREAT system call
 *   do_mknod:	perform the MKNOD system call
 *   do_open:	perform the OPEN system call
 *   do_close:	perform the CLOSE system call
 *   do_lseek:  perform the LSEEK system call
 *   do_mkdir:	perform the MKDIR system call
 */
#if VMDEXT_SYMLINK
/*
 *   do_slink:	perform the SLINK system call
 */
#endif /* VMDEXT_SYMLINK */

#include "fs.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "buf.h"
#include "assert.h"
INIT_ASSERT
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "super.h"
#include "param.h"

PRIVATE char mode_map[] = {R_BIT, W_BIT, R_BIT|W_BIT, 0};

FORWARD _PROTOTYPE( int common_open, (int oflags, Mode_t omode)		);
#if VMDEXT_SYMLINK
FORWARD _PROTOTYPE( struct inode *new_node, (struct inode **ldirp, char *path,
	char string[NAME_MAX], Mode_t bits, zone_t z0,
				int opaque, int *created, int *err_p)	);
#else /* !VMDEXT_SYMLINK */
FORWARD _PROTOTYPE( struct inode *new_node, (struct inode **ldirp, char *path,
	char string[NAME_MAX], Mode_t bits, zone_t z0,
						int *created, int *err_p) );
#endif /* VMDEXT_SYMLINK */

/*===========================================================================*
 *				do_creat				     *
 *===========================================================================*/
PUBLIC int do_creat()
{
/* Perform the creat(name, mode) system call. */
  int fs_err= EGENERIC;

  assert(fs_call == CREAT);

  /* get name */
  if ((fs_err= fetch_name(creat_name, creat_namelen, M3)) != OK)
  	return(fs_err);
  return common_open(O_WRONLY | O_CREAT | O_TRUNC, (mode_t) creat_mode);
}


/*===========================================================================*
 *				do_mknod				     *
 *===========================================================================*/
PUBLIC int do_mknod()
{
/* Perform the mknod(name, mode, addr) system call. */

  register mode_t bits, mode_bits;
  struct inode *ip;
  struct inode *ldirp;
  char string[NAME_MAX];
  int fs_err= EGENERIC;
  int created;

  assert(fs_call == MKNOD);

  /* Only the super_user may make nodes other than fifos. */
  mode_bits= mknod_mode;
  if (!super_user && ((mode_bits & I_TYPE) != I_NAMED_PIPE)) return(EPERM);
  if ((fs_err= fetch_name(mknod_name, mknod_namelen, M1)) != OK)
  	return(fs_err);
  bits = (mode_bits & I_TYPE) | (mode_bits & ALL_MODES & fp->fp_umask);

#if VMDEXT_SYMLINK
  ip = new_node(&ldirp, user_path, string, bits, (zone_t) mknod_addr,
						TRUE, &created, &fs_err);
  put_inode(ldirp);
#else /* !VMDEXT_SYMLINK */
  ip = new_node(&ldirp, user_path, string, bits, (zone_t) mknod_addr,
						&created, &fs_err);
#endif /* VMDEXT_SYMLINK */
  if (ip != NIL_INODE)
  {
  	if (!created)
  		fs_err= EEXIST;
  	else
  		fs_err= OK;
	put_inode(ip);
	return(fs_err);
  }
  return fs_err;
}


/*===========================================================================*
 *				new_node				     *
 *===========================================================================*/
#if VMDEXT_SYMLINK
PRIVATE struct inode *new_node(ldirp, path, string, bits, z0, opaque,
								created, err_p)
#else /* !VMDEXT_SYMLINK */
PRIVATE struct inode *new_node(ldirp, path, string, bits, z0,
								created, err_p)
#endif /* VMDEXT_SYMLINK */
struct inode **ldirp;		/* the directory the node is made in */
char *path;			/* pointer to path name */
char string[NAME_MAX];		/* returned basename of the entry */
mode_t bits;			/* mode of the new inode */
zone_t z0;			/* zone number 0 for new inode */
#if VMDEXT_SYMLINK
int opaque;			/* opaque to symbolic link */
#endif /* VMDEXT_SYMLINK */
int *created;			/* did we just create the file */
int *err_p;			/* pointer to error variable */
{
/* New_node() is called by common_open(), do_mknod(), and do_mkdir().  
 * In all cases it allocates a new inode, makes a directory entry for it on 
 * the path 'path', and initializes it.  It returns a pointer to the inode if 
 * it can do this; otherwise it returns NIL_INODE.
 */

  register struct inode *rip;
  register int r;
  int fs_err= EGENERIC;

  *created= 1;			/* assume that we create the inode */

  /* See if the path can be opened down to the last directory. */
#if VMDEXT_SYMLINK
  *ldirp = parse_path(path, string, opaque ? LAST_DIR : LAST_DIR_EATSYM, 
  								&fs_err);
  if (*ldirp == NIL_INODE)
  {
  	assert(fs_err != EGENERIC);
  	*err_p= fs_err; 
  	return(NIL_INODE);
  }
#else /* !VMDEXT_SYMLINK */
  if ((*ldirp = last_dir(path, string, &fs_err)) == NIL_INODE)
  {
  	assert(fs_err != EGENERIC);
  	*err_p= fs_err;
  	return(NIL_INODE);
  }
#endif /* VMDEXT_SYMLINK */

  /* The final directory is accessible. Get final component of the path. */
  rip = advance(ldirp, string, &fs_err);
  if (S_ISDIR(bits) && (*ldirp)->i_nlinks >= (*ldirp)->i_sp->s_link_max) {
	/* New entry is a directory, alas we can't give it a ".." */
where();
	put_inode(rip);
	*err_p = EMLINK;
	return(NIL_INODE);
  }
  if ( rip == NIL_INODE && fs_err == ENOENT) {
	/* Last path component does not exist.  Make new directory entry. */

	fs_err= EGENERIC;
	if ( (rip = alloc_inode((*ldirp)->i_sp, bits, &fs_err)) == NIL_INODE)
		/* Can't creat new inode: out of inodes. */
	{
		*err_p= fs_err;
		assert(fs_err != EGENERIC);
		fs_err= EGENERIC;
		return(NIL_INODE);
	}

	/* Inherit the group id of the parent dir if the parent is set-gid,
	 * or if the 'grpid' mount option is set.
	 */
	if (rip->i_sp->s_grpid || (*ldirp)->i_mode & I_SET_GID_BIT)
		rip->i_gid = (*ldirp)->i_gid;

	/* Force inode to the disk before making directory entry to make
	 * the system more robust in the face of a crash: an inode with
	 * no directory entry is much better than the opposite.
	 */
	rip->i_nlinks++;
	rip->i_zone[0] = z0;		/* major/minor device numbers */
	rw_inode(rip, WRITING);		/* force inode to disk now */

	/* New inode acquired.  Try to make directory entry. */
	if ((r = search_dir(*ldirp, string, &rip->i_num, ENTER)) != OK)
	{
		rip->i_nlinks--;	/* pity, have to free disk inode */
		rip->i_dirt = DIRTY;	/* dirty inodes are written out */
		put_inode(rip);	/* this call frees the inode */
		assert(r != EGENERIC);
		*err_p = r;
where();
		return(NIL_INODE);
	}

  } else {
	/* Either last component exists, or there is some problem. */
	if (rip != NIL_INODE)
	{
		*created= 0;
	}
	else
	{
		r = fs_err;
		assert(fs_err != EGENERIC);
		fs_err= EGENERIC;
where();
	}
  }

  if (rip == NIL_INODE)
  {
  	assert(r != EGENERIC);
	*err_p = r;
where();
  }
  return(rip);
}


/*===========================================================================*
 *				do_open					     *
 *===========================================================================*/
PUBLIC int do_open()
{
/* Perform the open(name, flags,...) system call. */

  int create_mode = 0;		/* is really mode_t but this gives problems */
  int r;

  assert(fs_call == OPEN);

  /* If O_CREAT is set, open has three parameters, otherwise two. */
  if (open_mode & O_CREAT) {
	create_mode = open_cmode;	
	r = fetch_name(open_cname, open_cnamelen, M1);
  } else {
	r = fetch_name(open_name, open_namelen, M3);
  }

  if (r != OK) return(r); /* name was bad */
  r= common_open(open_mode, create_mode);

  return r;
}


/*===========================================================================*
 *				common_open				     *
 *===========================================================================*/
PRIVATE int common_open(oflags, omode)
register int oflags;
mode_t omode;
{
/* Common code from do_creat and do_open. */

  register struct inode *rip;
  int r, fd, exist = TRUE;
  dev_t dev;
  mode_t bits;
  struct filp *fil_ptr;
  struct od *odp;
  struct inode *ldirp;
  char string[NAME_MAX];
  int fs_err= EGENERIC;
  int created;

  /* Remap the bottom two bits of oflags. */
  bits = (mode_t) mode_map[oflags & O_ACCMODE];

  /* See if file descriptor and filp slots are available. */
  if ( (r = new_fd(0, bits, &fd, &odp)) != OK) return(r);
  fil_ptr= odp->od_filp;
  curr_fd= fd;	/* Global variable for net_open() */

  /* If O_CREATE is set, try to make the file. */ 
  if (oflags & O_CREAT) {
  	/* Create a new inode by calling new_node(). */
        omode = I_REGULAR | (omode & ALL_MODES & fp->fp_umask);
#if VMDEXT_SYMLINK
    	rip = new_node(&ldirp, user_path, string, omode, NO_ZONE,
    					oflags & O_EXCL, &created, &fs_err);
#else /* !VMDEXT_SYMLINK */
    	rip = new_node(&ldirp, user_path, string, omode, NO_ZONE,
							&created, &fs_err);
#endif /* VMDEXT_SYMLINK */
	put_inode(ldirp);
	if (rip == NIL_INODE)
		return fs_err;
    	if (created) {
    		exist = FALSE;      /* we just created the file */
	} else {
		/* File exists, if the O_EXCL flag is set this is an error. */
		if (oflags & O_EXCL) {
			put_inode(rip);
			return(EEXIST);
		}
	}
  } else {
	 /* Scan path name. */
    	if ( (rip = eat_path(user_path, &fs_err)) == NIL_INODE) 
    		return(fs_err);
  }

  /* Claim the file descriptor and filp slot and fill them in. */
  fp->fp_filp[fd] = odp;
  odp->od_flags |= ODF_INUSE;
  fil_ptr->filp_count = 1;
  fil_ptr->filp_ino = rip;
  fil_ptr->filp_flags = oflags;

  /* Only do the normal open code if we didn't just create the file. */
  if (exist) {
  	/* Check protections. */
  	if ((r = forbidden(rip, bits)) == OK) {
  		/* Opening reg. files directories and special files differ. */
	  	switch (rip->i_mode & I_TYPE) {
    		   case I_REGULAR: 
			/* Truncate regular file if O_TRUNC. */
			if (oflags & O_TRUNC) {
				if ((r = forbidden(rip, W_BIT)) !=OK) break;
				freespace(rip, 0, MAX_FILE_POS);
				rip->i_size = 0;
				/* Send the inode from the inode cache to the
				 * block cache, so it gets written on the next
				 * cache flush.
				 */
				rw_inode(rip, WRITING);
			}
			break;
 
	    	   case I_DIRECTORY: 
			/* Directories may be read but not written. */
			r = (bits & W_BIT ? EISDIR : OK);
			break;

	     	   case I_CHAR_SPECIAL:
     		   case I_BLOCK_SPECIAL:
			/* Device files don't work on a nosuid mounted fs. */
			if (rip->i_sp->s_nosuid) {
				r = EACCES;
				break;
			}
			/* Invoke the driver for special processing. */
			dev = (dev_t) rip->i_zone[0];
			r= dev_opcl(1, dev, fp-fproc,
						bits | (oflags & ~O_ACCMODE));
			if (r != OK) break;

			/* The process may want to access FS structures.
			 * It could do a sync(), but this is better.
			 */
			if (search_super(dev) != NIL_SUPER) {
				flushall_inodes(dev);
				if ((rip->i_mode & I_TYPE) == I_CHAR_SPECIAL)
					flushall(dev);
			}
			break;

		   case I_NAMED_PIPE:
			fil_ptr->filp_flags = oflags;
			if (bits != R_BIT && bits != W_BIT)
			{
				r= EINVAL;
				break;
			}
			r = pipe_open(fd, fil_ptr, rip, bits, oflags);
			break;
 		}
  	}
  }

  if (r == SUSPEND)
  	return SUSPEND;

  /* If error, release inode. */
  if (r != OK) {
	fp->fp_filp[fd] = NIL_ODP;
	odp->od_flags= 0;
	fil_ptr->filp_count= 0;
	put_inode(rip);
	return(r);
  }
  
  return(fd);
}


/*===========================================================================*
 *				do_close				     *
 *===========================================================================*/
PUBLIC int do_close()
{
/* Perform the close(fd) system call. */

  int r;

  assert(fs_call == CLOSE);

  r= closefd(close_fd);
  return r;
}


/*===========================================================================*
 *				closefd					     *
 *===========================================================================*/
PUBLIC int closefd(pfd)
int pfd;
{
  register struct filp *rfilp;
  struct od *odp;
  register struct inode *rip;
  int major, status = OK;
  mode_t mode_word;
  dev_t dev;
  int fs_err= EGENERIC;

  /* First locate the inode that belongs to the file descriptor. */
  if ( (odp = get_fd(pfd, &fs_err)) == NIL_ODP) return(fs_err);
  rfilp= odp->od_filp;

  /* Check if there are operations in progress. */
  if (odp->od_flags & (ODF_RD_IP|ODF_WR_IP|ODF_IOC_IP))
  	cancel_fdops(fp, pfd);
  rip = rfilp->filp_ino;	/* 'rip' points to the inode */

  if (rip->i_lock != NULL)
  	lock_unlock(rfilp);

  /* Check to see if the file is special. */
  mode_word = rip->i_mode & I_TYPE;
  if (mode_word == I_CHAR_SPECIAL || mode_word == I_BLOCK_SPECIAL) {
	dev = (dev_t) rip->i_zone[0];
	if (mode_word == I_BLOCK_SPECIAL)  {
		/* Invalidate cache entries unless special is mounted or ROOT*/
		if (search_super(dev) == NIL_SUPER) {
			(void) flushall(dev);	/* purge cache */
			invalidate(dev);
		}
	}
	major = (dev >> MAJOR) & BYTE;	/* major device nr */
	if ((major == CTTY_MAJOR || rfilp->filp_count - 1 == 0) && 
					!(rfilp->filp_int_flags & FIF_CLOSED))
	{
		/* We only call close when the last fd refering to some
		 * device is closed, and the device is not already closed.
		 * Also call close for a controlling tty to update the
		 * fp_ttylink field.
		 */
		status = dev_opcl(0, dev, fp-fproc, 0);
		assert(status == OK || status == EIO);
	}
  }

  /* The controlling tty stuff sometimes causes fp->fp_filp[fd] to be assigned
   * a new value. We have lookup that value. */
  if ( (odp = get_fd(pfd, &fs_err)) == NIL_ODP) 
  {
	panic("unable to relookup fd", NO_NUM);
  }
  rfilp= odp->od_filp;

  /* If a write has been done, the inode is already marked as DIRTY. */
  if (rfilp->filp_count == 1)	/* Last user of the filepointer */
  {
	if (S_ISFIFO(rip->i_mode))
		pipe_close(rip, rfilp);
	put_inode(rip);
  }
  rfilp->filp_count--;

  fp->fp_cloexec &= ~(1L << pfd);	/* turn off close-on-exec bit */
  fp->fp_filp[pfd] = NIL_ODP;
  odp->od_flags= 0;

  return(status);
}


/*===========================================================================*
 *				do_lseek				     *
 *===========================================================================*/
PUBLIC int do_lseek()
{
/* Perform the lseek(fd, offset, whence) system call. */

  register struct filp *rfilp;
  u64_t pos;
  uoff_t offset;
  int fs_err= EGENERIC;

  assert(fs_call == LSEEK);

  /* Check to see if the file descriptor is valid. */
  if ( (rfilp = get_filp(lseek_fd, &fs_err)) == NIL_FILP) return(fs_err);

  /* No lseek on pipes. */
  if (S_ISFIFO(rfilp->filp_ino->i_mode))
  	return ESPIPE;	/* Can't seek on pipe or fifo */

  /* Byte offset to seek to. */
  offset = lseek_offset;

  /* Compute new position and check for overflow.  This is not simple,
   * because to be able to address really big disks we use 64 bit byte
   * offsets, but lseek can only handle a 32 bit offset.  (Signed or
   * unsigned as appropriate.)  We allow the operation if the end result
   * is a valid position that lseek can seek to (i.e. <= MAX_FILE_POS).
   */
  switch (lseek_whence) {
  case SEEK_SET:		/* Set position.  (Already apply offset) */
	pos = cvul64((unsigned long) offset);
	offset = 0;
	break;
  case SEEK_CUR:		/* Offset from current position. */
	pos = rfilp->filp_pos;
	break;
  case SEEK_END:		/* Offset from the end of the file. */
	pos = cvul64(rfilp->filp_ino->i_size);
	break;
  default:
	return(EINVAL);
  }

  if ((long) offset < 0) {
	if (cmp64ul(pos, -offset) < 0) return(EINVAL);
	pos = sub64ul(pos, -offset);
  } else {
	if (cmp64ul(pos, MAX_FILE_POS - offset) > 0) return(EINVAL);
	pos = add64ul(pos, offset);
  }

  if (cmp64(pos, rfilp->filp_pos) != 0)
	rfilp->filp_ino->i_seek = ISEEK;	/* inhibit read ahead */
  rfilp->filp_pos = pos;

  reply_l1 = cv64ul(pos);	/* insert the long into the output message */
  return(OK);
}


/*===========================================================================*
 *				do_mkdir				     *
 *===========================================================================*/
PUBLIC int do_mkdir()
{
/* Perform the mkdir(name, mode) system call. */

  int r1, r2;			/* status codes */
  ino_t dot, dotdot;		/* inode numbers for . and .. */
  mode_t bits;			/* mode bits for the new inode */
  char string[NAME_MAX];	/* last component of the new dir's path name */
  register struct inode *rip;
  struct inode *ldirp;
  int fs_err= EGENERIC;
  int created;

  assert(fs_call == MKDIR);

  /* Check to see if it is possible to make another link in the parent dir. */
  if ((fs_err= fetch_name(mkdir_name, mkdir_namelen, M1)) != OK)
  	return(fs_err);

  /* Next make the inode. If that fails, return error code. */
  bits = I_DIRECTORY | (mkdir_mode & ALL_MODES & fp->fp_umask);

#if VMDEXT_SYMLINK
  rip = new_node(&ldirp, user_path, string, bits, (zone_t) 0, TRUE,
  						&created, &fs_err);
#else /* !VMDEXT_SYMLINK */
  rip = new_node(&ldirp, user_path, string, bits, (zone_t) 0,
  						&created, &fs_err);
#endif /* VMDEXT_SYMLINK */
  if (rip == NIL_INODE || !created) {
  	if (rip != NIL_INODE)
  	{
		put_inode(rip);		/* can't make dir: it already exists */
		fs_err= EEXIST;
	}
	put_inode(ldirp);	/* return parent too */
	return(fs_err);
  }

  /* Spread the set-gid virus down from the parent dir. */
  if (ldirp->i_mode & I_SET_GID_BIT) bits |= I_SET_GID_BIT;

  /* Get the inode numbers for . and .. to enter in the directory. */
  dotdot = ldirp->i_num;	/* parent's inode number */
  dot = rip->i_num;		/* inode number of the new dir itself */

  /* Now make dir entries for . and .. unless the disk is completely full. */
  /* Use dot1 and dot2, so the mode of the directory isn't important. */
  rip->i_mode = bits;	/* set mode */
  r1 = search_dir(rip, dot1, &dot, ENTER);	/* enter . in the new dir */
  r2 = search_dir(rip, dot2, &dotdot, ENTER);	/* enter .. in the new dir */

  /* If both . and .. were successfully entered, increment the link counts. */
  if (r1 == OK && r2 == OK) {
	/* Normal case.  It was possible to enter . and .. in the new dir. */
	rip->i_nlinks++;	/* this accounts for . */
	ldirp->i_nlinks++;	/* this accounts for .. */
	ldirp->i_dirt = DIRTY;	/* mark parent's inode as dirty */
  } else {
	/* It was not possible to enter . or .. probably disk was full. */
	(void) search_dir(ldirp, string, (ino_t *) 0, DELETE);
	rip->i_nlinks--;	/* undo the increment done in new_node() */
  }
  rip->i_dirt = DIRTY;		/* either way, i_nlinks has changed */

  put_inode(ldirp);		/* return the inode of the parent dir */
  put_inode(rip);		/* return the inode of the newly made dir */
  return(OK);
}


#if VMDEXT_SYMLINK
/*===========================================================================*
 *				do_slink				     *
 *===========================================================================*/
PUBLIC int do_slink()
{
/* Perform the symlink(name1, name2) system call. */

  register int r;		/* error code */
  char string[NAME_MAX];	/* last component of the new dir's path name */
  struct inode *sip;		/* inode containing symbolic link */
  struct buf *bp;		/* disk buffer for link */
  struct inode *ldirp;		/* directory containing link */
  int fs_err= EGENERIC;
  int created;

  assert(fs_call == SYMLINK);

  if ((fs_err= fetch_name(symlink_name2, symlink_namelen2, M1) != OK))
	return(fs_err);

  if (symlink_namelen <= 1 || symlink_namelen > BLOCK_SIZE+1)
  	return(ENAMETOOLONG);

  r= OK;
  sip = new_node(&ldirp, user_path, string,
			(mode_t) (I_SYMBOLIC_LINK | RWX_MODES),
			(zone_t) 0, TRUE, &created, &fs_err);
                 
  if (!created)
  	r= EEXIST;

  if (sip == NIL_INODE)
  {
  	r= fs_err;
  	assert(fs_err != EGENERIC);
	fs_err= EGENERIC;
  }
  if (r == OK) {
	if ((bp = new_block(sip, (uoff_t) 0, FALSE, &fs_err)) == NIL_BUF)
	{
		r= fs_err;
		assert(fs_err != EGENERIC);
		fs_err= EGENERIC;
	}
	else
	{
		sip->i_size = symlink_namelen-1;
		r= sys_copy(who, SEG_D, (phys_bytes) symlink_name,
			FS_PROC_NR, SEG_D, (phys_bytes) bp->b_data,
			(phys_bytes) sip->i_size);
		put_block(bp);
	}
	
	if (r != OK) {
		sip->i_nlinks = 0;
		if (search_dir(ldirp, string, (ino_t *) 0, DELETE) != OK)
			panic("Symbolic link vanished", NO_NUM);
	}
  }
  put_inode(sip);
  put_inode(ldirp);
  return(r);
}
#endif /* VMDEXT_SYMLINK */

/*
 * $PchId: open.c,v 1.6 1996/02/29 23:09:24 philip Exp $
 */
