/* This file contains a collection of miscellaneous procedures.  Some of them
 * perform simple system calls.  Some others do a little part of system calls
 * that are mostly performed by the Memory Manager.
 *
 * The entry points into this file are
 *   do_dup:	  perform the DUP system call
 *   do_fcntl:	  perform the FCNTL system call
 *   do_reboot:	  sync disks and mark file systems clean
 *   do_sync:	  perform the SYNC system call
 *   do_fork:	  adjust the tables after MM has performed a FORK system call
 *   do_exec:	  handle files with FD_CLOEXEC on after MM has done an EXEC
 *   do_exit:	  a process has exited; note that in the tables
 *   do_sysuname: set or get machine description strings
 *   do_set:	  set uid or gid for some process
 *   do_revive:	  revive a process that was waiting for something (e.g. TTY)
 *   do_svrctl:   file system control
 */

#include "fs.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>		/* cc runs out of memory with unistd.h :-( */
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/queryparam.h>
#include <sys/svrctl.h>
#include "assert.h"
INIT_ASSERT
#include "buf.h"
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "param.h"
#include "super.h"
#include "dev.h"

FORWARD _PROTOTYPE( int unpause, (int proc_nr)				);
FORWARD _PROTOTYPE( int fsqueryparam, (int proc_nr, vir_bytes argp)	);
FORWARD _PROTOTYPE( int fq_getc, (void)					);
FORWARD _PROTOTYPE( void fq_flush, (void)				);


/*===========================================================================*
 *				do_dup					     *
 *===========================================================================*/
PUBLIC int do_dup()
{
/* Perform the dup(fd) or dup(fd,fd2) system call. These system calls are
 * obsolete.  In fact, it is not even possible to invoke them using the
 * current library because the library routines call fcntl().  They are
 * provided to permit old binary programs to continue to run.
 */

  register int rfd;
  register struct filp *f;
  struct od *odp;
  struct inode *rip;
  int fd, r;
  int fs_err= EGENERIC;

  assert(fs_call == DUP);

  /* Is the file descriptor valid? */
  rfd = dup_fd & ~DUP_MASK;		/* kill off dup2 bit, if on */
  if ((f = get_filp(rfd, &fs_err)) == NIL_FILP) return(fs_err);

  /* Distinguish between dup and dup2. */
  if (dup_fd == rfd) {			/* bit not on */
	/* dup(fd) */
	if ( (r = new_fd(0, 0, &dup_fd2, &odp)) != OK) return(r);
  } else {
	/* dup2(fd, fd2) */
	if (dup_fd2 < 0 || dup_fd2 >= OPEN_MAX) return(EBADF);
	if (rfd == dup_fd2) return(dup_fd2);	/* ignore the call: dup2(x, x) */
	fd = dup_fd2;		/* prepare to close fd2 */
	(void) closefd(dup_fd2);	/* cannot fail */
	if ( (r = new_fd(fd, 0, &dup_fd2, &odp)) != OK) return(r);
	assert(fd == dup_fd2);
  }

  /* Success. Set up new file descriptors. */
  f->filp_count++;
  fp->fp_filp[dup_fd2] = odp;
  odp->od_flags |= ODF_INUSE;
  odp->od_filp= f;

  /* Unfortunately, we have to maintain fp_ttylink for controlling ttys. */
  rip= f->filp_ino;
  if (S_ISCHR(rip->i_mode) && ((rip->i_zone[0] >> MAJOR) & BYTE) == CTTY_MAJOR)
  {
  	assert(fp->fp_ttylink != 0);
  	fp->fp_ttylink++;
  }

  return(dup_fd2);
}

/*===========================================================================*
 *				do_fcntl				     *
 *===========================================================================*/
PUBLIC int do_fcntl()
{
/* Perform the fcntl(fd, request, addr) system call. */

  register struct filp *f;
  struct od *odp;
  int n_fd, r, fl;
  long cloexec_mask;		/* bit map for the FD_CLOEXEC flag */
  long clo_value;		/* FD_CLOEXEC flag in proper position */
  struct inode *rip;
  int fs_err= EGENERIC;

  assert(fs_call == FCNTL);

  /* Is the file descriptor valid? */
  if ((f = get_filp(fcntl_fd, &fs_err)) == NIL_FILP) return(fs_err);
  odp= get_fd(fcntl_fd, &fs_err);
  assert (odp != NIL_ODP);

  /* XXX - F_FREESP and F_SEEK were renumbered, but old code persists. */
  if (fcntl_request == 8) fcntl_request = F_FREESP;
  if (fcntl_request == 9) fcntl_request = F_SEEK;

  switch (fcntl_request) {
     case F_DUPFD: 
	/* This replaces the old dup() system call. */
	if (fcntl_iarg < 0 || fcntl_iarg >= OPEN_MAX) return(EINVAL);
	if ((r = new_fd(fcntl_iarg, 0, &n_fd, &odp)) != OK) return(r);
	odp->od_filp= f;
	odp->od_flags |= ODF_INUSE;
   	f->filp_count++;
  	fp->fp_filp[n_fd] = odp;

	/* Unfortunately, we have to maintain fp_ttylink for controlling 
	 * ttys.
	 */
	rip= f->filp_ino;
	if (S_ISCHR(rip->i_mode) 
			&& ((rip->i_zone[0] >> MAJOR) & BYTE) == CTTY_MAJOR)
	{
		assert(fp->fp_ttylink != 0);
		fp->fp_ttylink++;
	}

  	return(n_fd);

     case F_GETFD: 
	fl= 0;
	/* Get close-on-exec flag (FD_CLOEXEC in POSIX Table 6-2). */
	if ((fp->fp_cloexec >> fcntl_fd) & 1) 
		fl |= FD_CLOEXEC;
	if (odp->od_flags & ODF_ASYNCH)
		fl |= FD_ASYNCHIO;
	return (fl);

     case F_SETFD: 
	/* Check if the user doesn't try to set unsupported flags. */
	if (fcntl_iarg & ~(FD_CLOEXEC|FD_ASYNCHIO))
		return (EINVAL);
	/* Set close-on-exec flag (FD_CLOEXEC in POSIX Table 6-2). */
	cloexec_mask = 1L << fcntl_fd;	/* singleton set position ok */
	clo_value = (fcntl_iarg & FD_CLOEXEC ? cloexec_mask : 0L);
	fp->fp_cloexec = (fp->fp_cloexec & ~cloexec_mask) | clo_value;
	if (fcntl_iarg & FD_ASYNCHIO)
		odp->od_flags |= ODF_ASYNCH;
	else
		odp->od_flags &= ~ODF_ASYNCH;
	return(OK);

     case F_GETFL: 
	/* Get file status flags (O_NONBLOCK and O_APPEND). */
	fl = f->filp_flags & (O_NONBLOCK | O_APPEND | O_ACCMODE);
	return(fl);	

     case F_SETFL: 
	/* Set file status flags (O_NONBLOCK and O_APPEND). */
	fl = O_NONBLOCK | O_APPEND;
	f->filp_flags = (f->filp_flags & ~fl) | (fcntl_iarg & fl);
	return(OK);

     case F_GETLK:
     case F_SETLK:
     case F_SETLKW:
     case F_FREESP:
	/* Set or clear a file lock / Free storage space. */
	r = lock_op(f, fcntl_request, fcntl_carg);
	return(r);

     case F_SEEK:
	return(sys_copy(who, SEG_D, (phys_bytes) fcntl_carg,
		FS_PROC_NR, SEG_D, (phys_bytes) &f->filp_pos,
		(phys_bytes) sizeof(u64_t)));

     default:
	return(EINVAL);
  }
}


/*===========================================================================*
 *				do_sync					     *
 *===========================================================================*/
PUBLIC int do_sync()
{
/* Perform the sync() system call.  Flush all the tables. */

  register struct super_block *sp;

  assert(fs_call == SYNC);

  sync_all(DIRTY);

  /* Superblocks are marked clean after N syncs that do nothing. */
  for (sp = &super_block[0]; sp < &super_block[NR_SUPERS]; sp++) {
	if (sp->s_dev != NO_DEV) {
		if (sp->s_sync_ct == N_SYNCS_CLEAN) {
			super_clean(sp, TRUE);
		} else {
			sp->s_sync_ct++;
		}
	}
  }

  return(OK);		/* sync() can't fail */
}

/*===========================================================================*
 *				sync_all				     *
 *===========================================================================*/
PUBLIC void sync_all(dirt_level)
int dirt_level;		/* DIRTY or GRIMY: don't always flush ghost blocks. */
{

  register struct inode *rip;
  register struct buf *bp;

  /* The order in which the various tables are flushed is critical.  The
   * blocks must be flushed last, since rw_inode() leaves its results in
   * the block cache.
   */

  /* Write all the dirty inodes to the disk. */
  for (rip = &inode_table[0]; rip < &inode_table[NR_INODES]; rip++) {
	if (rip->i_dev != NO_DEV) {
		if (rip->i_nlinks == 0 && rip->i_count != 0) {
			/* Inodes with no links in the filesystem are not very
			 * interesting, so need not be flushed. (Dirty blocks
			 * in these files are marked as GRIMY and are thus not
			 * flushed alone on a sync() call.)  The only problem
			 * is that they keep the FS in a bad state.
			 */
			rip->i_sp->s_sync_ct = 0;
		} else {
			if (rip->i_dirt >= dirt_level) rw_inode(rip, WRITING);
		}
	}
  }

  sync_all_bufs(dirt_level);
}


/*===========================================================================*
 *				do_reboot				     *
 *===========================================================================*/
PUBLIC int do_reboot()
{
/* Perform the FS side of the reboot call. */

	struct mount *mp;
	struct super_block *sp;
	struct inode *rip;
	struct inode root_mp;
	int fs_err= EGENERIC;
	int i;

	assert(fs_call == REBOOT);

	/* Only MM may make this call directly. */
	if (who != MM_PROC_NR) return(EGENERIC);

	/* Release the root and working directories of the servers. */
	for (fp= fproc; fp < &fproc[NR_PROCS]; fp++)
	{
		rip= fp->fp_workdir;
		if (rip != NIL_INODE)
		{
			put_inode(rip);
			fp->fp_workdir= NIL_INODE;
		}
		rip= fp->fp_rootdir;
		if (rip != NIL_INODE)
		{
			put_inode(rip);
			fp->fp_rootdir= NIL_INODE;
		}
	}

	/* Unmount all filesystems. */
	for (i= 0; i < NR_MOUNTS; i++)
	{
		/* Unmount at least one. */
		for (mp= &mount_table[0]; mp < &mount_table[NR_MOUNTS]; mp++)
		{
			if (mp->m_imount != NIL_INODE)
				(void) unmount(mp);
		}
	}

	/* The root file system is not mounted on anything, which makes it
	 * kind of difficult to unmount it.  A good thing there is thin air
	 * to draw a fake mount point from.
	 */
	mp= &mount_table[0];
	sp= &super_block[0];
	mp->m_iroot= get_inode(sp->s_dev, ROOT_INODE, &fs_err);
	mp->m_iroot->i_mounted_on= &root_mp;
	root_mp.i_mounted_by= mp->m_iroot;
	root_mp.i_count= 2;
	mp->m_imount= &root_mp;
	(void) unmount(mp);

	for (sp = &super_block[0]; sp < &super_block[NR_SUPERS]; sp++)
		assert(sp->s_dev == NO_DEV);

#if PCH_DEBUG
	assert(!check_filedes());
#endif

	return(OK);
}


/*===========================================================================*
 *				do_fork					     *
 *===========================================================================*/
PUBLIC int do_fork()
{
/* Perform those aspects of the fork() system call that relate to files.
 * In particular, let the child inherit its parent's file descriptors.
 * The parent and child parameters tell who forked off whom. The file
 * system uses the same slot numbers as the kernel.  Only MM makes this call.
 */

  register struct fproc *cp;
  int i;
  int j;
  struct od *odp;

  assert(fs_call == FORK);

  /* Only MM may make this call directly. */
  if (who != MM_PROC_NR) return(EGENERIC);

  /* Copy the parent's fproc struct to the child. */
  fproc[fork_child]= fproc[fork_parent];

  /* Increase the counters in the 'filp' table. */
  cp = &fproc[fork_child];
  for (i = 0; i < OPEN_MAX; i++)
  {
	if (cp->fp_filp[i] == NIL_ODP) continue;
	for (j= 0, odp= &od_table[0]; j<NR_ODS; j++, odp++)
	{
		if (!(odp->od_flags & ODF_INUSE))
			break;
	}
	if (j == NR_ODS)	/* Bad luck */
	{
		cp->fp_filp[i]= NIL_ODP;
		continue;
	}
	odp->od_flags |= ODF_INUSE;
	odp->od_filp= cp->fp_filp[i]->od_filp;
	cp->fp_filp[i]= odp;
	odp->od_filp->filp_count++;
  }

  /* Fill in new process id and, if necessary, process group. */
  cp->fp_pid = fork_pid;
  if (cp->fp_ttyfilp)
	cp->fp_ttyfilp->filp_count++;
  cp->fp_sesldr= 0;	/* A child is not a session leader. */

  /* Record the fact that both root and working dir have another user. */
  dup_inode(cp->fp_rootdir);
  dup_inode(cp->fp_workdir);
  return(OK);
}


/*===========================================================================*
 *				do_exec					     *
 *===========================================================================*/
PUBLIC int do_exec()
{
/* Files can be marked with the FD_CLOEXEC bit (in fp->fp_cloexec).  When
 * MM does an EXEC, it calls FS to allow FS to find these files and close them.
 */

  register int i;
  long bitmap;

  assert(fs_call == EXEC);

  /* Only MM may make this call directly. */
  if (who != MM_PROC_NR) return(EGENERIC);

  /* The array of FD_CLOEXEC bits is in the fp_cloexec bit map. */
  fp = &fproc[exec_proc];		/* get_filp() needs 'fp' */
  bitmap = fp->fp_cloexec;
  if (bitmap == 0) return(OK);	/* normal case, no FD_CLOEXECs */

  /* Check the file desriptors one by one for presence of FD_CLOEXEC. */
  for (i = 0; i < OPEN_MAX; i++) {
	if ( (bitmap >> i) & 1) (void) closefd(i);
  }

  return(OK);
}


/*===========================================================================*
 *				do_exit					     *
 *===========================================================================*/
PUBLIC int do_exit()
{
/* Perform the file system portion of the exit(status) system call. */

  register int i, exitee;

  assert(fs_call == EXIT);

  /* Only MM may do the EXIT call directly. */
  if (who != MM_PROC_NR) return(EGENERIC);

  /* Nevertheless, pretend that the call came from the user. */
  fp = &fproc[exit_proc];		/* get_filp() needs 'fp' */
  exitee = exit_proc;

  switch(fp->fp_event)
  {
  case EV_NONE:
  	break;
  case EV_POPEN:
  case EV_READ:
  case EV_WRITE:
  case EV_IOCTL:
	(void) unpause(exitee);	/* this always succeeds for MM */
	break;
  case EV_FWAIT:
  case EV_LOCK:
	break;
  default:
  	panic("strange event", fp->fp_event);
  }
  fp->fp_event= EV_NONE;

  /* Loop on file descriptors, closing any that are open. */
  for (i = 0; i < OPEN_MAX; i++) {
	(void) closefd(i);
  }

  /* When a session leaders dies filedescriptors refering to the tty
   * assiosiated with that session should be made inaccessable.
   */

  if (fp->fp_sesldr && fp->fp_tty)
  {
	exit_sesldr();
  }

  /* Release root and working directories. */
  put_inode(fp->fp_rootdir);
  put_inode(fp->fp_workdir);
  fp->fp_rootdir = NIL_INODE;
  fp->fp_workdir = NIL_INODE;
  fp->fp_pid = exit_server ? PID_SERVER : PID_FREE;

  return(OK);
}


/*===========================================================================*
 *				do_sysuname				     *
 *===========================================================================*/
PUBLIC int do_sysuname()
{
/* Set or get uname strings. */

  int r;
  size_t n, len;
  char *string, *t;
  char tmp[sizeof(uts_val.nodename)];
  static short sizes[] = {
	0,	/* arch, (0 = read-only) */
	0,	/* kernel */
	0,	/* machine */
	sizeof(uts_val.hostname),
	sizeof(uts_val.nodename),
	0,	/* release */
	0,	/* version */
	0,	/* sysname */
  };

  if ((unsigned) sysuname_field >= _UTS_MAX) return(EINVAL);

  string = uts_tbl[sysuname_field];

  switch (sysuname_req) {
  case _UTS_GET:
	/* Copy an uname string to the user. */
	n = strlen(string) + 1;
	if (n > sysuname_len) n = sysuname_len;
	r = sys_copy(FS_PROC_NR, SEG_D, (phys_bytes) string, 
		who, SEG_D, (phys_bytes) sysuname_string, (phys_bytes) n);
	if (r < 0) return(r);
	break;

  case _UTS_SET:
	/* Set an uname string, needs root power. */
	len = sizes[sysuname_field];
	if (!super_user || len == 0) return(EPERM);
	n = len < sysuname_len ? len : sysuname_len;
	if (n <= 0) return(EINVAL);
	r = sys_copy(who, SEG_D, (phys_bytes) sysuname_string,
		FS_PROC_NR, SEG_D, (phys_bytes) tmp, (phys_bytes) n);
	if (r < 0) return(r);
	tmp[n-1] = 0;
#if VMDEXT_HIDDENDIRS
	/* Don't accept '/' and '@', can't use them in file names. */
	for (t = tmp; *t != 0; t++) {
		if (*t == '/' || *t == '@') return(EINVAL);
	}
#endif /* VMDEXT_HIDDENDIRS */
	strcpy(string, tmp);
	break;

  default:
	return(EINVAL);
  }
  /* Return the number of bytes moved. */
  return(n);
}


/*===========================================================================*
 *				do_set					     *
 *===========================================================================*/
PUBLIC int do_set()
{
/* Set uid_t or gid_t field. */

  register struct fproc *tfp;

  assert(fs_call == SETUID || fs_call == SETGID || fs_call == GSGROUPS);

  /* Only MM may make this call directly. */
  if (who != MM_PROC_NR) return(EGENERIC);

  tfp = &fproc[setxid_proc];
  if (fs_call == SETUID) {
	tfp->fp_realuid = (uid_t) setuid_ruid;
	tfp->fp_effuid =  (uid_t) setuid_euid;
  }
  if (fs_call == SETGID) {
	tfp->fp_effgid =  (gid_t) setgid_egid;
	tfp->fp_realgid = (gid_t) setgid_rgid;
  }
  if (fs_call == GSGROUPS) {
	if ((tfp->fp_ngroups = setgrps_index+1) > 0)
		tfp->fp_groups[setgrps_index] = setgrps_group;
  }
  return(OK);
}


/*===========================================================================*
 *				do_revive				     *
 *===========================================================================*/
PUBLIC int do_revive()
{
/* A task, typically TTY, has now gotten the characters that were needed for a
 * previous read.  The process did not get a reply when it made the call.
 * Instead it was suspended.  Now we can send the reply to wake it up.  This
 * business has to be done carefully, since the incoming message is from
 * a task (to which no reply can be sent), and the reply must go to a process
 * that blocked earlier.  The reply to the caller is inhibited by setting the
 * 'dont_reply' flag, and the reply to the blocked process is done explicitly
 * in revive().
 */

  int proc_nr, event, op;
  struct fproc *rfp;

  assert(fs_call == REVIVE);

#if !ALLOW_USER_SEND
  if (who >= 0 && fp->fp_pid != PID_SERVER) return(EPERM);
#endif

  proc_nr= m.REP_PROC_NR;
  if (proc_nr < 0 || proc_nr >= NR_PROCS) panic("bad REVIVE", proc_nr);
  rfp = &fproc[proc_nr];
  event= rfp->fp_event;
  op= ASIO_NOOP;
  if (event == EV_READ) op= ASIO_READ;
  else if (event == EV_WRITE) op= ASIO_WRITE;
  else if (event == EV_IOCTL) op= ASIO_IOCTL;
  assert(op != ASIO_NOOP);

  fd_revive(m.REP_PROC_NR, rfp->fp_state.fp_rwio.fp_fd, op, m.REP_STATUS);

  dont_reply = TRUE;		/* don't reply to the TTY task */
  return(OK);
}


/*===========================================================================*
 *				do_fcancel				     *
 *===========================================================================*/
PUBLIC int do_fcancel()
{
	int req_fd, operation, status, ipcmpl_flags;
	struct od *odp;
	int fs_err= EGENERIC;

	assert(fs_call == FCANCEL);

	req_fd= cancel_fd;
	operation= cancel_operation;

	/* Is the file descriptor valid? */
	odp= get_fd(req_fd, &fs_err);
	if (odp == NIL_ODP)
		return fs_err;

	if (operation < 0 || operation >= ASIO_NR)
		return EINVAL;

	if ((odp->od_flags & asio_compl[operation]) != 0)
	{
		assert ((odp->od_flags & asio_ip[operation]) != 0);
		return OK;
	}
	if ((odp->od_flags & asio_ip[operation]) == 0)
		return EINVAL;

	ipcmpl_flags= asio_ip[operation] | asio_compl[operation];
	status= fd_cancel(fp-fproc, req_fd, operation);
	assert(status == OK);
	assert((odp->od_flags & ipcmpl_flags) == ipcmpl_flags);
	return OK;
}


/*===========================================================================*
 *				do_fwait				     *
 *===========================================================================*/
PUBLIC int do_fwait()
{
	struct fwait fw;
	struct asio_fd_set fds;
	int i, j, r, rep_fd, rep_op;
	int ipf, compf;
	struct od *odp;
	int fs_err= EGENERIC;
	int k,l,mask,n;
	asio_fd_mask b, *bp;

	assert(fs_call == FWAIT);

	/* Fetch the fwait struct from user space. */
	r= sys_copy(who, SEG_D, (phys_bytes) fwait_ptr,
		FS_PROC_NR, SEG_D, (phys_bytes) &fw, (phys_bytes) sizeof(fw));
	if (r != OK)
	{
		assert(r == EFAULT);
		return r;
	}
	if (fw.fw_maxfd < 0)
	{
		return EINVAL;
	}
	if (fw.fw_maxfd > ASIO_FD_SETSIZE)
		fw.fw_maxfd= ASIO_FD_SETSIZE;

	/* fetch the fds. */
	ASIO_FD_ZERO(&fds);
	if (fw.fw_maxfd != 0)
	{
		r= sys_copy(who, SEG_D, (phys_bytes) fw.fw_bits,
			FS_PROC_NR, SEG_D, (phys_bytes) fds.afds_bits,
			(phys_bytes) (4*howmany(fw.fw_maxfd, NBBY)));
		if (r != OK)
		{
			assert(r == EFAULT);
			return r;
		}
	}

	/* save the fds in the process struct in case we have to block. */
	if (!(fw.fw_flags & FWF_NONBLOCK))
		fp->fp_state.fp_fwait.fp_fds= fds;

	/* examin the filedescriptors. */
	rep_fd= -1;
	rep_op= ASIO_NOOP;
	fw.fw_flags &= ~FWF_MORE;
#ifndef FD_SAFE_VERSION
	/* Bypass the ASIO_FD_* macros for speed */
	n= howmany(fw.fw_maxfd, ASIO_NFDBITS) * 4;
	for (k= 0, bp= &fds.afds_bits[0][0]; k<n; k++, bp++)
	{
		if (*bp == 0)
			continue;
		b= *bp;
		j= (k & 3);
		for (l=0, mask= 1; l<ASIO_NFDBITS; l++, mask <<= 1)
		{
			if (!(b & mask))
				continue;
			i= (unsigned)k/4*ASIO_NFDBITS + l;
#else
	for (i= 0; i<fw.fw_maxfd; i++)
	{
		for (j= 0; j<ASIO_NR; j++)
		{
			if (!ASIO_FD_ISSET(i, j, &fds))
				continue;	/* Nothing to do. */
#endif
			odp= get_fd(i, &fs_err);
			if (odp == NIL_ODP)
			{
				return fs_err;	/* Closed fd. */
			}
			ipf= asio_ip[j];
			if (!(odp->od_flags & ipf))
			{
				return EINVAL;	/* Not operation in progress. */
			}
			compf= asio_compl[j];
			if (!(odp->od_flags & compf))
			{
				ASIO_FD_CLR(i, j, &fds);
				continue;	/* For FWF_MORE */
			}
			if (rep_op == ASIO_NOOP)
			{
				/* First completed operation. */
				rep_fd= i;
				rep_op= j;
				ASIO_FD_CLR(i, j, &fds);
				continue;	/* For FWF_MORE */
			}
			/* second completed operation. */
			fw.fw_flags |= FWF_MORE;
			break;
		}
#ifndef FD_SAFE_VERSION
		if (l != ASIO_NFDBITS)
			break;
	}
#else
		if (j != ASIO_NR)
			break;
	}
#endif

	/* report. */
	if (rep_op != ASIO_NOOP)	/* Found something. */
	{
		if (fw.fw_flags & FWF_MORE)	/* reply fds. */
		{
			r= sys_copy(FS_PROC_NR,SEG_D,(phys_bytes)fds.afds_bits,
				who, SEG_D, (phys_bytes) fw.fw_bits,
				(phys_bytes) (4*howmany(fw.fw_maxfd, NBBY)));
			if (r != OK)
			{
				assert(r == EFAULT);
				return r;
			}
		}
		/* Fill in the fwait struct. */
		odp= get_fd(rep_fd, &fs_err);
		assert(odp != NIL_ODP);
		fw.fw_result= odp->od_result[rep_op];
		if (fw.fw_result < 0)	/* Error */
		{
			fw.fw_errno= -fw.fw_result;
			fw.fw_result= -1;
		}
		fw.fw_fd= rep_fd;
		fw.fw_operation= rep_op;

		/* Reply the fwait struct to the user. */
		r= sys_copy(FS_PROC_NR, SEG_D, (phys_bytes) &fw,
			 who, SEG_D, (phys_bytes) fwait_ptr,
			 (phys_bytes) sizeof(fw));
		if (r != OK)
		{
			assert(r == EFAULT);
			return r;
		}

		/* Now we safely turn off the in progress and completed bits. */
		odp->od_flags &= ~(asio_ip[rep_op] | asio_compl[rep_op]);

		return OK;
	}
	if (fw.fw_flags & FWF_NONBLOCK)
	{
		return(EAGAIN);
	}
	
	assert(fp->fp_event == EV_NONE);
	fp->fp_state.fp_fwait.fp_buffer= fwait_ptr;
	fp->fp_event= EV_FWAIT;
	assert(dont_reply == FALSE);
	dont_reply = TRUE;	/* do not send caller a reply message now */
	return SUSPEND;
}


/*===========================================================================*
 *				do_unpause				     *
 *===========================================================================*/
PUBLIC int do_unpause()
{
/* A signal has been sent to a user who is paused on the file system.
 * Abort the system call with the EINTR error message.
 */

  int proc_nr;

  assert(fs_call == UNPAUSE);

  if (who > MM_PROC_NR) return(EPERM);
  proc_nr = unpause_proc;
  return unpause(proc_nr);
}
  

/*===========================================================================*
 *				unpause					     *
 *===========================================================================*/
PRIVATE int unpause(proc_nr)
int proc_nr;
{
/* A signal has been sent to a user who is paused on the file system.
 * Abort the system call with the EINTR error message.
 */

  register struct fproc *rfp;
  int fild;
  struct od *odp;
  int ipcmpl_flags;
  int status;
  int r, op, operation;
  int event;
  int saved_who;
  struct fproc *saved_fp;

  if (proc_nr < 0 || proc_nr >= NR_PROCS) panic("unpause err 1", proc_nr);
  rfp = &fproc[proc_nr];
  event= rfp->fp_event;
  status= EINTR;		/* default error. */

  switch(event)
  {
  case EV_NONE:
  	return OK;
  case EV_LOCK:		/* process trying to set a lock with FCNTL */
	break;

  case EV_POPEN:		/* process trying to open a fifo */
  	/* we can simply undo the work done by pipe_open by closing the
  	 * new filedescriptor. We have to fake proc_nr to be the current
  	 * process.
  	 */
  	saved_who= who;
  	saved_fp= fp;
  	who= proc_nr;
  	fp= rfp;
  	r= closefd(fp->fp_state.fp_popen.fp_fd);
  	assert(r == OK);
  	who= saved_who;
  	fp= saved_fp;
	break;
  case EV_FWAIT:
	break;		/* process was blocked in an fwait */

  case EV_READ:
  case EV_WRITE:
  case EV_IOCTL:
	fild = rfp->fp_state.fp_rwio.fp_fd;
	if (fild < 0 || fild >= OPEN_MAX)panic("unpause err 2",NO_NUM);
	odp = rfp->fp_filp[fild];
	assert(odp);

	operation= ASIO_NOOP;
	if (rfp->fp_event == EV_READ) operation= ASIO_READ;
	else if (rfp->fp_event == EV_WRITE) operation= ASIO_WRITE;
	else if (rfp->fp_event == EV_IOCTL) operation= ASIO_IOCTL;
	assert(operation != ASIO_NOOP);
	op= asio_cancel[operation];
	ipcmpl_flags= asio_ip[operation] | asio_compl[operation];
	status= fd_cancel(rfp-fproc, fild, operation);
	assert(status == OK);
	assert((odp->od_flags & ipcmpl_flags) == ipcmpl_flags);
	odp->od_flags &= ~ipcmpl_flags;
	status= odp->od_result[operation];
	break;
  default:
  	panic("unpause_proc: unknown event", event);
  }

  rfp->fp_event = EV_NONE;
  reply(proc_nr, status);
  return(OK);
}


/*===========================================================================*
 *				do_svrctl				     *
 *===========================================================================*/
PUBLIC int do_svrctl()
{
  if (!super_user) return(EPERM);

  if (svrctl_request == 0) svrctl_request= m.m2_i1;

  switch (svrctl_request) {
  case FSSIGNON: {
	/* A server in user space calls in to manage a device. */
	struct fssignon device;
	int r, major;
	struct dmap *dmapp;

	if (fp->fp_pid != PID_SERVER) return(EPERM);
	r = sys_copy(who, SEG_D, (phys_bytes) svrctl_argp,
		FS_PROC_NR, SEG_D, (phys_bytes) &device,
		(phys_bytes) sizeof(device));
	if (r != OK) return(r);

	major= (device.dev >> MAJOR) & BYTE;
	if (major == 0 || (unsigned) device.style >= 4) return(EINVAL);
	if (dmap[major] != NULL) return(EBUSY);

	/* Find an empty dmap slot. */
	for (dmapp= dmap_table; dmapp < dmap_table + NR_DMAPS; dmapp++) {
		if (dmapp->dmap_major == 0) break;
	}
	if (dmapp == dmap_table + NR_DMAPS) return(ENOMEM);

	if (device.style == STYLE_DEV) {
		/* Old style device. */
		dmapp->dmap_open= gen_opcl;
		dmapp->dmap_close= gen_opcl;
		dmapp->dmap_rw= gen_rw;
		dmapp->dmap_ioctl= gen_ioctl;
		dmapp->dmap_cancel= gen_cancel;
	} else {
		/* One of the new style devices. */
		dmapp->dmap_open= ngen_opcl;
		dmapp->dmap_close= ngen_opcl;
		dmapp->dmap_rw= ngen_rw;
		dmapp->dmap_ioctl= ngen_ioctl;
		dmapp->dmap_cancel= ngen_cancel;
	}
	dmapp->dmap_mess= no_msg;
	dmapp->dmap_task= who;
	dmapp->dmap_major= major;
	if (device.style == STYLE_TTY) {
		dmapp->dmap_open= ntty_open;
		dmapp->dmap_close= ntty_close;
		dmapp->dmap_mess= gen_msg;
	}
	if (device.style == STYLE_CLONE) {
		dmapp->dmap_open= clone_open;
	}
	dmap[major]= dmapp;
	return(OK);	
	}
  case FSQUERYPARAM:
  	/* Query FS for the values of some key parameters. */
  	return(fsqueryparam(who, (vir_bytes) svrctl_argp));

  case FSGPIPEFS: {
  	struct fspipefs fspipefs;
  	int r;

  	fspipefs.dev= pipe_fs->s_dev;
	r = sys_copy(FS_PROC_NR, SEG_D, (phys_bytes) &fspipefs,
		who, SEG_D, (vir_bytes) svrctl_argp,
		(phys_bytes) sizeof(fspipefs));
	return r;
	}

  case FSSPIPEFS: {
  	struct fspipefs fspipefs;
  	struct super_block *sp;
  	int r;

	r = sys_copy(who, SEG_D, (vir_bytes) svrctl_argp,
		FS_PROC_NR, SEG_D, (phys_bytes) &fspipefs,
		(phys_bytes) sizeof(fspipefs));
	if (r != OK) return r;

	for (sp = &super_block[0]; sp < &super_block[NR_SUPERS]; sp++)
		if (sp->s_dev == fspipefs.dev) break;
	if (sp == &super_block[NR_SUPERS])
		return EINVAL;
	pipe_fs= sp;

	return OK;
	}

  case FSDEVMAP: {
	/* Translate a device number to its managing task and minor number. */
	struct fsdevmap devmap;
	int r, major;
	struct dmap *dmapp;

	r = sys_copy(who, SEG_D, (phys_bytes) svrctl_argp,
		FS_PROC_NR, SEG_D, (phys_bytes) &devmap,
		(phys_bytes) sizeof(devmap));
	if (r != OK) return(r);

	major= (devmap.dev >> MAJOR) & BYTE;
	if (major == 0) return(EINVAL);
	if ((dmapp = dmap[major]) == NULL
		|| dmapp->dmap_taskname == NULL
		|| sys_findproc(dmapp->dmap_taskname, &devmap.task_nr, 0) != OK
	) return(ENODEV);

	devmap.minor = (devmap.dev >> MINOR) & BYTE;
	return(sys_copy(FS_PROC_NR, SEG_D, (phys_bytes) &devmap,
		who, SEG_D, (phys_bytes) svrctl_argp,
		(phys_bytes) sizeof(devmap)));
	}
  default:
	return(EINVAL);
  }
}


/*===========================================================================*
 *				fsqueryparam				     *
 *===========================================================================*/
PRIVATE struct queryvars {
	int proc_nr;
	struct svrqueryparam qpar;
	char parbuf[256], valbuf[256];
	char *param, *value;
	size_t vcount;
	int r;
} *qvars;

PRIVATE int fsqueryparam(proc_nr, argp)
int proc_nr;
vir_bytes argp;
{
	/* Return values, sizes, or addresses of variables in FS space. */

	struct queryvars qv;
	void *addr;
	size_t n, size;
	int byte;
	int more;
	static char hex[]= "0123456789ABCDEF";
#define fq_putc(c) ((void)((qv.vcount == 0 ? fq_flush() : (void) 0), \
					*qv.value++= (c), qv.vcount--))

	qv.r= sys_copy(proc_nr, SEG_D, (phys_bytes) argp,
		FS_PROC_NR, SEG_D, (phys_bytes) &qv.qpar,
		(phys_bytes) sizeof(qv.qpar));
	/* Export these to fq_getc() and fq_flush(). */
	qvars= &qv;
	qv.proc_nr= proc_nr;
	qv.param= qv.parbuf + sizeof(qv.parbuf);
	qv.value= qv.valbuf;
	qv.vcount= sizeof(qv.valbuf);

	do {
		more= queryparam(fq_getc, &addr, &size);
		for (n= 0; n < size; n++) {
			byte= ((u8_t *) addr)[n];
			fq_putc(hex[byte >> 4]);
			fq_putc(hex[byte & 0x0F]);
		}
		fq_putc(more ? ',' : 0);
	} while (more);
	fq_flush();
	return qv.r;
}


/*===========================================================================*
 *				fq_getc					     *
 *===========================================================================*/
PRIVATE int fq_getc()
{
	/* Return one character of the names to search for. */
	struct queryvars *qv= qvars;
	size_t n;

	if (qv->r != OK || qv->qpar.psize == 0) return 0;
	if (qv->param == qv->parbuf + sizeof(qv->parbuf)) {
		/* Need to fill the parameter buffer. */
		n= qv->qpar.psize;
		if (n > sizeof(qv->parbuf)) n= sizeof(qv->parbuf);
		qv->r= sys_copy(qv->proc_nr, SEG_D, (phys_bytes) qv->qpar.param,
			FS_PROC_NR, SEG_D, (phys_bytes) qv->parbuf,
			(phys_bytes) n);
		if (qv->r != OK) return 0;
		qv->qpar.param+= n;
		qv->param= qv->parbuf;
	}
	qv->qpar.psize--;
	return (u8_t) *qv->param++;
}


/*===========================================================================*
 *				fq_flush				     *
 *===========================================================================*/
PRIVATE void fq_flush()
{
	/* Send gathered characters back to the user. */
	struct queryvars *qv= qvars;
	size_t n;

	if (qv->r != OK || qv->qpar.vsize == 0) return;

	if ((n= qv->value - qv->valbuf) > qv->qpar.vsize) n= qv->qpar.vsize;
	if (n > 0 && qv->r == OK) {
		/* Copy the value buffer to user space. */
		qv->r= sys_copy(FS_PROC_NR, SEG_D, (phys_bytes) qv->valbuf,
			qv->proc_nr, SEG_D, (phys_bytes) qv->qpar.value,
			(phys_bytes) n);
		qv->qpar.value+= n;
		qv->qpar.vsize-= n;
	}
	qv->value= qv->valbuf;
	qv->vcount= sizeof(qv->valbuf);
}


#if !NDEBUG
/*=========================================================================*
 *				bad_assertion				   *
 *=========================================================================*/
PUBLIC void bad_assertion(file, line, what)
char *file;
int line;
char *what;
{
  printf("fs panic at %s(%d): assertion \"%s\" failed\n", file, line, what);
  panic(NULL, NO_NUM);
}


/*=========================================================================*
 *				bad_compare				   *
 *=========================================================================*/
PUBLIC void bad_compare(file, line, lhs, what, rhs)
char *file;
int line;
int lhs;
char *what;
int rhs;
{
  printf("fs panic at %s(%d): compare (%d) %s (%d) failed\n",
	file, line, lhs, what, rhs);
  panic(NULL, NO_NUM);
}
#endif /* !NDEBUG */

/*
 * $PchId: misc.c,v 1.8 1996/02/29 23:09:24 philip Exp $
 */
