/* This file contains the main program of the File System.  It consists of
 * a loop that gets messages requesting work, carries out the work, and sends
 * replies.
 *
 * The entry points into this file are
 *   main:	main program of the File System
 *   reply:	send a reply to a process after the requested work is done
 */

struct super_block;		/* proto.h needs to know this */

#include "fs.h"
#include <fcntl.h>
#include <string.h>
#include <stddef.h>
#include <sys/ioctl.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/queryparam.h>
#include "assert.h"
INIT_ASSERT
#include "buf.h"
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "param.h"
#include "super.h"
#include <alloca.h>
#include <stdlib.h>

#include "../version/version.h"	/* Tells the O.S. release and version. */
extern int revision;		/* Revision number is linked in. */
#include <minix/minlib.h>

#define DEV_RAM		0x100	/* device number of the RAM disk */
#define MAX_RAM		16384	/* maximum RAM disk size in blocks */

FORWARD _PROTOTYPE( void fs_init, (void)				);
FORWARD _PROTOTYPE( void get_work, (void)				);
FORWARD _PROTOTYPE( void load_ram, (void)				);
FORWARD _PROTOTYPE( int fill_ram, (Dev_t dev, block_t block, block_t count) );
FORWARD _PROTOTYPE( struct inode *load_super, (void)			);


PUBLIC struct utsname uts_val = {
  "Minix-vmd",		/* system name */
  "noname.no.do.main",	/* node/network name */
  OS_RELEASE,		/* O.S. release (e.g. 1.5) */
  OS_VERSION,		/* O.S. version (e.g. 10) */
  "xyzzy",		/* machine (cpu) type (filled in later) */
#if __i386
  "i386",		/* architecture */
  "i386",		/* kernel architecture */
#else
#error			/* oops, no 'uname -mk' */
#endif
  "noname",		/* hostname */
  "ibm",		/* bus architecture */
};

PUBLIC char *uts_tbl[] = {
  uts_val.arch,
  uts_val.kernel,
  uts_val.machine,
  uts_val.hostname,
  uts_val.nodename,
  uts_val.release,
  uts_val.version,
  uts_val.sysname,
  uts_val.bus,
};

/* Variables that can be inspected by programs that like to dig around. */
PRIVATE struct export_param_list fs_ex_param_list[] = {
  QP_VARIABLE(fproc),
  QP_ARRAY(fproc),
  QP_FIELD(fp_tty, struct fproc),
  QP_FIELD(fp_event, struct fproc),
  QP_FIELD(fp_pid, struct fproc),
  QP_FIELD(fp_sesldr, struct fproc),
  QP_END()
};
PRIVATE struct export_params fs_ex_params = { fs_ex_param_list };


/*===========================================================================*
 *				main					     *
 *===========================================================================*/
PUBLIC void main()
{
/* This is the main program of the file system.  The main loop consists of
 * three major activities: getting new work, processing the work, and sending
 * the reply.  This loop never terminates as long as the file system runs.
 */
  int error;
  int fs_debug;

  /* This should go to buf_pool() once malloc() is implemented for FS. */
  char *env_nr_bufs;
  int freemem = atoi(getenv("memsize")) + atoi(getenv("emssize"));

  /* Default number of buffers is 1/16 of free memory. */
  nr_bufs = freemem * 1024L / 16 / sizeof(cache[0]);
#if (CHIP == INTEL)
  /* Don't use all of the "DMA-able" memory. */
  if (nr_bufs > 12L * 1024 * 1024 / sizeof(cache[0]))
	nr_bufs = 12L * 1024 * 1024 / sizeof(cache[0]);
#endif

  if ((env_nr_bufs = getenv("NR_BUFS")) != NULL) {
  	/* Boot environment setting overrides. */
  	nr_bufs = atoi(env_nr_bufs);
  }
  if (nr_bufs < NR_IOREQS) nr_bufs = NR_IOREQS;
  buf = alloca(nr_bufs * sizeof(buf[0]));
  cache = alloca((nr_bufs + 1) * sizeof(cache[0]));
  buf_limit = buf + nr_bufs;

  /* Prepare a hash table of at least twice nr_bufs. */
  for (nr_buf_hash = 1; nr_buf_hash < 2 * nr_bufs; nr_buf_hash <<= 1) {}
  buf_hash = alloca(nr_buf_hash * sizeof(buf_hash[0]));
  hash_mask = nr_buf_hash - 1;

  fs_init();
#if PCH_DEBUG
  assert(!check_filedes());
#endif
  fs_debug= (getenv("fsdebug") != NULL &&
	strcmp(getenv("fsdebug"), "on") == 0);
  if (fs_debug)
	printf("main: fsdebug enabled\n");

  qp_export(&fs_ex_params);

  /* This is the main loop that gets work, processes it, and sends replies. */
  while (TRUE) {
	get_work();		/* sets who and fs_call */

	if (who < 0)
	{
		fp= NULL;
		super_user= FALSE;
	}
	else
	{
		fp = &fproc[who];	/* pointer to proc table struct */
		if (fp->fp_pid == PID_FREE)
		{
			/* Race condition between MM, FS and a killed
			 * process.
			 */
#if PCH_DEBUG
 { printf("warning: got request %d from killed process %d\n", fs_call, who); }
#endif
			continue;	/* No reply */
		}
		/* Super user? */
		super_user = (fp->fp_effuid == SU_UID ? TRUE : FALSE);  
	}
	dont_reply = FALSE;	/* in other words, do reply is default */

	/* Call the internal function that does the work. */
	if (fs_call < 0 || fs_call >= NCALLS)
		error = no_sys();
	else
		error = (*scall_table[fs_call])();

	/* Copy the results back to the user and send reply. */
	if (!dont_reply) 
		reply(who, error);
	if (call_lock_revive)
		lock_revive();
	assert(bufs_in_use < 20);
	if (fs_debug)
	{
		assert(check_ihash());
	}
  }
}


/*===========================================================================*
 *				get_work				     *
 *===========================================================================*/
PRIVATE void get_work()
{  
  /* Normally wait for new input.  However, if 'reviving' is
   * nonzero, a suspended process must be awakened.
   */

  /* Normal case.  No one to revive. */
  if (receive(ANY, &m) != OK) panic("fs receive error", NO_NUM);

  who = m.m_source;
  fs_call = m.m_type;
}


/*===========================================================================*
 *				reply					     *
 *===========================================================================*/
PUBLIC void reply(whom, result)
int whom;			/* process to reply to */
int result;			/* result of the call (usually OK or error #) */
{
/* Send a reply to a user process. It may fail (if the process has just
 * been killed by a signal), so don't check the return code.  If the send
 * fails, just ignore it.
 */

  reply_type = result;
  send(whom, &m1);
}

/*===========================================================================*
 *				fs_init					     *
 *===========================================================================*/
PRIVATE void fs_init()
{
/* Initialize global variables, tables, etc. */

  register struct inode *root_inode;
  struct super_block *sp;
  int i, r;
  int cpu;
  struct mem_map dummy_map[NR_SEGS];

  r= sys_findproc(CLOCK_NAME, &clck_tasknr, 0);
  if (r != OK)
	panic("unable to find clock task:", r);

  /* First initialize the special devices */
  dev_init();
  lock_init();

  /* The following 2 initializations are needed to let dev_open succeed .*/
  fp = (struct fproc *) NULL;
  who = FS_PROC_NR;

  /* Initialize the super_block table. */
  for (sp = &super_block[0]; sp < &super_block[NR_SUPERS]; sp++)
  	sp->s_dev = NO_DEV;

  buf_pool();			/* initialize buffer pool */

  load_ram();			/* init RAM disk, load if it is root */
  root_inode= load_super();	/* load super block for root device */

  /* Wait for MM to sync up. */
  if (receive(MM_PROC_NR, &m1) != OK || send(MM_PROC_NR, &m1) != OK)
  	panic("FS can't sync up with MM", NO_NUM);

  if (m1.m1_i1 > revision)
  	revision = m1.m1_i1;	/* MM has a higher revision number */

  m1.m_type = SYS_REV;
  (void) sendrec(SYSTASK, &m1);
  if (m1.m_type > revision)
	revision = m1.m_type;	/* kernel has a higher revision number */

  /* Finish up the uname(2) info by setting the machine type and adding the
   * revision number.
   */
  if ((cpu = getprocessor()) >= 186) {
	strcpy(uts_val.machine, "i086");
	uts_val.machine[1] = '0' + cpu / 100;
  }
  strcat(uts_val.version, "r");
  strcat(uts_val.version, itoa(revision));
  
  /* Initialize the 'fproc' fields for the initially active processes. */
  for (i = 0; i < NR_PROCS; i++) {
	if (sys_getmap(i, dummy_map) != OK) continue;
	fp = &fproc[i];
	dup_inode(root_inode);
	fp->fp_rootdir = root_inode;
	dup_inode(root_inode);
	fp->fp_workdir = root_inode;
	fp->fp_realuid = (uid_t) SYS_UID;
	fp->fp_effuid = (uid_t) SYS_UID;
	fp->fp_realgid = (gid_t) SYS_GID;
	fp->fp_effgid = (gid_t) SYS_GID;
	fp->fp_ngroups = 0;
	fp->fp_umask = ~0;
	fp->fp_tty= 0;
	fp->fp_ttyfilp= NIL_FILP;
	fp->fp_ttylink= 0;
	fp->fp_sesldr= 0;	/* INIT (and MM) is not a session leader
				 * this makes opening ttys files in /etc/rc
				 * easier.
				 */
	/* Normal process or server? */
	if (i >= INIT_PROC_NR) {
		fp->fp_pid= (i - INIT_PROC_NR) + 1;
	} else {
		fp->fp_pid= PID_SERVER;
	}
  }

  /* Certain relations must hold for the file system to work at all. */
  if (SUPER_SIZE > BLOCK_SIZE) panic("SUPER_SIZE > BLOCK_SIZE", NO_NUM);
  if (BLOCK_SIZE % V2_INODE_SIZE != 0)	/* this checks V1_INODE_SIZE too */
	panic("BLOCK_SIZE % V2_INODE_SIZE != 0", NO_NUM);
  if (OPEN_MAX > 127) panic("OPEN_MAX > 127", NO_NUM);
  if (nr_bufs < 6) panic("nr_bufs < 6", NO_NUM);
  if (V1_INODE_SIZE != 32) panic("V1 inode size != 32", NO_NUM);
  if (V2_INODE_SIZE != 64) panic("V2 inode size != 64", NO_NUM);
  if (OPEN_MAX > 8 * sizeof(long)) panic("Too few bits in fp_cloexec", NO_NUM);

  /* release the extra reference to the root_inode. */
  put_inode(root_inode);
}


/*===========================================================================*
 *				load_ram				     *
 *===========================================================================*/
PRIVATE void load_ram()
{
/* If the root device is the RAM disk, copy the entire root image device
 * block-by-block to a RAM disk with the same size as the image.
 * Otherwise, just allocate a RAM disk with size given in the boot parameters.
 */

  dev_t image_dev;
  u32_t ram_size, fsmax;
  zone_t zones;
  struct super_block sb, *dsp;
  block_t b, image_size;
  int n;
  struct buf *bp;

  /* What are the root device, the RAM image device, and the RAM disk size? */
  root_dev = atoi(getenv("rootdev"));
  image_dev = atoi(getenv("ramimagedev"));
  ram_size = atoi(getenv("ramsize"));

  if (root_dev == DEV_RAM) {
	/* The root device is the RAM disk, so it must be filled from the
	 * image device.  The size of the RAM disk is determined as the
	 * maximum of the image file system size and the ramsize parameter.
	 */
	if (dev_opcl(1, image_dev, FS_PROC_NR, R_BIT) != OK)
		panic("Cannot open RAM image device", NO_NUM);

	sb.s_dev = image_dev;
	read_super(&sb);
	if (sb.s_version == 0) panic("Bad RAM image file system", NO_NUM);

	/* Image file system size. */
	image_size = (block_t) sb.s_zones << sb.s_log_zone_size;

	if (ram_size > image_size) {
		/* Stretch the RAM disk file system to the boot parameters size,
		 * but no further than the last zone bit map block allows.
		 */
		fsmax = (u32_t) sb.s_zmap_blocks * CHAR_BIT * BLOCK_SIZE;
		fsmax = (fsmax + (sb.s_firstdatazone-1)) << sb.s_log_zone_size;
		if (ram_size > fsmax) ram_size = fsmax;
	} else {
		ram_size = image_size;
	}
  }

  if (ram_size > MAX_RAM) panic("RAM disk is too big. # blocks > ", MAX_RAM);

  /* Tell RAM driver how big the RAM disk is. */
  if (dev_opcl(1, DEV_RAM, FS_PROC_NR, R_BIT|W_BIT) != OK)
	panic("Can't open ramdisk", NO_NUM);
  if (dev_ioctl(DEV_RAM, FS_PROC_NR, MIOCRAMSIZE, (void *)&ram_size,-1,0) != OK)
	panic("Can't set ramdisk size", NO_NUM);

  if (root_dev == DEV_RAM) {
	/* Fill the RAM disk from the image device. */

	printf("Loading %luK RAM disk.",
			((unsigned long) image_size * BLOCK_SIZE) / 1024);

	for (b = 0; b < image_size; b += n) {
		printf("\r\33[30CLoaded:%5luK ",
				((unsigned long) b * BLOCK_SIZE) / 1024);
		n = fill_ram(image_dev, b, image_size - b);
		if (n == 0) panic("I/O error", NO_NUM);
	}
	printf("\r\33[30CRAM disk loaded.\n\n");

	if (dev_opcl(0, image_dev, FS_PROC_NR, 0) != OK)
		panic("Can't close RAM image", NO_NUM);
	invalidate(image_dev);

	/* Resize the RAM disk file system. */
	bp = get_block(DEV_RAM, SUPER_BLOCK, BF_NORMAL);
	dsp = (struct super_block *) bp->b_data;
	zones = ram_size >> sb.s_log_zone_size;
	dsp->s_nzones = conv2(sb.s_native, (u16_t) zones);
	dsp->s_zones = conv4(sb.s_native, zones);
	bp->b_dirt = DIRTY;
	put_block(bp);
	sync_all(GRIMY);
  }
  if (dev_opcl(0, DEV_RAM, FS_PROC_NR, 0) != OK)
	panic("Can't close ram disk", NO_NUM);
}


/*===========================================================================*
 *				fill_ram				     *
 *===========================================================================*/
PRIVATE int fill_ram(dev, block, count)
Dev_t dev;
block_t block, count;
{
/* Copy up to 'count' blocks starting with block 'block' from device 'dev'
 * into the RAM disk.  Return the number of blocks loaded.
 */
  struct buf *bufq[NR_IOREQS];
  iovec_t iovec[NR_IOREQS];
  int b, ct, rct;

  ct = count > NR_IOREQS ? NR_IOREQS : (int) count;

  /* Set up a list of RAM disk blocks. */
  for (b = 0; b < ct; b++) {
	bufq[b] = get_block(DEV_RAM, block + b, BF_ALLOC);
	iovec[b].iov_addr = (vir_bytes) bufq[b]->b_data;
	iovec[b].iov_size = BLOCK_SIZE;
  }

  /* Fill the blocks from the image device. */
  (void) dev_rw(DEV_GATHER, dev, FS_PROC_NR, mul64u(block, BLOCK_SIZE),
						(char *) iovec, ct, 0, 0);
  /* See how many were actually read. */
  for (b = 0; b < ct && iovec[b].iov_size == 0; b++) {
	bufq[b]->b_valid = 1;
	bufq[b]->b_dirt = DIRTY;
  }
  rct = b;

  /* Release the blocks. */
  for (b = 0; b < ct; b++) put_block(bufq[b]);

  return(rct);
}


/*===========================================================================*
 *				load_super				     *
 *===========================================================================*/
PRIVATE struct inode *load_super()
{
  register struct super_block *sp;
  register struct inode *root_inode;
  int fs_err= EGENERIC;

  /* Open the root device. */
  if (dev_opcl(1, root_dev, FS_PROC_NR, R_BIT|W_BIT) != OK)
	panic("Cannot open root device", NO_NUM);

  /* Read in super_block for the root file system. */
  sp = &super_block[0];
  sp->s_dev = root_dev;
  read_super(sp);

  /* Check super_block for consistency. */
  if (sp->s_version == 0) panic("Invalid root file system.",NO_NUM);

  /* inode for root dir */
  root_inode = get_inode(root_dev, ROOT_INODE, &fs_err);
  if ((root_inode->i_mode & I_TYPE) != I_DIRECTORY)
	panic("Root inode is not a directory", NO_NUM);

  sp->s_rd_only = 1;	/* So it can be fsck'd first, and rw remounted later. */
  sp->s_nosuid = 0;
  sp->s_superblock = NULL;
  if (load_bit_maps(root_dev) != OK)
	panic("init: can't load root bit maps", NO_NUM);
  return root_inode;
}

/*
 * $PchId: main.c,v 1.8 1996/01/19 23:12:00 philip Exp $
 */
