/* This file handles advisory file locking as required by POSIX.
 *
 * The entry points into this file are
 *   lock_op:	perform locking operations for FCNTL system call
 *   lock_init:	initialize locking structures
 *   lock_unlock: release a lock
 *   lock_revive: revive processes when a lock is released
 */

#include "fs.h"
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <minix/com.h>
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "lock.h"
#include "param.h"
#include "assert.h"
INIT_ASSERT

FORWARD _PROTOTYPE( int lock_get, (struct filp *f, int type,
		unsigned long start, unsigned long end,
		struct flock *res_lock, struct fproc **block_fpp)	);
FORWARD _PROTOTYPE( int lock_set, (struct filp *f, int type,
				unsigned long start, unsigned long end,
				struct fproc **block_fpp)		);
FORWARD _PROTOTYPE( int lock_setw, (struct filp *f, int type,
				unsigned long start, unsigned long end)	);
FORWARD _PROTOTYPE( int freesp, (struct filp *f, unsigned long start,
				unsigned long end, int trunc)		);


/*===========================================================================*
 *				lock_op					     *
 *===========================================================================*/
PUBLIC int lock_op(f, req, arg)
struct filp *f;
int req;			/* F_GETLK, F_SETLK, F_SETLKW, or F_FREESP */
char *arg;
{
	struct flock flock, new_flock;
	mode_t rw_bits;
	int r, type, whence;
	off_t pos;
	unsigned long start, end;

	/* Check if f is a regular file */
	if (!S_ISREG(f->filp_ino->i_mode))
		return EINVAL;

	/* Get the lock structure */
	r= sys_copy(who, SEG_D, (phys_bytes) arg,
		FS_PROC_NR, SEG_D, (phys_bytes) &flock,
		(phys_bytes) sizeof(flock));
	if (r != OK)
		return EINVAL;

	/* Check l_type */
	type= req == F_FREESP ? F_WRLCK : flock.l_type;
	if (type != F_RDLCK && type != F_WRLCK && type != F_UNLCK)
		return EINVAL;
	if (req == F_GETLK && type == F_UNLCK)
		return EINVAL;

	/* Check l_type, file mode combinations */
	rw_bits= f->filp_mode;
	if ((type == F_RDLCK && !(rw_bits & R_BIT)) ||
		(type == F_WRLCK && !(rw_bits & W_BIT)))
	{
		return EBADF;
	}

	/* Check l_whence */
	whence= flock.l_whence;
	if (whence == SEEK_SET)
		pos= 0;
	else if (whence == SEEK_CUR)
		pos= cv64ul(f->filp_pos);
	else if (whence == SEEK_END)
		pos= f->filp_ino->i_size;
	else
		return EINVAL;

	/* Check l_start */
	start= pos + flock.l_start;
	if ((long)start < 0)
		return EINVAL;

	/* Check l_len */
	end= start + flock.l_len;
	if ((long)end < (long)start)
		return EINVAL;

	if (end == start)
		end= MAX_FILE_POS-1;
	else
		end--;

	switch(req)
	{
	case F_GETLK:
		r= lock_get(f, type, start, end, &new_flock, NULL);
		if (r == -1)
			return r;
		if (r == 0)
		{
			new_flock= flock;
			new_flock.l_type= F_UNLCK;
		}
		r= sys_copy(FS_PROC_NR, SEG_D, (phys_bytes) &new_flock,
			who, SEG_D, (phys_bytes) arg,
			(phys_bytes) sizeof(new_flock));
		if (r != OK)
			return EINVAL;
		return OK;
	case F_SETLK:
		r= lock_set(f, type, start, end, NULL);
		return r;
	case F_SETLKW:
		r= lock_setw(f, type, start, end);
		return r;
	case F_FREESP:
		r= freesp(f, start, end, flock.l_len == 0);
		return r;
	default:
		assert(0);
		/*NOTREACHED*/
	}
}


/*===========================================================================*
 *				lock_get				     *
 *===========================================================================*/
PRIVATE int lock_get(f, type, start, end, res_lock, block_fpp)
struct filp *f;
int type;
unsigned long start;
unsigned long end;
struct flock *res_lock;
struct fproc **block_fpp;
{
	struct file_lock *flp;

	if (type == F_UNLCK) return 0;

	for (flp= f->filp_ino->i_lock; flp; flp= flp->fl_next)
	{
		if (flp->fl_proc == fp)	
			continue;		/* One of my own locks */
		if (type == F_RDLCK && flp->fl_type == F_RDLCK)
			continue;		/* Compatible locks */
		if (flp->fl_start > end)
			continue;		/* No overlap */
		if (flp->fl_end < start)
			continue;		/* No overlap */
		break;				/* Got one */
	}
	if (flp == NULL)
		return 0;			/* No conflict */
	res_lock->l_type= flp->fl_type;
	res_lock->l_whence= SEEK_SET;
	res_lock->l_start= flp->fl_start;
	res_lock->l_len= (flp->fl_end == ULONG_MAX ? 0 :
		flp->fl_end-flp->fl_start+1);
	res_lock->l_pid= flp->fl_proc->fp_pid;
	if (block_fpp != NULL)
		*block_fpp= flp->fl_proc;
	return 1;
}


/*===========================================================================*
 *				lock_set				     *
 *===========================================================================*/
PRIVATE int lock_set(f, type, start, end, block_fpp)
struct filp *f;
int type;
unsigned long start;
unsigned long end;
struct fproc **block_fpp;
{
	struct flock new_flock;
	struct file_lock *flp, *curr_flp, *lock1, *lock2, *lock_list;
	int r;

	r= lock_get(f, type, start, end, &new_flock, block_fpp);
	if (r == -1)
		return r;
	if (r != 0)
		return EAGAIN;
	lock1= free_locks;
	if (lock1 == NULL)
		return ENOLCK;
	lock2= lock1->fl_next;
	if (lock2 == NULL)
		return ENOLCK;
	free_locks= lock2->fl_next;

	lock1->fl_type= lock2->fl_type= F_UNLCK;
	flp= f->filp_ino->i_lock;
	lock_list= NULL;
	while (flp)
	{
		curr_flp= flp;
		flp= flp->fl_next;

		if (curr_flp->fl_proc != fp || curr_flp->fl_end < start ||
			curr_flp->fl_start > end)
		{
			curr_flp->fl_next= lock_list;
			lock_list= curr_flp;
			continue;
		}

		/* Check if we need to release blocked processes */
		if (type == F_UNLCK || (type == F_RDLCK &&
			curr_flp->fl_type == F_WRLCK))
		{
			call_lock_revive= 1;
		}

		/* Does this lock extend past the end of the new lock? */
		if (curr_flp->fl_end > end)
		{
			assert(lock1 != NULL);
			*lock1= *curr_flp;
			lock1->fl_start= end+1;
			lock1->fl_next= lock_list;
			lock_list= lock1;
			lock1= lock2;
			lock2= NULL;
		}
		/* Does this lock start before the start of the new lock? */
		if (curr_flp->fl_start < start)
		{
			curr_flp->fl_end= start-1;
			curr_flp->fl_next= lock_list;
			lock_list= curr_flp;
			continue;
		}

		/* Free lock */
		curr_flp->fl_next= free_locks;
		free_locks= curr_flp;
	}

	/* Create the new lock */
	assert(lock1 != NULL);
	if (type == F_UNLCK)
	{
		lock1->fl_next= free_locks;
		free_locks= lock1;
		lock1= lock2;
		lock2= NULL;
	}
	else
	{
		lock1->fl_type= type;
		lock1->fl_start= start;
		lock1->fl_end= end;
		lock1->fl_proc= fp;
		lock1->fl_filp= f;
		lock1->fl_next= lock_list;
		lock_list= lock1;
		lock1= lock2;
		lock2= NULL;
	}

	f->filp_ino->i_lock= lock_list;
	if (lock1)
	{
		lock1->fl_next= free_locks;
		free_locks= lock1;
	}
 #if DEBUG & 256
  { struct file_lock *flp; where(); printf("locks: ");
  	for(flp= f->filp_ino->i_lock; flp; flp= flp->fl_next)
  		printf("(%d: %d) ", flp->fl_proc-fproc,
  		flp->fl_filp-filp_table);
  	printf("\n"); }
 { int i; struct od **odpp;
	for (i= 0, odpp= fp->fp_filp; i<OPEN_MAX; i++, odpp++)
	{
		if ((*odpp) != NULL && (*odpp)->od_filp == f)
			break;
	}
	assert(i != OPEN_MAX); }
#endif
	return OK;
}


/*===========================================================================*
 *				lock_setw				     *
 *===========================================================================*/
PRIVATE int lock_setw(f, type, start, end)
struct filp *f;
int type;
unsigned long start;
unsigned long end;
{
	int i, r;
	struct fproc *block_fp;

	r= lock_set(f, type, start, end, &block_fp);
	if (r != EAGAIN)
		return r;
	assert(fp->fp_event == EV_NONE);
	fp->fp_state.fp_lock.fp_filp= f;
	fp->fp_state.fp_lock.fp_type= type;
	fp->fp_state.fp_lock.fp_start= start;
	fp->fp_state.fp_lock.fp_end= end;
	fp->fp_state.fp_lock.fp_blkproc= block_fp;
	
	/* Try to detect deadlock */
	for (i= 0; i<2*NR_PROCS; i++)
	{
		if (block_fp == fp)
			return EDEADLK;
		if (block_fp->fp_event != EV_LOCK)
			break;
		block_fp= block_fp->fp_state.fp_lock.fp_blkproc;
	}
	if (i == 2*NR_PROCS)
		panic("lock_setw: cycle", NO_NUM);

	fp->fp_event= EV_LOCK;
	dont_reply= TRUE;
	return SUSPEND;
}


/*===========================================================================*
 *				lock_unlock				     *
 *===========================================================================*/
PUBLIC void lock_unlock(filp)
struct filp *filp;
{
	struct file_lock *flp, *curr_flp, *list;
	struct od **odpp;
	int i, cnt;

	/* Check if more open files refer to this filp, in that case we
	 * don't unlock.
	 */
	cnt= 0;
	for (i= 0, odpp= fp->fp_filp; i<OPEN_MAX; i++, odpp++)
	{
		if ((*odpp) != NULL && (*odpp)->od_filp == filp)
			cnt++;
	}
	assert(cnt >= 1);
	if (cnt > 1)
		return;

	flp= filp->filp_ino->i_lock;
	list= NULL;
	while(flp)
	{
		curr_flp= flp;
		flp= flp->fl_next;
		if (curr_flp->fl_proc != fp || curr_flp->fl_filp != filp)
		{
			curr_flp->fl_next= list;
			list= curr_flp;
		}
		else
		{
			call_lock_revive= 1;
			curr_flp->fl_next= free_locks;
			free_locks= curr_flp;
		}
	}
	filp->filp_ino->i_lock= list;

}


/*===========================================================================*
 *				freesp					     *
 *===========================================================================*/
PRIVATE int freesp(f, start, end, trunc)
struct filp *f;
unsigned long start;
unsigned long end;
int trunc;
{
	/* Drill a hole in a file. */
	struct inode *ip;

	ip= f->filp_ino;
	if (start < ip->i_size) freespace(ip, start, end + 1);
	if (trunc) ip->i_size= start;
	ip->i_update |= CTIME | MTIME;
	ip->i_dirt= DIRTY;
	return OK;
}


/*===========================================================================*
 *				lock_init				     *
 *===========================================================================*/
PUBLIC void lock_init()
{
	int i;

	free_locks= NULL;
	for (i= 0; i<NR_LOCKS; i++)
	{
		lock_table[i].fl_next= free_locks;
		free_locks= &lock_table[i];
	}
}


/*===========================================================================*
 *				lock_revive				     *
 *===========================================================================*/
PUBLIC void lock_revive()
{
	struct fproc *rfp;
	int i, r;

	call_lock_revive= 0;

	for (i= 0, rfp= fproc; i<NR_PROCS; i++, rfp++)
	{
		if (rfp->fp_event != EV_LOCK)
			continue;
		rfp->fp_event= EV_NONE;
		fp= rfp;
		who= i;
		r= lock_setw(rfp->fp_state.fp_lock.fp_filp,
			rfp->fp_state.fp_lock.fp_type,
			rfp->fp_state.fp_lock.fp_start,
			rfp->fp_state.fp_lock.fp_end);
		if (r != SUSPEND)
			reply(i, r);
		else
		{
			assert(rfp->fp_event == EV_LOCK);
		}
	}
}
