/* This file contains the procedures that manipulate file descriptors.
 *
 * The entry points into this file are
 *   get_fd:	look for free file descriptor and free filp slots
 *   get_filp:	look up the filp entry for a given file descriptor
 *   find_filp:	find a filp slot that points to a given inode
 */

#include "fs.h"
#include <sys/stat.h>
#include <minix/com.h>
#include <fcntl.h>
#include "assert.h"
INIT_ASSERT
#include "file.h"
#include "fproc.h"
#include "inode.h"
#include "pipe.h"


/*===========================================================================*
 *				new_fd					     *
 *===========================================================================*/
PUBLIC int new_fd(start, bits, k, fpt)
int start;			/* start of search (used for F_DUPFD) */
mode_t bits;			/* mode of the file to be created (RWX bits) */
int *k;				/* place to return file descriptor */
struct od **fpt;		/* place to return fd */
{
/* Look for a free file descriptor and a free filp slot.  Fill in the mode word
 * in the latter, but don't claim either one yet, since the open() or creat()
 * may yet fail.
 */

  register struct filp *f;
  register int i;
  struct od *odp;

  *k= -1;
  for (i= 0, odp= &od_table[0]; i<NR_ODS; i++, odp++)
  {
	if (!(odp->od_flags & ODF_INUSE))
		break;
  }
  if (i == NR_ODS) 
	return(ENFILE);
  odp->od_flags= 0;

  /* Search the fproc table for a free file descriptor. */
  for (i = start; i < OPEN_MAX; i++) {
	if (fp->fp_filp[i] == NIL_ODP) {
		break; /* A file descriptor has been located. */
	}
  }

  if (i == OPEN_MAX) return(EMFILE); /* file descriptor table full */
  *k = i;

  /* Now that a file descriptor has been found, look for a free filp slot. */
  for (i= 0, f = &filp_table[0]; i < NR_FILPS; i++, f++) {
	if (f->filp_count == 0)
		break;
  }

  if (i == NR_FILPS)
  {
	/* The filp table must be full.  Report that back. */
	return(ENFILE);
  }

  f->filp_mode = bits;
  f->filp_pos = cvu64(0);
  f->filp_flags = 0;
  f->filp_int_flags= 0;
  odp->od_filp= f;
  *fpt = odp;

  return(OK);
}


/*===========================================================================*
 *				get_fd					     *
 *===========================================================================*/
PUBLIC struct od *get_fd(fild, err_p)
int fild;			/* file descriptor */
int *err_p;			/* pointer to error variable */
{
/* See if 'fild' refers to a valid file descr.  If so, return its filp ptr. */

  struct od *odp;

  if (fild < 0 || fild >= OPEN_MAX )
  	odp= NIL_ODP;
  else
  	odp= fp->fp_filp[fild];	/* may also be NIL_ODP */
  if (odp == NIL_ODP)
  	*err_p= EBADF;
  return odp;
}


/*===========================================================================*
 *				get_filp				     *
 *===========================================================================*/
PUBLIC struct filp *get_filp(fild, err_p)
int fild;			/* file descriptor */
int *err_p;			/* pointer to error variable */
{
/* See if 'fild' refers to a valid file descr.  If so, return its filp ptr. */
  struct od *odp;
  struct filp *filp;

  if (fild < 0 || fild >= OPEN_MAX )
  	filp= NIL_FILP;
  else
  {
	odp= fp->fp_filp[fild];
	if (odp == NIL_ODP) 
		filp= NIL_FILP;
	else
	{
		assert(fp->fp_filp[fild]->od_filp);
		filp= fp->fp_filp[fild]->od_filp;	/* may also be
							 * NIL_FILP
							 */
	}
  }
  if (filp == NIL_FILP)
  	*err_p= EBADF;
  return filp;
}


/*===========================================================================*
 *				find_filp				     *
 *===========================================================================*/
PUBLIC struct filp *find_filp(rip, bits)
register struct inode *rip;	/* inode referred to by the filp to be found */
Mode_t bits;			/* mode of the filp to be found (RWX bits) */
{
/* Find a filp slot that refers to the inode 'rip' in a way as described
 * by the mode bit 'bits'. Used for determining whether somebody is still
 * interested in either end of a pipe.  Also used when opening a FIFO to
 * find partners to share a filp field with (to shared the file position).
 * Like 'get_fd' it performs its job by linear search through the filp table.
 */

  register struct filp *f;

  for (f = &filp_table[0]; f < &filp_table[NR_FILPS]; f++) {
	if (f->filp_count != 0 && f->filp_ino == rip && (f->filp_mode & bits)){
		return(f);
	}
  }

  /* If control passes here, the filp wasn't there.  Report that back. */
  return(NIL_FILP);
}

#if PCH_DEBUG
/*===========================================================================*
 *				check_filedes				     *
 *===========================================================================*/
PUBLIC int check_filedes()
{
	int i, j, ttylink;
	struct filp *f;
	struct fproc *fp;
	struct inode *rip, *rip1;
	int error;
	dev_t dev, major;
	struct od *odp;

	error= 0;

	for (i= 0, f= filp_table; i<NR_FILPS; i++, f++)
	{
		f->filp_checkcnt= 0;
	}
	for (i= 0, odp= od_table; i<NR_ODS; i++, odp++)
	{
		odp->od_inuse= 0;
	}
	for (i= 0, rip= inode_table; i<NR_INODES; i++, rip++)
	{
		rip->i_checkcnt= 0;
	}
	for (i= 0, rip= inode_table; i<NR_INODES; i++, rip++)
	{
		rip1= rip->i_mounted_by;
		if (rip1 != NIL_INODE)
			rip1->i_checkcnt++;
		rip1= rip->i_mounted_on;
		if (rip1 != NIL_INODE)
			rip1->i_checkcnt++;
	}
	for (i= 0, fp= fproc; i < NR_PROCS; i++, fp++)
	{
		rip1= fp->fp_workdir;
		if (rip1 != NIL_INODE) rip1->i_checkcnt++;
		rip1= fp->fp_rootdir;
		if (rip1 != NIL_INODE) rip1->i_checkcnt++;

		ttylink= 0;
		for (j= 0; j < OPEN_MAX; j++)
		{
			if (fp->fp_filp[j] == NULL)
				continue;
			fp->fp_filp[j]->od_inuse++;
			f= fp->fp_filp[j]->od_filp;
			assert(f);
			printf("p%d f%d:%d ", i, j, f-filp_table);
			f->filp_checkcnt++;
			if (f->filp_int_flags & FIF_CLOSED)
				continue;
			rip= f->filp_ino;
			if (f->filp_checkcnt == 1)
				rip->i_checkcnt++;
			if (!S_ISCHR(rip->i_mode))
				continue;
			dev = (dev_t) rip->i_zone[0];
			major = (dev >> MAJOR) & BYTE;	/* major device nr */
			if (major != CTTY_MAJOR)
				continue;
			ttylink++;
		}
		if (fp->fp_ttylink != ttylink)
		{
			printf("proc %d: fp_ttylink= %d, ttylink= %d\n",
				i, fp->fp_ttylink, ttylink);
			error++;
		}
		if (fp->fp_ttyfilp)
		{
			fp->fp_ttyfilp->filp_checkcnt++;
			if (fp->fp_ttylink == 0)
			{
				printf(
				"proc %d has fp_ttyfilp but no fp_ttylink\n", 
					i);
				error++;
			}
		}
		else if (fp->fp_ttylink)
		{
			printf("proc %d has fp_ttylink but no fp_ttyfilp\n", i);
			error++;
		}
	}
	for (i= 0, f= filp_table; i<NR_FILPS; i++, f++)
	{
		if (f->filp_checkcnt != f->filp_count)
		{
			printf("filp %d: filp_checkcnt= %d, filp_count= %d\n",
				i, f->filp_checkcnt, f->filp_count);
			error++;
		}
	}
	for (i= 0, odp= od_table; i<NR_ODS; i++, odp++)
	{
		if (((odp->od_flags & ODF_INUSE) && odp->od_inuse != 1) ||
			(!(odp->od_flags & ODF_INUSE) && odp->od_inuse != 0))
		{
			printf("od %d: od_flags= 0x%x, od_inuse= %d\n",
				i, odp->od_flags, odp->od_inuse);
			error++;
		}
	}
	for (i= 0, rip= inode_table; i<NR_INODES; i++, rip++)
	{
		if (rip->i_checkcnt != rip->i_count)
		{
			printf("rip %d: i_checkcnt= %d, i_count= %d\n",
				i, rip->i_checkcnt, rip->i_count);
			printf("\ti_dev= 0x%x, i_num= %d\n", rip->i_dev,
				rip->i_num);
			error++;
		}
	}
	return error;
}
#endif /* PCH_DEBUG */

/*===========================================================================*
 *				fd_revive				     *
 *===========================================================================*/
PUBLIC void fd_revive(proc_nr, fd_nr, operation, result)
int proc_nr;			/* process to revive */
int fd_nr;			/* which fd */
int operation;			/* which operation */
int result;			/* the result of the operation */
{
/* Revive a process blocked on a filedescriptor.
 */

  register struct fproc *rfp;
  int rfd, ip, compl, r, event;
  struct inode *rip;
  struct od *odp;
  struct fwait fw;

  if (proc_nr < 0 || proc_nr >= NR_PROCS) 
  {
	panic("revive err", proc_nr);
  }
  rfp = &fproc[proc_nr];

  assert(fd_nr >= 0 && fd_nr < OPEN_MAX);
  odp= rfp->fp_filp[fd_nr];
  assert(odp);

  if (operation < 0 || operation >= ASIO_NR)
  {
	panic("illegal operation", operation);
  }
  ip= asio_ip[operation];
  compl= asio_compl[operation];
  event= asio_event[operation];

  compare(odp->od_flags, &, ip);
  assert(!(odp->od_flags & compl));

  odp->od_flags |= compl;
  odp->od_result[operation]= result;

  rip= odp->od_filp->filp_ino;

  /* We also update times for character special devices. */
  if (result >= 0)
  {
  	if (operation == ASIO_READ)
  	{
  		rip->i_update |= ATIME;
  		rip->i_dirt |= GRIMY;
  	}
  	else if (operation == ASIO_WRITE)
  	{
  		rip->i_update |= CTIME | MTIME;
  		rip->i_dirt |= GRIMY;
  	}
  }

  switch(rfp->fp_event)
  {
  case EV_NONE:
  case EV_LOCK:
  case EV_POPEN:
  	return;
  case EV_FWAIT:
	/* Process was suspend on fwait. */
	if (!ASIO_FD_ISSET(fd_nr, operation, &rfp->fp_state.fp_fwait.fp_fds))
		return;		/* Not on this filedescriptor/operation. */

	/* We reply, no matter what. */
	rfp->fp_event = EV_NONE;

	/* Let's fill in the fwait struct. */
	fw.fw_flags= 0;
	fw.fw_maxfd= ASIO_FD_SETSIZE;

	if (result >= 0)
		fw.fw_result= result;
	else
	{
		fw.fw_result= -1;
		fw.fw_errno= -result;
	}
	fw.fw_fd= fd_nr;
	fw.fw_operation= operation;

	/* And copy the thing to the user. */
	r= sys_copy(FS_PROC_NR, SEG_D, (phys_bytes) &fw,
		proc_nr, SEG_D, (phys_bytes) rfp->fp_state.fp_fwait.fp_buffer,
		(phys_bytes) sizeof(fw));

	if (r != OK)
	{
		assert(r == EFAULT);
where(); printf("replying EFAULT\n");
		reply(proc_nr, EFAULT);
		return;
	}

	/* We can clear the in progress and the completed bits. */
	odp->od_flags &= ~(ip|compl);
	reply(proc_nr, OK);
	return;
  case EV_READ:
  case EV_WRITE:
  case EV_IOCTL:
	rfd= rfp->fp_state.fp_rwio.fp_fd;
	assert(rfd >= 0 && rfd < OPEN_MAX);
	if (rfd != fd_nr) 
		return;
	if (event != rfp->fp_event) 
		return;

	rfp->fp_event = EV_NONE;
	odp->od_flags &= ~(ip|compl);

	reply(proc_nr, result);	/* unblock the process */
	return;
  default:
  	panic("fd_revive: unknown event", rfp->fp_event);
  }

}


/*===========================================================================*
 *				fd_suspend				     *
 *===========================================================================*/
PUBLIC void fd_suspend(event, fd_nr)
int event;			/* who is proc waiting for? (PIPE = pipe) */
int fd_nr;			/* which fd */
{
/* Take measures to suspend the processing of the present system call.
 */

  assert(fp->fp_event == EV_NONE);
  switch(event)
  {
  case EV_NONE:
  case EV_LOCK:
  case EV_FWAIT:
  	panic("fd_suspend: don't how to suspend on event", event);
  case EV_POPEN:
	fp->fp_state.fp_popen.fp_fd= fd_nr;
	break;
  case EV_READ:
  case EV_WRITE:
  case EV_IOCTL:
	  fp->fp_state.fp_rwio.fp_fd = fd_nr;
#if PCH_DEBUG
  { int operation; 
    if (event == EV_READ) operation= ASIO_READ;
    else if (event == EV_WRITE) operation= ASIO_WRITE;
    else if (event == EV_IOCTL) operation= ASIO_IOCTL;
    else 
    	assert(0);
    assert(fp->fp_filp[fd_nr]->od_flags & asio_ip[operation]);
  }
#endif
	break;
  default:
  	panic("fd_suspend: illegal event", event);
  }
  fp->fp_event = event;
  dont_reply = TRUE;		/* do not send caller a reply message now */
}


/*===========================================================================*
 *				fd_cancel				     *
 *===========================================================================*/
PUBLIC int fd_cancel(task, fd_nr, operation)
int task;
int fd_nr;
int operation;
{
  struct fproc *rfp;
  struct od *odp;
  struct filp *f;
  int ip, compl, cancel;
  int status;
  dev_t dev;
  struct inode *rip;
	
  if (task < 0 || task >= NR_PROCS) panic("cancel err", task);
  rfp = &fproc[task];

  if (fd_nr < 0 || fd_nr >= OPEN_MAX)
  	return EBADF;
  odp= rfp->fp_filp[fd_nr];
  if (odp == NIL_ODP) 
  	return EBADF;

  if (operation < 0 || operation >= ASIO_NR)
	return EINVAL;
  ip= asio_ip[operation];
  compl= asio_compl[operation];
  cancel= asio_cancel[operation];
  if (!(odp->od_flags & ip) || (odp->od_flags & compl))
  	return EINVAL;
  
  f= odp->od_filp;
  assert(f);

  if (S_ISFIFO(f->filp_ino->i_mode))
  	status= pipe_cancel(f->filp_ino, operation, task, fd_nr);
  else
  {
	dev = (dev_t) f->filp_ino->i_zone[0];	/* device hung on */
	status= dev_cancel(cancel, dev, task, fd_nr);
  }
  
  /* We also update times in inodes even if they are interrupted. */
  if (status >= 0)
  {
  	rip= f->filp_ino;
  	if (operation == ASIO_READ)
  	{
  		rip->i_update |= ATIME;
  		rip->i_dirt |= GRIMY;
  	}
  	else if (operation == ASIO_WRITE)
  	{
  		rip->i_update |= CTIME | MTIME;
  		rip->i_dirt |= GRIMY;
  	}
  }
  odp->od_result[operation]= status;
	
  odp->od_flags |= compl;

  return(OK);
}


/*===========================================================================*
 *				do_nrevive				     *
 *===========================================================================*/
PUBLIC int do_nrevive()
{
  int operation;

  assert(fs_call == DEVICE_REPLY);

#if !ALLOW_USER_SEND
  if (who >= 0 && fp->fp_pid != PID_SERVER) return(EPERM);
#endif

  switch(m.REP_OPERATION)
  {
  case DEV_READ: operation= ASIO_READ; break;
  case DEV_WRITE: operation= ASIO_WRITE; break;
  case DEV_IOCTL3: operation= ASIO_IOCTL; break;
  default: panic("don't know how to revive call", m.REP_OPERATION);
  }
  fd_revive(m.REP_PROC_NR, m.REP_REF, operation, m.REP_STATUS);
  dont_reply = TRUE;		/* don't reply to the TTY task */
  return(OK);
}


/*===========================================================================*
 *				cancel_fdops				     *
 *===========================================================================*/
PUBLIC void cancel_fdops(fp, fd_nr)
struct fproc *fp;
int fd_nr;
{
	int proc_nr, r;
	struct od *odp;

	proc_nr= fp-fproc;
	odp = fp->fp_filp[fd_nr];
	assert(odp != NULL);

	if ((odp->od_flags & (ODF_RD_IP|ODF_RD_COMPL)) == ODF_RD_IP)
	{
		r= fd_cancel(proc_nr, fd_nr, ASIO_READ);
		assert(r == OK);
		r= odp->od_result[ASIO_READ];
		assert((odp->od_flags & (ODF_RD_IP|ODF_RD_COMPL)) == 
						(ODF_RD_IP|ODF_RD_COMPL));
		odp->od_flags &= ~ODF_RD_COMPL;
		fd_revive(proc_nr, fd_nr, ASIO_READ, r);
	}
	if ((odp->od_flags & (ODF_WR_IP|ODF_WR_COMPL)) == ODF_WR_IP)
	{
		r= fd_cancel(proc_nr, fd_nr, ASIO_WRITE);
		assert(r == OK);
		r= odp->od_result[ASIO_WRITE];
		assert((odp->od_flags & (ODF_WR_IP|ODF_WR_COMPL)) == 
						(ODF_WR_IP|ODF_WR_COMPL));
		odp->od_flags &= ~ODF_WR_COMPL;
		fd_revive(proc_nr, fd_nr, ASIO_WRITE, r);
	}
	if ((odp->od_flags & (ODF_IOC_IP|ODF_IOC_COMPL)) == ODF_IOC_IP)
	{
		r= fd_cancel(proc_nr, fd_nr, ASIO_IOCTL);
		assert(r == OK);
		r= odp->od_result[ASIO_IOCTL];
		assert((odp->od_flags & (ODF_IOC_IP|ODF_IOC_COMPL)) == 
						(ODF_IOC_IP|ODF_IOC_COMPL));
		odp->od_flags &= ~ODF_IOC_COMPL;
		fd_revive(proc_nr, fd_nr, ASIO_IOCTL, r);
	}
}

/*
 * $PchId: filedes.c,v 1.6 1996/02/29 23:09:24 philip Exp $
 */
