/*
tcpstat.c

Created:	June 1995 by Philip Homburg <philip@cs.vu.nl>
*/

#define _MINIX_SOURCE
#define _POSIX_C_SOURCE 2

#include <inet/inet.h>
#undef printf

#include <fcntl.h>
#include <stdio.h>
#include <time.h>
#include <unistd.h>
#include <sys/svrctl.h>
#include <net/netlib.h>
#include <net/gen/inet.h>
#include <net/gen/netdb.h>
#include <net/gen/socket.h>
#include <minix/queryparam.h>

#include <inet/generic/buf.h>
#include <inet/generic/clock.h>
#include <inet/generic/event.h>
#include <inet/generic/type.h>
#include <inet/generic/tcp_int.h>

char *prog_name;
tcp_conn_t tcp_conn_table[TCP_CONN_NR];
char values[2 * sizeof(tcp_conn_table) + 1];
int inclListen, numerical;

void print_conn(int i, clock_t now);
void usage(void);

int main(int argc, char*argv[])
{
	char *tcp_device;
	int fd, i;
	struct svrqueryparam qpar;
	char *pval;
	struct timeval uptime;
	clock_t now;
	int fl;
	int a_flag, n_flag;

	(prog_name=strrchr(argv[0], '/')) ? prog_name++ : (prog_name=argv[0]);

	a_flag= 0;
	n_flag= 0;
	while ((fl= getopt(argc, argv, "?an")) != -1)
	{
		switch(fl)
		{
		case '?':
			usage();
		case 'a':
			a_flag= 1;
			break;
		case 'n':
			n_flag= 1;
			break;
		default:
			fprintf(stderr, "%s: getopt failed: '%c'\n", 
				prog_name, fl);
			exit(1);
		}
	}
	inclListen= !!a_flag;
	numerical= !!n_flag;

	tcp_device= TCP_DEVICE;
	if ((fd= open(TCP_DEVICE, O_RDWR)) == -1)
	{
		fprintf(stderr, "%s: unable to open '%s': %s\n", prog_name,
			tcp_device, strerror(errno));
		exit(1);
	}

	qpar.param = "tcp_conn_table";
	qpar.psize = strlen(qpar.param);
	qpar.value = values;
	qpar.vsize = sizeof(values);
	if (ioctl(fd, NWIOQUERYPARAM, &qpar) == -1)
	{
		fprintf(stderr, "%s: queryparam failed: %s\n", prog_name,
			strerror(errno));
		exit(1);
	}
	pval= values;
	if (paramvalue(&pval, tcp_conn_table, sizeof(tcp_conn_table)) !=
		sizeof(tcp_conn_table))
	{
		fprintf(stderr,
			"%s: unable to decode the results from queryparam\n");
		exit(1);
	}

	/* Get the uptime in clock ticks. */
	if (sysutime(UTIME_UPTIME, &uptime) == -1)
	{
		fprintf(stderr, "%s: sysutime failed: %s\n", prog_name,
			strerror(errno));
		exit(1);
	}
	now= uptime.tv_sec * HZ + (uptime.tv_usec*HZ/1000000);

	for (i= 0; i<TCP_CONN_NR; i++)
		print_conn(i, now);
	exit(0);
}

void print_conn(int i, clock_t now)
{
	tcp_conn_t *tcp_conn;
	char *locaddr_str, *remaddr_str;
	struct hostent *hostent;
	struct servent *servent;

	tcp_conn= &tcp_conn_table[i];
	if (!(tcp_conn->tc_flags & TCF_INUSE))
		return;
	if (tcp_conn->tc_state == TCS_LISTEN && !inclListen)
		return;
	if (tcp_conn->tc_state == TCS_CLOSED && tcp_conn->tc_fd == NULL &&
		tcp_conn->tc_senddis < now)
	{
		return;
	}
	
	printf("%3d", i);

	if (!numerical &&
		(hostent= gethostbyaddr((char *)&tcp_conn->tc_locaddr,
		sizeof(ipaddr_t), AF_INET)) != NULL)
	{
		locaddr_str= hostent->h_name;
	}
	else
		locaddr_str= inet_ntoa(tcp_conn->tc_locaddr);
	printf(" %s.", locaddr_str);

	if (tcp_conn->tc_locport == 0)
		printf("*");
	else if ((servent= getservbyport(tcp_conn->tc_locport, "tcp")) !=
		NULL)
	{
		printf("%s", servent->s_name);
	}
	else
		printf("%u", ntohs(tcp_conn->tc_locport));

	printf(" -> ");

	if (tcp_conn->tc_remaddr == 0)
		remaddr_str= "*";
	else if (!numerical &&
		(hostent= gethostbyaddr((char *)&tcp_conn->tc_remaddr,
		sizeof(ipaddr_t), AF_INET)) != NULL)
	{
		remaddr_str= hostent->h_name;
	}
	else
		remaddr_str= inet_ntoa(tcp_conn->tc_remaddr);
	printf("%s.", remaddr_str);

	if (tcp_conn->tc_remport == 0)
		printf("*");
	else if ((servent= getservbyport(tcp_conn->tc_remport, "tcp")) !=
		NULL)
	{
		printf("%s", servent->s_name);
	}
	else
		printf("%u", ntohs(tcp_conn->tc_remport));

	printf(" ");
	switch(tcp_conn->tc_state)
	{
	case TCS_CLOSED:	printf("CLOSED");
				if (tcp_conn->tc_senddis >= now)
					printf("(time wait)");
				break;
	case TCS_LISTEN:	printf("LISTEN"); break;
	case TCS_SYN_RECEIVED:	printf("SYN_RECEIVED"); break;
	case TCS_SYN_SENT:	printf("SYN_SENT"); break;
	case TCS_ESTABLISHED:	printf("ESTABLISHED"); break;
	case TCS_CLOSING:	printf("CLOSING"); break;
	default:		printf("state(%d)", tcp_conn->tc_state);
				break;
	}

	if (tcp_conn->tc_flags & TCF_FIN_RECV)
		printf(" F<");
	if (tcp_conn->tc_flags & TCF_FIN_SENT)
	{
		printf(" F>");
		if (tcp_conn->tc_SND_UNA == tcp_conn->tc_SND_NXT)
			printf("+");
	}
	if (tcp_conn->tc_state != TCS_CLOSED &&
		tcp_conn->tc_state != TCS_LISTEN)
	{
		printf("\n\t");
		printf("RQ: %lu, SQ: %lu, SWnd: %lu",
			tcp_conn->tc_RCV_NXT - tcp_conn->tc_RCV_LO,
			tcp_conn->tc_SND_NXT - tcp_conn->tc_SND_UNA,
			tcp_conn->tc_snd_cwnd - tcp_conn->tc_SND_UNA);
	}

	printf("\n");
}

void usage(void)
{
	fprintf(stderr, "Usage: %s [-a] [-n]\n", prog_name);
	exit(1);
}

/*
 * $PchId: tcpstat.c,v 1.1 1995/11/27 22:03:53 philip Exp $
 */
