/*
keymap.c
*/

#define _MINIX_SOURCE
#define _POSIX_C_SOURCE 2

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/ioctl.h>

#include <time.h>		/* Grrr, needed for <sys/kbdio.h> */
#include <sys/kbdio.h>
#include <minix/keymap.h>

#define STRTAB_SIZE	(KEYMAP_STRINGS*KM_STRLEN)

struct keymap km[KEYMAP_ENTRIES];
u8_t km_used[KEYMAP_ENTRIES];
u8_t str_table[STRTAB_SIZE];
int str_index= 0;

unsigned char line[1024];	/* should be enough */
unsigned char *lineptr;
int lineno;
char *prog_name;
int stralign;
int error;

struct
{
	char *name;
	u16_t value;
} flagsmap[]=
{
	{ "caps",	KM_CAPSLOCK, },
	{ "num",	KM_NUMLOCK, },
	{ NULL, },
}, specialmap[]=
{
	{ "CapsLock",	KM_CAPSLOCK, },
	{ "NumLock",	KM_NUMLOCK, },
	{ "Shift-L",	KM_SHIFT_L, },
	{ "Shift-R",	KM_SHIFT_R, },
	{ "Ctrl-L",	KM_CTRL_L, },
	{ "Ctrl-R",	KM_CTRL_R, },
	{ "Alt-L",	KM_ALT_L, },
	{ "Alt-R",	KM_ALT_R, },
	{ "Alt-GR",	KM_ALT_GR, },
	{ "SysReq",	KM_SYSREQ, },
	{ "ScrLock",	KM_SCRLOCK, },
	{ NULL, },
};

static void read_keymap(FILE *file);
static void set_default(struct keymap *row, int col);
static unsigned char *get_token(FILE *file, unsigned char *token);
static unsigned char *maybe_token(FILE *file, unsigned char *token);
static char *getname(unsigned char *token, unsigned char **next);
static void getstr(unsigned char *token, unsigned char **next);
static int getflag(unsigned char *token, unsigned char **next);
static int getvalue(unsigned char *token, unsigned char **next);
static void read_oldmap(FILE *file);
static long read_value(FILE *file);
static u16_t cnv_value(u16_t value, u16_t *res, struct keymap *kme);
static int strtab(unsigned char *str);
static int strtab_label(char *str);
static void print_header(void);
static void print_hdrline(int i, struct keymap *kmp);
static void print_hdrstr(unsigned char *str, int maxlen, int width);
static void print_flags(u16_t flags, int width);
static void print_hdrvalue(u16_t v, int width);
static void print_hdrspecial(u16_t v, int width);
static void print_map(void);
static void print_mapline(int i, struct keymap *kmp);
static void print_mapstr(unsigned char *str);
static void print_mapvalue(u16_t v);
static void print_mapspecial(u16_t v);
static void load_map(void);
static void dprintf(char *fmt, ...);
static void fatal(char *fmt, ...);
static void usage(void);

u8_t set1tokeyno[]=
{
/* 00 */      0,    110,      2,      3,      4,      5,      6,      7,
/* 08 */      8,      9,     10,     11,     12,     13,     15,     16,
/* 10 */     17,     18,     19,     20,     21,     22,     23,     24,
/* 18 */     25,     26,     27,     28,     43,     58,     31,     32,
/* 20 */     33,     34,     35,     36,     37,     38,     39,     40,
/* 28 */     41,      1,     44,     29,     46,     47,     48,     49,
/* 30 */     50,     51,     52,     53,     54,     55,     57,    100,
/* 38 */     60,     61,     30,    112,    113,    114,    115,    116,
/* 40 */    117,    118,    119,    120,    121,     90,    125,     91,
/* 48 */     96,    101,    105,     92,     97,    102,    106,     93,
/* 50 */     98,    103,     99,    104,    108,      0,     45,    122,
/* 58 */    123,      0,      0,      0,      0,      0,      0,      0,
/* 60 */    255,      0,      0,      0,      0,      0,      0,      0,
};

int
main(int argc, char *argv[])
{
	int c, oldmap, genmap, genhdr, loadmap;
	int b_flag, l_flag, m_flag;
	char *map;
	FILE *map_fp;

	(prog_name=strrchr(argv[0],'/')) ? prog_name++ : (prog_name=argv[0]);

	b_flag= 0;
	l_flag= 0;
	m_flag= 0;
	while (c= getopt(argc, argv, "?blm"), c != -1)
	{
		switch(c)
		{
		case '?':
			usage();
			break;
		case 'b':
			b_flag= 1;
			break;
		case 'l':
			if (m_flag)
				usage();
			l_flag= 1;
			break;
		case 'm':
			if (l_flag)
				usage();
			m_flag= 1;
			break;
		default:
			fprintf(stderr, "%s: getopt returned '%c'\n",
				prog_name, c);
			exit(1);
		}
	}

	if (optind < argc)
	{
		map= argv[optind++];
		map_fp= fopen(map, "r");
		if (map_fp == NULL)
			fatal("can't open %s: %s\n", map, strerror(errno));
	}
	else
	{
		map= "stdin";
		map_fp= stdin;
	}

	if (optind != argc)
		usage();

	oldmap= b_flag;
	loadmap= l_flag;
	genmap= m_flag;
	genhdr= !genmap && !loadmap;

	if (genhdr)
		stralign= 1;

	if (b_flag)
		read_oldmap(map_fp);
	else
		read_keymap(map_fp);

	if (genmap)
		print_map();
	else if (loadmap)
		load_map();
	else
		print_header();

	if (error)
		exit(1);
	exit(0);
}

static void
read_keymap(FILE *file)
{
	unsigned char *t, *next;
	long keynum;
	struct keymap row;
	int col;
	u16_t v;

	t= NULL;

	for(;;)
	{
		memset(&row, '\0', sizeof(row));

		t= maybe_token(file, t);
		if (t == NULL)
			return;

		/* Get the key number */
		keynum= strtol((char *)t, (char **)&next, 0);
		if (next == t)
			fatal("unable to parse '%s', expected number", t);
		t= next;

		/* Check the number */
		if (keynum < 0 || keynum >= KEYMAP_ENTRIES)
		{
			fatal("key number (%ld) out of range [0..%d>\n",
				keynum, KEYMAP_ENTRIES);
		}
		dprintf("keynum= %d\n", keynum);

		/* Expect an '=' */
		t= get_token(file, t);
		if (*t != '=')
			fatal("expected '='");
		t++;

		/* Get the key name */
		t= get_token(file, t);
		row.l= getname(t, &next);
		t= next;

		if (row.l[0] == '\0')
		{
			free(row.l);
			row.l= NULL;
		}
		dprintf("row.l= '%s'\n", row.l);

		/* The key name may be followed by some flags */
		row.fl= 0;
		for(;;)
		{
			if (*t != ',' && *t != '\0')
				fatal("',' or end of token expected");

			t= get_token(file, t);

			/* See if we got a ',' */
			if (*t != ',')
				break;	/* end of flags */
			t++;

			t= get_token(file, t);

			/* Get the flag */
			row.fl |= getflag(t, &next);
			t= next;
		}
		dprintf("flags= 0x%x\n", row.fl);

		/* Read mappings until we hit a ';' */
		col= 0;
		for (;;)
		{
			t= get_token(file, t);
			if (*t == ';')
			{
				/* done */
				t++;
				break;
			}

			t= get_token(file, t);
			v= getvalue(t, &next);
			t= next;

			dprintf("col %d: %d\n", col, v);
			if (v & KM_SPECIAL)
			{
				/* Specials are only allowed in column 1,
				 * and there has to be only one column.
				 */
				if (col != 0)
				{
					fatal(
			"special is only allowed in the first column");
				}
				row.fl |= KM_SPECIAL;
				v &= ~KM_SPECIAL;
			}
			if (col != 0 && (row.fl & KM_SPECIAL))
			{
				fatal(
				"special keys should have only one column");
			}

			switch(col)
			{
			case 0: row.n= v; break;
			case 1: row.s= v; break;
			case 2: row.c= v; break;
			case 3: row.a= v; break;
			case 4: row.ag= v; break;
			case 5: row.sa= v; break;
			case 6: row.sc= v; break;
			case 7: row.ca= v; break;
			default:
				fatal("too many columns");
			}
			col++;
		}
		if (!(row.fl & KM_SPECIAL) && row.n != 0)
		{
			for (col= 1; col <= 7; col++)
				set_default(&row, col);
		}
		if (row.fl & KM_CAPSLOCK)
		{
			row.fl &= ~KM_CAPSLOCK;
			row.n |= KM_CAPSLOCK;
			row.s |= KM_CAPSLOCK;
		}
		if (row.fl & KM_NUMLOCK)
		{
			row.fl &= ~KM_NUMLOCK;
			row.n |= KM_NUMLOCK;
			row.s |= KM_NUMLOCK;
		}
		if (km_used[keynum])
		{
			fprintf(stderr, "redefinition of key %d\n",
				keynum);
		}
		km_used[keynum]= 1;
		km[keynum]= row;
	}
}

static void set_default(struct keymap *row, int col)
{
	/* Fill in undefined columns using rules that would
	 * make sense if there weren't so many.
	 */
	switch(col)
	{
	case 1:
		if (row->s != 0) break;
		if ('a' <= row->n && row->n <= 'z')
			row->s= (row->n - 'a') + 'A';
		else
			row->s= row->n;
		break;
	case 2:
		if (row->c != 0) break;
		if (row->n == ' ' || row->n == '@')
			row->c= 256;
		else
		if (row->n < ' ' || row->n == 0x7F)
			row->c= row->n;
		else
		if (row->n == '?')
			row->c= 0x7F;
		else
		if ('A' <= row->n && row->n <= '_')
			row->c= row->n & 0x1F;
		else
		if (row->s == ' ' || row->s == '@')
			row->c= 256;
		else
		if (row->s < ' ' || row->s == 0x7F)
			row->c= row->s;
		else
		if (row->s == '?')
			row->c= 0x7F;
		else
		if ('A' <= row->s && row->s <= '_')
			row->c= row->s & 0x1F;
		break;
	case 3:
		if (row->a != 0) break;
		row->a= row->n;
		break;
	case 4:
		if (row->ag != 0) break;
		row->ag= row->a;
		break;
	case 5:
		if (row->sa != 0) break;
		row->sa= row->s;
		break;
	case 6:
		if (row->sc != 0) break;
		if (row->s == ' ' || row->s == '@')
			row->sc= 256;
		else
		if (row->s < ' ' || row->s == 0x7F)
			row->sc= row->s;
		else
		if (row->s == '?')
			row->sc= 0x7F;
		else
		if ('A' <= row->s && row->s <= '_')
			row->sc= row->s & 0x1F;
		else
			row->sc= row->c;
		break;
	case 7:
		if (row->ca != 0) break;
		row->ca= row->ag;
	}
}

static unsigned char *get_token(FILE *file, unsigned char *token)
{
	unsigned char *t;

	t= maybe_token(file, token);
	if (t == NULL)
		fatal("unexpected end of input");
	return t;
}

static unsigned char *
maybe_token(FILE *file, unsigned char *token)
{
	unsigned char *t;

	if (token && *token != '\0')
		return token;

	for (;;)
	{
		if (lineptr == NULL || *lineptr == '\0')
		{
			/* Read a new line */
			lineptr= (unsigned char *)fgets((char *)line,
				sizeof(line), file);
			if (lineptr == NULL)
			{
				/* EOF or error */
				if (feof(file))
					return NULL;
				if (ferror(file))
					fatal("read error");
				/* Strange, continue */
				continue;
			}
			if (strlen((char *)lineptr) == sizeof(line)-1)
				fatal("line too long");
			lineno++;
		}

		/* Skip white space and comments */
		while (isspace(*lineptr))
			lineptr++;
		if (*lineptr == '#')
		{
			lineptr= NULL;
			continue;
		}
		if (*lineptr == '\0')
			continue;

		t= lineptr;

		/* Find the end of the token */
		while (*lineptr != '\0' && !isspace(*lineptr))
		{
			if (*lineptr == '"')
			{
				lineptr++;
				for(;;)
				{
					if (*lineptr == '\0')
					{
						fatal(
						"unexpected end of token");
					}
					if (*lineptr == '"')
						break;
					if (*lineptr == '\\')
					{
						lineptr++;
						if (*lineptr == '\0')
						{
							fatal(
						"unexpected end of token");
						}
					}
					lineptr++;
				}
			}
			lineptr++;
		}
		if (*lineptr != '\0')
		{
			*lineptr= '\0';
			lineptr++;
		}

		return t;
	}
}

static char *
getname(unsigned char *token, unsigned char **next)
{
	char *name, *s;

	dprintf("getname: token= '%s'\n", token);
	getstr(token, next);

	name= strdup((char *)token);
	if (name == NULL)
		fatal("out of memory");
	return name;
}

static void
getstr(unsigned char *token, unsigned char **next)
{
	char *s;
	unsigned char *n;
	long v;

	s= (char *)token;
	if (*token != '"')
		fatal("'\"' expected");
	token++;

	/* Remove '\' characters in place. */
	while (*token != '"')
	{
		if (*token == '\0')
			fatal("unexpected end of token");
		if (*token == '\\')
		{
			token++;
			if (*token == 'x')
			{
				v= strtol((char *)token+1, (char **)&n, 16);
				if (n == token+1)
					fatal("unexpected \\x construction");
				if (v >= 256)
					fatal("character value out of range");
				token= n-1;
				*token= v;
			}
			else if (*token == '\0')
				fatal("unexpected end of token");
		}
		*s++= *token;
		token++;
	}
	token++;
	*s= '\0';
	*next= token;
}

static int
getflag(unsigned char *token, unsigned char **next)
{
	int i;
	size_t len;

	for (i= 0; flagsmap[i].name != NULL; i++)
	{
		len= strlen(flagsmap[i].name);
		if (strncasecmp(flagsmap[i].name, (char *)token, len) == 0)
		{
			*next= token+len;
			return flagsmap[i].value;
		}
	}
	fatal("no flag found in token '%s'", token);
}

static int
getvalue(unsigned char *token, unsigned char **next)
{
	unsigned char *vstr;
	int i, v;
	size_t len;

	/* A value is a special key name, a string, a number or a
	 * control character
	 */
	for (i= 0; specialmap[i].name != NULL; i++)
	{
		len= strlen(specialmap[i].name);
		if (strncasecmp(specialmap[i].name, (char *)token, len) == 0)
		{
			*next= token+len;
			return KM_SPECIAL | specialmap[i].value;
		}
	}

	if (*token == '"')
	{
		getstr(token, next);
		vstr= token;
		if (strlen((char *)vstr) <= 1)
		{
			/* Only a single character */
			return *vstr;
		}
		return strtab(vstr);
	}
	if (*token == '^')
	{
		token++;
		if (*token == '"')
		{
			getstr(token, next);
			vstr= token;
			return KM_HOTKEY | strtab(vstr);
		}
		if (*token == '\0')
			fatal("unexpected end of token");
		v= *token;
		token++;
		*next= token;

		if (v == '?')
			v= 127;
		else
			v &= 0x1F;
		if (v == 0)
			v= 256;
		return v;
	}
	if (*token == '-' && token[1] == 0)
	{
		return 0;	/* explicitly undefined */
	}
	else
	{
		v= strtol((char *)token, (char **)next, 0);
		if (*next == token)
			fatal("unable to parse token '%s'\n", token);
		if (v <= 0 || v >= KM_STRING)
			fatal("invalid value: %ld\n", v);
		return v;
	}
}

static void
read_oldmap(FILE *file)
{
	char magic[4];
	u16_t row[6];
	int i, j, kn, r;
	long v;
	u16_t vcum;
	struct keymap kme;

	/* Check the magic number */
	if (fread(magic, sizeof(magic), 1, file) != 1)
	{
		fprintf(stderr, "%s: unable to read magic number\n",
			prog_name);
		exit(1);
	}
	if (strncmp(magic, "KMAZ", sizeof(magic)) != 0)
	{
		fprintf(stderr, "%s: wrong magic number\n", prog_name);
		exit(1);
	}
	for (i= 0;; i++)
	{
		vcum= 0;
		for (j= 0; j<6; j++)
		{
			v= read_value(file);
			if (v == -1)
			{
				if (j == 0)
					break;
				row[j]= 0;
				continue;
			}
			row[j]= v;
			vcum |= v;
		}
		if (j == 0)
			break;
		if (vcum == 0)
			continue;
		if (i >= sizeof(set1tokeyno))
		{
			fprintf(stderr, "%s: row %d out of range\n",
				prog_name, i);
			error= 1;
			continue;
		}
		kn= set1tokeyno[i];
		if (kn == 0)
		{
			fprintf(stderr, "%s: no mapping from key %d\n",
				prog_name, i);
			error= 1;
			continue;
		}
		if (kn == 255)
		{
			continue;	/* to be ignored */
		}
		assert(kn > 0 && kn < KEYMAP_ENTRIES);
		assert(!km_used[kn]);
		kme.l= NULL;	/* No key label */
		kme.fl= 0;	/* Initially no flags. */
		cnv_value(row[0], &kme.n, &kme);
		cnv_value(row[1], &kme.s, &kme);
		cnv_value(row[2], &kme.a, &kme);
		cnv_value(row[3], &kme.ag, &kme);
		cnv_value(row[4], &kme.sa, &kme);
		cnv_value(row[5], &kme.c, &kme);
		kme.sc= 0;
		kme.ca= kme.ag;
		km[kn]= kme;
		km_used[kn]= 1;
	}
}


static long
read_value(FILE *file)
{
	static int n= 8;
	static u8_t fb;

	int r;
	u8_t bl, bh;

	if (n == 8)
	{
		n= 0;
		r= fread(&fb, 1, 1, file);
		if (r != 1)
		{
			assert(r == 0);
			if (!feof(file))
			{
				fprintf(stderr, "%s: read error\n",
					prog_name);
				exit(1);
			}
			fb= 0;
		}
	}
	r= fread(&bl, 1, 1, file);
	if (r != 1)
	{
		assert(r == 0);
		if (!feof(file))
		{
			fprintf(stderr, "%s: read error\n", prog_name);
			exit(1);
		}
		n++;
		return -1;
	}
	if (fb & (1<<n))
	{
		r= fread(&bh, 1, 1, file);
		if (r != 1)
		{
			assert(r == 0);
			if (!feof(file))
			{
				fprintf(stderr, "%s: read error\n", prog_name);
				exit(1);
			}
			bh= 0;
		}
	}
	else
		bh= 0;
	n++;
	return (((u16_t)bh) << 8) | bl;
}


static u16_t
cnv_value(u16_t value, u16_t *res, struct keymap *kme)
{
	u16_t v;
	u16_t special;
	char *str;

	v= 0;
	*res= 0;
	switch(value)
	{
	case 0x10D:	/* Caps lock */
		special= KM_CAPSLOCK;
		break;
	case 0x10E:	/* Num lock */
		special= KM_NUMLOCK;
		break;
	case 0x10F:	/* Num lock */
		special= KM_SCRLOCK;
		break;
	case 0x200:	/* Control key */
		special= KM_CTRL_L;
		break;
	case 0x400:	/* Control key */
		special= KM_SHIFT_L;
		break;
	case 0x800:	/* Control key */
		special= KM_ALT_L;
		break;
	default:
		special= 0;
		break;
	}
	if (special)
	{
		if (res == &kme->n)
		{
			/* First column, set KM_SPECIAL */
			assert(!(kme->fl & KM_SPECIAL));
			kme->fl |= KM_SPECIAL;
			*res= special;
			return;
		}
		/* See if KM_SPECIAL is set and n == special */
		if ((kme->fl & KM_SPECIAL) && kme->n == special)
			return;
		error= 1;
		fprintf(stderr, "%s: ignoring special key\n", prog_name);
		return;
	}
	switch(value)
	{
	case 0x101:	str= "\x1b[H";	break;	/* HOME */
	case 0x102:	str= "\x1b[Y";	break;	/* END */
	case 0x103:	str= "\x1b[A";	break;	/* UP */
	case 0x104:	str= "\x1b[B";	break;	/* DOWN */
	case 0x105:	str= "\x1b[D";	break;	/* LEFT */
	case 0x106:	str= "\x1b[C";	break;	/* RIGHT */
	case 0x107:	str= "\x1b[V";	break;	/* PGUP */
	case 0x108:	str= "\x1b[U";	break;	/* PGDN */
	case 0x109:	str= "\x1b[G";	break;	/* MID */
	case 0x10A:	str= "\x1b[S";	break;	/* MIN */
	case 0x10B:	str= "\x1b[T";	break;	/* PLUS */
	case 0x10C:	str= "\x1b[@";	break;	/* INS */
	case 0x110:	str= "F1";	break;	/* F1 */
	case 0x111:	str= "F2";	break;	/* F2 */
	case 0x112:	str= "F3";	break;	/* F3 */
	case 0x113:	str= "F4";	break;	/* F4 */
	case 0x114:	str= "F5";	break;	/* F5 */
	case 0x115:	str= "F6";	break;	/* F6 */
	case 0x116:	str= "F7";	break;	/* F7 */
	case 0x117:	str= "F8";	break;	/* F8 */
	case 0x118:	str= "F9";	break;	/* F9 */
	case 0x119:	str= "F10";	break;	/* F10 */
	case 0x11A:	str= "F11";	break;	/* F11 */
	case 0x11B:	str= "F12";	break;	/* F12 */

	case 0x201:	str= "C-HOME";	break;	/* C-HOME */
	case 0x202:	str= "C-END";	break;	/* C-END */
	case 0x203:	str= "C-UP";	break;	/* C-UP */
	case 0x204:	str= "C-DOWN";	break;	/* C-DOWN */
	case 0x205:	str= "C-LEFT";	break;	/* C-LEFT */
	case 0x206:	str= "C-RIGHT";	break;	/* C-RIGHT */
	case 0x207:	str= "C-PGUP";	break;	/* C-PGUP */
	case 0x208:	str= "C-PGDN";	break;	/* C-PGDN */
	case 0x209:	str= "C-MID";	break;	/* C-MID */
	case 0x20A:	str= "C-MIN";	break;	/* C-MIN */
	case 0x20B:	str= "C-PLUS";	break;	/* C-PLUS */
	case 0x20C:	str= "C-INS";	break;	/* C-INS */
	case 0x210:	str= "C-F1";	break;	/* C-F1 */
	case 0x211:	str= "C-F2";	break;	/* C-F2 */
	case 0x212:	str= "C-F3";	break;	/* C-F3 */
	case 0x213:	str= "C-F4";	break;	/* C-F4 */
	case 0x214:	str= "C-F5";	break;	/* C-F5 */
	case 0x215:	str= "C-F6";	break;	/* C-F6 */
	case 0x216:	str= "C-F7";	break;	/* C-F7 */
	case 0x217:	str= "C-F8";	break;	/* C-F8 */
	case 0x218:	str= "C-F9";	break;	/* C-F9 */
	case 0x219:	str= "C-F10";	break;	/* C-F10 */
	case 0x21A:	str= "C-F11";	break;	/* C-F11 */
	case 0x21B:	str= "C-F12";	break;	/* C-F12 */

	case 0x410:	str= "S-F1";	break;	/* S-F1 */
	case 0x411:	str= "S-F2";	break;	/* S-F2 */
	case 0x412:	str= "S-F3";	break;	/* S-F3 */
	case 0x413:	str= "S-F4";	break;	/* S-F4 */
	case 0x414:	str= "S-F5";	break;	/* S-F5 */
	case 0x415:	str= "S-F6";	break;	/* S-F6 */
	case 0x416:	str= "S-F7";	break;	/* S-F7 */
	case 0x417:	str= "S-F8";	break;	/* S-F8 */
	case 0x418:	str= "S-F9";	break;	/* S-F9 */
	case 0x419:	str= "S-F10";	break;	/* S-F10 */
	case 0x41A:	str= "S-F11";	break;	/* S-F11 */
	case 0x41B:	str= "S-F12";	break;	/* S-F12 */

	case 0x801:	str= "A-HOME";	break;	/* A-HOME */
	case 0x802:	str= "A-END";	break;	/* A-END */
	case 0x803:	str= "A-UP";	break;	/* A-UP */
	case 0x804:	str= "A-DOWN";	break;	/* A-DOWN */
	case 0x805:	str= "A-LEFT";	break;	/* A-LEFT */
	case 0x806:	str= "A-RIGHT";	break;	/* A-RIGHT */
	case 0x807:	str= "A-PGUP";	break;	/* A-PGUP */
	case 0x808:	str= "A-PGDN";	break;	/* A-PGDN */
	case 0x809:	str= "A-MID";	break;	/* A-MID */
	case 0x80A:	str= "A-MIN";	break;	/* A-MIN */
	case 0x80B:	str= "A-PLUS";	break;	/* A-PLUS */
	case 0x80C:	str= "A-INS";	break;	/* A-INS */
	case 0x810:	str= "A-F1";	break;	/* A-F1 */
	case 0x811:	str= "A-F2";	break;	/* A-F2 */
	case 0x812:	str= "A-F3";	break;	/* A-F3 */
	case 0x813:	str= "A-F4";	break;	/* A-F4 */
	case 0x814:	str= "A-F5";	break;	/* A-F5 */
	case 0x815:	str= "A-F6";	break;	/* A-F6 */
	case 0x816:	str= "A-F7";	break;	/* A-F7 */
	case 0x817:	str= "A-F8";	break;	/* A-F8 */
	case 0x818:	str= "A-F9";	break;	/* A-F9 */
	case 0x819:	str= "A-F10";	break;	/* A-F10 */
	case 0x81A:	str= "A-F11";	break;	/* A-F11 */
	case 0x81B:	str= "A-F12";	break;	/* A-F12 */

	case 0xC10:	str= "AS-F1";	break;	/* AS-F1 */
	case 0xC11:	str= "AS-F2";	break;	/* AS-F2 */
	case 0xC12:	str= "AS-F3";	break;	/* AS-F3 */
	case 0xC13:	str= "AS-F4";	break;	/* AS-F4 */
	case 0xC14:	str= "AS-F5";	break;	/* AS-F5 */
	case 0xC15:	str= "AS-F6";	break;	/* AS-F6 */
	case 0xC16:	str= "AS-F7";	break;	/* AS-F7 */
	case 0xC17:	str= "AS-F8";	break;	/* AS-F8 */
	case 0xC18:	str= "AS-F9";	break;	/* AS-F9 */
	case 0xC19:	str= "AS-F10";	break;	/* AS-F10 */
	case 0xC1A:	str= "AS-F11";	break;	/* AS-F11 */
	case 0xC1B:	str= "AS-F12";	break;	/* AS-F12 */

	default:	str= NULL;	break;
	}
	if (str)
	{
		*res= strtab((unsigned char *)str);
		return;
	}

	if (value & 0x8000)
	{
		/* Has CAPS */
		value &= ~0x8000;
		v |= KM_CAPSLOCK;
	}
	if (value < 256)
	{
		if (value == 0)
			value= 256;
		if (kme->fl & KM_SPECIAL)
		{
			fprintf(stderr, "%s: ignoring key, KM_SPECIAL is set\n",
				prog_name);
			error= 1;
			return;
		}
		*res= v | value;
		return;
	}
	fprintf(stderr, "%s: don't know how to convert value 0x%x\n",
		prog_name, value);
	error= 1;
}


static int
strtab(unsigned char *str)
{
	int inc, v, len;

	if (stralign && !str_index)
		str_index= KM_STRLEN;

	len= strlen((char *)str)+1;
	inc= 1;
	if (stralign)
	{
		inc= KM_STRLEN;
		len= (len+KM_STRLEN-1)/KM_STRLEN*KM_STRLEN;
	}

	/* Try to merge strings. */
	for (v= 0; v+len <= str_index; v += inc)
	{
		if (strcmp((char *)str, (char *)str_table+v) == 0)
			return KM_STRING | v;
	}

	v= str_index;

	str_index += len;
	if (str_index > STRTAB_SIZE)
		fatal("string table full");
	strcpy((char *)str_table+v, (char *)str);
	return KM_STRING | v;
}

static int
strtab_label(char *str)
{
	int v, len;
	
	assert(!stralign);
	if (!str_index)
		str_index= 1;

	len= strlen(str)+1;

	/* Try to merge strings. */
	for (v= 1; v+len <= str_index; v += 1)
	{
		if (strcmp(str, (char *)str_table+v) == 0)
			return v;
	}

	v= str_index;

	str_index += len;
	if (str_index > STRTAB_SIZE)
		return 0;
	strcpy((char *)str_table+v, str);
	return v;
}

static void
print_header(void)
{
	int i, v, len, high;

	/* Do boilerplate */
	printf(
"/* Generated keymap */\n"
"\n"
"#include <sys/types.h>\n"
"#include <minix/keymap.h>\n"
"\n"
"struct keymap keymap[KEYMAP_ENTRIES]=\n"
"{\n"	/* } */
"\n"
"#define c	0x1f\n"
"#define C	KM_CAPSLOCK\n"
"#define N	KM_NUMLOCK\n"
"#define H	KM_HOTKEY\n"
"#define S	KM_SPECIAL\n"
"#define F(x)	(KM_STRING | ((x)*KM_STRLEN))\n"
"\n"
"/*\n"
"code	label	  flags    normal shifted  control  alt    alt-gr  sh-alt  sh-ctrl ctrl-alt\n"
" */\n"
	);

	high= 0;
	for (i= 0; i<KEYMAP_ENTRIES; i++)
	{
		if (km_used[i])
			high= i;
	}
	for (i= 0; i<=high; i++)
		print_hdrline(i, &km[i]);

	printf(
/* { */ "};\n"
"\n"
"u8_t keymap_str[KEYMAP_STRINGS][KM_STRLEN]=\n"
"{\n" /* } */
	);
	for (v= STRTAB_SIZE - KM_STRLEN; v > 0; v -= KM_STRLEN)
	{
		if (strlen((char *)str_table+v) != 0)
			break;
	}
	high= v;
	for (v= 0; v<= high; v += KM_STRLEN)
	{
		len= strlen((char *)str_table+v);
		if (len > KM_STRLEN)
			fatal("not implemented");
		printf("/* %3d */\t", v / KM_STRLEN);
		print_hdrstr(str_table+v, -1, 1);
		printf("\n");
	}
/* { */	printf("};\n");
}

static void
print_hdrline(int i, struct keymap *kmp)
{
	int col;

	/* print only necessary columns */
	col= 0;
	if (kmp->n) col= 1;
	if (kmp->s) col= 2;
	if (kmp->c) col= 3;
	if (kmp->a) col= 4;
	if (kmp->ag) col= 5;
	if (kmp->sa) col= 6;
	if (kmp->sc) col= 7;
	if (kmp->ca) col= 8;

	printf("/*%03d*/{ ", i);

	if (col == 0 && kmp->fl == 0)
	{
		print_hdrstr((unsigned char *)kmp->l, -1, 0);
		printf(" },\n");
		return;
	}
	print_hdrstr((unsigned char *)kmp->l, -1, 11);
	print_flags(kmp->fl, 8);
	if (kmp->fl & KM_SPECIAL)
		print_hdrspecial(kmp->n, 8);
	else
	{
		if (col >= 1) print_hdrvalue(kmp->n, 8);
	}
	if (col >= 2) print_hdrvalue(kmp->s, 8);
	if (col >= 3) print_hdrvalue(kmp->c, 8);
	if (col >= 4) print_hdrvalue(kmp->a, 8);
	if (col >= 5) print_hdrvalue(kmp->ag, 8);
	if (col >= 6) print_hdrvalue(kmp->sa, 8);
	if (col >= 7) print_hdrvalue(kmp->sc, 8);
	if (col >= 8) print_hdrvalue(kmp->ca, 8);
/*{*/	printf("},\n");
}

static void
print_hdrstr(unsigned char *str, int maxlen, int width)
{
	int cnt, len, i, c;

	cnt= 0;

	if (str == NULL)
	{
		printf("0,");
		cnt= 2;
	}
	else
	{
		len= strlen((char *)str);
		if (maxlen != -1 && len > maxlen)
			len= maxlen;

		printf("\"");
		cnt= 1;
		for (i= 0; i<len; i++)
		{
			c= str[i];
			if (c == '\\')
			{
				printf("\\%c", c);
				cnt += 2;
				continue;
			}
			if (c >= ' ' && c <= '~')
			{
				putchar(c);
				cnt++;
				continue;
			}
			printf("\\x%02x", c);
			cnt += 4;
			if (isxdigit(str[i+1]))
			{
				printf("\"\"");
				cnt += 2;
			}
		}
		printf("\",");
		cnt += 2;
	}
	for (; cnt<width; cnt++)
		putchar(' ');
}

static void
print_flags(u16_t flags, int width)
{
	int cnt;

	if (flags == 0)
	{
		printf("0");
		cnt= 1;
	}
	else
	{
		if (flags & KM_SPECIAL)
		{
			printf("S");
			flags &= ~KM_SPECIAL;
			cnt= 1;
		}
		if (flags)
			fatal("invalid flags: 0x%x\n", flags);
	}
	putchar(',');
	cnt++;
	for (; cnt<width; cnt++)
		putchar(' ');
}

static void
print_hdrvalue(u16_t v, int width)
{
	u16_t v1;
	int cnt;
	char str[20];

	v1= v & (KM_STRING-1);
	if (v & KM_STRING)
	{
		if (v1 % KM_STRLEN)
			fatal("string not aligned: %d", v1);
		sprintf(str, "F(%d)", v1 / KM_STRLEN);
		printf("%s", str);
		cnt= strlen(str);
		v1 |= KM_STRING;
	}
	else if (v1 == 0)
	{
		printf("0");
		cnt= 1;
	}
	else if (v1 == '\\' - 64)
	{
		printf("'\\\\'&c");
		cnt= 6;
	}
	else if (v1<32)
	{
		printf("'%c'&c", v1+64);
		cnt= 5;
	}
	else if (v1 == '\\' || v1 == '\'')
	{
		printf("'\\%c'", v1);
		cnt= 4;
	}
	else if (v1<127)
	{
		printf("'%c'", v1);
		cnt= 3;
	}
	else if (v1<=256)
	{
		printf("%d", v1);
		cnt= 3;
	}
	else
		fatal("value out of range: %d\n", v1);
	if (v & KM_CAPSLOCK)
	{
		printf("|C");
		cnt += 2;
		v1 |= KM_CAPSLOCK;
	}
	if (v & KM_NUMLOCK)
	{
		printf("|N");
		cnt += 2;
		v1 |= KM_NUMLOCK;
	}
	if (v & KM_HOTKEY)
	{
		printf("|H");
		cnt += 2;
		v1 |= KM_HOTKEY;
	}
	putchar(',');
	cnt++;
	if ((v ^ v1) != 0)
		fatal("print_hdrvalue: unrecognized flags: 0x%x", v ^ v1);
	for (; cnt<width; cnt++)
		putchar(' ');
}

static void
print_hdrspecial(u16_t v, int width)
{
	char *s;
	int cnt;

	switch(v)
	{
	case KM_SHIFT_L: s= "KM_SHIFT_L"; break;
	case KM_SHIFT_R: s= "KM_SHIFT_R"; break;
	case KM_CTRL_L: s= "KM_CTRL_L"; break;
	case KM_CTRL_R: s= "KM_CTRL_R"; break;
	case KM_ALT_L: s= "KM_ALT_L"; break;
	case KM_ALT_R: s= "KM_ALT_R"; break;
	case KM_ALT_GR: s= "KM_ALT_GR"; break;
	case KM_SYSREQ: s= "KM_SYSREQ"; break;
	case KM_SCRLOCK: s= "KM_SCRLOCK"; break;
	case KM_NUMLOCK: s= "KM_NUMLOCK"; break;
	case KM_CAPSLOCK: s= "KM_CAPSLOCK"; break;
	default:
		fatal("print_special: unknown code: %d\n", v);
	}
	printf("%s,", s);
	cnt= strlen(s)+1;
	for (; cnt<width; cnt++)
		putchar(' ');
}


static void
print_map(void)
{
	int i, high;

	/* Do boilerplate */

	high= 0;
	for (i= 0; i<KEYMAP_ENTRIES; i++)
	{
		if (km_used[i])
			high= i;
	}
	for (i= 0; i<=high; i++)
		print_mapline(i, &km[i]);
}


static void
print_mapline(int i, struct keymap *kmp)
{
	int col;

	/* print only necessary columns */
	col= 0;
	if (kmp->n) col= 1;
	if (kmp->s) col= 2;
	if (kmp->c) col= 3;
	if (kmp->a) col= 4;
	if (kmp->ag) col= 5;
	if (kmp->sa) col= 6;
	if (kmp->sc) col= 7;

	if (col == 0 && kmp->fl == 0 && kmp->l == NULL)
		return;

	printf("%3d=\t", i);

	if (col == 0 && kmp->fl == 0)
	{
		print_mapstr((unsigned char *)kmp->l);
		printf(" ;\n");
		return;
	}
	print_mapstr((unsigned char *)kmp->l);
	if (kmp->fl & KM_SPECIAL)
	{
		printf("\t\t");
		print_mapspecial(kmp->n);
		printf(";\n");
		return;
	}
	if (kmp->n & (KM_CAPSLOCK|KM_NUMLOCK))
	{
		if ((kmp->n & KM_CAPSLOCK) && (kmp->s & KM_CAPSLOCK))
		{
			printf(", Caps");
			kmp->n &= ~KM_CAPSLOCK;
			kmp->s &= ~KM_CAPSLOCK;
		}
		if ((kmp->n & KM_NUMLOCK) && (kmp->s & KM_NUMLOCK))
		{
			printf(", Num");
			kmp->n &= ~KM_NUMLOCK;
			kmp->s &= ~KM_NUMLOCK;
		}
	}
	else
		printf("\t");
	printf("\t");
	if (col >= 1) print_mapvalue(kmp->n);
	if (col >= 2) print_mapvalue(kmp->s);
	if (col >= 3) print_mapvalue(kmp->c);
	if (col >= 4) print_mapvalue(kmp->a);
	if (col >= 5) print_mapvalue(kmp->ag);
	if (col >= 6) print_mapvalue(kmp->sa);
	if (col >= 7) print_mapvalue(kmp->sc);
	printf(";\n");
}


static void
print_mapstr(unsigned char *str)
{
	int cnt, len, i, c;

	cnt= 0;

	len= strlen((char *)str);

	printf("\"");
	cnt= 1;
	for (i= 0; i<len; i++)
	{
		c= str[i];
		if (c == '\\')
		{
			putchar('\\');
			putchar(c);
			cnt += 2;
			continue;
		}
		if (c >= ' ' && c <= '~')
		{
			putchar(c);
			cnt++;
			continue;
		}
		printf("\\x%02x", c);
		cnt += 4;
		if (isxdigit(str[i+1]))
		{
			printf("\"\"");
			cnt += 2;
		}
	}
	printf("\"");
	cnt++;
}


static void
print_mapvalue(u16_t v)
{
	u16_t v1;
	int cnt;
	char str[20];

	v1= v & (KM_STRING-1);
	if (v & KM_STRING)
	{
		print_mapstr(str_table+v1);
		cnt= 0;
		v1 |= KM_STRING;
	}
	else if (v1 == 0)
	{
		printf("0");
		cnt= 1;
	}
	else if (v1<32)
	{
		printf("^%c", v1+64);
		cnt= 2;
	}
	else if (v1 == '\\' || v1 == '"')
	{
		printf("\"\\%c\"", v1);
		cnt= 4;
	}
	else if (v1<127)
	{
		printf("\"%c\"", v1);
		cnt= 3;
	}
	else if (v1 == 127)
	{
		printf("^?");
		cnt= 2;
	}
	else if (v1<256)
	{
		printf("%d", v1);
		cnt= 3;
	}
	else if (v1 == 256)
	{
		printf("^@");
		cnt= 2;
	}
	else
		fatal("value out of range: %d\n", v1);

	if ((v ^ v1) != 0)
		fatal("print_value: unrecognized flags: 0x%x", v ^ v1);

	printf("\t");
}


static void
print_mapspecial(u16_t v)
{
	char *s;
	int cnt;

	switch(v)
	{
	case KM_SHIFT_L: s= "Shift-L"; break;
	case KM_SHIFT_R: s= "Shift-R"; break;
	case KM_CTRL_L: s= "Ctrl-L"; break;
	case KM_CTRL_R: s= "Ctrl-R"; break;
	case KM_ALT_L: s= "Alt-L"; break;
	case KM_ALT_R: s= "Alt-R"; break;
	case KM_ALT_GR: s= "Alt-GR"; break;
	case KM_SYSREQ: s= "SysReq"; break;
	case KM_SCRLOCK: s= "ScrLock"; break;
	case KM_NUMLOCK: s= "NumLock"; break;
	case KM_CAPSLOCK: s= "CapsLock"; break;
	default:
		fatal("print_special: unknown code: %d\n", v);
	}
	printf("%s", s);
	cnt= strlen(s)+1;
}

static void
load_map(void)
{
	int fd, i, r, v;
	kio_map_t maphdr;

	/* Add key labels to the string table */
	for (i= 0; i<KEYMAP_ENTRIES; i++)
	{
		if (!km[i].l)
			continue;
		v= strtab_label(km[i].l);
		if (v == 0)
		{
			fprintf(stderr, "%s: warning no room for key labels\n",
				prog_name);
			break;	/* Table is full */
		}
		km[i].l= (char *)v;
	}
	if (i != KEYMAP_ENTRIES)
	{
		/* Clear label entries. */
		for (i= 0; i<KEYMAP_ENTRIES; i++)
			km[i].l= NULL;
	}

	/* Header structure */
	maphdr.km_entries= KEYMAP_ENTRIES;
	maphdr.km_keymap= km;
	maphdr.km_strsize= STRTAB_SIZE;
	maphdr.km_strtab= (char *)str_table;

	/* Open /dev/kbd */
	fd= open("/dev/kbd", O_RDWR);
	if (fd == -1)
	{
		fprintf(stderr, "%s: unable to open /dev/kbd: %s\n",
			prog_name, strerror(errno));
		error= 1;
		return;
	}

	r= ioctl(fd, KIOCSMAP, &maphdr);
	if (r == -1)
	{
		fprintf(stderr, "%s: unable to load keymap: %s\n",
			prog_name, strerror(errno));
		error= 1;
		error= 1;
	}
	close(fd);
}

static void
dprintf(char *fmt, ...)
{
	va_list ap;

	return;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}

static void
fatal(char *fmt, ...)
{
	va_list ap;

	fprintf(stderr, "%s, line %d: ", prog_name, lineno);

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fprintf(stderr, "\n");
	exit(1);
}

static void
usage(void)
{
	fprintf(stderr, "Usage: %s [-blm] [map-file]\n", prog_name);
	exit(1);
}

/*
 * $PchId: keymap.c,v 1.2 1996/03/12 21:57:51 philip Exp $
 */
